/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef mrpt_vision_utils_H
#define mrpt_vision_utils_H

#include <mrpt/vision/CFeature.h>
#include <mrpt/utils/CMRPTImage.h>
#include <mrpt/math/utils.h>
#include <mrpt/utils/CLoadableOptions.h>
#include <mrpt/slam/CMetricMap.h>

namespace mrpt
{
	namespace slam
	{
		class CLandmarksMap;
		class CObservationVisualLandmarks;
	}
	namespace poses
	{
		class CPoint3D;
	}

	/** Classes for computer vision, detectors, features, etc.
	 */
	namespace vision
	{
		using namespace mrpt::math;

		// Here follow utility declarations, methods, etc. (Old "VisionUtils" namespace).
		//  Implementations are in "vision.cpp"

		/** Landmark ID
		  */
		typedef	 uint64_t TLandmarkID;

		/** Parameters associated to a stereo system
		  */
		struct MRPTDLLIMPEXP TStereoSystemParams : public mrpt::utils::CLoadableOptions
		{
			/** Initilization of default parameters
			 */
			TStereoSystemParams(	);

			/** See utils::CLoadableOptions
			  */
			void  loadFromConfigFile(
				const mrpt::utils::CConfigFileBase	&source,
				const std::string		&section);

			/** See utils::CLoadableOptions
			  */
			void  dumpToTextStream(utils::CStream		&out);

			/** Method for propagating the feature's image coordinate uncertainty into 3D space. Default value: Prop_Linear
			  */
			enum TUnc_Prop_Method
			{
				/** Linear propagation of the uncertainty
				  */
				Prop_Linear = -1,
				/** Uncertainty propagation through the Unscented Transformation
				  */
				Prop_UT,
				/** Uncertainty propagation through the Scaled Unscented Transformation
				  */
				Prop_SUT
			};

			TUnc_Prop_Method uncPropagation;

			/** Intrinsic parameters
			  */
			CMatrix		K;
			/** Baseline. Default value: baseline = 0.119f;	[Bumblebee]
			  */
			float		baseline;
			/** Standard deviation of the error in feature detection. Default value: stdPixel = 1
			  */
			float		stdPixel;
			/** Standard deviation of the error in disparity computation. Default value: stdDisp = 1
			  */
			float		stdDisp;
			/** Maximum allowed distance. Default value: maxZ = 20.0f
			  */
			float		maxZ;
			/** Maximum allowed distance. Default value: minZ = 0.5f
			  */
			float		minZ;
			/** Maximum allowed height. Default value: maxY = 3.0f
			  */
			float		maxY;
			/** K factor for the UT. Default value: k = 1.5f
			  */
			float		factor_k;
			/** Alpha factor for SUT. Default value: a = 1e-3
			  */
			float		factor_a;
			/** Beta factor for the SUT. Default value: b = 2.0f
			  */
			float		factor_b;

			/** Parameters initialization
			  */
			//TStereoSystemParams();

		}; // End struct TStereoSystemParams

		/** A structure for storing a 3D ROI
		  */
		struct MRPTDLLIMPEXP TROI
		{
			// Constructors
			TROI();
			TROI(float x1, float x2, float y1, float y2, float z1, float z2);

			// Members
			float	xMin;
			float	xMax;
			float	yMin;
			float	yMax;
			float	zMin;
			float	zMax;
		}; // end struct TROI

		/** A structure for defining a ROI within an image
		  */
		struct MRPTDLLIMPEXP TImageROI
		{
			// Constructors
			TImageROI();
			TImageROI( float x1, float x2, float y1, float y2 );

			// Members
			/** X coordinate limits [0,imageWidth)
			  */
			float	xMin, xMax;
			/** Y coordinate limits [0,imageHeight)
			  */
			float	yMin, yMax;
		}; // end struct TImageROI

		/** A structure containing options for the matching
		  */
		struct MRPTDLLIMPEXP TMatchingOptions : public mrpt::utils::CLoadableOptions
		{

			/** Method for propagating the feature's image coordinate uncertainty into 3D space. Default value: Prop_Linear
			  */
			enum TMatchingMethod
			{
				/** Matching by cross correlation of the image patches
				  */
				mmCorrelation = 0,
				/** Matching by Euclidean distance between SIFT descriptors
				  */
				mmDescriptorSIFT,
				/** Matching by Euclidean distance between SURF descriptors
				  */
				mmDescriptorSURF
			};

			// General
			TMatchingMethod	matching_method;
			float			epipolar_TH;
			float			rowCheck_TH;
			float			ccCheck_TH;

			// SIFT
			float			maxEDD_TH;
			float			EDD_RATIO;

			// KLT
			float			minCC_TH;
			float			minDCC_TH;
			float			rCC_TH;

			// SURF

			/** Constructor
			  */
			TMatchingOptions( );

			/** See utils::CLoadableOptions
			  */
			void  loadFromConfigFile(
				const mrpt::utils::CConfigFileBase	&source,
				const std::string		&section);

			/** See utils::CLoadableOptions
			  */
			void  dumpToTextStream(utils::CStream		&out);

		}; // end struct TMatchingOptions

			/**	Computes the correlation between this image and another one, encapsulating the openCV function cvMatchTemplate
			*   This implementation reduced computation time.
			* \param patch_img The "patch" image, which must be equal, or smaller than "this" image. This function supports gray-scale (1 channel only) images.
			* \param x_search_ini The "x" coordinate of the search window.
			* \param y_search_ini The "y" coordinate of the search window.
			* \param x_search_size The width of the search window.
			* \param y_search_size The height of the search window.
			* \param x_max The x coordinate where found the maximun cross correlation value.
			* \param y_max The y coordinate where found the maximun cross correlation value
			* \param max_val The maximun value of cross correlation which we can find
			*  Note: By default, the search area is the whole (this) image.
			* \sa cross_correlation
			*/
			void MRPTDLLIMPEXP openCV_cross_correlation(
													const CMRPTImage	&img,
													const CMRPTImage	&patch_img,
													size_t				&x_max,
													size_t				&y_max,
													double				&max_val,
													int					x_search_ini=-1,
													int					y_search_ini=-1,
													int					x_search_size=-1,
													int					y_search_size=-1);

			/**	Invert an image using OpenCV function
			*
			*/
			void MRPTDLLIMPEXP flip(CMRPTImage		&img);

			/** Extract a UNITARY 3D vector in the direction of a 3D point, given from its (x,y) pixels coordinates, and the camera intrinsic coordinates.
			  *  \param x Pixels coordinates, from the top-left corner of the image.
			  *  \param y Pixels coordinates, from the top-left corner of the image.
			  *  \param A The 3x3 intrinsic parameters matrix for the camera.
			  *
			  * \sa buildIntrinsicParamsMatrix, defaultIntrinsicParamsMatrix
			  */
			poses::CPoint3D MRPTDLLIMPEXP pixelTo3D(float x,float y, const CMatrix &A);

			/** Builds the intrinsic parameters matrix A from parameters:
			  * \param focalLengthX The focal length, in X (horizontal) pixels
			  * \param focalLengthY The focal length, in Y (vertical) pixels
			  * \param centerX The image center, horizontal, in pixels
			  * \param centerY The image center, vertical, in pixels
			  *
			  * <br>This method returns the matrix:
			  <table>
			  <tr><td>f_x</td><td>0</td><td>cX</td> </tr>
			  <tr><td>0</td><td>f_y</td><td>cY</td> </tr>
			  <tr><td>0</td><td>0</td><td>1</td> </tr>
			  </table>
			  *  See also the tutorial discussing the <a rhref="http://babel.isa.uma.es/mrpt/index.php/Camera_Parameters">camera model parameters</a>.
			  * \sa defaultIntrinsicParamsMatrix, pixelTo3D
			  */
			CMatrix MRPTDLLIMPEXP buildIntrinsicParamsMatrix(
							float	focalLengthX,
							float	focalLengthY,
							float	centerX,
							float	centerY);

			/** Returns the stored, default intrinsic params matrix for a given camera:
			  * \param camIndex  Posible values are listed next.
			  * \param resolutionX The number of pixel columns
			  * \param resolutionY The number of pixel rows
			  *
			  * The matrix is generated for the indicated camera resolution configuration.
			  * The following table summarizes the current supported cameras and the values as
			  *  ratios of the corresponding horz. or vert. resolution:<br>

			  <center><table>
			  <tr>
			   <td><center><b>camIndex</b></center></td>
			   <td><center><b>Manufacturer</b></center></td>
			   <td><center><b>Camera model</b></center></td>
			   <td><center><b>fx</b></center></td>
			   <td><center><b>fy</b></center></td>
			   <td><center><b>cx</b></center></td>
			   <td><center><b>cy</b></center></td>
			  </tr>

			  <tr>
			   <td><center>0</center></td>
			   <td><center>Point Grey Research</center></td>
			   <td><center>Bumblebee</center></td>
			   <td><center>0.79345</center></td>
			   <td><center>1.05793</center></td>
			   <td><center>0.55662</center></td>
			   <td><center>0.52692</center></td>
			  </tr>

			  <tr>
			   <td><center>1</center></td>
			   <td><center>Sony</center></td>
			   <td><center>???</center></td>
			   <td><center>0.95666094</center></td>
			   <td><center>1.3983423f</center></td>
			   <td><center>0.54626328f</center></td>
			   <td><center>0.4939191f</center></td>
			  </tr>
			  </table>
			  </center>

			  * \sa buildIntrinsicParamsMatrix, pixelTo3D
			  */
			CMatrix MRPTDLLIMPEXP defaultIntrinsicParamsMatrix(
							unsigned int camIndex = 0,
							unsigned int resolutionX = 320,
							unsigned int resolutionY = 240 );

			/** Explore the feature list and removes features which are in the same coordinates
			  * \param list (Input). The list of features.
			  */
			void MRPTDLLIMPEXP deleteRepeatedFeats( CFeatureList &list );

			/** Computes ONLY the SIFT descriptors of a set of features ...
			  * ...
			  */
			/** /
			void  computeSiftDescriptors( const CMRPTImage						&in_img,
										  CKanadeLucasTomasi::TKLTFeatureList	&in_features,
										  TSIFTFeatureList						&out_features);
			/ ** Computes a set of features with the SIFT algorithm...
			  * ...
			  * /
			void  computeSiftFeatures( const CMRPTImage		&in_img,
									   TSIFTFeatureList	&out_features,
									   int				wantedNumberOfFeatures = 150 );

			/ ** Search for correspondences which are not in the same row and deletes them
			  * ...
			  */
			void MRPTDLLIMPEXP rowChecking(	CFeatureList &leftList,
								CFeatureList &rightList,
								float threshold = 0.0);

			/** Search for correspondences which are not in the same row and deletes them
			  * ...
			  */
			void MRPTDLLIMPEXP checkTrackedFeatures( CFeatureList &leftList,
							    CFeatureList &rightList,
								vision::TMatchingOptions options);


			/** Computes the dispersion of the features in the image
			  * \param list (IN) Inpute list of features
			  * \param std	(OUT) 2 element vector containing the standard deviations in the 'x' and 'y' coordinates.
			  * \param mean	(OUT) 2 element vector containing the mean in the 'x' and 'y' coordinates.
			  */
			void MRPTDLLIMPEXP getDispersion( const CFeatureList &list,
											  vector_float &std,
											  vector_float &mean );

			/** Tracks a set of features in an image.
			  */
			void MRPTDLLIMPEXP trackFeatures( const CMRPTImage &inImg1,
								 const CMRPTImage &inImg2,
								 vision::CFeatureList &featureList,
								 const unsigned int &window_width = 7,
								 const unsigned int &window_height = 7);

			/** Filter bad correspondences by distance
			  * ...
			  */
			void MRPTDLLIMPEXP filterBadCorrsByDistance( mrpt::slam::CMetricMap::TMatchingPairList &list,	// The list of correspondences
											unsigned int numberOfSigmas );				// Threshold


			/** Returns a new image where distortion has been removed.
			  * \param A The 3x3 intrinsic parameters matrix
			  * \param dist_coeffs The 1x4 vector of distortion coefficients
			  */
			void MRPTDLLIMPEXP correctDistortion(
					CMRPTImage		&in_img,
					CMRPTImage		&out_img,
					CMatrix			A,
					CMatrix			dist_coeffs );

			/** Transform HSV color components to RGB, all of them in the range [0,1]
			  * \sa rgb2hsv
			  */
			void MRPTDLLIMPEXP hsv2rgb(
				float	h,
				float	s,
				float	v,
				float	&r,
				float	&g,
				float	&b);

			/** Transform RGB color components to HSV, all of them in the range [0,1]
			  * \sa hsv2rgb
			  */
			void MRPTDLLIMPEXP rgb2hsv(
				float	r,
				float	g,
				float	b,
				float	&h,
				float	&s,
				float	&v );

			/** Different colormaps
			  * \sa mrpt::vision::colormap
			  */
			enum TColormap
			{
				cmGRAYSCALE = 0,
				cmJET
			};

			/** Transform a float number in the range [0,1] into RGB components. Different colormaps are available.
			  */
			void MRPTDLLIMPEXP colormap(
				const TColormap &color_map,
				const float	&color_index,
				float	&r,
				float	&g,
				float	&b);

			/** Computes the RGB color components (range [0,1]) for the corresponding color index in the range [0,1] using the MATLAB 'jet' colormap.
			  * \sa colormap
			  */
			void MRPTDLLIMPEXP jet2rgb(
				const float	&color_index,
				float	&r,
				float	&g,
				float	&b);

			/** Interpolation methods (as defined in OpenCV)
			  *  Used for OpenCV related operations with images, but also with MRPT native classes.
			  * \sa mrpt::utils::CMappedImage
			  */
			enum TInterpolationMethod
			{
				imNEAREST = 0, //!< No interpolation
				imBILINEAR,		//!< Bilinear interpolation
				imBICUBIC,		//!< Bilinear interpolation
				imAREA			//!< Resampling using pixel-area relation
			};


			/** Computes the mean squared distance between a set of 3D correspondences
			  * ...
			  */
			double MRPTDLLIMPEXP computeMsd( const mrpt::slam::CMetricMap::TMatchingPairList &list,
								const mrpt::poses::CPose3D &Rt );

			/** Transform two clouds of 3D points into a matched list of points
			  * ...
			  */
			void MRPTDLLIMPEXP cloudsToMatchedList( const mrpt::slam::CObservationVisualLandmarks &cloud1,
									   const mrpt::slam::CObservationVisualLandmarks &cloud2,
											 mrpt::slam::CMetricMap::TMatchingPairList &outList);

			/** Computes the main orientation of a set of points with an image (for using in SIFT-based algorithms)
			  * \param image (Input). The input image.
			  * \param x (Input). A vector containing the 'x' coordinates of the image points.
			  * \param y (Input). A vector containing the 'y' coordinates of the image points.
			  * \param orientation (Output). A vector containing the main orientation of the image points.
			  */
			float MRPTDLLIMPEXP computeMainOrientation( const CMRPTImage &image,
										  const unsigned int &x,
										  const unsigned int &y );

			/** Find the matches between two lists of features. They must be of the same type. Return value: the number of matched pairs of features
			  * \param list1 (Input). One list.
			  * \param list2 (Input). Other list.
			  * \param matches (Output). A vector of pairs of correspondences.
			  * \param options (Optional Input). A struct containing matching options
			  */
			size_t MRPTDLLIMPEXP matchFeatures( const CFeatureList &list1,
								  const CFeatureList &list2,
								  CMatchedFeatureList &matches,
								  const TMatchingOptions &options = TMatchingOptions() );

			/** Project a list of matched features into the 3D space, using the provided options for the stereo system
			  * \param matches (Input). The list of matched features.
			  * \param options (Input). The options of the stereo system.
			  * \param landmarks (Output). A map containing the projected landmarks.
			  */
			void MRPTDLLIMPEXP projectMatchedFeatures(
				CMatchedFeatureList		&mfList,		// The set of matched features
				const vision::TStereoSystemParams	&param,			// Parameters for the stereo system
				mrpt::slam::CLandmarksMap				&landmarks );	// Output map of 3D landmarks

	}
}


#endif
