/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CVisualOdometryStereo_H
#define CVisualOdometryStereo_H

#include <mrpt/slam/CObservationStereoImages.h>
#include <mrpt/poses/CPose3DPDFGaussian.h>
#include <mrpt/poses/CPose3DPDFParticles.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/slam/CMetricMap.h>
#include <mrpt/slam/CObservationVisualLandmarks.h>
#include <mrpt/scan_matching.h>

#include <mrpt/vision/CFeatureExtraction.h>
#include <mrpt/vision/utils.h>

#include <mrpt/math/CMatrix.h>
#include <mrpt/utils/CMRPTImage.h>
#include <mrpt/gui/CDisplayWindow.h>

#define	VISUALODOMETRY_USES_GUI 1

namespace mrpt
{
	namespace vision
	{
		/** A structure containing options for the visual odometry
		  */
		struct MRPTDLLIMPEXP TOdometryOptions : public mrpt::utils::CLoadableOptions
		{
			enum TCov_Computation_Method
			{
				TCov_Linear = -1,			// Linear propagation of the uncertainty
				TCov_UT,					// Using the Unscented Transform (UT)
				TCov_Particles				// Using particles
			};

			// General
			TCov_Computation_Method		covMethod;
			unsigned int				minNumFeatures;
			double						minDispX, minDispY, minSetDist;
			bool						VERBOSE, PLOT_IMAGES, SAVE_FILES;

			TOdometryOptions( );

			/** See utils::CLoadableOptions
			  */
			void  loadFromConfigFile(
				const mrpt::utils::CConfigFileBase	&source,
				const std::string		&section);

			/** See utils::CLoadableOptions
			  */
			void  dumpToTextStream(mrpt::utils::CStream		&out);

		}; // end struct TOdometryOptions

		/** Visual odometry for stereo cameras.
		  *  The method is described in the paper:
		  *   - F.-A. Moreno, J.-L. Blanco, and J. Gonzalez, "An Efficient Closed-form Solution to Probabilistic 6D Visual Odometry for a Stereo Camera," in Proc. Advanced Concepts for Intelligent Vision Systems, 2007, pp. 932-942.
		  *
 		  *  \note This class is undergoing changes and it's in an experimental status.
 		  */
		class MRPTDLLIMPEXP CVisualOdometryStereo
		{

		public:

			/** Constructor
			  */
			CVisualOdometryStereo();

			/** Destructor
			  */
			virtual ~CVisualOdometryStereo();

			/** Resets the state of the Visual Odometry system.
			  */
			void  reset();

			/** Performs one step of the Visual Odometry system. It takes the input stereo observation and estimates the
			  *	camera movement between the previous observation and this one.
			  * If this is the first observation inserted into the system, this functions return FALSE.
			  * Otherwise it returns TRUE and yields a CPose3DPDFGaussian object containing the change in the 6D camera pose.
			  */
			bool  process( const mrpt::slam::CObservationStereoImages &inObs, mrpt::poses::CPose3DPDFGaussian &outEst );

			vision::TStereoSystemParams	stereoParams;
			vision::TMatchingOptions		matchingOptions;
			vision::TOdometryOptions		odometryOptions;

			unsigned int				nIter;

		protected:

			bool										firstObs;

			vision::CMatchedFeatureList					mfList;
			vision::CFeatureExtraction					fExtract;
			mrpt::slam::CObservationVisualLandmarks			Prev_cloud;
			utils::CMRPTImage							imgL, imgR;

			vision::CFeatureList							leftFeats, rightFeats;

#if VISUALODOMETRY_USES_GUI
			gui::CDisplayWindow						*wind1,*wind2;
#endif

			/** Compute the covariance of the change in pose using the method defined in the options
			  * (for internal use only)
			  */
			void  computeRTCovariance(
				const mrpt::poses::CPose3D &meanPose,
				const mrpt::slam::CMetricMap::TMatchingPairList &list,
				const mrpt::slam::CObservationVisualLandmarks &Pre_Cloud,
				const mrpt::slam::CObservationVisualLandmarks &Cur_Cloud,
				math::CMatrixD &RTCov );

			/** Fills in the KLT lists from the matched SIFTS
			  * (for internal use only)
			  */
			void  storeMatchedFeatureList(
				const CMatchedFeatureList &mfList );			// The list of matched features

			/** Adds the new set of matched points into the internal lists of points
			  * (for internal use only)
			  */
			void  addMatches(
				const CMatchedFeatureList &mfList );

			/** Construct a CMatchedFeatureList from the internal lists of features
			  * (for internal use only)
			  */
			void getMatchedList(
				CMatchedFeatureList &mfList );


		}; // end of class CVisualOdometryStereo

	} // end of namespace
} // end of namespace

#endif
