/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef UTILSDEFS_H
#define UTILSDEFS_H

/* ------------------------------------
          Disable some warnings
   ------------------------------------ */
#if defined(_MSC_VER)
	// (Compiler: Visual C++) Disable warning for too long debug names:
	#pragma warning(disable:4786)
	// (Compiler: Visual C++) Disable warning for unreachable code (I don't know why some of these errors appear in the STANDARD LIBRARY headers with Visual Studio 2003!):
	#pragma warning(disable:4702)
	// (Compiler: Visual C++) Conversion double->float
	#pragma warning (disable:4244)
	#pragma warning (disable:4305)
	#pragma warning (disable:4267)
	// Visual C++ does not implement decl. specifiers: throw(A,B,...)
	#pragma warning( disable : 4290 )
	// Visual C++ 2003+ warnings on STL classes when exporting to DLL...
	#pragma warning(disable: 4251)
	#pragma warning(disable: 4275)
#endif

#if defined(__BORLANDC__)
	// (Compiler: Borland C++) Disable a warning for inline functions
	#pragma warn -8027
        // (Compiler: Borland C++) Disable a warning for override of virtual functions, while in reality there are many virtual function with different parameters
	#pragma warn -8012
	#pragma warn -8022
#endif

// For the new more secure library in VC++8
#if defined(_MSC_VER) && (_MSC_VER>=1400)
	#if !defined(_CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES)
		#define _CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES 1
	#endif
#endif


#include <mrpt/config.h>
#include <mrpt/system/os.h>

// If we are building a DLL, do not link against the .lib files:
#if defined(MRPT_BUILT_AS_DLL) && defined(mrpt_core_EXPORTS)
#	define MRPT_SKIP_AUTOINCLUDE_LIBS
#endif

#define _IAMINUTILSDEFS_H
#include <mrpt/utils/utils_impexp.h>  // DLL import/export definitions
#undef _IAMINUTILSDEFS_H

/* ------------------------------------------------
    Automatically tell the linker to include
	 the correct MRPT ".lib" file into the
	 user's application.
   ----------------------------------------------- */
// Borland C++
#if (defined(_MSC_VER) || defined(__BORLANDC__)) && !defined(MRPT_SKIP_AUTOINCLUDE_LIBS)
#	if defined(_DEBUG)
#		pragma comment (lib,"libmrpt-core-dbg.lib")
#		pragma comment (lib,"libmrpt-ann-dbg.lib")
#		pragma comment (lib,"libmrpt-3ds-dbg.lib")
#		pragma comment (lib,"libmrpt-sift-hess-dbg.lib")
#		if !MRPT_HAS_WXWIDGETS && MRPT_HAS_ZLIB && !MRPT_HAS_ZLIB_SYSTEM
#			pragma comment (lib,"libmrpt-zlib-dbg.lib")
#		endif
#	else
#		pragma comment (lib,"libmrpt-core.lib")
#		pragma comment (lib,"libmrpt-ann.lib")
#		pragma comment (lib,"libmrpt-3ds.lib")
#		pragma comment (lib,"libmrpt-sift-hess.lib")
#		if !MRPT_HAS_WXWIDGETS && MRPT_HAS_ZLIB && !MRPT_HAS_ZLIB_SYSTEM
#			pragma comment (lib,"libmrpt-zlib.lib")
#		endif
#	endif
#endif


// A cross-compiler definition for "deprecated"-warnings
#if defined(__GNUC__) && (__GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 >= 2))
   /* gcc >= 3.2 */
#   define MRPT_DEPRECATED_PRE
#   define MRPT_DEPRECATED_POST __attribute__ ((deprecated))
# elif defined(_MSC_VER) && (_MSC_VER >= 1300)
  /* msvc >= 7 */
#   define MRPT_DEPRECATED_PRE  __declspec(deprecated)
#   define MRPT_DEPRECATED_POST
# else
#  define MRPT_DEPRECATED_PRE
#  define MRPT_DEPRECATED_POST
# endif


// Utils C++ headers:
#include <stdlib.h>
#include <cstdlib>
#include <cmath>

// C++ STD Library:
#include <algorithm>
#include <iostream>
#include <fstream>
#include <cstring>
#include <exception>
#include <stdexcept>
#include <limits>
#include <sstream>

// STL+ library:
#include <mrpt/otherlibs/stlplus/smart_ptr.hpp>

// Standard elemental types:
#include <mrpt/utils/types.h>


// Forward declarations:
namespace mrpt
{
	/** A std::string version of C sprintf.
	  *  You can call this to obtain a std::string using printf-like syntax.
	  *  Based on very nice code by Paul Senzee, published at http://senzee.blogspot.com/2006/05/c-formatting-stdstring.html
	  *  Function implemented in format.cpp
	  */
	std::string MRPTDLLIMPEXP format(const char *fmt, ...) MRPT_printf_format_check(1,2);

	namespace utils
	{
		class CFileStream;
	}
	namespace system
	{
		// Forward definition: (Required for Visual C++ 6 implementation of THROW_EXCEPTION...)
		std::string  MRPTDLLIMPEXP extractFileName(const std::string &filePath);
	}


	namespace math
	{
	    bool MRPTDLLIMPEXP isNan(float v);
	    bool MRPTDLLIMPEXP isNan(double v);
	    bool MRPTDLLIMPEXP isFinite(float v);
	    bool MRPTDLLIMPEXP isFinite(double v);
	}
} // End of namespace


/** A macro for obtaining the name of the current function:
  */
#if defined(__BORLANDC__)
		#define	__CURRENT_FUNCTION_NAME__	__FUNC__
#elif defined(_MSC_VER) && (_MSC_VER>=1300)
		#define	__CURRENT_FUNCTION_NAME__	__FUNCTION__
#elif defined(_MSC_VER) && (_MSC_VER<1300)
		// Visual C++ 6 HAS NOT A __FUNCTION__ equivalent.
#define	__CURRENT_FUNCTION_NAME__	::system::extractFileName(__FILE__).c_str()
#else
		#define	__CURRENT_FUNCTION_NAME__	__PRETTY_FUNCTION__
#endif


/** \def THROW_EXCEPTION(msg)
 * \param msg This can be a char*, a std::string, or a literal string.
 * Defines a unified way of reporting exceptions
 * \sa MRPT_TRY_START, MRPT_TRY_END, THROW_EXCEPTION_CUSTOM_MSG1
 */
#define THROW_EXCEPTION(msg)	\
	{\
		std::ostringstream auxCompStr;\
		auxCompStr << "\n\n =============== MRPT EXCEPTION =============\n";\
		auxCompStr << __CURRENT_FUNCTION_NAME__ << ", line " << __LINE__ << ":\n";\
		auxCompStr << msg << "\nCALL STACK:\n";\
		throw std::logic_error( auxCompStr.str() );\
	}\

/** \def THROW_EXCEPTION_CUSTOM_MSG1
  * \param e The caught exception.
  *	\param msg Is a char* or literal string.
  */
#define THROW_EXCEPTION_CUSTOM_MSG1(msg,param1)	\
	{\
		std::ostringstream auxCompStr;\
		auxCompStr << "\n\n =============== MRPT EXCEPTION =============\n";\
		auxCompStr << __CURRENT_FUNCTION_NAME__ << ", line " << __LINE__ << ":\n";\
		auxCompStr << format(msg,param1);\
		auxCompStr << "\nCALL STACK:\n";\
		throw std::logic_error( auxCompStr.str() );\
	}\


/** \def THROW_TYPED_EXCEPTION(msg,exceptionClass)
 * Defines a unified way of reporting exceptions of type different from "std::exception"
 * \sa MRPT_TRY_START, MRPT_TRY_END
 */
#define THROW_TYPED_EXCEPTION(msg,exceptionClass)	\
	{\
		std::ostringstream auxCompStr;\
		auxCompStr << "\n\n =============== MRPT EXCEPTION =============\n";\
		auxCompStr << __CURRENT_FUNCTION_NAME__ << ", line " << __LINE__ << ":\n";\
		auxCompStr << msg << "\nCALL STACK:\n";\
		throw exceptionClass( auxCompStr.str() );\
	}\

/** \def THROW_EXCEPTION_CUSTOM_MSG1
  * \param e The caught exception.
  *	\param msg Is a char* or literal string.
  */
#define THROW_TYPED_EXCEPTION_CUSTOM_MSG1(msg,param1,exceptionClass)	\
	{\
		std::ostringstream auxCompStr;\
		auxCompStr << "\n\n =============== MRPT EXCEPTION =============\n";\
		auxCompStr << __CURRENT_FUNCTION_NAME__ << ", line " << __LINE__ << ":\n";\
		auxCompStr << format(msg,param1);\
		auxCompStr << "\nCALL STACK:\n";\
		throw exceptionClass( auxCompStr.str() );\
	}\


/** \def THROW_STACKED_EXCEPTION
 * \sa MRPT_TRY_START, MRPT_TRY_END
 */
#define THROW_STACKED_EXCEPTION(e)	\
	{\
		std::string str( e.what() );\
		str+= __CURRENT_FUNCTION_NAME__;\
		str+= format(", line %i:\n", __LINE__ );\
		if (str.size()>3000) { std::cerr << "TOO MANY STACKED EXCEPTIONS!: " << std::endl << str << std::endl; abort(); } \
		throw std::logic_error( str );\
	}\

/** \def THROW_STACKED_EXCEPTION_CUSTOM_MSG
  * \param e The caught exception.
  *	\param msg Is a char* or std::string.
  */
#define THROW_STACKED_EXCEPTION_CUSTOM_MSG1(e,msg)	\
	{\
		std::ostringstream auxCompStr;\
		auxCompStr << e.what() ; \
		auxCompStr << msg << __CURRENT_FUNCTION_NAME__ << ", line " << __LINE__ << "(User msg above)\n";\
		throw std::logic_error( auxCompStr.str() );\
	}\

/** \def THROW_STACKED_EXCEPTION_CUSTOM_MSG
  * \param e The caught exception.
  *	\param stuff Is a printf-like sequence of params, e.g: "The error happens for x=%i",x
  */
#define THROW_STACKED_EXCEPTION_CUSTOM_MSG2(e,stuff,param1)	\
	{\
		std::ostringstream auxCompStr;\
		auxCompStr << e.what(); \
		auxCompStr << format( stuff, param1 ) << "\n" << __CURRENT_FUNCTION_NAME__ << ", line " << __LINE__ << ":\n";\
		throw std::logic_error( auxCompStr.str() );\
	}\

/** For use in CSerializable implementations */
#define MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(V) THROW_EXCEPTION(format("Cannot parse object: unknown serialization version number: '%i'",static_cast<int>(version)))


#if MRPT_HAS_ASSERT
	/**
	 * Defines an assertion mechanism.
	 * \note Do NOT put code that must be always executed inside this statement, but just comparisons. This is because users might require ASSERT_'s to be ignored for optimized releases.
	 * \sa MRPT_TRY_START, MRPT_TRY_END
	 */
#	define ASSERT_(f) \
	{ \
	if (!(f)) \
		{ \
			::std::string tmpStr("Assert condition failed: "); \
			tmpStr+= #f; \
			THROW_EXCEPTION( tmpStr.c_str() ); \
		} \
	}

/** Throws an exception if the number is NaN, IND, or +/-INF, or return the same number otherwise.
  */
#define MRPT_CHECK_NORMAL_NUMBER(val) \
	{ \
		double	v=(val); \
		if (math::isNan(v)) THROW_EXCEPTION("Check failed (value is NaN)"); \
		if (!math::isFinite(v)) THROW_EXCEPTION("Check failed (value is not finite)"); \
	}

	// Use some stuff from boost here
	//  (C) Copyright John Maddock 2001 - 2003.
	//  (C) Copyright Darin Adler 2001.
	//  (C) Copyright Peter Dimov 2001.
	//  (C) Copyright Bill Kempf 2002.
	//  (C) Copyright Jens Maurer 2002.
	//  (C) Copyright David Abrahams 2002 - 2003.
	//  (C) Copyright Gennaro Prota 2003.
	//  (C) Copyright Eric Friedman 2003.
	// License: Boost Software License   See LICENSE.txt for the full license.
	#ifndef BOOST_JOIN
	#define BOOST_JOIN( X, Y ) BOOST_DO_JOIN( X, Y )
	#define BOOST_DO_JOIN( X, Y ) BOOST_DO_JOIN2(X,Y)
	#define BOOST_DO_JOIN2( X, Y ) X##Y
	#endif

	// The following macro is based on dclib:
	// Copyright (C) 2003  Davis E. King (davisking@users.sourceforge.net)
	// License: Boost Software License   See LICENSE.txt for the full license.
	namespace mrpt
	{
		namespace utils
		{
			template <bool value> struct compile_time_assert;
			template <> struct compile_time_assert<true> { enum {value=1};  };
		}
	}
	#define MRPT_COMPILE_TIME_ASSERT(expression) \
			typedef char BOOST_JOIN(MRPT_CTA, __LINE__)[::mrpt::utils::compile_time_assert<(bool)(expression)>::value];

#else
#	define ASSERT_(f)
#	define MRPT_CHECK_NORMAL_NUMBER(val)
#	define MRPT_COMPILE_TIME_ASSERT(f)
#endif



/** Can be used to avoid "not used parameters" warnings from the compiler
 */
#define MRPT_UNUSED_PARAM(a)		(void)(a)

#if MRPT_HAS_STACKED_EXCEPTIONS
	/** The start of a standard MRPT "try...catch()" block that allows tracing throw the call stack after an exception.
	  * \sa MRPT_TRY_END,MRPT_TRY_END_WITH_CLEAN_UP
	  */
#	define MRPT_TRY_START	\
	try { \

	/** The end of a standard MRPT "try...catch()" block that allows tracing throw the call stack after an exception.
	  * \sa MRPT_TRY_START,MRPT_TRY_END_WITH_CLEAN_UP
	  */
#	define MRPT_TRY_END	\
	} \
	catch (std::bad_alloc &e) \
	{ \
		throw e; \
	} \
	catch (std::exception &e) \
	{ \
		THROW_STACKED_EXCEPTION(e); \
	} \
	catch (...) \
	{ \
		THROW_EXCEPTION("Unexpected runtime error!"); \
	} \

	/** The end of a standard MRPT "try...catch()" block that allows tracing throw the call stack after an exception, including a "clean up" piece of code to be run before throwing the exceptions.
	  * \sa MRPT_TRY_END,MRPT_TRY_START
	  */
#	define MRPT_TRY_END_WITH_CLEAN_UP(stuff)	\
	} \
	catch (std::bad_alloc &e) \
	{ \
		throw e; \
	} \
	catch (std::exception &e) \
	{ \
		{stuff} \
		THROW_STACKED_EXCEPTION(e); \
	} \
	catch (...) \
	{ \
		{ stuff } \
		THROW_EXCEPTION("Unexpected runtime error!"); \
	} \

#else
#	define MRPT_TRY_START
#	define MRPT_TRY_END
#	define MRPT_TRY_END_WITH_CLEAN_UP(stuff)
#endif

// Generic constants and defines:
// ---------------------------------------------------------

#ifndef M_PI
/** The PI constant
 */
	#define M_PI 3.14159265358979323846264338327950288
#endif

#ifndef M_2PI
/** The 2*PI constant
 */
	#define M_2PI 6.283185307179586476925286766559
#endif

#define M_PIf  3.14159265358979f
#define M_2PIf 6.28318530717959f


// Avoid conflicting declaration of max macro in windows headers
#ifdef MRPT_OS_WINDOWS
#	define NOMINMAX
#	ifdef max
#		undef	max
#		undef	min
#	endif
#endif

//  We want to avoid defining "max" & "min" as #define's since it create conflicts
//    with methods, variables, etc... with the same name in some compilers.
// Use std::max & std::min for all compilers by default, but for MSVC6 it does not exist:
#if defined(_MSC_VER) && (_MSC_VER<1300)
#	ifndef max
		namespace std
		{
			template<class T> inline const T max(const T& A,const T& B) { return A>B ? A:B; }
			template<class T> inline const T min(const T& A,const T& B) { return A<B ? A:B; }
		}
#	else
#		define  MAX3_MSVC6_VERSION
#	endif
#endif

// Min & Max:
#ifndef MAX3_MSVC6_VERSION
	template<typename T> inline const T  min3(const T& A, const T& B,const T& C) { return std::min<T>(A, std::min<T>(B,C) ); }
	template<typename T> inline const T  max3(const T& A, const T& B,const T& C) { return std::max<T>(A, std::max<T>(B,C) ); }
#else
#	define max3(A,B,C) max(A,max(B,C))
#	define min3(A,B,C) min(A,min(B,C))
#endif

namespace mrpt
{
	namespace utils
	{
		/** Degrees to radians */
		inline double DEG2RAD(const double &x) { return x*M_PI/180.0;	}

		/** Degrees to radians */
		inline float DEG2RAD(const float &x) { return x*M_PIf/180.0f; }

		/** Degrees to radians */
		inline float DEG2RAD(const int &x) { return x*M_PIf/180.0f; }

		/** Radians to degrees */
		inline double RAD2DEG(const double &x) { return x*180.0/M_PI; }

		/** Radians to degrees */
		inline float RAD2DEG(const float &x) { return x*180.0f/M_PIf; }

		/** Returns the sign of X as "1" or "-1" */
		template <typename T>
		int sign(T x) { return x<0 ? -1:1; }

		/** Returns the closer integer (int) to x */
		template <typename T>
		int round(T x) { return static_cast<int>(floor(static_cast<double>(x+0.5))); }

		/** Returns the closer integer (long) to x */
		template <typename T>
		long round_long(T x) { return static_cast<long>(floor(static_cast<double>(x+0.5))); }

		/** Rounds toward zero  */
		template <typename T>
		int fix(T x) { return  x>0 ? static_cast<int>(floor(static_cast<double>(x))) : static_cast<int>(ceil(static_cast<double>(x))) ; }

		/** Inline function for the square of a number. */
		template<class T>
		inline T square(const T& x)    { return x*x; }

		// --------------------------
		// Functors:
		//  Ret: NO  In:1-3
		// --------------------------

		/** A generic functor type for functions accepting 1 input arguments and returning nothing.
		*/
		typedef void (*TFunctor_noRet_1inputs)(const void *);

		/** A generic functor type for functions accepting 2 input arguments and returning nothing.
		*/
		typedef void (*TFunctor_noRet_2inputs)(const void *,const void *);

		/** A generic functor type for functions accepting 3 input arguments and returning nothing.
		*/
		typedef void (*TFunctor_noRet_3inputs)(const void *,const void *,const void *);

		// --------------------------
		// Functors:
		//  Ret: double  In:1-3
		// --------------------------

		/** A generic functor type for functions accepting 1 input arguments and returning a double value.
		*/
		typedef double (*TFunctor_retDouble_1inputs)(const void *);

		/** A generic functor type for functions accepting 2 input arguments and returning a double value.
		*/
		typedef double (*TFunctor_retDouble_2inputs)(const void *,const void *);

		/** A generic functor type for functions accepting 3 input arguments and returning a double value.
		*/
		typedef double (*TFunctor_retDouble_3inputs)(const void *,const void *,const void *);

		// --------------------------
		// Functors:
		//  Ret: vector  In: vector
		// --------------------------

		/** A generic functor type for functions accepting 1 vector and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecDbl_inpVecDbl)(const vector_double &in, vector_double &out);

		/** A generic functor type for functions accepting 1 vector and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecFlt_inpVecFlt)(const vector_float &in, vector_float &out);

		/** A generic functor type for functions accepting 1 vector and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecInt_inpVecInt)(const vector_int &in, vector_int &out);

		// -----------------------------
		// Functors:
		//  Ret: vector  In: 2 x vector
		// -----------------------------

		/** A generic functor type for functions accepting 2 vectors and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecDbl_inp2VecDbl)(const vector_double &x,const vector_double &y, vector_double &out);

		/** A generic functor type for functions accepting 2 vectors and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecFlt_inp2VecFlt)(const vector_float &x,const vector_float &y, vector_float &out);

		/** A generic functor type for functions accepting 2 vectors and returning 1 vector.
		*/
		typedef void (*TFunctor_retVecInt_inp2VecInt)(const vector_int &x,const vector_int &y, vector_int &out);

		// --------------------------
		// Functors:
		//  Ret: double  In: vectors
		// --------------------------

		/** A generic functor type for functions accepting 1 vector and returning 1 double.
		*/
		typedef double (*TFunctor_retDbl_inp1VecDbl)(const vector_double &in1);

		/** A generic functor type for functions accepting 2 vectors and returning 1 double.
		*/
		typedef double (*TFunctor_retDbl_inp2VecDbl)(const vector_double &in1,const vector_double &in2);

		/** A generic functor type for functions accepting 3 vectors and returning 1 double.
		*/
		typedef double (*TFunctor_retDbl_inp3VecDbl)(const vector_double &in1,const vector_double &in2,const vector_double &in3);


		/** For performing type casting from a pointer to its numeric value.
		*/
		#if defined(_MSC_VER) && (_MSC_VER>=1300)
			typedef unsigned long long POINTER_TYPE;
		#else
			typedef unsigned long POINTER_TYPE;
		#endif

		/** An object for deleting pointers (intended for STL algorithms) */
		struct ObjectDelete
		{
			template<typename T>
			void operator()(const T *ptr)
			{
				delete ptr;
			}
		};

		/** An object for clearing an object (invokes its method "clear()") given a pointer or smart-pointer, intended for being used in STL algorithms. */
		struct ObjectClear
		{
			template<typename T>
			void operator()(T ptr)
			{
				ptr->clear();
			}
		};

		/** Utility to get a cast'ed pointer from a smart pointer */
		template <class R, class P>
		R* getAs(stlplus::smart_ptr_clone<P> &o) { return static_cast<R*>( & (*o) ); }

		/** Utility to get a cast'ed pointer from a smart pointer */
		template <class R, class P>
		const R* getAs(const stlplus::smart_ptr_clone<P> &o) { return static_cast<const R*>( & (*o) ); }


		/** Reverse the order of the bytes of a given type (useful for transforming btw little/big endian)  */
		template <class T> void reverseBytes(const T &v_in, T& v_out)
		{
			v_out = v_in;
			uint8_t *ptr = reinterpret_cast<uint8_t*>(&v_out);
			std::reverse(ptr,ptr+sizeof(T));
		}

	} // End of namespace
} // end of namespace
#endif
