/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CObservationGPS_H
#define CObservationGPS_H

#include <mrpt/utils/CSerializable.h>
#include <mrpt/slam/CObservation.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPose2D.h>

namespace mrpt
{
namespace slam
{
	using namespace mrpt::utils;

	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CObservationGPS , CObservation )

	/** Declares a class derived from "CObservation" that represents a Global Positioning System (GPS) reading.
	 *
	 * \sa CObservation
	 */
	class MRPTDLLIMPEXP CObservationGPS : public CObservation
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CObservationGPS )

	 public:
		/** Constructor.
		 */
		CObservationGPS(  );

		/** Dumps the contents of the observation in a human-readable form to a given output stream
		  */
		void  dumpToStream( CStream &out );

		/** Dumps the contents of the observation in a human-readable form to the console
		  */
		void  dumpToConsole( );


		 /** The sensor pose on the robot.
		  */
		CPose3D			sensorPose;

		/** A UTC time-stamp structure for GPS messages
		  */
		struct MRPTDLLIMPEXP TUTCTime
		{
			TUTCTime();

			uint8_t	hour;
			uint8_t	minute;
			double  sec;

			bool operator == (const TUTCTime& o) const { return hour==o.hour && minute==o.minute && sec==o.sec; }
			bool operator != (const TUTCTime& o) const { return hour!=o.hour || minute!=o.minute || sec!=o.sec; }
		};

		/** The GPS datum for GGA commands
		  */
		struct MRPTDLLIMPEXP TGPSDatum_GGA
		{
			TGPSDatum_GGA();

			/** The GPS sensor measured timestamp (in UTC time)
			*/
			TUTCTime	UTCTime;

			/** The measured latitude, in degrees (North:+ , South:-)
			*/
			double			latitude_degrees;

			/** The measured longitude, in degrees (East:+ , West:-)
			*/
			double			longitude_degrees;

			/** The values defined in the NMEA standard are the following:
			  *
			  *	0 = invalid
              *	1 = GPS fix (SPS)
			  *	2 = DGPS fix
              * 3 = PPS fix
			  * 4 = Real Time Kinematic
			  * 5 = Float RTK
			  * 6 = estimated (dead reckoning) (2.3 feature)
			  * 7 = Manual input mode
			  * 8 = Simulation mode
			  */
			uint8_t		fix_quality;

			/** The measured altitude, in meters.
			*/
			double			altitude_meters;

			/** The number of satelites used to compute this estimation.
			*/
			uint32_t		satellitesUsed;

			/** This states whether to take into account the value in the HDOP field.
			*/
			bool			thereis_HDOP;

			/** The HDOP (Horizontal Dilution of Precision) as returned by the sensor.
			*/
			float			HDOP;
		};

		/** The GPS datum for RMC commands
		  */
		struct MRPTDLLIMPEXP TGPSDatum_RMC
		{
			TGPSDatum_RMC();

			/** The GPS sensor measured timestamp (in UTC time)
			*/
			TUTCTime	UTCTime;

			/** This will be: 'A'=OK or 'V'=void
			 */
			int8_t		validity_char;

			/** The measured latitude, in degrees (North:+ , South:-)
			  */
			double		latitude_degrees;

			/** The measured longitude, in degrees (East:+ , West:-)
			  */
			double		longitude_degrees;

			/** The measured speed (in knots)
			  */
			double		speed_knots;

			/** The measured speed direction (in degrees)
			  */
			double		direction_degrees;
		};

		/** Will be true if the corresponding field contains data read from the sensor, or false if it is not available.
		  * \sa GGA_datum
		  */
		bool				has_GGA_datum;

		/** Will be true if the corresponding field contains data read from the sensor, or false if it is not available.
		  * \sa RMC_datum
		  */
		bool				has_RMC_datum;

		/** If "has_GGA_datum" is true, this contains the read GGA datum.
		  */
		TGPSDatum_GGA		GGA_datum;

		/** If "has_RMC_datum" is true, this contains the read GGA datum.
		  */
		TGPSDatum_RMC		RMC_datum;

		 /** Implements the virtual method in charge of finding the likelihood between this
		  *   and another observation, probably only of the same derived class. The operator
		  *   may be asymmetric.
		  *
		  * \param anotherObs The other observation to compute likelihood with.
		  * \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.
		  *
		  * \return Returns a likelihood measurement, in the range [0,1].
		  *	\exception std::exception On any error, as another observation being of an invalid class.
		  */
		 float  likelihoodWith( const CObservation *anotherObs, const CPosePDF *anotherObsPose = NULL ) const;

		/** A general method to retrieve the sensor pose on the robot.
		  *  Note that most sensors will return a full (6D) CPose3D, but see the derived classes for more details or special cases.
		  * \sa setSensorPose
		  */
		void getSensorPose( CPose3D &out_sensorPose ) const { out_sensorPose = sensorPose; }


		/** A general method to change the sensor pose on the robot.
		  *  Note that most sensors will use the full (6D) CPose3D, but see the derived classes for more details or special cases.
		  * \sa getSensorPose
		  */
		void setSensorPose( const CPose3D &newSensorPose ) { sensorPose = newSensorPose; }

	}; // End of class def.


	} // End of namespace
} // End of namespace

#endif
