/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CPOSEPDF_H
#define CPOSEPDF_H

#include <mrpt/utils/CSerializable.h>
#include <mrpt/poses/CPose2D.h>

namespace mrpt
{
namespace poses
{
	// This must be added to any CSerializable derived class:
	DEFINE_SERIALIZABLE_PRE( CPosePDF )

	/** Declares a class that represents a probability density function (pdf) of a 2D pose (x,y,phi).
	 *   This class is just the base class for unifying many diferent ways this pdf can be implemented.
	 *
	 *  For convenience, a pose composition is also defined for any pdf derived class,
	 *   changeCoordinatesReference, in the form of a method rather than an operator.
     *
	 *
	 *  See also the tutorial on <a href="http://babel.isa.uma.es/mrpt/index.php/Probability_Density_Distributions_Over_Spatial_Representations">probabilistic spatial representations in the MRPT</a>.
	 *
	 * \sa CPose2D, CPose3DPDF, CPoseRandomSampler
	 */
	class MRPTDLLIMPEXP CPosePDF : public mrpt::utils::CSerializable
	{
		DEFINE_VIRTUAL_SERIALIZABLE( CPosePDF )

	public:
		/** Returns the entropy of the estimated pose covariance matrix.
		  */
		double  getEstimatedCovarianceEntropy();

		 /** Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
		  */
		virtual CPose2D  getEstimatedPose() const = 0 ;

		 /** Returns an estimate of the pose covariance matrix (3x3 cov.matrix  for x,y,phi variables)
		  */
		virtual math::CMatrixD  getEstimatedCovariance() const = 0;

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		virtual void  copyFrom(const CPosePDF &o) = 0;

		/** Save PDF's particles to a text file. See derived classes for more information about the format of generated files.
		 */
		virtual void  saveToTextFile(const std::string &file) const = 0;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		virtual void  changeCoordinatesReference( const CPose2D &newReferenceBase ) = 0;

		/** Draws a single sample from the distribution
		  */
		virtual void  drawSingleSample( CPose2D &outPart ) const = 0;

		/** Draws a number of samples from the distribution, and saves as a list of 1x3 vectors, where each row contains a (x,y,phi) datum.
		  */
		virtual void  drawManySamples( size_t N, std::vector<vector_double> & outSamples ) const = 0;

		/** Bayesian fusion of two pose distributions (product of two distributions->new distribution), then save the result in this object (WARNING: See implementing classes to see classes that can and cannot be mixtured!)
		  * \param p1 The first distribution to fuse
		  * \param p2 The second distribution to fuse
		  * \param minMahalanobisDistToDrop If set to different of 0, the result of very separate Gaussian modes (that will result in negligible components) in SOGs will be dropped to reduce the number of modes in the output.
		  */
		virtual void  bayesianFusion( CPosePDF &p1, CPosePDF &p2, const double&minMahalanobisDistToDrop = 0)  = 0 ;

		/** Returns a new PDF such as: NEW_PDF = (0,0,0) - THIS_PDF
		  */
		virtual void  inverse(CPosePDF &o) const = 0;


	}; // End of class def.


	} // End of namespace
} // End of namespace

#endif
