/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

/*---------------------------------------------------------------
    APPLICATION: Camera calibration
    FILE: calib_main.cpp
    AUTHOR: Vicente Arevalo Espejo <varevalo@ctima.uma.es>
             based on code from the OpenCV library.

	See README.txt for instructions.

   Part of the MRPT Library - http://babel.isa.uma.es/mrpt/
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/

#include <stdio.h>

#include "Cam.h"
#include "CalibFilter.h"


#include <mrpt/config.h>
#include <mrpt/system.h>

using namespace mrpt;
using namespace mrpt::system;
using namespace mrpt::utils;

#if MRPT_HAS_OPENCV
	// Include the OPENCV libraries:
	#if defined(_MSC_VER)
		//#pragma comment (lib,"vfw32.lib")
		/*#pragma comment (lib,"cv.lib")
		#pragma comment (lib,"cvaux.lib")
		#pragma comment (lib,"cxcore.lib")
		#pragma comment (lib,"highgui.lib")*/
	#endif
#endif


int   CHECKBOARD_SIZE_X = 10, CHECKBOARD_SIZE_Y=7;
float CHECKBOARD_DIMENSION_MM = 23;



int main(int argc, char* argv[])
{
	const char *filename="calib.txt";
	IplImage *frame = NULL;
	int salir = 0;


	printf(" camera-calib - Part of the MRPT\n");
	printf(" MRPT C++ Library: %s - BUILD DATE %s\n", MRPT_getVersion().c_str(), MRPT_getCompilationDate().c_str());
	printf("-------------------------------------------------------------------\n");

	if (argc==2 && !os::_strcmpi(argv[1],"--help") )
	{
		printf(" Usage: \n");
		printf(" camera-calib             - Default parameters assumed, or\n");
		printf(" camera-calib NX NY SIZE  - State checkerboard values:\n");
		printf("     NX x NY: Number of squares in the checkerboard (default=10x7)\n");
		printf("     SIZE   : Dimension of each square in mm (default=23mm)\n\n");
		return 0;
	}


	printf("Instructions:\n");
	printf(" - Press 'c' to start camera calibration with checkerboard.\n");
	printf(" - Press 'x' to save calibration data to %s and exit.\n", filename);
	printf(" - Press 'q' to exit without saving.\n");
	printf(" - Press 'l' to load camera parameters from %s.\n", filename);
	printf(" - Press 'r' to reset the undistorted view.\n");
	printf(" - Press 's' to save a frame to frame_XXXX.bmp.\n");

	printf("\n");

	printf("Checkerboard dimensions: %ux%u (square length=%.01fmm)\n", CHECKBOARD_SIZE_X, CHECKBOARD_SIZE_Y, CHECKBOARD_DIMENSION_MM );
	printf("(This can be changed through command line arguments. Run 'camera-calib --help')\n");


	CCam cam;
	CCalibFilter cf;


	cam.open(CV_CAP_DC1394);
	cam.openVideoSourceDlg();
	if(cam.isOpen())
    {
		char buffer[1024];

//		cam.openVideoSourceDlg();
		cam.openVideoFormatDlg();
		cvNamedWindow( "output", 1 );

		frame = cam.getFrame();
		if(frame)
			printf("\nCapturing images at %dx%d\n", frame->width, frame->height);

		long nframe = 0;
		while(!salir)
		{
			frame = cam.getFrame();
			if( !frame )
				break;

			nframe++;

			switch(cvWaitKey(5))
			{
			case 'c':
			case 'C':
				printf("\nStarting calibration...\n");
				cf.StartCalibrate();
				break;

			case 's':
			case 'S':
				os::sprintf(buffer,sizeof(buffer), "frame_%lu.bmp", nframe);
				printf("\nSaving %s...\n",buffer);
				cvSaveImage(buffer, frame);
				break;

			case 'r':
			case 'R':
				printf("\nReseting...\n");
				cf.reset();
				break;

			case 'x':
			case 'X':
				printf("Saving camera parameters to %s\n",filename);
				cf.SaveCameraParams(filename);
			case 'q':
			case 'Q':
				printf("\nBye!\n");
				salir = 1;
				break;

			case 'l':
			case 'L':
				if (1!=cf.LoadCameraParams(filename))
				{
					printf("Error loading file %s...\n",filename);
				}
				else
				{
					printf("Loaded from %s OK\n",filename);
					cf.setEnableUndistortion(1);
				}
				break;
			}

			cf.ProcessFrame(frame);

			cvShowImage("output", frame);
		}

		cam.close();

		cf.StopCalibrate();

		cvDestroyWindow("output");
    }
	else
	{
		printf("ERROR: Cannot open camera.\n");
	}

	return 0;
}
