#include <wx/filename.h>
#include <wx/stdpaths.h>

#include <algorithm>
#include <set>
#include <math.h>

#include "Dictionary.h"
#include "DicomFile.h"
#include "StringConvert.h"
#include "Volume.h"
#include "SeriesHandler.h"

using namespace jcs;


SeriesHandler::SeriesHandler(const std::string& seriesUid)
  : mSeriesUid(seriesUid)
{ 
    seriesInfo.subject_name = "";
    seriesInfo.subject_id = "";
    seriesInfo.study_number = "";
    seriesInfo.series_number = "";
    seriesInfo.StudyDate = "";
    seriesInfo.SeriesDate = "";
    seriesInfo.study_description = "";
    seriesInfo.series_description = "";
    multiecho = false;
}


SeriesHandler::~SeriesHandler()
{
}


///
/**
 */
std::vector<double> 
SeriesHandler::GetIppForFirstSlice(const VolId& id)
{ 
    return position[id]; 
}


///
/**
 */
std::vector<double>
SeriesHandler::GetVoxelSize()
{
    std::vector<double> voxel_size(3, 0);
    std::string s;
    if(Find("PixelSpacing", s)) {
	voxel_size[0] = stof(s, 0);
	voxel_size[1] = stof(s, 1);
    }
    Find("SliceSpacing", voxel_size[2]);
    if (voxel_size[2] == 0)
	Find("SliceThickness", voxel_size[2]);
    return voxel_size;
}


///
/**
 */
int
SeriesHandler::GetNumberOfSlices() const
{
    int nVolumes = GetNumberOfVolumes();
    int nFiles = GetNumberOfFiles();
    if (nVolumes != 0)
	return nFiles/nVolumes;
    else
	return nFiles;

}


///
/**
   \return A vector of filenames.
*/
std::vector<std::string> 
SeriesHandler::GetFilenames() const
{
    std::vector<std::string> names;
    for (FileMapType::const_iterator it = mFileMap.begin(); it != mFileMap.end(); ++it) 
	names.push_back(it->first);
    return names;

}


///
/**
 */
bool
SeriesHandler::IsBigEndian() const
{
    if (mFileMap.empty()) return 0;
    DicomFile dicomFile((*mFileMap.begin()).first.c_str());
    return dicomFile.IsBigEndian();
}


///
/**
 */
int
SeriesHandler::GetNumberOfVolumes() const
{
    std::set<VolId> volSet = GetVolIds();
    return volSet.size();
}


///
/**
 */
std::string
SeriesHandler::GetImagePositionPatient(DicomFile& dfile, int frame)
{
    std::string ipp;

    if (!dfile.Find("ImagePositionPatient", ipp))
	ipp = "0\\0\\" + itos(frame);
    return ipp;

}


///
/**
 */
int
SeriesHandler::GetRescaleSlopeAndIntercept(const VolId& id, double& slope, double& intercept) const
{
    std::string filename;
    if (!GetFirstFilename(id, filename)) return 0;
    DicomFile dicomFile(filename.c_str());
    return GetRescaleSlopeAndIntercept(dicomFile, slope, intercept);
}


///
/** Seeks RescaleSlope and RescaleIntercept attributes. If
    not found, will set slope to 1 and intercept to 0.
*/
int
SeriesHandler::GetRescaleSlopeAndIntercept(DicomFile& dfile, double& slope, double& intercept, int frame) const
{
    if (!dfile.Find("RescaleSlope", slope)) slope = 1;
    if (!dfile.Find("RescaleIntercept", intercept))  intercept = 0;
    return 1;
}


///
/**
 */
double
SeriesHandler::GetVolumeInterval() const
{
    std::map<VolId, std::vector<double> > vtmap;
    FileMapType::const_iterator it = mFileMap.begin();
    while (it != mFileMap.end()) { 
	DicomFile file(it->first.c_str());
	double time;
	file.Find("AcquisitionTime", time);
	vtmap[it->second.front()].push_back(time);
	++it;
    }

    if (vtmap.size() <= 1) return 0;

    std::vector<double> vol_times;
    std::map<VolId, std::vector<double> >::iterator vit = vtmap.begin();
    while (vit != vtmap.end()) { 
	vol_times.push_back(*min_element(vit->second.begin(), vit->second.end()));
	++vit;
    }

    sort(vol_times.begin(), vol_times.end());

    return (vol_times[1] - vol_times[0]);

}


///
/**
 */
double
SeriesHandler::GetSliceDuration() const
{
    return 0;
}


///
/**
 */
int
SeriesHandler::GetColumns()
{
    int columns;
    Find("ImageColumns", columns);
    return columns;
}


///
/**
 */
int
SeriesHandler::GetRows()
{
    int rows;
    Find("ImageRows", rows);
    return rows;
}


///
/**
 */
int
SeriesHandler::GetFrames() const
{
    int frames = 0;
    Find("NumberOfFrames", frames);
    return frames;
}


///
/** Adds a file name to mFileMap.
    \param filename Name of file to add.
    \return 0 if filename already in mFileMap, 1 otherwise.
*/
int
SeriesHandler::AddFile(const char* filename)
{
    if (mFileMap.count(filename)) return 0;

    DicomFile dicom_file(filename);

    std::string series_uid;
    dicom_file.Find("SeriesInstanceUid", series_uid);

    // Fix problems with termination
    std::string temp(series_uid.c_str());
    series_uid.clear();
    series_uid = temp;

    assert(mSeriesUid == series_uid);

    std::string uid;
    dicom_file.Find("SopInstance", uid);
    if (mInstanceUids.count(uid) == 0 ) {
	mInstanceUids.insert(uid);
	VolListType info = ReadVolIds(dicom_file);
	mFileMap[filename] = info;
    }
  
    if (seriesInfo.SeriesDate == "") {
	Find("PatientId", seriesInfo.subject_id);
	Find("PatientName", seriesInfo.subject_name);
	Find("StudyId", seriesInfo.study_number);
	Find("StudyDescription", seriesInfo.study_description);
	int series_number;
	Find("SeriesNumber", series_number);
	seriesInfo.series_number = itos(series_number, 3);
	Find("StudyDate", seriesInfo.StudyDate);
	Find("SeriesDate", seriesInfo.SeriesDate);
	Find("SeriesDescription", seriesInfo.series_description);
    }

    return 1;
}


///
/** Retreieves a bunch of information about a series.
    \return A vector of strings with series information.
*/
std::vector<std::string>
SeriesHandler::GetStringInfo()
{
    std::vector<std::string> v;

    v.push_back(std::string("Series UID: ") + GetSeriesUid());
    v.push_back(std::string("Study date: ") + Find("StudyDate"));
    v.push_back(std::string("Study time: ") + Find("StudyTime"));
    v.push_back(std::string("Series date: ") + Find("SeriesDate"));
    v.push_back(std::string("Series time: ") + Find("SeriesTime"));
    v.push_back(std::string("Subject: ") + Find("PatientName"));
    v.push_back(std::string("Series description: ") + Find("SeriesDescription"));
    v.push_back(std::string("Manufacturer: ") + Find("Manufacturer"));
    v.push_back(std::string("Model name: ") + Find("ModelName"));
    v.push_back(std::string("Study id: ") + Find("StudyId"));
    v.push_back(std::string("Series number: ") + Find("SeriesNumber"));
    v.push_back(std::string("Repetition time (ms): ") + Find("RepetitionTime"));

    GetEchoTimes();
    //  if (multiecho)
    for(unsigned int i = 0; i < echo_times.size(); i++)
	v.push_back(std::string("Echo time[" + itos(i) + "] (ms): ") + echo_times[i]);
    //else
    //v.push_back(std::string("Echo time (ms): ") + echo_times[0]);
 
    v.push_back(std::string("Inversion time (ms): ") + Find("InversionTime"));
    v.push_back(std::string("Flip angle: ") + Find("FlipAngle"));
    v.push_back(std::string("Number of averages: ") + Find("NumberOfAverages"));
    v.push_back(std::string("Slice thickness (mm): ") + Find("SliceThickness"));
    v.push_back(std::string("Slice spacing (mm): ") + Find("SliceSpacing"));
    v.push_back(std::string("Image columns: ") + Find("ImageColumns"));
    v.push_back(std::string("Image rows: ") + Find("ImageRows"));
    v.push_back(std::string("Number of frames: ") + Find("NumberOfFrames"));
    v.push_back(std::string("Phase encoding direction: ") + Find("PhaseEncodingDirection"));

    std::vector<double> voxels = GetVoxelSize();
    v.push_back(std::string("Voxel size x (mm): ") + ftos(voxels[0]));
    v.push_back(std::string("Voxel size y (mm): ") + ftos(voxels[1]));

    int nVolumes = GetNumberOfVolumes();
    v.push_back(std::string("Number of volumes: ") + itos(nVolumes));
    v.push_back(std::string("Number of slices: ") + itos(GetNumberOfSlices()));
    v.push_back(std::string("Number of files: ") + itos(GetNumberOfFiles()));
    v.push_back(std::string("Number of frames: ") + itos(GetFrames()));
    v.push_back(std::string("Slice duration (ms) : ") + ftos(GetSliceDuration()));

    if (nVolumes > 1)
	v.push_back(std::string("Volume interval (s): ") + ftos(GetVolumeInterval()));

    if (!orientations.empty()) {
	if (orientations.front().at(1) > orientations.front().at(0)) // sagital slices
	    v.push_back(std::string("Orientation: sag"));
	else if (orientations.front().at(4) > fabs(orientations.front().at(5))) // transverse slices
	    v.push_back(std::string("Orientation: tra"));
	else // coronal slices
	    v.push_back(std::string("Orientation: cor"));
    }

    return v;
}


///
/** Retrieves echo times for the series. Sets multiecho boolean. Sets echo_times.
    Note that echo_times is indexed by the EchoNumber.
*/
void
SeriesHandler::GetEchoTimes()
{
    std::map<int, std::string>::iterator eterator;
    std::string s;
    int idx;
    echo_times.clear();
    set<VolId> volids = GetVolIds();
    set<VolId>::iterator vol_it = volids.begin();
    set<VolId>::iterator vol_it_end = volids.end();
    for (; vol_it != vol_it_end; vol_it++)
	if (Find("EchoTime", s, *vol_it) && Find("EchoNumber", idx, *vol_it))
	    echo_times[idx] = s;
    multiecho = (echo_times.size() > 1);
}


/// DicomFile will set str to empty std::string if value not found, so we don't have to check.
/**
 */
SeriesHandler::VolListType
SeriesHandler::ReadVolIds(DicomFile& file)
{
    // should use FrameIncrementPointer, but doing it the easy way for now
    int nframes;
    if (!file.Find("NumberOfFrames", nframes)) nframes = 1;

    VolListType v;

    for (int frameno = 0; frameno < nframes; ++frameno) {

	VolId info;

	info.ids.push_back(GetSeriesUid());

	std::string str;
	file.Find("NumberOfTemporalPositions", str);
	int ndigits = str.size();
  
	str.clear();
	int tpid = 0;
	file.Find("TemporalPositionId", tpid);
	str = itos(tpid, ndigits);
	info.ids.push_back(str);

	str.clear();
	int echonumber = 0;
	file.Find("EchoNumber", echonumber);
	str = itos(echonumber, 2);
	info.ids.push_back(str);
  
	std::string ImageType;
	file.Find("ImageType", ImageType);

	if (ImageType.find("DERIVED") != std::string::npos) info.ids.push_back("D");
	else info.ids.push_back("");
	if (ImageType.find("T2 MAP") != std::string::npos) info.ids.push_back("T2MAP");
	else info.ids.push_back("");
	if (ImageType.find("\\M\\") != std::string::npos) info.ids.push_back("M");
	else
	    if (ImageType.find("\\P\\") != std::string::npos) info.ids.push_back("P");
	    else info.ids.push_back("");

	std::vector<double> rotation;
	rotation.resize(9, 0);

	std::string iop;
	if (!file.Find("ImageOrientationPatient", iop))
	    iop = "1\\0\\0\\0\\1\\0";
  
	for (int i = 0; i < 6; ++i) {
	    double f = stof(iop, i);
	    // round off
	    rotation[i] = static_cast<double>(static_cast<int>(f * 10000 + 0.5)) / 10000;
	}

	rotation[6] = rotation[1]*rotation[5] - rotation[2]*rotation[4];
	rotation[7] = rotation[2]*rotation[3] - rotation[0]*rotation[5];
	rotation[8] = rotation[0]*rotation[4] - rotation[1]*rotation[3];
  
	std::vector<std::vector <double> >::iterator pos;
	pos = find(orientations.begin(), orientations.end(), rotation);
	if (pos != orientations.end()) {
	    info.orientNumber = distance(orientations.begin(), pos);
	}
	else {
	    info.orientNumber = orientations.size();
	    orientations.push_back(rotation);
	}

	info.ids.push_back(itos(info.orientNumber + 1));

	v.push_back(info);
    }
    return v;
}


///
/**
 */
set<VolId>
SeriesHandler::GetVolIds() const
{
    std::set<VolId> volSet;
    FileMapType::const_iterator it = mFileMap.begin();
    FileMapType::const_iterator it_end = mFileMap.end();
    for(; it != it_end; it++) {
	VolListType::const_iterator vit = it->second.begin();
	VolListType::const_iterator vit_end = it->second.end();
	for (; vit != vit_end; vit++) { 
	    volSet.insert(*vit);
	}
    }
    return volSet;
}


///
/** returns volid + last file with that id
    \return
*/
std::map<VolId, std::string>
SeriesHandler::GetVolIdsAndFiles() const
{
    std::map<VolId, std::string> volMap;

    FileMapType::const_iterator it;
    VolListType::const_iterator vit;
    for (it = mFileMap.begin(); it != mFileMap.end(); it++)
	for (vit = it->second.begin(); vit != it->second.end(); vit++)
	    volMap[*vit] = it->first;

    return volMap;
}


///
/**
   \param id 
   \param str Reference to std::string
   \return True if match to str is found.
*/
bool
SeriesHandler::GetFirstFilename(const VolId& id, std::string& str) const
{
    FileMapType::const_iterator it;
    VolListType::const_iterator vit;
    for (it = mFileMap.begin(); it != mFileMap.end(); it++)
	for (vit = it->second.begin(); vit != it->second.end(); vit++)
	    if(*vit == id) {
		str = it->first;
		return true;
	    }

    return false;
}


///
/**
 */
std::vector<double>
SeriesHandler::GetRotationMatrix(const VolId& id)
{
    std::vector<double> r(9,0);
    try {
	r = orientations.at(id.orientNumber);
    }
    catch (const std::exception& error) {
	wxLogError(_("orientNumber: %d; number of orientations: %d"), id.orientNumber, orientations.size());
	wxLogError(wxString(error.what(), wxConvLocal));
    }
    return r;
}


///
/**
 */
void 
SeriesHandler::GetImageComments(std::vector<std::string>& comments) const 
{
    std::map<VolId, std::string> volFileMap = GetVolIdsAndFiles();
    std::map<VolId, std::string>::iterator it = volFileMap.begin();
    std::map<VolId, std::string>::iterator it_end = volFileMap.end();

    for (;it != it_end; it++) {
	std::string comment;
	DicomFile file(it->second.c_str());
	file.Find("ImageComments", comment);
	comments.push_back(comment);
    }
}


///
/**
 */
SyngoHandler::SyngoHandler(const std::string& seriesUid):SeriesHandler(seriesUid)
{
}


///
/**
 */
int
SyngoHandler::ReadCSAImageHeader(const std::string& tag, std::string& value) const
{
    if (mFileMap.empty()) return 0;
    DicomFile dicomFile((*mFileMap.begin()).first.c_str());
    return dicomFile.ReadCSAImageHeader(tag, value);
}


///
/**
 */
int
SyngoHandler::ReadCSASeriesHeader(const std::string& tag, std::string& value) const
{
    if (mFileMap.empty()) return 0;
    DicomFile dicomFile((*mFileMap.begin()).first.c_str());
    return dicomFile.ReadCSASeriesHeader(tag, value);
}


///
/**
 */
double
SyngoHandler::GetSliceDuration() const
{
    double duration = 0;
    std::string str;
    int err = ReadCSAImageHeader("SliceMeasurementDuration", str);
    if (err > 0) 
	duration = stof(str);
  
    return duration;
}


///
/**
 */
AoCode
SyngoHandler::GetAcquisitionOrder() const
{
    std::string protocol;
    int err = ReadCSASeriesHeader("MrProtocol", protocol);
  
    if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);

    if (err > 0) {
	std::string::size_type pos = protocol.find("sSliceArray.ucMode");
	pos = protocol.find('=', pos);
	pos = protocol.find_first_not_of(' ', pos+1);
	if (pos != std::string::npos) {
	    std::string acquisition = protocol.substr(pos, 3);
	    if (acquisition == "0x1") return ASCENDING;
	    if (acquisition == "0x2") return DESCENDING;
	    if (acquisition == "0x4")
		if (GetNumberOfSlices() % 2 == 0)
		    return INTERLEAVED_EVEN;
		else
		    return INTERLEAVED_ODD;
	    else
		return OTHER;
	}
    }
    return UNKNOWN;
}


///
/**
 */
bool
SyngoHandler::IsMoCo() const
{
    if (!mFileMap.empty()) {
	std::string imageType;
	DicomFile dicomFile((*mFileMap.begin()).first.c_str());
	dicomFile.Find("ImageType", imageType);
	if (imageType.find("MOCO") != std::string::npos) 
	    return true;
    }
    return false; 
}


///
/**
 */
SeriesHandler::VolListType
SyngoHandler::ReadVolIds(DicomFile& file)
{
    VolListType v = SeriesHandler::ReadVolIds(file);

    // needed for DTI
    std::string str;
    file.Find("SequenceName", str);
    std::string::size_type pos = str.find('#');
    if (pos != std::string::npos) {
	int zeros_to_add = 4 + pos - str.length();
	str.insert(pos+1, zeros_to_add, '0');

	pos = str.find('b') + 1;
	if (pos != std::string::npos) {
	    std::string::size_type pos2 = str.find_first_not_of("0123456789", pos);
	    zeros_to_add = 4 + pos - pos2;
	    str.insert(pos, zeros_to_add, '0');
	}
    }
    str = RemoveInvalidChars(str);
    v.front().ids.push_back(str);

    int an;
    file.Find("AcquisitionNumber", an);
  
    int width = 4;
    str = itos(an, width);
    v.front().ids.push_back(str);

    str.clear();  
    file.ReadCSAImageHeader("UsedChannelMask", str);
    v.front().ids.push_back(RemoveInvalidChars(str));

    return v;
}


///
/**
 */
bool
SyngoHandler::IsDti() const
{
    std::string bValue;
    int err = ReadCSAImageHeader("B_value", bValue);
    return (err > 0);

}


///
/**
 */
GradientInfo
SyngoHandler::GetGradientInfo()
{

    GradientInfo info;

    std::map<VolId, std::string> volFileMap = GetVolIdsAndFiles();
    std::map<VolId, std::string>::iterator it = volFileMap.begin();
    std::map<VolId, std::string>::iterator it_end = volFileMap.end();

    DicomFile first_file(it->second.c_str());

    std::string version;
    first_file.Find("SoftwareVersion", version);
    if (version.find("B15") != std::string::npos) 
	wxLogWarning(_("Warning: Polarity of the bvecs may be wrong for VB15 data."));
    if (version.find("B13") != std::string::npos)
	wxLogWarning(_("Warning: bvecs are sometimes wrong for VB13 data, due to a bug in the algorithm by which Siemen's converted the B_matrix to a principle eigendirection.  Use with caution."));

    for (; it != it_end; it++) {
	std::string bValue;
	std::string directionality;
	std::vector<std::string> gradDir;

	DicomFile file(it->second.c_str());

	int err = file.ReadCSAImageHeader("B_value", bValue);

	if (err > 0) info.values.push_back(stof(bValue));

	err = file.ReadCSAImageHeader("DiffusionGradientDirection", gradDir);
	if ((err < 0) || (gradDir.size() != 3)) {
	    info.xGrads.push_back(0);
	    info.yGrads.push_back(0);
	    info.zGrads.push_back(0);
	}
	else {
	    info.xGrads.push_back(stof(gradDir.at(0)));
	    info.yGrads.push_back(stof(gradDir.at(1)));
	    info.zGrads.push_back(stof(gradDir.at(2)));
	}
    }

    std::vector<double> r = GetRotationMatrix(volFileMap.begin()->first);
    RotateGradInfo(info, r);

    return info;
}


///
/**
 */
std::vector<std::string>
NumarisMosaicHandler::GetStringInfo()
{
    std::vector<std::string> info = SeriesHandler::GetStringInfo();
    info.push_back(std::string("Mosaic rows: ") + itos(GetRows()));
    info.push_back(std::string("Mosaic columns: ") + itos(GetColumns()));
  
    std::string a_order;
    switch (GetAcquisitionOrder()) {
    case ASCENDING :
	a_order = "Ascending";
	break;

    case DESCENDING :
	a_order = "Descending";
	break;

    case INTERLEAVED_ODD :
	a_order = "Interleaved";
	break;

    case OTHER :
	a_order = "Other";
	break;

    case UNKNOWN :
    default:
	a_order = "Unknown";
	break;
    }
    info.push_back(std::string("Acquisition order: ") + a_order);

    return info;
}


///
/**
 */
NumarisMosaicHandler::NumarisMosaicHandler(const std::string& seriesUid)
  : SeriesHandler(seriesUid)
{
}


///
/**
 */
SeriesHandler::VolListType
NumarisMosaicHandler::ReadVolIds(DicomFile& file) 
{
    VolListType v = SeriesHandler::ReadVolIds(file);
    std::string str;
    file.Find("InstanceNumber", str);
    v.front().ids.push_back(str);
    return v;
}


///
/**
 */
AoCode
NumarisMosaicHandler::GetAcquisitionOrder() const
{
    Dictionary* Numaris = Numaris_Dictionary::Instance();
    std::string acquisition_order;
    Find(Numaris->Lookup("ORDER_OF_SLICES"), acquisition_order);
    if (acquisition_order == "ASCENDING") return ASCENDING;
    if (acquisition_order == "DECREASING") return DESCENDING;
    if (acquisition_order == "INTERLEAVED") return INTERLEAVED_ODD;
    return OTHER;
}


///
/**
 */
double
NumarisMosaicHandler::GetVolumeInterval() const
{
    double tr;
    Find("RepetitionTime", tr);
    return tr/1000.0;
}


///
/**
 */
std::vector<double>
NumarisMosaicHandler::GetSliceOrder()
{
    Dictionary* Numaris = Numaris_Dictionary::Instance();
    int n_slices = GetNumberOfSlices();
    std::vector<double> slice_order;
    slice_order.reserve(n_slices);

    double delta_z = GetVoxelSize().back();

    AoCode acquisition_order = GetAcquisitionOrder();

    switch (acquisition_order) {

    case INTERLEAVED_ODD: 
	for (int i = n_slices; i > 0; i-=2)
	    slice_order.push_back(delta_z * i);
	for (int i = (n_slices - 1); i > 0; i-=2)
	    slice_order.push_back(delta_z * i);
        break;
  
    case DESCENDING: 
	for (int i = 1; i <= n_slices; ++i) 
	    slice_order.push_back(i * delta_z);

        break;

    default: 
	for (int i = n_slices; i > 0; --i)
	    slice_order.push_back(i * delta_z);
    }

    return slice_order;
}


///
/**
 */
int
NumarisMosaicHandler::GetRows()
{
    Dictionary* Numaris = Numaris_Dictionary::Instance();
    int mosaic_size = 0;
    Find(Numaris->Lookup("BASE_RAW_MATRIX_SIZE"), mosaic_size);
    return mosaic_size;
}


///
/**
 */
int NumarisMosaicHandler::GetColumns()
{
    return GetRows();
}


///
/**
 */
std::vector<double>
NumarisMosaicHandler::GetVoxelSize()
{
    std::vector<double> voxel_size(3, 0);
  
    std::string s;
    if(Find("PixelSpacing", s)) {
	voxel_size[0] = stof(s, 0);
	voxel_size[1] = stof(s, 1);
    }
  
    // Pixel spacing is reported incorrectly
    // for mosaic files -- must correct
    int mosaic_size = GetRows();
    double image_size;
    Find("ImageColumns", image_size);
    voxel_size[0] *= image_size/mosaic_size;
    Find("ImageRows", image_size);
    voxel_size[1] *= image_size/mosaic_size;
  
    Find("SliceThickness", voxel_size[2]);
    double slice_spacing = 0;
    Find("SliceSpacing", slice_spacing);
    voxel_size[2] += slice_spacing;

    return voxel_size;
}


///
/**
 */
int
NumarisMosaicHandler::GetNumberOfSlices() const
{
    Dictionary* Numaris = Numaris_Dictionary::Instance();
    int slices = 0;
    Find(Numaris->Lookup("NUMBER_OF_SLICES_CURRENT"), slices);
    return slices;
}


///
/**
 */
SyngoMosaicHandler::SyngoMosaicHandler(const std::string& seriesUid)
  : SyngoHandler(seriesUid)
{
}



///
/**
 */
SeriesHandler::VolListType
SyngoMosaicHandler::ReadVolIds(DicomFile& file) 
{
    VolListType v = SyngoHandler::ReadVolIds(file);

    std::string description;
    file.Find("SeriesDescription", description);
    if ((description.find("EvaSeries") != std::string::npos) ||
	(description.find("Superimposed Data") != std::string::npos)) {

	std::string str;
	int in;
	file.Find("InstanceNumber", in);
  
	int width = 4;
	str = itos(in, width);
	v.front().ids.push_back(str);
    }

    return v;
}


///
/**
 */
int
SyngoMosaicHandler::GetRows()
{
    std::string pe_direction;
    Find("PhaseEncodingDirection", pe_direction);
    double fov_rows;
    if (pe_direction == "ROW")
	fov_rows = GetRoFov();
    else
	fov_rows = GetPhaseFov();

    double pixel_size_y = GetVoxelSize()[1];
    return static_cast<int>(floor(fov_rows/pixel_size_y + 0.5));

}


///
/**
 */
int
SyngoMosaicHandler::GetColumns()
{
    std::string pe_direction;
    Find("PhaseEncodingDirection", pe_direction);
    double fov_cols;
    if (pe_direction == "COL")
	fov_cols = GetRoFov();
    else
	fov_cols = GetPhaseFov();

    double pixel_size_x = GetVoxelSize()[0];
    return static_cast<int>(floor(fov_cols/pixel_size_x + 0.5));
}


///
/**
 */
double
SyngoMosaicHandler::GetPhaseFov() const
{
    double phase_fov = 256;
    std::string protocol;
    int err = ReadCSASeriesHeader("MrProtocol", protocol);
    if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);

    if (err > 0) {
	std::string::size_type pos = protocol.find("sSliceArray.asSlice[0].dPhaseFOV");
	std::stringstream ss(protocol.substr(pos, 256));
	ss.ignore(256, '=');
	ss >> phase_fov;
    }
    return phase_fov;
}


///
/**
 */
double
SyngoMosaicHandler::GetRoFov() const
{
    double ro_fov = 256;
    std::string protocol;
    int err = ReadCSASeriesHeader("MrProtocol", protocol);
    if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);

    if (err > 0) {
	std::string::size_type pos = protocol.find("sSliceArray.asSlice[0].dReadoutFOV");
	std::stringstream ss(protocol.substr(pos, 256));
	ss.ignore(256, '=');
	ss >> ro_fov;
    }
    return ro_fov;
}


///
/**
 */
int
SyngoMosaicHandler::GetNumberOfSlices() const
{
    std::string slices;
    int n_slices = 1;

    int err = ReadCSAImageHeader("NumberOfImagesInMosaic", slices);
    if (err == 0)
	wxLogError(_("Unable to find number of slices"));
    else if (err == -1)
	wxLogError(_("Number of slices not provided"));
    else
	n_slices = stoi(slices);
  
    return n_slices;

}


///
/**
 */ 
std::vector<std::string>
SyngoMosaicHandler::GetStringInfo()
{
    std::vector<std::string> info = SeriesHandler::GetStringInfo();
    info.push_back(std::string("Mosaic rows: ") + itos(GetRows()));
    info.push_back(std::string("Mosaic columns: ") + itos(GetColumns()));
  
    std::string a_order;
    switch (GetAcquisitionOrder()) {
    case ASCENDING :
	a_order = "Ascending";
	break;

    case DESCENDING :
	a_order = "Descending";
	break;

    case INTERLEAVED_ODD :
	a_order = "Interleaved Odd";
	break;

    case INTERLEAVED_EVEN :
	a_order = "Interleaved Even";
	break;

    case OTHER :
	a_order = "Other";
	break;

    case UNKNOWN :
    default:
	a_order = "Unknown";
	break;
    }
    info.push_back(std::string("Acquisition order: ") + a_order);


    DicomFile dicomFile((*mFileMap.begin()).first.c_str());
    std::vector<std::string> temp;
    dicomFile.ReadCSAImageHeader("MosaicRefAcqTimes", temp);
    info.insert(info.end(), temp.begin(), temp.end());


    return info;
}

GeDtiRbHandler::GeDtiRbHandler(const std::string& seriesUid)
  : SeriesHandler(seriesUid)
{
}

SeriesHandler::VolListType
GeDtiRbHandler::ReadVolIds(DicomFile& file)
{ 
    VolListType v = SeriesHandler::ReadVolIds(file);
    // Can't call default ReadVolId because of problem with trigger time.

    std::string acquisition_number;
    if (!file.Find("AcquisitionNumber", acquisition_number))
	wxLogError(_("Acquisition number not found"));
    v.front().ids.push_back(acquisition_number);

    int image_number, number_of_slices;
    if (!file.Find("InstanceNumber", image_number))
	wxLogError(_("Instance number not found"));

    Dictionary* Excite = Excite_Dictionary::Instance();
    if (!file.Find(Excite->Lookup("Locations_in_acquisition"), number_of_slices))
	wxLogError(_("Locations_in_acquisition not found"));

    int vol_no = static_cast<int>(floor(static_cast<double>((image_number - 1))/number_of_slices) + 1);

    v.front().ids.push_back(itos(vol_no, 3));

    return v;

}

GradientInfo
GeDtiRbHandler::GetGradientInfo()
{

    GradientInfo info;

    int gradfileno;
    Dictionary* Excite = Excite_Dictionary::Instance();

    double bvalue;
    Find(Excite->Lookup("User_data_9"), bvalue);

    if (Find(Excite->Lookup("User_data_11"), gradfileno)) {
	wxString gradfilename = wxString::Format(_T("dwepi.%d.grads"), gradfileno);
    
	wxFileName gradfile(wxStandardPaths::Get().GetDataDir());
	gradfile.SetFullName(gradfilename);

	std::ifstream input(gradfile.GetFullPath().mb_str(wxConvLocal));

	double value;
	input >> value;
	while (input.good()) {
	    info.xGrads.push_back(value);
	    input >> value;
	    info.yGrads.push_back(value);
	    input >> value;
	    info.zGrads.push_back(value);
	    info.values.push_back(bvalue);
	    input >> value;
	}

	// GE -- do not correct
	// assuming one orientation in series
	//std::vector<double> r = orientations.at(0);
	//RotateGradInfo(info, r);

    }

    return info;
}

GeEpiHandler::GeEpiHandler(const std::string& seriesUid)
  : SeriesHandler(seriesUid)
{
}

SeriesHandler::VolListType
GeEpiHandler::ReadVolIds(DicomFile& file)
{ 
    VolListType v = SeriesHandler::ReadVolIds(file);

    int image_number, number_of_slices;

    if (!file.Find("InstanceNumber", image_number))
	wxLogError(_("Instance number not found"));

    Dictionary* Excite = Excite_Dictionary::Instance();
    if (!file.Find(Excite->Lookup("Locations_in_acquisition"), number_of_slices))
	wxLogError(_("Locations_in_acquisition not found"));

    int vol_no = static_cast<int>(floor(static_cast<double>((image_number - 1))
					/number_of_slices) + 1);

    v.front().ids.push_back(itos(vol_no, 3));

    return v;

}

GeDti2Handler::GeDti2Handler(const std::string& seriesUid)
: GeEpiHandler(seriesUid)
{
}

SeriesHandler::VolListType
GeDti2Handler::ReadVolIds(DicomFile& file)
{ 
    VolListType v = SeriesHandler::ReadVolIds(file);
    Dictionary* Excite = Excite_Dictionary::Instance();

    int image_number, number_of_slices;

    if (!file.Find("InstanceNumber", image_number))
	wxLogError(_("Instance number not found"));

    if (!file.Find(Excite->Lookup("Locations_in_acquisition"), number_of_slices))
	wxLogError(_("Locations_in_acquisition not found"));

    int vol_no = static_cast<int>(floor(static_cast<double>((image_number - 1))
					/number_of_slices) + 1);

    v.front().ids.push_back(itos(vol_no, 3));

    std::vector<double> g_value;
    std::string b_direction;
    double value;
    file.Find(Excite->Lookup("User_data_20"), value);
    g_value.push_back(value);

    file.Find(Excite->Lookup("User_data_21"), value);
    g_value.push_back(value);

    file.Find(Excite->Lookup("User_data_22"), value);
    g_value.push_back(value);

    int g_number;
    std::vector<std::vector <double> >::iterator pos;
    pos = find(gradients.begin(), gradients.end(), g_value);
    if (pos != gradients.end()) {
	g_number = distance(gradients.begin(), pos);
    }
    else {
	g_number = gradients.size();
	gradients.push_back(g_value);
    }

    v.front().ids.push_back(itos(g_number, 3));

    return v;

}


///
/**
 */
GradientInfo
GeDti2Handler::GetGradientInfo()
{

    GradientInfo info;

    Dictionary* Excite = Excite_Dictionary::Instance();

    std::map<VolId, std::string> volFileMap = GetVolIdsAndFiles();
    std::map<VolId, std::string>::iterator it = volFileMap.begin();
    std::map<VolId, std::string>::iterator it_end = volFileMap.end();

    while (it != it_end) {

	//std::vector<std::string> test = it->first.ids;
	//for (int t = 0; t < test.size(); ++t)
	//  wxLogMessage(test[t].c_str());
    
	info.values.push_back(0); // bvalue not found in these files.

	DicomFile file(it->second.c_str());

	double value;
	int err = file.Find(Excite->Lookup("User_data_20"), value);
	if (err > 0) info.xGrads.push_back(value);

	err = file.Find(Excite->Lookup("User_data_21"), value);
	if (err > 0) info.yGrads.push_back(value);

	err = file.Find(Excite->Lookup("User_data_22"), value);
	if (err > 0) info.zGrads.push_back(value);

	++it;
    }

    // GE -- do not correct!
    //  std::vector<double> r = GetRotationMatrix(volFileMap.begin()->first);
    //  RotateGradInfo(info, r);

    return info;
}


///
/**
 */
AchievaDtiHandler::AchievaDtiHandler(const std::string& seriesUid)
  : SeriesHandler(seriesUid)
{
}


///
/**
 */
SeriesHandler::VolListType
AchievaDtiHandler::ReadVolIds(DicomFile& file)
{ 
    VolListType v = SeriesHandler::ReadVolIds(file);
    Dictionary* Achieva = PhilipsMr_Dictionary::Instance();

    std::vector<double> g_value;
    std::string b_direction;

    double value;
    file.Find(Achieva->Lookup("Diffusion_B-Factor"), value);
    g_value.push_back(value);

    file.Find(Achieva->Lookup("Diffusion_Direction_RL"), value);
    g_value.push_back(value);

    file.Find(Achieva->Lookup("Diffusion_Direction_AP"), value);
    g_value.push_back(value);

    file.Find(Achieva->Lookup("Diffusion_Direction_FH"), value);
    g_value.push_back(value);

    int g_number;
    std::vector<std::vector <double> >::iterator pos;
    pos = find(gradients.begin(), gradients.end(), g_value);
    if (pos != gradients.end()) {
	g_number = distance(gradients.begin(), pos);
    }
    else {
	g_number = gradients.size();
	gradients.push_back(g_value);
    }

    v.front().ids.push_back(itos(g_number, 3));

    return v;

}


///
/**
 */
GradientInfo
AchievaDtiHandler::GetGradientInfo()
{

    GradientInfo info;

    Dictionary* Achieva = PhilipsMr_Dictionary::Instance();

    std::map<VolId, std::string> volFileMap = GetVolIdsAndFiles();
    std::map<VolId, std::string>::iterator it = volFileMap.begin();
    std::map<VolId, std::string>::iterator it_end = volFileMap.end();

    while (it != it_end) {

	//std::vector<std::string> test = it->first.ids;
	//for (int t = 0; t < test.size(); ++t)
	//  wxLogMessage(test[t].c_str());

	DicomFile file(it->second.c_str());

	double value;
	file.Find(Achieva->Lookup("Diffusion_B-Factor"), value);
	info.values.push_back(value);

	file.Find(Achieva->Lookup("Diffusion_Direction_RL"), value);
	info.xGrads.push_back(value);

	file.Find(Achieva->Lookup("Diffusion_Direction_AP"), value);
	info.yGrads.push_back(value);

	file.Find(Achieva->Lookup("Diffusion_Direction_FH"), value);
	info.zGrads.push_back(value);

	++it;
    }

    std::vector<double> r = GetRotationMatrix(volFileMap.begin()->first);

    RotateGradInfo(info, r);
    return info;
}


///
/**
 */
void 
jcs::Normalize(GradientInfo& info)
{
    std::vector<double> norms;
    for (unsigned int i = 0; i < info.xGrads.size(); ++i) {
	double norm = sqrt( pow(info.xGrads.at(i), 2) +
			    pow(info.yGrads.at(i), 2) +
			    pow(info.zGrads.at(i), 2) );
	//    norm = sqrt(norm);

	norms.push_back(norm);
    }
    for (unsigned int i = 0; i < info.xGrads.size(); ++i) {
	if ((norms.at(i) != 1.0) && (norms.at(i) != 0)) {
	    info.xGrads.at(i) /= norms.at(i);
	    info.yGrads.at(i) /= norms.at(i);
	    info.zGrads.at(i) /= norms.at(i);
  
	}
    }
}


///
/**
 */
// nb - inverse(r)*info.xGrads
void
jcs::RotateGradInfo(GradientInfo& info, std::vector<double>& r)
{
    std::vector<double> temp (3,0);
    for (unsigned int i = 0; i < info.xGrads.size(); ++i) {
	temp[0] = r[0]*info.xGrads[i] + r[1]*info.yGrads[i] + r[2]*info.zGrads[i];
	temp[1] = r[3]*info.xGrads[i] + r[4]*info.yGrads[i] + r[5]*info.zGrads[i];
	temp[2] = r[6]*info.xGrads[i] + r[7]*info.yGrads[i] + r[8]*info.zGrads[i];

	info.xGrads[i] = temp[0];
	info.yGrads[i] = temp[1];
	info.zGrads[i] = temp[2];
    }

    return;
}


///
/**
 */
EnhancedMrHandler::EnhancedMrHandler(const std::string& seriesUid)
  : SeriesHandler(seriesUid)
{
}


///
/**
 */
std::string
EnhancedMrHandler::GetImagePositionPatient(DicomFile& dfile, int frame)
{
    return ipps.at(frame);
}


///
/**
 */
int
EnhancedMrHandler::GetRescaleSlopeAndIntercept(DicomFile& dfile, double& slope, double& intercept, int frame) const
{
    // use a private std::vector
    // if it's empty, fill it on the first call
    // or fill it in getvolids

    slope = slopes.at(frame);
    intercept = intercepts.at(frame);

    return 1;
}


///
/**
 */
std::vector<std::string>
EnhancedMrHandler::GetStringInfo()
{
    std::vector<std::string> info = SeriesHandler::GetStringInfo();
    return info;
}


///
/**
 */
SeriesHandler::VolListType 
EnhancedMrHandler::ReadVolIds(DicomFile& file)
{
    std::string imageType;
    file.Find("ImageType", imageType);
    bool isdti = (imageType.find("DIFFUSION") != std::string::npos);

    VolListType v;

    std::string sequence;
    std::vector<std::string> dips;
    file.Find("DimensionIndexSequence", sequence);
    file.FindInSequence(sequence, "DimensionIndexPointer", dips);

    unsigned int n_indices = dips.size();

    sequence.clear();
    std::vector<std::string> fc_sequence;
    std::vector<std::string> po_sequence;
    std::vector<std::string> pp_sequence;
    std::vector<std::string> diff_sequence;
    std::vector<std::string> pm_sequence;
    std::vector<std::string> pvt_sequence;

    // This is a type 2 element. Could probably comment this out?
    if (file.Find("SharedFunctionalGroupsSequence", sequence)) {
	file.FindInSequence(sequence, "PlaneOrientationSequence", po_sequence);
	file.FindInSequence(sequence, "PlanePositionSequence", pp_sequence);
    }

    sequence.clear();

    if (!file.Find("PerFrameFunctionalGroupsSequence", sequence)) {
	wxLogError(_("Unable to find PerFrameFunctionalGroupsSequence"));
	return v;
    }

  
    if (!file.FindInSequence(sequence, "FrameContentSequence", fc_sequence)) {
	wxLogError(_("Unable to find FrameContentSequence"));
	return v;
    }

    if (po_sequence.size() == 0) {
	if (!file.FindInSequence(sequence, "PlaneOrientationSequence", po_sequence)) {
	    wxLogError(_("Unable to find PlaneOrientationSequence"));
	    return v;
	}
    }

    if (pp_sequence.size() == 0) {
	if (!file.FindInSequence(sequence, "PlanePositionSequence", pp_sequence)) {
	    wxLogError(_("Unable to find PlanePositionSequence"));
	    return v;
	}
    }
    file.FindInSequence(sequence, "PixelMeasuresSequence", pm_sequence);

    if (isdti) 
	file.FindInSequence(sequence, "MrDiffusionSequence", diff_sequence);

    int pvt_found = file.FindInSequence(sequence, "PixelValueTransformationSequence", pvt_sequence);

    std::vector<std::string>::iterator fc_it = fc_sequence.begin();
    std::vector<std::string>::iterator po_it = po_sequence.begin();
    std::vector<std::string>::iterator pp_it = pp_sequence.begin();
    std::vector<std::string>::iterator diff_it = diff_sequence.begin();
    std::vector<std::string>::iterator pvt_it = pvt_sequence.begin();

    std::vector<std::vector <double> > rotations;
    std::vector<std::vector <int> > divs(n_indices);
    std::vector<int> inStackPositions;
    std::vector<std::string> bvecs;
    std::vector<std::string> bvals;

    // Assumes pixel spacing same in every frame -- not, strictly speaking,
    // a safe assumption
    std::vector<std::string> vals;
    if (!pm_sequence.empty()) {
	file.FindInSequence(pm_sequence.front(), "PixelSpacing", vals);
	if (!vals.empty()) pixelSpacing = vals.front();
    }

    while (fc_it != fc_sequence.end()) {

	vals.clear();
	if (file.FindInSequence(*fc_it, "DimensionIndexValues", vals)) {
	    std::cout << "found DimensionIndexValues" << std::endl;
	    for (unsigned int i = 0; i < n_indices; ++i) 
		divs.at(i).push_back(stoi(vals.front(), i));
	    vals.clear();
	}

	if (file.FindInSequence(*fc_it, "InStackPositionNumber", vals)) {
	    inStackPositions.push_back(stoi(vals.front()));
	    vals.clear();
	}
	else inStackPositions.push_back(0);

	double rescale_val;
	if (pvt_found && file.FindInSequence(*pvt_it, "RescaleIntercept", vals)) {
	    intercepts.push_back(stof(vals.front()));
	    vals.clear();
	}
	else if (file.Find("RescaleIntercept", rescale_val))
	    intercepts.push_back(rescale_val);
	else intercepts.push_back(0);

	if (pvt_found && file.FindInSequence(*pvt_it, "RescaleSlope", vals)) {
	    slopes.push_back(stof(vals.front()));
	    vals.clear();
	}
	else if (file.Find("RescaleSlope", rescale_val))
	    intercepts.push_back(rescale_val);
	else slopes.push_back(1);

	if (file.FindInSequence(*pp_it, "ImagePositionPatient", vals))
	    ipps.push_back(vals.front());
	else ipps.push_back("0/0/0");
	vals.clear();

	if (!file.FindInSequence(*po_it, "ImageOrientationPatient", vals))
	    vals.push_back("1/0/0/0/1/0");
	std::vector<double> rotation;
	rotation.resize(9, 0);

	for (int i = 0; i < 6; ++i) {
	    double f = stof(vals.front(), i);
	    // round off
	    rotation[i] = static_cast<double>(static_cast<int>(f * 100000 + 0.5)) / 100000;
	}

	rotation[6] = rotation[1]*rotation[5] - rotation[2]*rotation[4];
	rotation[7] = rotation[2]*rotation[3] - rotation[0]*rotation[5];
	rotation[8] = rotation[0]*rotation[4] - rotation[1]*rotation[3];

	rotations.push_back(rotation);

	if (isdti && diff_sequence.size() != 0) {
	    vals.clear();
	    file.FindInSequence(*diff_it, "DiffusionBValue", vals);
	    if (vals.size() != 0) bvals.push_back(vals.front());
	    else bvals.push_back("-1");
	    vals.clear();
	    file.FindInSequence(*diff_it, "DiffusionDirectionality", vals);
	    if (vals.front() == "DIRECTIONAL") {
		vals.clear();
		file.FindInSequence(*diff_it, "DiffusionGradientDirectionSequence", vals);
		std::vector<std::string> dgo;
		file.FindInSequence(vals.front(), "DiffusionGradientOrientation", dgo);
		bvecs.push_back(dgo.front());
	    }
	    else bvecs.push_back("0/0/0");
	    ++diff_it;
	};

	++fc_it;
	if (po_sequence.size() == fc_sequence.size()) ++po_it;
	if (pp_sequence.size() == fc_sequence.size()) ++pp_it;
	wxTheApp->Yield();
    }

    std::vector<bool> use_index;
    for (unsigned int i = 0; i < divs.size(); ++i) {
	// Don't use if indices simply reflect inStackPositions
	bool use = divs.at(i) != inStackPositions;
	// or if they just count up from 1 to nslices
	unsigned int j = 0;
	bool match = use;
	while (match && j < divs.at(i).size()) {
	    match = (divs.at(i).at(j) == j+1);
	    ++j;
	}
	use &= !match;
	use_index.push_back(use);

    }

    for (unsigned int i = 0; i < rotations.size(); ++i) {
	VolId info;
	info.ids.push_back(GetSeriesUid());

	for (unsigned int j = 0; j < divs.size(); ++j) 
	    if (use_index.at(j)) info.ids.push_back(itos(divs.at(j).at(i), 3));

	std::vector<std::vector <double> >::iterator pos;
	pos = find(orientations.begin(), orientations.end(), rotations.at(i));

	if (pos != orientations.end()) {
	    info.orientNumber = distance(orientations.begin(), pos);
	}
	else {
	    info.orientNumber = orientations.size();
	    orientations.push_back(rotations.at(i));
	}


	v.push_back(info);
	if (isdti && diff_sequence.size() != 0) {
	    bvalMap[info] = bvals.at(i);
	    bvecMap[info] = bvecs.at(i);
	}

	wxTheApp->Yield();
    }
    return v;
}


///
/**
 */
GradientInfo 
EnhancedMrHandler::GetGradientInfo()
{
    GradientInfo info;

    set<VolId> vols = GetVolIds();
    std::vector<double> r = GetRotationMatrix(*vols.begin());

    for (set<VolId>::iterator it = vols.begin(); it != vols.end(); ++it) {
	info.values.push_back(stof(bvalMap[*it]));
	info.xGrads.push_back(stof(bvecMap[*it], 0));
	info.yGrads.push_back(stof(bvecMap[*it], 1));
	info.zGrads.push_back(stof(bvecMap[*it], 2));
    };

    wxLogWarning(_("Warning: bvecs have NOT been verified as correct for enhanced MR DICOM files. If you would like to help with this, please email jolinda@uoregon.edu."));

    RotateGradInfo(info, r);

    return info;
}


///
/**
 */
bool 
EnhancedMrHandler::IsDti() const 
{
    std::string imageType;
    Find("ImageType", imageType);
    if (imageType.find("DIFFUSION") != std::string::npos)
	return true;
    else return false;
}
  

///
/**
 */
std::vector<double> 
EnhancedMrHandler::GetVoxelSize() 
{
    std::vector<double> voxel_size(3, 0);

    if (pixelSpacing.empty()) Find("PixelSpacing", pixelSpacing);
    if (!pixelSpacing.empty()) {
	voxel_size[0] = stof(pixelSpacing, 0);
	voxel_size[1] = stof(pixelSpacing, 1);
    }
    Find("SliceSpacing", voxel_size[2]);
    if (voxel_size[2] == 0)
	Find("SliceThickness", voxel_size[2]);
    
    return voxel_size;
}


///
/**
 */
double
SyngoHandler::GetVolumeInterval() const
{
    std::map<VolId, std::vector<double> > vtmap;
    FileMapType::const_iterator it = mFileMap.begin();
    while (it != mFileMap.end()) { 
	DicomFile file(it->first.c_str());
	std::string str;
	int err = file.ReadCSAImageHeader("TimeAfterStart", str);
	if (err <= 0) return 0;
	double time = stof(str);
	vtmap[it->second.front()].push_back(time);
	++it;
    }

    if (vtmap.size() <= 1) return 0;

    std::vector<double> vol_times;
    std::map<VolId, std::vector<double> >::iterator vit = vtmap.begin();
    while (vit != vtmap.end()) { 
	vol_times.push_back(*min_element(vit->second.begin(), vit->second.end()));
	++vit;
    }

    sort(vol_times.begin(), vol_times.end());

    return (vol_times[1] - vol_times[0]);

}
