#include <wx/wxprec.h>
#include <wx/spinctrl.h>
#include <wx/statline.h>

#include <string>

#include "OptionsDialogs.h"
#include "OutputterBase.h"
#include "OutputFactory.h"

#include "NewSpmOutputter.h"
#include "NewMetaOutputter.h"
#include "NewBvOutputter.h"
#include "NiftiOutputter.h"
#include "AnalyzeOutputter.h"

using namespace std;
using namespace jcs;

int 
jcs::ShowOptionsDlg(int type, OutputterBase* outputter)
{
  bool rebuild = 0;

  switch (type) {

  case OutputFactory::SPM : {
    SpmOptionsDlg* dlg = new SpmOptionsDlg(dynamic_cast<NewSpmOutputter*>(outputter));
    if (dlg->ShowModal() == wxID_OK) rebuild = dlg->NeedsRebuild();
    dlg->Destroy();}
    break;

  case OutputFactory::META : {
    MetaOptionsDlg* dlg = new MetaOptionsDlg(dynamic_cast<NewMetaOutputter*>(outputter));
    if (dlg->ShowModal() == wxID_OK) rebuild = dlg->NeedsRebuild();
    dlg->Destroy(); }
    break;

  case OutputFactory::NIFTI : {
    NiftiOptionsDlg* dlg = new NiftiOptionsDlg(dynamic_cast<NiftiOutputterBase*>(outputter));
    if (dlg->ShowModal() == wxID_OK) rebuild = dlg->NeedsRebuild();
    dlg->Destroy(); }
    break;

  case OutputFactory::ANALYZE : {
    AnalyzeOptionsDlg* dlg = new AnalyzeOptionsDlg(dynamic_cast<AnalyzeOutputter*>(outputter));
    if (dlg->ShowModal() == wxID_OK) rebuild = dlg->NeedsRebuild();
    dlg->Destroy(); }
    break;

  case OutputFactory::FSL : {
    NiftiOptionsDlg* dlg = new NiftiOptionsDlg(dynamic_cast<NiftiOutputterBase*>(outputter));
    if (dlg->ShowModal() == wxID_OK) rebuild = dlg->NeedsRebuild();
    dlg->Destroy(); }
    break;

  case OutputFactory::BV : {
    BvOptionsDlg* dlg = new BvOptionsDlg(dynamic_cast<NewBvOutputter*>(outputter));
    if (dlg->ShowModal() == wxID_OK) rebuild = dlg->NeedsRebuild();
    dlg->Destroy(); }
    break;

  default : {
    NoOptionsDlg* dlg = new NoOptionsDlg();
    dlg->ShowModal();
    dlg->Destroy(); }
  }

  return static_cast<int>(rebuild);
}



BEGIN_EVENT_TABLE(NoOptionsDlg, wxDialog)

EVT_BUTTON(wxID_OK, NoOptionsDlg::OnOkay)

END_EVENT_TABLE()

///
/** We should never see this dialog, as all output formats have options.
 */
NoOptionsDlg::NoOptionsDlg()
: wxDialog(NULL, -1, wxString(_("Output format options")))
{
  wxBoxSizer* dlgSizer = new wxBoxSizer(wxVERTICAL);

  dlgSizer->Add(new wxStaticText(this, -1, 
				 _("No options exist for this format.")), 0, wxALL, 10);

  dlgSizer->Add(new wxButton(this, wxID_OK, _("Okay")), 
		0, wxALIGN_RIGHT|wxALL, 10);

  SetSizer(dlgSizer);
  dlgSizer->Fit(this);

}

void
NoOptionsDlg::OnOkay(wxCommandEvent& event)
{
  EndModal(event.GetId());
}


/// A basic panel for our Options dialog.
/**
 */
BasicOptionsDlg::BasicPanel::BasicPanel(wxWindow* parent) 
  : wxPanel(parent)
{
  wxBoxSizer* panelSizer = new wxBoxSizer(wxVERTICAL);

  panelSizer->Add(new wxStaticText(this, -1, _("Items for default file names")),
		  0, wxTOP|wxLEFT|wxRIGHT, 10);

  mpNameFields = new wxCheckListBox(this, -1);

  panelSizer->Add(mpNameFields, 0, wxEXPAND|wxALL, 10);

  mpSplitDirsCheck = new wxCheckBox(this, -1, 
				    _("Save each series in separate directory"));
  panelSizer->Add(mpSplitDirsCheck, 0, wxALL, 10);

  mpSplitSubjCheck = new wxCheckBox(this, -1, 
				    _("Save each subject in separate directory"));
  panelSizer->Add(mpSplitSubjCheck, 0, wxALL, 10);

  mpDimCheck = new wxCheckBox(this, -1,
			      _("Save multivolume series as 4D files"));
  panelSizer->Add(mpDimCheck, 0, wxLEFT|wxRIGHT|wxBOTTOM, 10);

  panelSizer->Add(new wxStaticText(this, -1,
				   _("Skip volumes for multivolume series:")), 
		  0, wxLEFT|wxRIGHT|wxBOTTOM, 10);
  mpSkipSpin = new wxSpinCtrl(this, -1, _T("0"));
  panelSizer->Add(mpSkipSpin, 0, wxLEFT|wxRIGHT|wxBOTTOM, 10);

  mpRescaleCheck = new wxCheckBox(this, -1,
				  _("Apply rescale slope and intercept to data"));
  panelSizer->Add(mpRescaleCheck, 0, wxALL, 10);

  SetSizer(panelSizer);
  panelSizer->Fit(this);

}


BasicOptionsDlg::BasicOptionsDlg(const wxString& title, Basic3DOutputter* outputter)
  : wxDialog(NULL, -1, title), mOutputter(outputter)
{
  myPanel = new BasicPanel(this);
  myPanel->mpSplitDirsCheck->SetValue(mOutputter->GetSplit());
  myPanel->mpSplitSubjCheck->SetValue(mOutputter->GetSplitSubj());
  myPanel->mpDimCheck->SetValue(mOutputter->GetDimensionality() == 4);
  myPanel->mpSkipSpin->SetValue(mOutputter->GetSkip());
  myPanel->mpRescaleCheck->SetValue(mOutputter->GetRescale());

  OutputterBase::FieldMap::iterator it = mOutputter->defaultNameFields.begin();
  OutputterBase::FieldMap::iterator it_end = mOutputter->defaultNameFields.end();
  for (int i = 0; it != it_end; it++, i++) {
    myPanel->mpNameFields->Append(wxString(it->second.name.c_str(), wxConvLocal));
    myPanel->mpNameFields->Check(i, it->second.value);
  }

  // default min size has extra white space vertically
  wxSize boxSize = myPanel->mpNameFields->GetEffectiveMinSize();
  boxSize.SetHeight(boxSize.GetHeight()*3/4);
  myPanel->mpNameFields->SetMinSize(boxSize);

}

bool
BasicOptionsDlg::SplitDirs() const 
{ return myPanel->mpSplitDirsCheck->GetValue(); }

bool
BasicOptionsDlg::SplitSubj() const 
{ return myPanel->mpSplitSubjCheck->GetValue(); }

int
BasicOptionsDlg::Dimensionality() const 
{ return (myPanel->mpDimCheck->GetValue()) ? 4 : 3; }


int
BasicOptionsDlg::Skip() const 
{ return myPanel->mpSkipSpin->GetValue(); }

bool
BasicOptionsDlg::Rescale() const
{ return myPanel->mpRescaleCheck->GetValue(); }

///
/** All sub-classes of BasicOptionsDlg want these actions done.
 */
void
BasicOptionsDlg::OnOkay(wxCommandEvent& event)
{
  mOutputter->SetSplit (SplitDirs());
  mOutputter->SetSplitSubj (SplitSubj());
  mOutputter->SetDimensionality (Dimensionality());
  mOutputter->SetSkip (Skip());
  mOutputter->SetRescale (Rescale());
}

bool
BasicOptionsDlg::Rebuild() const
{
  return ((SplitDirs()      != mOutputter->GetSplit()) || 
	  (SplitSubj()      != mOutputter->GetSplitSubj()) ||
	  (Dimensionality() != mOutputter->GetDimensionality()) || 
	  (Skip()           != mOutputter->GetSkip()));
}

bool
BasicOptionsDlg::SaveNameFields()
{
  bool needsRebuild = false;
  for (unsigned int i = 0; i < myPanel->mpNameFields->GetCount(); ++i) {
    string name(myPanel->mpNameFields->GetString(i).mb_str(wxConvLocal));
    bool value = myPanel->mpNameFields->IsChecked(i);
    OutputterBase::FieldMap::iterator it = mOutputter->defaultNameFields.begin();
    OutputterBase::FieldMap::iterator it_end = mOutputter->defaultNameFields.end();
    while (it != it_end) {
      if (it->second.name == name) {
        if (it->second.value != value) needsRebuild = true;
        it->second.value = value;
        break;
      }
      ++it;
    }
  }
  return needsRebuild;
}

BEGIN_EVENT_TABLE(AnalyzeOptionsDlg, wxDialog)

EVT_BUTTON(wxID_OK, AnalyzeOptionsDlg::OnOkay)

END_EVENT_TABLE()

AnalyzeOptionsDlg::AnalyzeOptionsDlg(AnalyzeOutputter* outputter)
: BasicOptionsDlg(_("Analyze 7.5 options"), outputter), 
  mOutputter(outputter)
{
  wxBoxSizer* dlgSizer = new wxBoxSizer(wxVERTICAL);

  dlgSizer->Add(myPanel, 0, wxEXPAND);

  wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);

  wxButton* okayButton = new wxButton(this, wxID_OK, _("Okay"));
  buttonSizer->Add(okayButton, 0, wxRIGHT, 10);
  buttonSizer->Add(new wxButton(this, wxID_CANCEL, _("Cancel")));
  okayButton->SetDefault();

  dlgSizer->Add(buttonSizer, 0, wxALIGN_RIGHT|wxALL, 10);

  SetSizer(dlgSizer);
  dlgSizer->Fit(this);

  mNeedsRebuild = false;

}

void
AnalyzeOptionsDlg::OnOkay(wxCommandEvent& event)
{
  mNeedsRebuild = Rebuild() || SaveNameFields();
  BasicOptionsDlg::OnOkay(event);
  EndModal(event.GetId());
}



BEGIN_EVENT_TABLE(SpmOptionsDlg, wxDialog)

EVT_BUTTON(wxID_OK, SpmOptionsDlg::OnOkay)

END_EVENT_TABLE()


SpmOptionsDlg::SpmOptionsDlg(NewSpmOutputter* outputter)
: BasicOptionsDlg(_("SPM Analyze options"), outputter), 
  mOutputter(outputter)
{
  wxBoxSizer* dlgSizer = new wxBoxSizer(wxVERTICAL);

  dlgSizer->Add(myPanel);

  wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);

  wxButton* okayButton = new wxButton(this, wxID_OK, _("Okay"));
  buttonSizer->Add(okayButton, 0, wxRIGHT, 10);
  buttonSizer->Add(new wxButton(this, wxID_CANCEL, _("Cancel")));
  okayButton->SetDefault();

  dlgSizer->Add(buttonSizer, 0, wxALIGN_RIGHT|wxALL, 10);

  SetSizer(dlgSizer);
  dlgSizer->Fit(this);

  mNeedsRebuild = false;

}

void
SpmOptionsDlg::OnOkay(wxCommandEvent& event)
{
  mNeedsRebuild = Rebuild() || SaveNameFields();
  BasicOptionsDlg::OnOkay(event);
  EndModal(event.GetId());
}




BEGIN_EVENT_TABLE(NiftiOptionsDlg, wxDialog)

EVT_BUTTON(wxID_OK, NiftiOptionsDlg::OnOkay)

END_EVENT_TABLE()


NiftiOptionsDlg::NiftiOptionsDlg(NiftiOutputterBase* outputter)
: BasicOptionsDlg(_("Nifti options"), outputter),
  mOutputter(outputter)
{
  wxBoxSizer* dlgSizer = new wxBoxSizer(wxVERTICAL);

  dlgSizer->Add(myPanel);

  mpNiiCheck = new wxCheckBox(this, -1, _("Save as .nii file"));
  mpNiiCheck->SetValue(mOutputter->saveNii);
  dlgSizer->Add(mpNiiCheck, 0, wxLEFT|wxRIGHT|wxBOTTOM, 10);

  wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);

  wxButton* okayButton = new wxButton(this, wxID_OK, _("Okay"));
  buttonSizer->Add(okayButton, 0, wxRIGHT, 10);
  buttonSizer->Add(new wxButton(this, wxID_CANCEL, _("Cancel")));
  okayButton->SetDefault();

  dlgSizer->Add(buttonSizer, 0, wxALIGN_RIGHT|wxALL, 10);

  SetSizer(dlgSizer);
  dlgSizer->Fit(this);

  mNeedsRebuild = false;
}

void
NiftiOptionsDlg::OnOkay(wxCommandEvent& event)
{
  mNeedsRebuild = Rebuild() || (SaveNii() != mOutputter->saveNii) ||
    SaveNameFields();
  BasicOptionsDlg::OnOkay(event);
  mOutputter->SetSaveNii(SaveNii());
  EndModal(event.GetId());
}



BEGIN_EVENT_TABLE(MetaOptionsDlg, wxDialog)

EVT_BUTTON(wxID_OK, MetaOptionsDlg::OnOkay)
EVT_BUTTON(wxID_CLEAR, MetaOptionsDlg::OnClear)

END_EVENT_TABLE()

MetaOptionsDlg::MetaOptionsDlg(NewMetaOutputter* outputter)
: BasicOptionsDlg(_("MetaImage options"), outputter),
  mOutputter(outputter)
{
  wxBoxSizer* dlgSizer = new wxBoxSizer(wxVERTICAL);

  dlgSizer->Add(myPanel);

  wxString check_text(_("Save header only "));
  mpHeaderOnlyCheck = new wxCheckBox(this, -1, check_text);
  mpHeaderOnlyCheck->SetValue(mOutputter->SaveHeaderOnly());
  dlgSizer->Add(mpHeaderOnlyCheck, 0, wxALL, 10);

  dlgSizer->Add(new wxStaticText(this, -1, _("Add fields")), 0, wxTOP|wxLEFT, 10);
  mpCheckList = new wxCheckListBox(this, -1);

  Dictionary* dicom = Dicom_Dictionary::Instance();
  vector<string> tagList = dicom->TagList();
  for (unsigned int i = 0; i < tagList.size(); ++i) {
    mAddTag(tagList.at(i));
  }

  mpCheckList->SetSize(mpCheckList->GetBestSize());
  dlgSizer->Add(mpCheckList, 1, wxEXPAND|wxALL, 10);

  wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);

  buttonSizer->Add(new wxButton(this, wxID_CLEAR, _("Clear fields")), 0, wxRIGHT, 10);
  wxButton* okayButton = new wxButton(this, wxID_OK, _("Okay"));
  buttonSizer->Add(okayButton, 0, wxRIGHT, 10);
  buttonSizer->Add(new wxButton(this, wxID_CANCEL, _("Cancel")));
  okayButton->SetDefault();

  dlgSizer->Add(buttonSizer, 0, wxALIGN_RIGHT|wxALL, 10);

  SetSizer(dlgSizer);
  dlgSizer->Fit(this);

  mNeedsRebuild = false;
}

void
MetaOptionsDlg::SetFields(vector<string> fields)
{
  for (unsigned int i = 0; i < fields.size(); ++i) {
    mCheckItem(fields.at(i));
  }

}

std::vector<std::string> 
MetaOptionsDlg::GetFields() const
{
  vector<string> fields;
  int n_items = mpCheckList->GetCount();
  for (int i = 0; i < n_items; ++i) {
    if (mpCheckList->IsChecked(i))
      fields.push_back(static_cast<const char*>(mpCheckList->GetString(i).mb_str(wxConvLocal)));
  }

  return fields;

}


int
MetaOptionsDlg::mAddTag(const string tag)
{
  if (tag.compare(0, 2, "Mf"))  // ignore metafile elements
    mpCheckList->Append(wxString(tag.c_str(), wxConvLocal));
  return 1;
}

int
MetaOptionsDlg::mCheckItem(const string item)
{
  int index = mpCheckList->FindString(wxString(item.c_str(), wxConvLocal));
  mpCheckList->Check(index);
  return 1;
}


void
MetaOptionsDlg::OnClear(wxCommandEvent& event)
{
  int n_items = mpCheckList->GetCount();
  for (int i = 0; i < n_items; ++i)
    mpCheckList->Check(i, false);
}

void
MetaOptionsDlg::OnOkay(wxCommandEvent& event)
{
  mNeedsRebuild = Rebuild() || (SaveHeaderOnly() != mOutputter->SaveHeaderOnly()) ||
    SaveNameFields();
  BasicOptionsDlg::OnOkay(event);
  mOutputter->SetSaveHeader(SaveHeaderOnly());
  mOutputter->fields = GetFields();

  EndModal(event.GetId());

}


BEGIN_EVENT_TABLE(BvOptionsDlg, wxDialog)

EVT_BUTTON(wxID_OK, BvOptionsDlg::OnOkay)

END_EVENT_TABLE()


BvOptionsDlg::BvOptionsDlg(NewBvOutputter* outputter)
: wxDialog(NULL, -1, wxString(_("BrainVoyager options"))), 
  mOutputter(outputter)
{
  wxBoxSizer* dlgSizer = new wxBoxSizer(wxVERTICAL);

  dlgSizer->Add(new wxStaticText(this, -1, _("Items for default file names")),
		0, wxTOP|wxLEFT|wxRIGHT, 10);

  mpNameFields = new wxCheckListBox(this, -1);

  OutputterBase::FieldMap::iterator it = mOutputter->defaultNameFields.begin();
  OutputterBase::FieldMap::iterator it_end = mOutputter->defaultNameFields.end();
  int i = 0;
  while (it != it_end) {
    mpNameFields->Append(wxString(it->second.name.c_str(), 
				  wxConvLocal));
    mpNameFields->Check(i, it->second.value);
    ++i;
    ++it;
  }

  // default min size has extra white space vertically
  wxSize boxSize = mpNameFields->GetEffectiveMinSize();
  boxSize.SetHeight(boxSize.GetHeight()*3/4);
  mpNameFields->SetMinSize(boxSize);

  dlgSizer->Add(mpNameFields, 0, wxEXPAND|wxALL, 10);


  wxBoxSizer* bvSizer = new wxBoxSizer(wxHORIZONTAL);
  bvSizer->Add(new wxStaticText(this, -1, _("Skip volumes:")), 0, wxTOP|wxALIGN_CENTER, 5);
  mpBvSpin = new wxSpinCtrl(this, -1, _T("0"));
  mpBvSpin->SetValue(mOutputter->mSkip);
  bvSizer->Add(mpBvSpin);

  dlgSizer->Add(bvSizer, 0, wxALL, 10);

  mpBvCheck = new wxCheckBox(this, -1, _("Save .v16 file"));
  mpBvCheck->SetValue(mOutputter->mSaveV16);
  dlgSizer->Add(mpBvCheck, 0, wxALL, 10);

  mpSplitDirsCheck = new wxCheckBox(this, -1, _("Save each series in separate directory"));
  mpSplitDirsCheck->SetValue(mOutputter->GetSplit());
  dlgSizer->Add(mpSplitDirsCheck, 0, wxLEFT|wxRIGHT|wxBOTTOM, 10);

  mpSplitSubjCheck = new wxCheckBox(this, -1, _("Save each subject in separate directory"));
  mpSplitSubjCheck->SetValue(mOutputter->GetSplitSubj());
  dlgSizer->Add(mpSplitSubjCheck, 0, wxLEFT|wxRIGHT|wxBOTTOM, 10);


  wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);

  wxButton* okayButton = new wxButton(this, wxID_OK, _("Okay"));
  buttonSizer->Add(okayButton, 0, wxRIGHT, 10);
  buttonSizer->Add(new wxButton(this, wxID_CANCEL, _("Cancel")));
  okayButton->SetDefault();

  dlgSizer->Add(buttonSizer, 0, wxALIGN_RIGHT|wxALL, 10);

  SetSizer(dlgSizer);
  dlgSizer->Fit(this);

  mNeedsRebuild = false;
}

void
BvOptionsDlg::OnOkay(wxCommandEvent& event)
{

  mNeedsRebuild = (SplitDirs() != mOutputter->GetSplit()) || 
    (SplitSubj() != mOutputter->GetSplitSubj()) ||
    (SaveV16()   != mOutputter->mSaveV16) || 
    (Skip()      != mOutputter->mSkip ) ||
    SaveNameFields();

  mOutputter->SetSplit(SplitDirs());
  mOutputter->SetSplitSubj(SplitSubj());
  mOutputter->mSaveV16 = SaveV16();
  mOutputter->mSkip = Skip();

  EndModal(event.GetId());

}

bool
BvOptionsDlg::SaveNameFields()
{
  bool needsRebuild = false;
  for (unsigned int i = 0; i < mpNameFields->GetCount(); ++i) {
    string name(mpNameFields->GetString(i).mb_str(wxConvLocal));
    bool value = mpNameFields->IsChecked(i);
    OutputterBase::FieldMap::iterator it = mOutputter->defaultNameFields.begin();
    OutputterBase::FieldMap::iterator it_end = mOutputter->defaultNameFields.end();
    while (it != it_end) {
      if (it->second.name == name) {
        if (it->second.value != value) needsRebuild = true;
        it->second.value = value;
        break;
      }
      ++it;
    }
  }
  return needsRebuild;
}

bool
BvOptionsDlg::SplitDirs() const 
{ 
  return mpSplitDirsCheck->GetValue();
}

bool
BvOptionsDlg::SplitSubj() const 
{ 
  return mpSplitSubjCheck->GetValue();
}

bool 
BvOptionsDlg::SaveV16() const
{ 
  return mpBvCheck->GetValue(); 
}

int 
BvOptionsDlg::Skip() const 
{ 
  return mpBvSpin->GetValue(); 
}

int 
jcs::ShowOverride(int type, OutputterBase* outputter, const string& series_uid)
{

  int needsRebuild = 0;
  if (type == OutputFactory::BV) {
    wxLogMessage(_("It is not possible override default options for BrainVoyager output."));
    return needsRebuild;
  }
  Basic3DOutputter* basicOutputter = dynamic_cast<Basic3DOutputter*>(outputter);
  OverrideDlg dlg;
  dlg.dimCheck->SetValue(basicOutputter->GetDimensionality(series_uid) == 4);
  dlg.skipSpin->SetValue(basicOutputter->GetSkip(series_uid));
  if (dlg.ShowModal() == wxID_OK) {
    if (dlg.dimCheck->GetValue())
      basicOutputter->SetDimensionality(series_uid, 4);
    else
      basicOutputter->SetDimensionality(series_uid, 3);
    basicOutputter->SetSkip(series_uid, dlg.skipSpin->GetValue());
    needsRebuild = 1;
  }
  return needsRebuild;
  
}

OverrideDlg::OverrideDlg()
  : wxDialog(NULL, -1, wxString(_("Override defaults")))
{
  wxBoxSizer* dlgSizer = new wxBoxSizer(wxVERTICAL);

  dimCheck = new wxCheckBox(this, -1, _("Save this series as 4d"));
  //  checkBox->SetValue(init_value);
  dlgSizer->Add(dimCheck, 0, wxALL, 10);

  dlgSizer->Add(new wxStaticText(this, -1, _("Skip volumes for multivolume series:")), 
		0, wxLEFT|wxRIGHT|wxBOTTOM, 10);

  skipSpin = new wxSpinCtrl(this);
  dlgSizer->Add(skipSpin, 0, wxLEFT|wxRIGHT|wxBOTTOM, 10);

  wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);

  wxButton* okayButton = new wxButton(this, wxID_OK, _("Okay"));
  buttonSizer->Add(okayButton, 0, wxRIGHT, 10);
  buttonSizer->Add(new wxButton(this, wxID_CANCEL, _("Cancel")));
  okayButton->SetDefault();

  dlgSizer->Add(buttonSizer, 0, wxALIGN_RIGHT|wxALL, 10);

  SetSizer(dlgSizer);
  dlgSizer->Fit(this);

}



