#ifndef DICTIONARY_H_
#define DICTIONARY_H_

#include <map>
#include <vector>
#include <string>

#include <wx/defs.h>

namespace jcs {
  
  const wxUint32 UNDEFINED = 0xFFFFFFFF; // Undefined value length
  const std::string fnDicom = "dicom";
  const std::string fnExcite = "excite";
  const std::string fnNumaris = "numaris";
  const std::string fnPhilips = "philips_mr";

  ///
  /** Structure represents a DICOM tag as a group and element tuple.
   */
  struct DicomTag {
    
    DicomTag() {}
    DicomTag(wxUint16 g, wxUint16 e) : group(g), element(e) {}
    
    wxUint16 group;
    wxUint16 element;
    
    bool operator== (const DicomTag &rhs) const;
    bool operator!= (const DicomTag &rhs) const;
    bool operator<  (const DicomTag &rhs) const;
    bool operator>  (const DicomTag &rhs) const;
  };

  std::ostream &operator<<( std::ostream &out, const DicomTag &rhs );

  ///
  /** Provides the relationship between a DICOM tag and text 
      description of the tag.
  */
  struct DicomElement {
  
    DicomElement() {}
    DicomElement(const DicomTag& tag,
		 const char* desc = "", 
		 const char* vr = "UN") :
      tag(tag), description(desc), vr(vr) {}
    
    /// The DICOM group and element numbers.
    DicomTag tag;
    /// The DICOM field name.
    std::string description;
    /// The DICOM field data type identifier.
    std::string vr;
    std::string value;
    wxUint32 value_length;
  };
  
  ///
  /** Top-level class for various DICOM dictionaries.
   */
  class Dictionary {

  public:
    const DicomElement Lookup(const DicomTag& tag) const;
    const DicomElement Lookup(const std::string& desc) const;

    std::vector<std::string> TagList() const;

  protected:
    Dictionary(const std::string& dname);
    Dictionary& operator= (const Dictionary&);

  private:
    typedef std::map<DicomTag, DicomElement> TagMap;
    typedef std::map<std::string, DicomElement> StringMap;

    std::string mDictionaryName;
    TagMap mTagMap;
    StringMap mStringMap;
    DicomElement NULL_ENTRY;

    void mAddEntry(const DicomElement& e);

  };

  /// Minimum DICOM definitions.
  class Dicom_Dictionary : public Dictionary
  {
  public:
    /// Self-instantiator.
    static Dictionary* Instance()
    {
      static Dicom_Dictionary inst;
      return &inst;
    }

  protected:
    Dicom_Dictionary() : Dictionary(fnDicom) {}

  };

  /// Adds to Dicom_Dictionary field definitions.
  class Numaris_Dictionary : public Dictionary
  {
  public:
    /// Self-instantiator.
    static Dictionary* Instance()
    {
      static Numaris_Dictionary inst;
      return &inst;
    }

  protected:
    Numaris_Dictionary() : Dictionary(fnNumaris) {}

  };

  /// Adds to Dicom_Dictionary field definitions.
  class Excite_Dictionary : public Dictionary
  {
  public:
    /// Self-instantiator.
    static Dictionary* Instance()
    {
      static Excite_Dictionary inst;
      return &inst;
    }

  protected:
    Excite_Dictionary() : Dictionary(fnExcite) {}

  };

  /// Adds to Dicom_Dictionary field definitions.
  class PhilipsMr_Dictionary : public Dictionary
  {
  public:
    /// Self-instantiator.
    static Dictionary* Instance()
    {
      static PhilipsMr_Dictionary inst;
      return &inst;
    }

  protected:
    PhilipsMr_Dictionary() : Dictionary(fnPhilips) {}

  };


  DicomTag Tag(const std::string& s, 
	       const Dictionary* dict = Dicom_Dictionary::Instance());
}

#endif
