# -*- coding: utf-8 -*-

# Copyright (c) 2009, 2010 Jack Kaliko <efrim@azylum.org> {{{
#
#  This file is part of MPD_sima
#
#  MPD_sima is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#  
#  MPD_sima is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#  
#  You should have received a copy of the GNU General Public License
#  along with MPD_sima.  If not, see <http://www.gnu.org/licenses/>. 
#
#
#  }}}

"""
Deal with configuration and data files.
Parse configuration file and set defaults for missing options.
"""

# DOC {{{
#    http://mail.python.org/pipermail/python-list/2004-February/248674.html
# }}}

# IMPORTS {{{
from __future__ import with_statement
import ConfigParser
import sys

from ConfigParser import (MissingSectionHeaderError, ParsingError)
from os import (makedirs, environ, stat, chmod)
from os.path import (join, isdir, isfile, expanduser)
from stat import (S_IMODE, ST_MODE, S_IRWXO, S_IRWXG)
#}}}

#DEFAULTS{{{
DIRNAME = 'mpd_sima'
CONF_FILE = 'mpd_sima.cfg'

# Mandatory user options
M_SECTIONS = ['MPD']
# Optional user options
O_SECTIONS = ['sima', 'log']
O_SECTIONS.extend(M_SECTIONS)

DEFAULT_CONF = {
        'MPD': {
            'host': "localhost",
            'port': "6600",
            'password': "false"},
        'sima': {
            'similarity': "22",
            'queue_mode': "track", #TODO control values
            'user_db': "false",
            'history_duration': "8",
            'queue_length': "1",
            'track_to_add': "1",
            'album_to_add': "1",
            'main_loop_time': "4",
            'consume': "0",
            'single_album': "false",
            'check_new_version':"false",},
        'log': {
            'verbosity': "info"}}
#}}}


def get_mpd_environ():#{{{
    """
    Retrieve MPD env. var.
    """
    # TODO: get_mpd_environ() should return actual password or None
    mpd_host_and_pass = environ.get('MPD_HOST')
    if mpd_host_and_pass:
        # If password is set:
        # mpd_host_and_pass = ['pass', 'host']
        mpd_host_and_pass = mpd_host_and_pass.split('@')
        mpd_host_and_pass.reverse()

    mpd_port = environ.get('MPD_PORT')
    return (mpd_host_and_pass, mpd_port)#}}}


class ConfMan(object):#CONFIG MANAGER CLASS{{{
    """
    Configuration manager.
    Default configuration is stored in DEFAULT_CONF dictionnary.
    Then init_config() is run and retrieve configuration from defaults if not
    set in conf files.
    These settings are then updated with command line options with
    supersedes_config_with_cmd_line_options().
    Order of priority for the origin of an option is then (lowest to highest):
        * DEFAULT_CONF
        * Env. Var for host and port
        * configuration file (override previous)
        * command line options (override previous)
    """

    def __init__(self, logger, options=None):#{{{
        # options settings priority:
        # defauts < conf. file < command line
        self.conf_file = options.get('conf_file')
        self.config = None
        self.defaults = dict(DEFAULT_CONF)
        self.startopt = options
        ## Sima sqlite DB
        self.userdb_file = None

        self.log = logger
        ## INIT CALLS
        self.use_envar()
        self.init_config()
        self.control_conf()
        self.supersedes_config_with_cmd_line_options()
        #}}}

    def control_mod(self):#{{{
        """
        Controls conf file permissions.
        """
        mode = S_IMODE(stat(self.conf_file)[ST_MODE])
        self.log.debug(u'file permision is: %o' % mode)
        if mode & S_IRWXO or mode & S_IRWXG:
            self.log.warning(u'File is readable by "other" and/or' +
                             u' "group" (actual permission %o octal).' %
                             mode)
            self.log.warning(u'Consider setting permissions' +
                             u' to 600 octal.')
        #}}}

    def supersedes_config_with_cmd_line_options(self):#{{{
        """Updates defaults settings with command line options"""
        for sec in self.config.sections():
            for opt in self.config.options(sec):
                if opt in self.startopt.keys():
                    self.config.set(sec, opt, str(self.startopt.get(opt)))
            #}}}

    def use_envar(self):#{{{
        """Use MPD en.var. to set defaults"""
        mpd_host_pass, mpd_port = get_mpd_environ()
        if mpd_host_pass:
            self.log.debug(u'MPD_HOST set to "%s"' % mpd_host_pass[0])
            self.defaults['MPD']['host'] = mpd_host_pass[0]
            if len(mpd_host_pass) == 2 and mpd_host_pass[1]:
                self.log.debug(u'MPD_HOST contains password.')
                self.defaults['MPD']['password'] = mpd_host_pass[1]
        if mpd_port:
            self.log.debug(u'MPD_PORT set to "%i".'
                                  % int(environ.get('MPD_PORT')))
            self.defaults['MPD']['port'] = environ.get('MPD_PORT')
        #}}}

    def control_conf(self):#{{{
        """Get through options/values and set defaults if not in conf file."""
        # Control presence of obsolete settings
        for option in ['history', 'history_length', 'top_tracks']:
            if self.config.has_option('sima', option):
                self.log.warning('Obsolete setting found in conf file: "%s"'
                        % option)
        # Setting default if not specified
        for section in O_SECTIONS:
            if section not in self.config.sections():
                self.log.debug(u'[%s] NOT in conf file' % section)
                self.config.add_section(section)
                for option in self.defaults[section]:
                    self.config.set(section,
                            option,
                            self.defaults[section][option])
                    self.log.debug(
                            u'Setting option with default value: %s = %s' %
                            (option, self.defaults[section][option]))
            elif section in self.config.sections():
                self.log.debug(u'[%s] present in conf file' % section)
                for option in self.defaults[section]:
                    if self.config.has_option(section, option):
                        self.log.debug(u'option "%s" set to "%s" in conf. file' %
                                      (option, self.config.get(section, option)))
                    else:
                        self.log.debug(
                                u'Option "%s" missing in section "%s"' %
                                (option, section))
                        self.log.debug(u'=> setting default "%s" (may not suit you…)' %
                                       self.defaults[section][option])
                        self.config.set(section, option,
                                        self.defaults[section][option])#}}}

    def init_config(self):#{{{
        """
        Use XDG directory standard if exists
        else use "HOME/(.config|.local/share)/mpd_sima/"
        http://standards.freedesktop.org/basedir-spec/basedir-spec-0.6.html
        """

        homedir = expanduser('~')

        if environ.get('XDG_DATA_HOME'):
            data_dir = join(environ.get('XDG_DATA_HOME'), DIRNAME)
        else:
            data_dir = join(homedir, '.local', 'share', DIRNAME)

        # If var folder is provided via CLI set data_dir accordingly
        if self.startopt.get('var_dir'):
            data_dir = join(self.startopt.get('var_dir'))

        if not isdir(data_dir):
            makedirs(data_dir)
            chmod(data_dir, 0700)

        if environ.get('XDG_CONFIG_HOME'):
            conf_dir = join(environ.get('XDG_CONFIG_HOME'), DIRNAME)
        else:
            conf_dir = join(homedir, '.config', DIRNAME)

        if not isdir(conf_dir):
            makedirs(conf_dir)
            chmod(conf_dir, 0700)
 
        self.userdb_file = join(data_dir, 'sima.db')
        # If no conf_file is provided via CLI set default path to XDG
        if not self.startopt.get('conf_file'):
            self.conf_file = join(conf_dir, CONF_FILE)

        config = ConfigParser.SafeConfigParser()

        # If no conf file present, uses defaults
        if not isfile(self.conf_file):
            self.config = config
            return

        self.log.info(u'Loading configuration from:  %s' % self.conf_file)
        self.control_mod()

        try:
            config.read(self.conf_file)
        except (MissingSectionHeaderError, ParsingError), err_conf:
            self.log.error(u'error loading %s:\n%s' %
                           (self.conf_file, err_conf))
            sys.exit(1)

        self.config = config
        ##}}}

#}}}

if __name__ == '__main__':#{{{
    pass
#}}}

# VIM MODLINE
# vim: ai ts=4 sw=4 sts=4 expandtab
