// Copyright 2010-2011, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// Converts collocation text file into existence filter header file.
// input format:
// <collocation1>
// <collocation2>
// ...
//
// example:
// ./gen_collocation_data_main.cc collocation.txt >
// embedded_collocation_data.h

#include <algorithm>
#include <iostream>
#include <vector>

#include "base/util.h"
#include "base/file_stream.h"
#include "storage/existence_filter.h"

DEFINE_string(collocation_data, "", "collocation data text");
DEFINE_string(output, "", "output file name (default: stdout)");
DEFINE_double(error_rate, 0.00001, "error rate");
DECLARE_bool(logtostderr);

namespace mozc {
namespace {

void Convert() {
  InputFileStream ifs(FLAGS_collocation_data.c_str());
  string line;
  vector<string> entries;
  while (getline(ifs, line)) {
    if (line.empty()) {
      continue;
    }
    entries.push_back(line);
  }

  const float e = FLAGS_error_rate;
  const int n = entries.size();
  const int m =  ExistenceFilter::MinFilterSizeInBytesForErrorRate(e, n);
  LOG(INFO) << "entry: " << n << " err: " << e << " bytes: " << m;

  scoped_ptr<mozc::ExistenceFilter>
      filter(ExistenceFilter::CreateOptimal(m, n));
  DCHECK(filter.get());

  for (size_t i = 0; i < entries.size(); ++i) {
    const uint64 id = Util::Fingerprint(entries[i]);
    filter->Insert(id);
  }

  char *buf = NULL;
  size_t size = 0;
  filter->Write(&buf, &size);

  const char *begin = buf;
  const char *end = buf + size;

  ostream *ofs = &cout;
  if (!FLAGS_output.empty()) {
    ofs = new OutputFileStream(FLAGS_output.c_str());
  }
  *ofs << "// This header file is generated by "
       << "collocation_data_converter_main."
       << endl;

  // TODO(toshiyuki): Make a common function for making byte array header.
  *ofs << "static const size_t kCollocationSize = "
       << size << ";" << endl;

  // Windows does not accept static string of size >= 65536,
  // so we represent string in an array of uint64
#ifdef OS_WINDOWS
  *ofs << "static const uint64 kCollocation[] = {" << endl;
  ofs->setf(ios::hex, ios::basefield);  // in hex
  ofs->setf(ios::showbase);             // add 0x
  int num = 0;
  while (begin < end) {
    uint64 n = 0;
    uint8 *buf = reinterpret_cast<uint8 *>(&n);
    const size_t size = min(static_cast<size_t>(end - begin),
                            static_cast<size_t>(8));
    for (size_t i = 0; i < size; ++i) {
      buf[i] = static_cast<uint8>(begin[i]);
    }
    begin += 8;
    *ofs << n << ", ";
    if (++num % 8 == 0) {
      *ofs << endl;
    }
  }
  *ofs << "};" << endl;
#else
  *ofs << "static const char kCollocation[] =" << endl;
  static const size_t kBucketSize = 20;
  while (begin < end) {
    const size_t size = min(static_cast<size_t>(end - begin), kBucketSize);
    string buf;
    Util::Escape(string(begin, size), &buf);
    *ofs << "\"" << buf << "\"";
    *ofs << endl;
    begin += kBucketSize;
  }
  *ofs << ";" << endl;
#endif

  if (ofs != &cout) {
    delete ofs;
  }
}
}  // namespace
}  // namespace mozc

int main(int argc, char *argv[]) {
  FLAGS_logtostderr = true;
  InitGoogle(argv[0], &argc, &argv, true);

  if (FLAGS_collocation_data.empty() && argc > 1) {
    FLAGS_collocation_data = argv[1];
  }

  LOG(INFO) << FLAGS_collocation_data;

  mozc::Convert();

  return 0;
}
