# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2007-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampann <benjamin@fluendo.com>


from elisa.core.components.model import Model
from elisa.core.utils.i18n import install_translation


_ = install_translation('search')


"""
Contains the basic model that need to be subclassed by any plugin that wants to
return search results.

Also contains some convenience subclasses for the basic types of media handled
by default by Moovida (video, music and pictures).
"""

class BaseSearchResultModel(Model):
    """
    Base class for all search results handled by Moovida's search framework.
    All plugins that need to return search results through this framework need
    to subclass this and implement the necessary methods.
    
    This model represent the output of a search containing results for one or
    more different subtypes (for example the media type "audio" has subtypes
    such as "track", "artits", "album")

    @ivar supported_subtypes: the subtypes that this model may contain
    @type supported_subtypes: C{list} of L{str}
    """
    
    def _warn_not_implemented(key):
        raise NotImplementedError("BaseSearchResultModel does not implement"
                                  "supported_subtypes, but it is mandatory.")
    supported_subtypes = property(_warn_not_implemented)

    def __init__(self):
        super(BaseSearchResultModel, self).__init__()
    
    def label_for_subtype(self, subtype):
        """
        Given a certain subtype, it should return the label to be used when
        displaying the result group for that subtype (e.g. the "Album" part
        in "Album Results")
        
        @param subtype: the name of the subtype for which we need a label
        @param subtype: L{str}

        @return: the label to be used for the subtype
        @rtype: L{str}
        """
        raise NotImplementedError("%s does not implement label_for_subtype, "
                                  "but it is mandatory." % self.__class__)

    def results_for_subtype(self, subtype):
        """
        Given a certain subtype, it should return a list of the currently
        available result items for that subtype
        
        @param subtype: the name of the subtype for which we need results
        @param subtype: L{str}

        @return: all the available results for that subtype
        @rtype: C{list} TODO: do we want NotifyingLists here?
        """
        raise NotImplementedError("%s does not implement results_for_subtype, "
                                  "but it is mandatory." % self.__class__)

class MusicSearchResultModel(BaseSearchResultModel):
    """
    Contains the result of a music search (C{elisa://search/music}).

    @ivar   artists:    the artists found for that search
    @type   artists:    C{list} of
                        L{elisa.plugins.base.models.audio.ArtistModel}'s
    @ivar   albums:     the albums found for that search
    @type   albums:     C{list} of
                        L{elisa.plugins.base.models.audio.AlbumModel}'s
    @ivar   tracks:     the tracks found for that search
    @type   tracks:     C{list} of
                        L{elisa.plugins.base.models.audio.TrackModel}'s
    """
    
    _subtype_to_label_mappings = {
        'artists' : _('Artist'),
        'albums' : _('Album'),
        'tracks' : _('Track'),
    }

    supported_subtypes = _subtype_to_label_mappings.keys()
    
    def __init__(self):
        super(MusicSearchResultModel, self).__init__()
        self.artists = []
        self.albums = []
        self.tracks = []

    def label_for_subtype(self, media_type):
        return self._subtype_to_label_mappings.get(media_type)

    def results_for_subtype(self, media_type):
        return getattr(self, media_type, None)


class VideosSearchResultModel(BaseSearchResultModel):
    """
    Contains the result of a video search (C{elisa://search/videos}).

    @ivar   videos:     the videos found
    @type   videos:     C{list} of
                        L{elisa.plugins.base.models.video.VideoModel}'s
    """

    _subtype_to_label_mappings = {
        'videos' : _('Video'),
    }

    supported_subtypes = _subtype_to_label_mappings.keys()

    def __init__(self):
        super(VideosSearchResultModel, self).__init__()
        self.videos = []

    def label_for_subtype(self, media_type):
        return self._subtype_to_label_mappings.get(media_type)

    def results_for_subtype(self, media_type):
        return getattr(self, media_type, None)


class PicturesSearchResultModel(BaseSearchResultModel):
    """
    Contains the result of a pictures search (C{elisa://search/pictures}).

    @ivar   pictures:   the pictures found for that search
    @type   pictures:   C{list}
    @ivar   groups:     the groups found for that search
    @type   groups:     C{list}
    """

    _subtype_to_label_mappings = {
        'pictures' : _('Picture'),
        'groups' : _('Group'),
    }

    supported_subtypes = _subtype_to_label_mappings.keys()

    def __init__(self):
        super(PicturesSearchResultModel, self).__init__()
        self.pictures = []
        self.groups = []

    def label_for_subtype(self, media_type):
        return self._subtype_to_label_mappings.get(media_type)

    def results_for_subtype(self, media_type):
        return getattr(self, media_type, None)
