# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

from elisa.core.utils import defer
from elisa.core.utils.i18n import install_translation

from elisa.plugins.pigment.pigment_controller import PigmentController

from elisa.plugins.poblesec import modal_popup
from elisa.plugins.poblesec.widgets.keyboard import Keyboard, TextEntry, \
                                                    PasswordTextEntry


_ = install_translation('poblesec')


class GenericEntryValidationController(PigmentController):

    """
    A generic controller presenting the user with a number of text entries and
    a virtual keyboard to validate some data (e.g. username and password).

    @cvar layouts: a list of layout resources for the OSK
    @type layouts: C{list} of C{str}
    """

    layouts = ['elisa.plugins.poblesec.osk_qwerty',
               'elisa.plugins.poblesec.osk_123',
               'elisa.plugins.poblesec.osk_symbols']

    def set_frontend(self, frontend):
        dfr = super(GenericEntryValidationController,
                    self).set_frontend(frontend)
        dfr.addCallback(self._create_keyboard)
        return dfr

    def _create_keyboard(self, result):
        self.keyboard = Keyboard(self.layouts)
        self.set_title()
        self.create_entries()
        self.keyboard.connect('validated', self._on_validated)
        self.keyboard.visible = True
        self.widget.add(self.keyboard)
        self.widget.set_focus_proxy(self.keyboard)
        return result

    def set_title(self):
        """
        Set the title of the keyboard widget.

        This method should be overridden by subclasses.
        """
        raise NotImplementedError('Override me!')

    def create_entries(self):
        """
        Create the desired text entries and add them to the keyboard widget.

        This method should be overridden by subclasses.
        """
        raise NotImplementedError('Override me!')

    def clean(self):
        self.keyboard.disconnect_by_func(self._on_validated)
        return super(GenericEntryValidationController, self).clean()

    def _on_validated(self, keyboard):
        dfr = self.validate()
        dfr.addCallbacks(self.success, self.failure)
        return dfr

    def validate(self):
        """
        Attempt to validate the contents of the entries against whatever
        validation process the subclasses will implement.

        This method should be overridden by subclasses to implement the desired
        validation logic. The default implementation fails.

        @return: a deferred fired when the validation is done
                 (successful or failed) with the result
        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return defer.fail(NotImplementedError())

    def success(self, result):
        """
        Callback invoked when the validation succeeds.

        This method should be overridden by subclasses to implement the desired
        action to be taken when the validation succeeds. The default
        implementation return the original result as is.

        @param result: the result of the validation (may be of any type)

        @return: a deferred fired when the callback is executed
        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        # To be overridden by subclasses.
        return defer.succeed(result)

    def failure(self, failure):
        """
        Callback invoked when the validation fails.

        This method may be overridden by subclasses to implement a custom
        behaviour. The default implementation propagates the failure.

        @param failure: the failure object
        @type failure:  L{twisted.python.failure.Failure}

        @return: a deferred fired when the callback is executed
        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return defer.fail(failure)


class GenericLoginController(GenericEntryValidationController):

    """
    A generic login controller presenting the user with two text entries
    (username and password) and a virtual keyboard to login to any service.
    """

    def set_title(self):
        # Overridden from base class.
        self.keyboard.title.foreground.label = \
            _('ENTER YOUR USERNAME & PASSWORD')

    def create_entries(self):
        # Overridden from base class.
        self._username = TextEntry()
        self._username.instructions = _('Username')
        self._username.decorations.set_name('username')
        self._username.visible = True
        self.keyboard.add_entry(self._username)

        self._password = PasswordTextEntry()
        self._password.instructions = _('Password')
        self._password.decorations.set_name('password')
        self._password.visible = True
        self.keyboard.add_entry(self._password)

    def validate(self):
        # Overridden from base class.
        return self.login(self._username.content, self._password.content)

    def login(self, username, password):
        """
        Attempt to log in with the supplied credentials
        (username and password).

        This method should be overridden by subclasses to implement the desired
        login logic. The default implementation fails.

        @param username: the user name
        @type username:  C{unicode}
        @param password: the password
        @type password:  C{unicode}

        @return: a deferred fired when the login is done (successful or failed)
                 with the result
        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return defer.fail(NotImplementedError())


class LoginController(GenericLoginController):

    """
    A login controller specific to poblesec's browser controller with a default
    behaviour that should fit the most common cases when the login fails
    (show an error popup and offer the user to cancel or retry).
    """

    def go_back(self):
        # Go back one screen.
        main = self.frontend.controller
        return main.browser.history.go_back()

    def close(self):
        """
        Close this controller.

        @return: a deferred fired when done closing
        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return self.go_back()

    def failure(self, failure):
        # Overridden from base class.
        main = self.frontend.controller
        title = _('LOGIN FAILED')
        subtitle = _('Could Not Login')
        text = \
            _("Moovida was unable to login using the details you entered.\n" \
              "Please recheck the login information you provided and " \
              "attempt to login again by selecting 'Check Login Details'.\n" \
              "Alternatively, you can cancel the login process by selecting " \
              "'Cancel Login'.")

        def close_and_callback(callback):
            dfr = main.hide_popup()
            dfr.addCallback(lambda result: callback())
            return dfr

        buttons = []
        buttons.append((_('Cancel Login'),
                        lambda: close_and_callback(self.close)))
        buttons.append((_('Check Login Details'), main.hide_popup))

        main.enqueue_popup(title, subtitle, text, buttons,
                           modal_popup.ErrorPopup)

        # FIXME: the popup system does not return deferreds.
        return defer.succeed(None)
