// Copyright (C) 2009 Stephen Leake <stephen_leake@stephe-leake.org>
// Copyright (C) 2002 Graydon Hoare <graydon@pobox.com>
//
// This program is made available under the GNU GPL version 2.0 or
// greater. See the accompanying file COPYING for details.
//
// This program is distributed WITHOUT ANY WARRANTY; without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE.

#include "base.hh"
#include "work.hh"

#include <ostream>
#include <sstream>
#include <cstring>
#include <cerrno>
#include <queue>

#include "lexical_cast.hh"
#include "basic_io.hh"
#include "cset.hh"
#include "file_io.hh"
#include "platform-wrapped.hh"
#include "restrictions.hh"
#include "sanity.hh"
#include "safe_map.hh"
#include "simplestring_xform.hh"
#include "revision.hh"
#include "inodeprint.hh"
#include "merge_content.hh"
#include "charset.hh"
#include "app_state.hh"
#include "database.hh"
#include "roster.hh"
#include "transforms.hh"
#include "vocab_cast.hh"

using std::deque;
using std::exception;
using std::make_pair;
using std::map;
using std::pair;
using std::set;
using std::string;
using std::vector;

using boost::lexical_cast;

// workspace / book-keeping file code

static char const inodeprints_file_name[] = "inodeprints";
static char const local_dump_file_name[] = "debug";
static char const options_file_name[] = "options";
static char const user_log_file_name[] = "log";
static char const revision_file_name[] = "revision";
static char const update_file_name[] = "update";
static char const bisect_file_name[] = "bisect";

static void
get_revision_path(bookkeeping_path & m_path)
{
  m_path = bookkeeping_root / revision_file_name;
  L(FL("revision path is %s") % m_path);
}

static void
get_options_path(bookkeeping_path & o_path)
{
  o_path = bookkeeping_root / options_file_name;
  L(FL("options path is %s") % o_path);
}

static void
get_options_path(system_path const & workspace, system_path & o_path)
{
  o_path = workspace / bookkeeping_root_component / options_file_name;
  L(FL("options path is %s") % o_path);
}

static void
get_inodeprints_path(bookkeeping_path & ip_path)
{
  ip_path = bookkeeping_root / inodeprints_file_name;
  L(FL("inodeprints path is %s") % ip_path);
}

static void
get_user_log_path(bookkeeping_path & ul_path)
{
  ul_path = bookkeeping_root / user_log_file_name;
  L(FL("user log path is %s") % ul_path);
}

static void
get_update_path(bookkeeping_path & update_path)
{
  update_path = bookkeeping_root / update_file_name;
  L(FL("update path is %s") % update_path);
}

static void
get_bisect_path(bookkeeping_path & bisect_path)
{
  bisect_path = bookkeeping_root / bisect_file_name;
  L(FL("bisect path is %s") % bisect_path);
}

//

bool
directory_is_workspace(system_path const & dir)
{
  // as far as the users of this function are concerned, a version 0
  // workspace (MT directory instead of _MTN) does not count.
  return directory_exists(dir / bookkeeping_root_component);
}

bool workspace::found;
bool workspace::branch_is_sticky;

void
workspace::require_workspace()
{
  E(workspace::found, origin::user,
    F("workspace required but not found"));
}

void
workspace::require_workspace(i18n_format const & explanation)
{
  E(workspace::found, origin::user,
    F("workspace required but not found\n%s") % explanation.str());
}

void
workspace::create_workspace(options const & opts,
                            lua_hooks & lua,
                            system_path const & new_dir)
{
  E(!new_dir.empty(), origin::user, F("invalid directory ''"));

  L(FL("creating workspace in %s") % new_dir);

  mkdir_p(new_dir);
  go_to_workspace(new_dir);
  mark_std_paths_used();

  E(!directory_exists(bookkeeping_root), origin::user,
    F("monotone bookkeeping directory '%s' already exists in '%s'")
    % bookkeeping_root % new_dir);

  L(FL("creating bookkeeping directory '%s' for workspace in '%s'")
    % bookkeeping_root % new_dir);

  mkdir_p(bookkeeping_root);

  workspace::found = true;
  workspace::set_options(opts, true);
  workspace::write_format();

  data empty;
  bookkeeping_path ul_path;
  get_user_log_path(ul_path);
  write_data(ul_path, empty);

  if (lua.hook_use_inodeprints())
    {
      bookkeeping_path ip_path;
      get_inodeprints_path(ip_path);
      write_data(ip_path, empty);
    }

  bookkeeping_path dump_path;
  workspace::get_local_dump_path(dump_path);
  // The 'false' means that, e.g., if we're running checkout,
  // then it's okay for dumps to go into our starting working
  // dir's _MTN rather than the new workspace dir's _MTN.
  global_sanity.set_dump_path(system_path(dump_path, false).as_external());
}

// Normal-use constructor.
workspace::workspace(app_state & app, bool writeback_options)
  : lua(app.lua)
{
  require_workspace();
  if (writeback_options)
    set_options(app.opts, false);
}

workspace::workspace(app_state & app, i18n_format const & explanation,
                     bool writeback_options)
  : lua(app.lua)
{
  require_workspace(explanation);
  if (writeback_options)
    set_options(app.opts, false);
}

workspace::workspace(options const & opts, lua_hooks & lua,
                     i18n_format const & explanation, bool writeback_options)
  : lua(lua)
{
  require_workspace(explanation);
  if (writeback_options)
    set_options(opts, false);
}

// routines for manipulating the bookkeeping directory

// revision file contains a partial revision describing the workspace
void
workspace::get_work_rev(revision_t & rev)
{
  bookkeeping_path rev_path;
  get_revision_path(rev_path);
  data rev_data;
  MM(rev_data);
  try
    {
      read_data(rev_path, rev_data);
    }
  catch(exception & e)
    {
      E(false, origin::system,
        F("workspace is corrupt: reading %s: %s")
        % rev_path % e.what());
    }

  read_revision(rev_data, rev);
  // Mark it so it doesn't creep into the database.
  rev.made_for = made_for_workspace;
}

void
workspace::put_work_rev(revision_t const & rev)
{
  MM(rev);
  I(rev.made_for == made_for_workspace);
  rev.check_sane();

  data rev_data;
  write_revision(rev, rev_data);

  bookkeeping_path rev_path;
  get_revision_path(rev_path);
  write_data(rev_path, rev_data);
}

void
workspace::get_update_id(revision_id & update_id)
{
  data update_data;
  bookkeeping_path update_path;
  get_update_path(update_path);
  E(file_exists(update_path), origin::user,
    F("no update has occurred in this workspace"));

  read_data(update_path, update_data);

  update_id = revision_id(decode_hexenc(update_data(), origin::internal),
                          origin::internal);
  E(!null_id(update_id), origin::internal,
    F("no update revision available"));
}

void
workspace::put_update_id(revision_id const & update_id)
{
  data update_data(encode_hexenc(update_id.inner()(), origin::internal),
                   origin::internal);
  bookkeeping_path update_path;
  get_update_path(update_path);
  write_data(update_path, update_data);
}

// structures derived from the work revision, the database, and possibly
// the workspace

static void
get_roster_for_rid(database & db,
                   revision_id const & rid,
                   cached_roster & cr)
{
  // We may be asked for a roster corresponding to the null rid, which
  // is not in the database.  In this situation, what is wanted is an empty
  // roster (and marking map).
  if (null_id(rid))
    {
      cr.first = boost::shared_ptr<roster_t const>(new roster_t);
      cr.second = boost::shared_ptr<marking_map const>(new marking_map);
    }
  else
    {
      E(db.revision_exists(rid), origin::user,
        F("base revision %s does not exist in database") % rid);
      db.get_roster(rid, cr);
    }
  L(FL("base roster has %d entries") % cr.first->all_nodes().size());
}

void
workspace::require_parents_in_db(database & db,
                                 revision_t const & rev)
{
  for (edge_map::const_iterator i = rev.edges.begin();
       i != rev.edges.end(); i++)
    {
      revision_id const & parent(edge_old_revision(i));
      E(null_id(parent) || db.revision_exists(parent), origin::user,
        F("parent revision %s does not exist, did you specify the wrong database?")
        % parent);
    }
}

void
workspace::get_parent_rosters(database & db, parent_map & parents)
{
  revision_t rev;
  get_work_rev(rev);
  require_parents_in_db(db, rev);

  parents.clear();
  for (edge_map::const_iterator i = rev.edges.begin();
       i != rev.edges.end(); i++)
    {
      cached_roster cr;
      get_roster_for_rid(db, edge_old_revision(i), cr);
      safe_insert(parents, make_pair(edge_old_revision(i), cr));
    }
}

void
workspace::get_current_roster_shape(database & db,
                                    node_id_source & nis,
                                    roster_t & ros)
{
  revision_t rev;
  get_work_rev(rev);
  require_parents_in_db(db, rev);
  revision_id new_rid(fake_id());

  // If there is just one parent, it might be the null ID, which
  // make_roster_for_revision does not handle correctly.
  if (rev.edges.size() == 1 && null_id(edge_old_revision(rev.edges.begin())))
    {
      I(ros.all_nodes().empty());
      editable_roster_base er(ros, nis);
      edge_changes(rev.edges.begin()).apply_to(er);
    }
  else
    {
      marking_map dummy;
      make_roster_for_revision(db, nis, rev, new_rid, ros, dummy);
    }
}

bool
workspace::has_changes(database & db)
{
  parent_map parents;
  get_parent_rosters(db, parents);

  // if we have more than one parent roster then this workspace contains
  // a merge which means this is always a committable change
  if (parents.size() > 1)
    return true;

  temp_node_id_source nis;
  roster_t new_roster, old_roster = parent_roster(parents.begin());

  get_current_roster_shape(db, nis, new_roster);
  update_current_roster_from_filesystem(new_roster);

  return !(old_roster == new_roster);
}

// user log file

void
workspace::read_user_log(utf8 & dat)
{
  bookkeeping_path ul_path;
  get_user_log_path(ul_path);

  if (file_exists(ul_path))
    {
      data tmp;
      read_data(ul_path, tmp);
      system_to_utf8(typecast_vocab<external>(tmp), dat);
    }
}

void
workspace::write_user_log(utf8 const & dat)
{
  bookkeeping_path ul_path;
  get_user_log_path(ul_path);

  external tmp;
  utf8_to_system_best_effort(dat, tmp);
  write_data(ul_path, typecast_vocab<data>(tmp));
}

void
workspace::blank_user_log()
{
  data empty;
  bookkeeping_path ul_path;
  get_user_log_path(ul_path);
  write_data(ul_path, empty);
}

bool
workspace::has_contents_user_log()
{
  utf8 user_log_message;
  read_user_log(user_log_message);
  return user_log_message().length() > 0;
}

// _MTN/options handling.

static void
read_options_file(any_path const & optspath,
                  system_path & workspace_database,
                  branch_name & workspace_branch,
                  external_key_name & workspace_key,
                  system_path & workspace_keydir)
{
  data dat;
  try
    {
      read_data(optspath, dat);
    }
  catch (exception & e)
    {
      W(F("Failed to read options file %s: %s") % optspath % e.what());
      return;
    }

  basic_io::input_source src(dat(), optspath.as_external(), origin::workspace);
  basic_io::tokenizer tok(src);
  basic_io::parser parser(tok);

  while (parser.symp())
    {
      string opt, val;
      parser.sym(opt);
      parser.str(val);

      if (opt == "database")
        workspace_database = system_path(val, origin::workspace);
      else if (opt == "branch")
        workspace_branch = branch_name(val, origin::workspace);
      else if (opt == "key")
        workspace_key = external_key_name(val, origin::workspace);
      else if (opt == "keydir")
        workspace_keydir = system_path(val, origin::workspace);
      else
        W(F("unrecognized key '%s' in options file %s - ignored")
          % opt % optspath);
    }
  E(src.lookahead == EOF, src.made_from,
    F("Could not parse entire options file %s") % optspath);
}

static void
write_options_file(bookkeeping_path const & optspath,
                   system_path const & workspace_database,
                   branch_name const & workspace_branch,
                   external_key_name const & workspace_key,
                   system_path const & workspace_keydir)
{
  basic_io::stanza st;
  if (!workspace_database.as_internal().empty())
    st.push_str_pair(symbol("database"), workspace_database.as_internal());
  if (!workspace_branch().empty())
    st.push_str_pair(symbol("branch"), workspace_branch());
  if (!workspace_key().empty())
    {
      st.push_str_pair(symbol("key"), workspace_key());
    }
  if (!workspace_keydir.as_internal().empty())
    st.push_str_pair(symbol("keydir"), workspace_keydir.as_internal());

  basic_io::printer pr;
  pr.print_stanza(st);
  try
    {
      write_data(optspath, data(pr.buf, origin::internal));
    }
  catch(exception & e)
    {
      W(F("Failed to write options file %s: %s") % optspath % e.what());
    }
}

void
workspace::get_options(options & opts)
{
  if (!workspace::found)
    return;

  system_path workspace_database;
  branch_name workspace_branch;
  external_key_name workspace_key;
  system_path workspace_keydir;

  bookkeeping_path o_path;
  get_options_path(o_path);
  read_options_file(o_path,
                    workspace_database, workspace_branch,
                    workspace_key, workspace_keydir);

  // Workspace options are not to override the command line.
  if (!opts.dbname_given)
    {
      opts.dbname = workspace_database;
    }

  if (!opts.key_dir_given && !opts.conf_dir_given && !workspace_keydir.empty())
    { // if empty/missing, we want to keep the default
      opts.key_dir = workspace_keydir;
      opts.key_dir_given = true;
    }

  if (opts.branch().empty() && !workspace_branch().empty())
    {
      opts.branch = workspace_branch;
      branch_is_sticky = true;
    }

  L(FL("branch name is '%s'") % opts.branch);

  if (!opts.key_given)
    opts.signing_key = workspace_key;
}

void
workspace::get_database_option(system_path const & workspace,
                               system_path & workspace_database)
{
  branch_name workspace_branch;
  external_key_name workspace_key;
  system_path workspace_keydir;

  system_path o_path = (workspace
                        / bookkeeping_root_component
                        / options_file_name);
  read_options_file(o_path,
                    workspace_database, workspace_branch,
                    workspace_key, workspace_keydir);
}

void
workspace::set_options(options const & opts, bool branch_is_sticky)
{
  E(workspace::found, origin::user, F("workspace required but not found"));

  bookkeeping_path o_path;
  get_options_path(o_path);

  // If any of the incoming options was empty, we want to leave that option
  // as is in _MTN/options, not write out an empty option.
  system_path workspace_database;
  branch_name workspace_branch;
  external_key_name workspace_key;
  system_path workspace_keydir;

  if (file_exists(o_path))
    read_options_file(o_path,
                      workspace_database, workspace_branch,
                      workspace_key, workspace_keydir);

  // FIXME: we should do more checks here, f.e. if this is a valid sqlite
  // file and if it contains the correct identifier, but these checks would
  // duplicate those in database.cc. At the time it is checked there, however,
  // the options file for the workspace is already written out...
  if (!opts.dbname.as_internal().empty() &&
      get_path_status(opts.dbname.as_internal()) == path::file)
    workspace_database = opts.dbname;
  if (!opts.key_dir.as_internal().empty() &&
      get_path_status(opts.key_dir.as_internal()) == path::directory)
    workspace_keydir = opts.key_dir;
  if ((branch_is_sticky || workspace::branch_is_sticky)
      && !opts.branch().empty())
    workspace_branch = opts.branch;
  if (opts.key_given)
    workspace_key = opts.signing_key;

  write_options_file(o_path,
                     workspace_database, workspace_branch,
                     workspace_key, workspace_keydir);
}

void
workspace::print_option(utf8 const & opt, std::ostream & output)
{
  E(workspace::found, origin::user, F("workspace required but not found"));

  bookkeeping_path o_path;
  get_options_path(o_path);

  system_path workspace_database;
  branch_name workspace_branch;
  external_key_name workspace_key;
  system_path workspace_keydir;
  read_options_file(o_path,
                    workspace_database, workspace_branch,
                    workspace_key, workspace_keydir);

  if (opt() == "database")
    output << workspace_database << '\n';
  else if (opt() == "branch")
    output << workspace_branch << '\n';
  else if (opt() == "key")
    output << workspace_key << '\n';
  else if (opt() == "keydir")
    output << workspace_keydir << '\n';
  else
    E(false, origin::user, F("'%s' is not a recognized workspace option") % opt);
}

// _MTN/bisect handling.

namespace syms
{
    symbol const start("start");
    symbol const good("good");
    symbol const bad("bad");
    symbol const skipped("skipped");
};

void
workspace::get_bisect_info(vector<bisect::entry> & bisect)
{
  bookkeeping_path bisect_path;
  get_bisect_path(bisect_path);
  
  if (!file_exists(bisect_path))
    return;

  data dat;
  read_data(bisect_path, dat);

  string name("bisect");
  basic_io::input_source src(dat(), name, origin::workspace);
  basic_io::tokenizer tok(src);
  basic_io::parser parser(tok);

  while (parser.symp())
    {
      string rev;
      bisect::type type;
      if (parser.symp(syms::start))
        {
          parser.sym();
          parser.hex(rev);
          type = bisect::start;
        }
      else if (parser.symp(syms::good))
        {
          parser.sym();
          parser.hex(rev);
          type = bisect::good;
        }
      else if (parser.symp(syms::bad))
        {
          parser.sym();
          parser.hex(rev);
          type = bisect::bad;
        }
      else if (parser.symp(syms::skipped))
        {
          parser.sym();
          parser.hex(rev);
          type = bisect::skipped;
        }
      else
        I(false);

      revision_id rid = 
        decode_hexenc_as<revision_id>(rev, parser.tok.in.made_from);
      bisect.push_back(make_pair(type, rid));
    }
}

void
workspace::put_bisect_info(vector<bisect::entry> const & bisect)
{
  bookkeeping_path bisect_path;
  get_bisect_path(bisect_path);

  basic_io::stanza st;
  for (vector<bisect::entry>::const_iterator i = bisect.begin(); 
       i != bisect.end(); ++i)
    {
      switch (i->first)
        {
        case bisect::start:
          st.push_binary_pair(syms::start, i->second.inner());
          break;

        case bisect::good:
          st.push_binary_pair(syms::good, i->second.inner());
          break;

        case bisect::bad:
          st.push_binary_pair(syms::bad, i->second.inner());
          break;

        case bisect::skipped:
          st.push_binary_pair(syms::skipped, i->second.inner());
          break;

        case bisect::update:
          // this value is not persisted, it is only used by the bisect
          // update command to rerun a selection and update based on current
          // bisect information
          I(false);
          break;
        }
    }

  basic_io::printer pr;
  pr.print_stanza(st);
  data dat(pr.buf, origin::internal);

  write_data(bisect_path, dat);
}

void
workspace::remove_bisect_info()
{
  bookkeeping_path bisect_path;
  get_bisect_path(bisect_path);
  delete_file(bisect_path);
}

// local dump file

void
workspace::get_local_dump_path(bookkeeping_path & d_path)
{
  E(workspace::found, origin::user, F("workspace required but not found"));

  d_path = bookkeeping_root / local_dump_file_name;
  L(FL("local dump path is %s") % d_path);
}

// inodeprint file

bool
workspace::in_inodeprints_mode()
{
  bookkeeping_path ip_path;
  get_inodeprints_path(ip_path);
  return file_exists(ip_path);
}

void
workspace::read_inodeprints(data & dat)
{
  I(in_inodeprints_mode());
  bookkeeping_path ip_path;
  get_inodeprints_path(ip_path);
  read_data(ip_path, dat);
}

void
workspace::write_inodeprints(data const & dat)
{
  I(in_inodeprints_mode());
  bookkeeping_path ip_path;
  get_inodeprints_path(ip_path);
  write_data(ip_path, dat);
}

void
workspace::enable_inodeprints()
{
  bookkeeping_path ip_path;
  get_inodeprints_path(ip_path);
  data dat;
  write_data(ip_path, dat);
}

void
workspace::maybe_update_inodeprints(database & db)
{
  if (!in_inodeprints_mode())
    return;

  inodeprint_map ipm_new;
  temp_node_id_source nis;
  roster_t new_roster;

  get_current_roster_shape(db, nis, new_roster);
  update_current_roster_from_filesystem(new_roster);

  parent_map parents;
  get_parent_rosters(db, parents);

  node_map const & new_nodes = new_roster.all_nodes();
  for (node_map::const_iterator i = new_nodes.begin(); i != new_nodes.end(); ++i)
    {
      node_id nid = i->first;
      if (!is_file_t(i->second))
        continue;
      file_t new_file = downcast_to_file_t(i->second);
      bool all_same = true;

      for (parent_map::const_iterator parent = parents.begin();
           parent != parents.end(); ++parent)
        {
          roster_t const & parent_ros = parent_roster(parent);
          if (parent_ros.has_node(nid))
            {
              const_node_t old_node = parent_ros.get_node(nid);
              I(is_file_t(old_node));
              const_file_t old_file = downcast_to_file_t(old_node);

              if (new_file->content != old_file->content)
                {
                  all_same = false;
                  break;
                }
            }
        }

      if (all_same)
        {
          file_path fp;
          new_roster.get_name(nid, fp);
          hexenc<inodeprint> ip;
          if (inodeprint_file(fp, ip))
            ipm_new.insert(inodeprint_entry(fp, ip));
        }
    }
  data dat;
  write_inodeprint_map(ipm_new, dat);
  write_inodeprints(dat);
}

bool
workspace::ignore_file(file_path const & path)
{
  return lua.hook_ignore_file(path);
}

bool
ignored_file::operator()(file_path const & f) const
{
  return work.ignore_file(f);
}

void
workspace::init_attributes(file_path const & path, editable_roster_base & er)
{
  map<string, string> attrs;
  lua.hook_init_attributes(path, attrs);
  if (!attrs.empty())
    for (map<string, string>::const_iterator i = attrs.begin();
         i != attrs.end(); ++i)
      er.set_attr(path, attr_key(i->first, origin::user),
                  attr_value(i->second, origin::user));
}

// objects and routines for manipulating the workspace itself
namespace {

struct file_itemizer : public tree_walker
{
  database & db;
  workspace & work;
  set<file_path> & known;
  set<file_path> & unknown;
  set<file_path> & ignored;
  path_restriction const & mask;
  file_itemizer(database & db, workspace & work,
                set<file_path> & k,
                set<file_path> & u,
                set<file_path> & i,
                path_restriction const & r)
    : db(db), work(work), known(k), unknown(u), ignored(i), mask(r) {}
  virtual bool visit_dir(file_path const & path);
  virtual void visit_file(file_path const & path);
};


bool
file_itemizer::visit_dir(file_path const & path)
{
  this->visit_file(path);
  return known.find(path) != known.end();
}

void
file_itemizer::visit_file(file_path const & path)
{
  if (mask.includes(path) && known.find(path) == known.end())
    {
      if (work.ignore_file(path) || db.is_dbfile(path))
        ignored.insert(path);
      else
        unknown.insert(path);
    }
}


struct workspace_itemizer : public tree_walker
{
  roster_t & roster;
  set<file_path> const & known;
  node_id_source & nis;

  workspace_itemizer(roster_t & roster, set<file_path> const & paths,
                     node_id_source & nis);
  virtual bool visit_dir(file_path const & path);
  virtual void visit_file(file_path const & path);
};

workspace_itemizer::workspace_itemizer(roster_t & roster,
                                       set<file_path> const & paths,
                                       node_id_source & nis)
    : roster(roster), known(paths), nis(nis)
{
  node_id root_nid = roster.create_dir_node(nis);
  roster.attach_node(root_nid, file_path_internal(""));
}

bool
workspace_itemizer::visit_dir(file_path const & path)
{
  node_id nid = roster.create_dir_node(nis);
  roster.attach_node(nid, path);
  return known.find(path) != known.end();
}

void
workspace_itemizer::visit_file(file_path const & path)
{
  file_id fid;
  node_id nid = roster.create_file_node(fid, nis);
  roster.attach_node(nid, path);
}


class
addition_builder
  : public tree_walker
{
  database & db;
  workspace & work;
  roster_t & ros;
  editable_roster_base & er;
  bool respect_ignore;
  bool recursive;
public:
  addition_builder(database & db, workspace & work,
                   roster_t & r, editable_roster_base & e,
                   bool i, bool rec)
    : db(db), work(work), ros(r), er(e), respect_ignore(i), recursive(rec)
  {}
  virtual bool visit_dir(file_path const & path);
  virtual void visit_file(file_path const & path);
  void add_nodes_for(file_path const & path, file_path const & goal);
};

void
addition_builder::add_nodes_for(file_path const & path,
                                file_path const & goal)
{
  // this check suffices to terminate the recursion; our caller guarantees
  // that the roster has a root node, which will be a directory.
  if (ros.has_node(path))
    {
      E(is_dir_t(ros.get_node(path)), origin::user,
        F("cannot add %s, because %s is recorded as a file "
          "in the workspace manifest") % goal % path);
      return;
    }

  add_nodes_for(path.dirname(), goal);
  P(F("adding %s to workspace manifest") % path);

  node_id nid = the_null_node;
  switch (get_path_status(path))
    {
    case path::nonexistent:
      return;
    case path::file:
      {
        file_id ident;
        I(ident_existing_file(path, ident));
        nid = er.create_file_node(ident);
      }
      break;
    case path::directory:
      nid = er.create_dir_node();
      break;
    }

  I(nid != the_null_node);
  er.attach_node(nid, path);

  work.init_attributes(path, er);
}


bool
addition_builder::visit_dir(file_path const & path)
{
  struct directory_has_unignored_files_exception {};
  struct directory_has_unignored_files : public dirent_consumer
  {
    directory_has_unignored_files(workspace & work, file_path const & p)
      : work(work), p(p) {}
    virtual void consume(char const * s)
    {
      try
        {
          file_path entry = p / path_component(s);
          if (!work.ignore_file(entry))
            throw directory_has_unignored_files_exception();
        }
      catch (std::logic_error)
        {
          // ignore this file for purposes of the warning; this file
          // wouldn't have been added by a recursive add anyway.
        }
    }
  private:
    workspace & work;
    file_path const & p;
  };

  if (!recursive)
    {
      bool warn = false;

      // If the db can ever be stored in a dir
      // then revisit this logic
      I(!db.is_dbfile(path));

      if (!respect_ignore)
        warn = !directory_empty(path);
      else if (!work.ignore_file(path))
        {
          directory_has_unignored_files dhuf(work, path);
          try
            {
              read_directory(path, dhuf, dhuf, dhuf);
            }
          catch (directory_has_unignored_files_exception)
            {
              warn = true;
            }
        }

      if (warn)
        W(F("Non-recursive add: Files in the directory '%s' "
            "will not be added automatically.") % path);
    }

  this->visit_file(path);
  return true;
}

void
addition_builder::visit_file(file_path const & path)
{
  if ((respect_ignore && work.ignore_file(path)) || db.is_dbfile(path))
    {
      P(F("skipping ignorable file %s") % path);
      return;
    }

  if (ros.has_node(path))
    {
      if (!path.empty())
        P(F("skipping %s, already accounted for in workspace") % path);
      return;
    }

  I(ros.has_root());
  add_nodes_for(path, path);
}

struct editable_working_tree : public editable_tree
{
  editable_working_tree(workspace & work, lua_hooks & lua,
                        content_merge_adaptor const & source,
                        bool const messages)
    : work(work), lua(lua), source(source), next_nid(1),
      root_dir_attached(true), messages(messages)
  {};

  virtual node_id detach_node(file_path const & src);
  virtual void drop_detached_node(node_id nid);

  virtual node_id create_dir_node();
  virtual node_id create_file_node(file_id const & content);
  virtual void attach_node(node_id nid, file_path const & dst);

  virtual void apply_delta(file_path const & pth,
                           file_id const & old_id,
                           file_id const & new_id);
  virtual void clear_attr(file_path const & path,
                          attr_key const & key);
  virtual void set_attr(file_path const & path,
                        attr_key const & key,
                        attr_value const & val);

  virtual void commit();

  virtual ~editable_working_tree();
private:
  workspace & work;
  lua_hooks & lua;
  content_merge_adaptor const & source;
  node_id next_nid;
  std::map<bookkeeping_path, file_path> rename_add_drop_map;
  bool root_dir_attached;
  bool messages;
};


struct simulated_working_tree : public editable_tree
{
  roster_t & workspace;
  node_id_source & nis;

  set<file_path> blocked_paths;
  set<file_path> conflicting_paths;
  int conflicts;
  map<node_id, file_path> nid_map;

  simulated_working_tree(roster_t & r, temp_node_id_source & n)
    : workspace(r), nis(n), conflicts(0) {}

  virtual node_id detach_node(file_path const & src);
  virtual void drop_detached_node(node_id nid);

  virtual node_id create_dir_node();
  virtual node_id create_file_node(file_id const & content);
  virtual void attach_node(node_id nid, file_path const & dst);

  virtual void apply_delta(file_path const & pth,
                           file_id const & old_id,
                           file_id const & new_id);
  virtual void clear_attr(file_path const & path,
                          attr_key const & key);
  virtual void set_attr(file_path const & path,
                        attr_key const & key,
                        attr_value const & val);

  virtual void commit();

  virtual bool has_conflicting_paths() const { return conflicting_paths.size() > 0; }
  virtual set<file_path> get_conflicting_paths() const { return conflicting_paths; }

  virtual ~simulated_working_tree();
};


// editable_working_tree implementation

static inline bookkeeping_path
path_for_detached_nids()
{
  return bookkeeping_root / "detached";
}

static inline bookkeeping_path
path_for_detached_nid(node_id nid)
{
  return path_for_detached_nids() / path_component(lexical_cast<string>(nid),
                                                   origin::internal);
}

// Attaching/detaching the root directory:
//   This is tricky, because we don't want to simply move it around, like
// other directories.  That would require some very snazzy handling of the
// _MTN directory, and never be possible on windows anyway[1].  So, what we do
// is fake it -- whenever we want to move the root directory into the
// temporary dir, we instead create a new dir in the temporary dir, move
// all of the root's contents into this new dir, and make a note that the root
// directory is logically non-existent.  Whenever we want to move some
// directory out of the temporary dir and onto the root directory, we instead
// check that the root is logically nonexistent, move its contents, and note
// that it exists again.
//
// [1] Because the root directory is our working directory, and thus locked in
// place.  We _could_ chdir out, then move _MTN out, then move the real root
// directory into our newly-moved _MTN, etc., but aside from being very finicky,
// this would require that we know our root directory's name relative to its
// parent.

node_id
editable_working_tree::detach_node(file_path const & src_pth)
{
  I(root_dir_attached);
  node_id nid = next_nid++;
  bookkeeping_path dst_pth = path_for_detached_nid(nid);
  safe_insert(rename_add_drop_map, make_pair(dst_pth, src_pth));
  if (src_pth == file_path())
    {
      // root dir detach, so we move contents, rather than the dir itself
      mkdir_p(dst_pth);

      vector<file_path> files, dirs;
      fill_path_vec<file_path> fill_files(src_pth, files, false);
      fill_path_vec<file_path> fill_dirs(src_pth, dirs, true);
      read_directory(src_pth, fill_files, fill_dirs);

      for (vector<file_path>::const_iterator i = files.begin();
           i != files.end(); ++i)
        move_file(*i, dst_pth / (*i).basename());
      for (vector<file_path>::const_iterator i = dirs.begin();
           i != dirs.end(); ++i)
        move_dir(*i, dst_pth / (*i).basename());

      root_dir_attached = false;
    }
  else
    move_path(src_pth, dst_pth);
  return nid;
}

void
editable_working_tree::drop_detached_node(node_id nid)
{
  bookkeeping_path pth = path_for_detached_nid(nid);
  map<bookkeeping_path, file_path>::const_iterator i
    = rename_add_drop_map.find(pth);
  I(i != rename_add_drop_map.end());
  P(F("dropping %s") % i->second);
  safe_erase(rename_add_drop_map, pth);
  delete_file_or_dir_shallow(pth);
}

node_id
editable_working_tree::create_dir_node()
{
  node_id nid = next_nid++;
  bookkeeping_path pth = path_for_detached_nid(nid);
  require_path_is_nonexistent(pth,
                              F("path %s already exists") % pth);
  mkdir_p(pth);
  return nid;
}

node_id
editable_working_tree::create_file_node(file_id const & content)
{
  node_id nid = next_nid++;
  bookkeeping_path pth = path_for_detached_nid(nid);
  require_path_is_nonexistent(pth,
                              F("path %s already exists") % pth);
  file_data dat;
  source.get_version(content, dat);
  write_data(pth, dat.inner());

  return nid;
}

void
editable_working_tree::attach_node(node_id nid, file_path const & dst_pth)
{
  bookkeeping_path src_pth = path_for_detached_nid(nid);

  map<bookkeeping_path, file_path>::const_iterator i
    = rename_add_drop_map.find(src_pth);
  if (i != rename_add_drop_map.end())
    {
      if (messages)
        P(F("renaming %s to %s") % i->second % dst_pth);
      safe_erase(rename_add_drop_map, src_pth);
    }
  else if (messages)
     P(F("adding %s") % dst_pth);

  if (dst_pth == file_path())
    {
      // root dir attach, so we move contents, rather than the dir itself
      vector<bookkeeping_path> files, dirs;
      fill_path_vec<bookkeeping_path> fill_files(src_pth, files, false);
      fill_path_vec<bookkeeping_path> fill_dirs(src_pth, dirs, true);
      read_directory(src_pth, fill_files, fill_dirs);

      for (vector<bookkeeping_path>::const_iterator i = files.begin();
           i != files.end(); ++i)
        move_file(*i, dst_pth / (*i).basename());
      for (vector<bookkeeping_path>::const_iterator i = dirs.begin();
           i != dirs.end(); ++i)
        move_dir(*i, dst_pth / (*i).basename());

      delete_dir_shallow(src_pth);
      root_dir_attached = true;
    }
  else
    // This will complain if the move is actually impossible
    move_path(src_pth, dst_pth);
}

void
editable_working_tree::apply_delta(file_path const & pth,
                                   file_id const & old_id,
                                   file_id const & new_id)
{
  require_path_is_file(pth,
                       F("file '%s' does not exist") % pth,
                       F("file '%s' is a directory") % pth);
  file_id curr_id;
  calculate_ident(pth, curr_id);
  E(curr_id == old_id, origin::system,
    F("content of file '%s' has changed, not overwriting") % pth);
  P(F("updating %s") % pth);

  file_data dat;
  source.get_version(new_id, dat);
  write_data(pth, dat.inner());
}

void
editable_working_tree::clear_attr(file_path const & path,
                                  attr_key const & key)
{
  L(FL("calling hook to clear attribute %s on %s") % key % path);
  lua.hook_clear_attribute(key(), path);
}

void
editable_working_tree::set_attr(file_path const & path,
                                attr_key const & key,
                                attr_value const & value)
{
  L(FL("calling hook to set attribute %s on %s to %s") % key % path % value);
  lua.hook_set_attribute(key(), path, value());
}

void
editable_working_tree::commit()
{
  I(rename_add_drop_map.empty());
  I(root_dir_attached);
}

editable_working_tree::~editable_working_tree()
{
}


node_id
simulated_working_tree::detach_node(file_path const & src)
{
  node_id nid = workspace.detach_node(src);
  nid_map.insert(make_pair(nid, src));
  return nid;
}

void
simulated_working_tree::drop_detached_node(node_id nid)
{
  const_node_t node = workspace.get_node(nid);
  if (is_dir_t(node))
    {
      const_dir_t dir = downcast_to_dir_t(node);
      if (!dir->children.empty())
        {
          map<node_id, file_path>::const_iterator i = nid_map.find(nid);
          I(i != nid_map.end());
          W(F("cannot drop non-empty directory '%s'") % i->second);
          conflicts++;
          for (dir_map::const_iterator j = dir->children.begin();
               j != dir->children.end(); ++j)
            conflicting_paths.insert(i->second / j->first);
        }
    }
}

node_id
simulated_working_tree::create_dir_node()
{
  return workspace.create_dir_node(nis);
}

node_id
simulated_working_tree::create_file_node(file_id const & content)
{
  return workspace.create_file_node(content, nis);
}

void
simulated_working_tree::attach_node(node_id nid, file_path const & dst)
{
  // this check is needed for checkout because we're using a roster to
  // represent paths that *may* block the checkout. however to represent
  // these we *must* have a root node in the roster which will *always*
  // block us. so here we check for that case and avoid it.
  if (dst.empty() && workspace.has_root())
    return;

  if (workspace.has_node(dst))
    {
      W(F("attach node %d blocked by unversioned path '%s'") % nid % dst);
      blocked_paths.insert(dst);
      conflicting_paths.insert(dst);
      conflicts++;
    }
  else if (dst.empty())
    {
      // the parent of the workspace root cannot be in the blocked set
      // this attach would have been caught above if it were a problem
      workspace.attach_node(nid, dst);
    }
  else
    {
      file_path parent = dst.dirname();

      if (blocked_paths.find(parent) == blocked_paths.end())
        workspace.attach_node(nid, dst);
      else
        {
          W(F("attach node %d blocked by blocked parent '%s'")
            % nid % parent);
          blocked_paths.insert(dst);
        }
    }
}

void
simulated_working_tree::apply_delta(file_path const & path,
                                    file_id const & old_id,
                                    file_id const & new_id)
{
  // this may fail if path is not a file but that will be caught
  // earlier in update_current_roster_from_filesystem
}

void
simulated_working_tree::clear_attr(file_path const & path,
                                   attr_key const & key)
{
}

void
simulated_working_tree::set_attr(file_path const & path,
                                 attr_key const & key,
                                 attr_value const & val)
{
}

void
simulated_working_tree::commit()
{
  // This used to error out on any conflicts, but now some can be resolved
  // (by --move-conflicting-paths), so we just warn. The non-resolved
  // conflicts generate other errors downstream.
  if (conflicts > 0)
    F("%d workspace conflicts") % conflicts;
}

simulated_working_tree::~simulated_working_tree()
{
}


}; // anonymous namespace

static void
move_conflicting_paths_into_bookkeeping(set<file_path> const & leftover_paths)
{
  I(leftover_paths.size() > 0);

  // There is some concern that this fixed bookkeeping path will cause
  // problems, if a user forgets to clean up, and then does something that
  // involves the same name again. However, I can't think of a reasonable
  // use case that does that, so I can't think of a reasonable solution. One
  // solution is to generate a random directory name, another is to use the
  // current time in some format to generate a directory name.
  //
  // now().as_iso_8601_extended doesn't work on Windows, because it has
  // colons in it.
  //
  // Random or time based directory names significantly complicate testing,
  // since you can't predict the directory name.
  //
  // If this turns out to be a problem, a modification of
  // now().as_iso_8601_extended to eliminate the colons, or some appropriate
  // format for now().as_formatted_localtime would be simple and
  // probably adequate.
  bookkeeping_path leftover_path = bookkeeping_root / "resolutions";

  mkdir_p(leftover_path);

  for (set<file_path>::const_iterator i = leftover_paths.begin();
        i != leftover_paths.end(); ++i)
    {
      L(FL("processing %s") % *i);

      file_path basedir = (*i).dirname();
      if (!basedir.empty())
        mkdir_p(leftover_path / basedir);

      bookkeeping_path new_path = leftover_path / *i;
      if (directory_exists(*i))
        move_dir(*i, new_path);
      else if (file_exists(*i))
        move_file(*i, new_path);
      else
        I(false);

      P(F("moved conflicting path %s to %s") % *i % new_path);
    }
}

static void
add_parent_dirs(database & db, node_id_source & nis, workspace & work,
                file_path const & dst, roster_t & ros)
{
  editable_roster_base er(ros, nis);
  addition_builder build(db, work, ros, er, false, true);

  // FIXME: this is a somewhat odd way to use the builder
  build.visit_dir(dst.dirname());
}

// updating rosters from the workspace

void
workspace::update_current_roster_from_filesystem(roster_t & ros)
{
  update_current_roster_from_filesystem(ros, node_restriction());
}

void
workspace::update_current_roster_from_filesystem(roster_t & ros,
                                                 node_restriction const & mask)
{
  temp_node_id_source nis;
  inodeprint_map ipm;

  if (in_inodeprints_mode())
    {
      data dat;
      read_inodeprints(dat);
      read_inodeprint_map(dat, ipm);
    }

  size_t missing_items = 0;

  // this code is speed critical, hence the use of inode fingerprints so be
  // careful when making changes in here and preferably do some timing tests

  if (!ros.has_root())
    return;

  node_map const & nodes = ros.all_nodes();
  for (node_map::const_iterator i = nodes.begin(); i != nodes.end(); ++i)
    {
      node_id nid = i->first;
      node_t node = i->second;

      // Only analyze restriction-included files and dirs
      if (!mask.includes(ros, nid))
        continue;

      file_path fp;
      ros.get_name(nid, fp);

      const path::status status(get_path_status(fp));

      if (is_dir_t(node))
        {
          if (status == path::nonexistent)
            {
              W(F("missing directory '%s'") % (fp));
              missing_items++;
            }
          else if (status != path::directory)
            {
              W(F("not a directory '%s'") % (fp));
              missing_items++;
            }
        }
      else
        {
          // Only analyze changed files (or all files if inodeprints mode
          // is disabled).
          if (inodeprint_unchanged(ipm, fp))
            continue;

          if (status == path::nonexistent)
            {
              W(F("missing file '%s'") % (fp));
              missing_items++;
            }
          else if (status != path::file)
            {
              W(F("not a file '%s'") % (fp));
              missing_items++;
            }

          file_id fid;
          ident_existing_file(fp, fid, status);
          file_t file = downcast_to_file_t(node);
          if (file->content != fid)
            {
              ros.unshare(node);
              downcast_to_file_t(node)->content = fid;
            }
        }

    }

  E(missing_items == 0, origin::user,
    F("%d missing items; use '%s ls missing' to view\n"
      "To restore consistency, on each missing item run either\n"
      " '%s drop ITEM' to remove it permanently, or\n"
      " '%s revert ITEM' to restore it.\n"
      "To handle all at once, simply use\n"
      " '%s drop --missing' or\n"
      " '%s revert --missing'")
    % missing_items % prog_name % prog_name % prog_name
    % prog_name % prog_name);
}

void
workspace::find_missing(roster_t const & new_roster_shape,
                        node_restriction const & mask,
                        set<file_path> & missing)
{
  node_map const & nodes = new_roster_shape.all_nodes();
  for (node_map::const_iterator i = nodes.begin(); i != nodes.end(); ++i)
    {
      node_id nid = i->first;

      if (!new_roster_shape.is_root(nid)
          && mask.includes(new_roster_shape, nid))
        {
          file_path fp;
          new_roster_shape.get_name(nid, fp);
          if (!path_exists(fp))
            missing.insert(fp);
        }
    }
}

void
workspace::find_unknown_and_ignored(database & db,
                                    path_restriction const & mask,
                                    vector<file_path> const & roots,
                                    set<file_path> & unknown,
                                    set<file_path> & ignored)
{
  set<file_path> known;
  roster_t new_roster;
  temp_node_id_source nis;

  get_current_roster_shape(db, nis, new_roster);
  new_roster.extract_path_set(known);

  file_itemizer u(db, *this, known, unknown, ignored, mask);
  for (vector<file_path>::const_iterator
         i = roots.begin(); i != roots.end(); ++i)
    {
      walk_tree(*i, u);
    }
}

void
workspace::perform_additions(database & db, set<file_path> const & paths,
                             bool recursive, bool respect_ignore)
{
  if (paths.empty())
    return;

  temp_node_id_source nis;
  roster_t new_roster;
  MM(new_roster);
  get_current_roster_shape(db, nis, new_roster);

  editable_roster_base er(new_roster, nis);

  if (!new_roster.has_root())
    {
      er.attach_node(er.create_dir_node(), file_path_internal(""));
    }

  I(new_roster.has_root());
  addition_builder build(db, *this, new_roster, er, respect_ignore, recursive);

  for (set<file_path>::const_iterator i = paths.begin(); i != paths.end(); ++i)
    {
      if (recursive)
        {
          // NB.: walk_tree will handle error checking for non-existent paths
          walk_tree(*i, build);
        }
      else
        {
          // in the case where we're just handed a set of paths, we use the
          // builder in this strange way.
          switch (get_path_status(*i))
            {
            case path::nonexistent:
              E(false, origin::user,
                F("no such file or directory: '%s'") % *i);
              break;
            case path::file:
              build.visit_file(*i);
              break;
            case path::directory:
              build.visit_dir(*i);
              break;
            }
        }
    }

  parent_map parents;
  get_parent_rosters(db, parents);

  revision_t new_work;
  make_revision_for_workspace(parents, new_roster, new_work);
  put_work_rev(new_work);
}

static bool
in_parent_roster(const parent_map & parents, const node_id & nid)
{
  for (parent_map::const_iterator i = parents.begin();
       i != parents.end();
       i++)
    {
      if (parent_roster(i).has_node(nid))
        return true;
    }

  return false;
}

void
workspace::perform_deletions(database & db,
                             set<file_path> const & paths,
                             bool recursive, bool bookkeep_only)
{
  if (paths.empty())
    return;

  temp_node_id_source nis;
  roster_t new_roster;
  MM(new_roster);
  get_current_roster_shape(db, nis, new_roster);

  parent_map parents;
  get_parent_rosters(db, parents);

  // we traverse the the paths backwards, so that we always hit deep paths
  // before shallow paths (because set<file_path> is lexicographically
  // sorted).  this is important in cases like
  //    monotone drop foo/bar foo foo/baz
  // where, when processing 'foo', we need to know whether or not it is empty
  // (and thus legal to remove)

  deque<file_path> todo;
  set<file_path>::const_reverse_iterator i = paths.rbegin();
  todo.push_back(*i);
  ++i;

  while (todo.size())
    {
      file_path const & name(todo.front());

      E(!name.empty(), origin::user,
        F("unable to drop the root directory"));

      if (!new_roster.has_node(name))
        P(F("skipping %s, not currently tracked") % name);
      else
        {
          const_node_t n = new_roster.get_node(name);
          if (is_dir_t(n))
            {
              const_dir_t d = downcast_to_dir_t(n);
              if (!d->children.empty())
                {
                  E(recursive, origin::user,
                    F("cannot remove %s/, it is not empty") % name);
                  for (dir_map::const_iterator j = d->children.begin();
                       j != d->children.end(); ++j)
                    todo.push_front(name / j->first);
                  continue;
                }
            }
          if (!bookkeep_only && path_exists(name)
              && in_parent_roster(parents, n->self))
            {
              if (is_dir_t(n))
                {
                  if (directory_empty(name))
                    delete_file_or_dir_shallow(name);
                  else
                    W(F("directory %s not empty - "
                        "it will be dropped but not deleted") % name);
                }
              else
                {
                  const_file_t file = downcast_to_file_t(n);
                  file_id fid;
                  I(ident_existing_file(name, fid));
                  if (file->content == fid)
                    delete_file_or_dir_shallow(name);
                  else
                    W(F("file %s changed - "
                        "it will be dropped but not deleted") % name);
                }
            }
          P(F("dropping %s from workspace manifest") % name);
          new_roster.drop_detached_node(new_roster.detach_node(name));
        }
      todo.pop_front();
      if (i != paths.rend())
        {
          todo.push_back(*i);
          ++i;
        }
    }

  revision_t new_work;
  make_revision_for_workspace(parents, new_roster, new_work);
  put_work_rev(new_work);
}

void
workspace::perform_rename(database & db,
                          set<file_path> const & srcs,
                          file_path const & dst,
                          bool bookkeep_only)
{
  temp_node_id_source nis;
  roster_t new_roster;
  MM(new_roster);
  set< pair<file_path, file_path> > renames;

  I(!srcs.empty());

  get_current_roster_shape(db, nis, new_roster);

  // validation.  it's okay if the target exists as a file; we just won't
  // clobber it (in !--bookkeep-only mode).  similarly, it's okay if the
  // source does not exist as a file.
  if (srcs.size() == 1 && !new_roster.has_node(dst))
    {
      // "rename SRC DST" case
      file_path const & src = *srcs.begin();
      file_path dpath = dst;

      E(!src.empty(), origin::user,
        F("cannot rename the workspace root (try '%s pivot_root' instead)")
        % prog_name);
      E(new_roster.has_node(src), origin::user,
        F("source file %s is not versioned") % src);

      //this allows the 'magic add' of a non-versioned directory to happen in
      //all cases.  previously, mtn mv fileA dir/ woudl fail if dir/ wasn't
      //versioned whereas mtn mv fileA dir/fileA would add dir/ if necessary
      //and then reparent fileA.
      if (get_path_status(dst) == path::directory)
        dpath = dst / src.basename();
      else
        {
          //this handles the case where:
          // touch foo
          // mtn mv foo bar/foo where bar doesn't exist
          file_path parent = dst.dirname();
          E(get_path_status(parent) == path::directory, origin::user,
            F("destination path's parent directory %s/ doesn't exist") % parent);
        }

      renames.insert(make_pair(src, dpath));
      add_parent_dirs(db, nis, *this, dpath, new_roster);
    }
  else
    {
      // "rename SRC1 [SRC2 ...] DSTDIR" case
      E(get_path_status(dst) == path::directory, origin::user,
        F("destination %s/ is not a directory") % dst);

      for (set<file_path>::const_iterator i = srcs.begin();
           i != srcs.end(); i++)
        {
          E(!i->empty(), origin::user,
            F("cannot rename the workspace root (try '%s pivot_root' instead)")
            % prog_name);
          E(new_roster.has_node(*i), origin::user,
            F("source file %s is not versioned") % *i);

          file_path d = dst / i->basename();
          E(!new_roster.has_node(d), origin::user,
            F("destination %s already exists in the workspace manifest") % d);

          renames.insert(make_pair(*i, d));

          add_parent_dirs(db, nis, *this, d, new_roster);
        }
    }

  // do the attach/detaching
  for (set< pair<file_path, file_path> >::const_iterator i = renames.begin();
       i != renames.end(); i++)
    {
      node_id nid = new_roster.detach_node(i->first);
      new_roster.attach_node(nid, i->second);
      P(F("renaming %s to %s in workspace manifest") % i->first % i->second);
    }

  parent_map parents;
  get_parent_rosters(db, parents);

  revision_t new_work;
  make_revision_for_workspace(parents, new_roster, new_work);
  put_work_rev(new_work);

  if (!bookkeep_only)
    for (set< pair<file_path, file_path> >::const_iterator i = renames.begin();
         i != renames.end(); i++)
      {
        file_path const & s(i->first);
        file_path const & d(i->second);
        // silently skip files where src doesn't exist or dst does
        bool have_src = path_exists(s);
        bool have_dst = path_exists(d);
        if (have_src && !have_dst)
          {
            move_path(s, d);
          }
        else if (!have_src && !have_dst)
          {
            W(F("%s doesn't exist in workspace, skipping") % s);
          }
        else if (have_src && have_dst)
          {
            W(F("destination %s already exists in workspace, "
                "skipping filesystem rename") % d);
          }
        else
          {
            W(F("%s doesn't exist in workspace and %s does, "
                "skipping filesystem rename") % s % d);
          }
      }
}

void
workspace::perform_pivot_root(database & db,
                              file_path const & new_root,
                              file_path const & put_old,
                              bool bookkeep_only,
                              bool move_conflicting_paths)
{
  temp_node_id_source nis;
  roster_t old_roster, new_roster;
  MM(old_roster);
  MM(new_roster);
  get_current_roster_shape(db, nis, old_roster);

  I(old_roster.has_root());
  E(old_roster.has_node(new_root), origin::user,
    F("proposed new root directory '%s' is not versioned or does not exist")
    % new_root);
  E(is_dir_t(old_roster.get_node(new_root)), origin::user,
    F("proposed new root directory '%s' is not a directory") % new_root);
  {
    E(!old_roster.has_node(new_root / bookkeeping_root_component), origin::user,
      F("proposed new root directory '%s' contains illegal path %s")
      % new_root % bookkeeping_root);
  }

  {
    file_path current_path_to_put_old = (new_root / put_old);
    file_path current_path_to_put_old_parent
      = current_path_to_put_old.dirname();

    E(old_roster.has_node(current_path_to_put_old_parent), origin::user,
      F("directory '%s' is not versioned or does not exist")
      % current_path_to_put_old_parent);
    E(is_dir_t(old_roster.get_node(current_path_to_put_old_parent)),
      origin::user,
      F("'%s' is not a directory")
      % current_path_to_put_old_parent);
    E(!old_roster.has_node(current_path_to_put_old),
      origin::user,
      F("'%s' is in the way") % current_path_to_put_old);
  }

  cset cs;
  safe_insert(cs.nodes_renamed, make_pair(file_path_internal(""), put_old));
  safe_insert(cs.nodes_renamed, make_pair(new_root, file_path_internal("")));

  {
    new_roster = old_roster;
    editable_roster_base e(new_roster, nis);
    cs.apply_to(e);
  }

  {
    parent_map parents;
    get_parent_rosters(db, parents);

    revision_t new_work;
    make_revision_for_workspace(parents, new_roster, new_work);
    put_work_rev(new_work);
  }
  if (!bookkeep_only)
    {
      content_merge_empty_adaptor cmea;
      perform_content_update(old_roster, new_roster, cs, cmea, true, move_conflicting_paths);
    }
}

void
workspace::perform_content_update(roster_t const & old_roster,
                                  roster_t const & new_roster,
                                  cset const & update,
                                  content_merge_adaptor const & ca,
                                  bool const messages,
                                  bool const move_conflicting_paths)
{
  roster_t test_roster;
  temp_node_id_source nis;
  set<file_path> known;
  bookkeeping_path detached = path_for_detached_nids();

  E(!directory_exists(detached), origin::user,
    F("workspace is locked\n"
      "you must clean up and remove the %s directory")
    % detached);

  old_roster.extract_path_set(known);

  workspace_itemizer itemizer(test_roster, known, nis);
  walk_tree(file_path(), itemizer);

  simulated_working_tree swt(test_roster, nis);
  update.apply_to(swt);

  // if we have found paths during the test-run which will conflict with
  // newly attached or to-be-dropped nodes, move these paths out of the way
  // into _MTN while keeping the path to these paths intact in case the user
  // wants them back
  if (swt.has_conflicting_paths())
    {
      E(move_conflicting_paths, origin::user,
        F("re-run this command with --move-conflicting-paths to move "
          "conflicting paths out of the way."));
      move_conflicting_paths_into_bookkeeping(swt.get_conflicting_paths());
    }

  mkdir_p(detached);

  editable_working_tree ewt(*this, this->lua, ca, messages);
  update.apply_to(ewt);

  // attributes on updated files must be reset because apply_delta writes
  // new versions of files to _MTN/tmp and then renames them over top of the
  // old versions and doesn't reset attributes (mtn:execute).

  for (map<file_path, pair<file_id, file_id> >::const_iterator
         i = update.deltas_applied.begin(); i != update.deltas_applied.end();
       ++i)
    {
      const_node_t node = new_roster.get_node(i->first);
      for (attr_map_t::const_iterator a = node->attrs.begin();
           a != node->attrs.end(); ++a)
        {
          if (a->second.first)
            this->lua.hook_set_attribute(a->first(), i->first,
                                         a->second.second());
        }
    }

  delete_dir_shallow(detached);
}

// Local Variables:
// mode: C++
// fill-column: 76
// c-file-style: "gnu"
// indent-tabs-mode: nil
// End:
// vim: et:sw=2:sts=2:ts=2:cino=>2s,{s,\:s,+s,t0,g0,^-2,e-2,n-2,p2s,(0,=s:
