
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apr_errno.h>
#include <apr_tables.h>
#include <string.h>

#include <libbtutil.h>
#include <libbtpeer/bitfield.h>
#include <libbtpeer/types/btp_torrent.h>
#include <libbtpeer/types/btp_peer.h>
#include <libbtpeer/hooks/peer_command.h>
#include <libbtpeer/metainfo.h>

void btp_torrent_destroy(btp_torrent* t) {
    if(t->myfiles)
        btp_file_pool_destroy(t->files);
    
    /* TODO: socket closing, etc? */
    
    apr_pool_destroy(t->pool);
    return;
}

btp_torrent* btp_torrent_create(
    apr_pool_t* p, bt_metainfo* info, btp_file_pool* file_pool
) {
    btp_torrent* rv;
    apr_pool_t* pool;
    
    apr_pool_create(&pool, p);
    rv = apr_pcalloc(pool, sizeof(btp_torrent));
    rv->pool = pool;
    rv->info = apr_pmemdup(pool, info, sizeof(bt_metainfo));
    rv->status = BTP_TORRENT_STATUS_NEW | BTP_TORRENT_STATUS_STOPPED;
    rv->block_size = BTP_TORRENT_BLOCK(info);
    rv->block_count = (info->total_size + rv->block_size - 1) / rv->block_size;
    rv->known_peers = apr_table_make(pool, BTP_TORRENT_MAX_PEERS);
    if(file_pool) {
        rv->files = file_pool;
    } else {
        rv->files = btp_file_pool_create(pool, 50);    
        rv->myfiles = 1;
    }
    rv->peers = btp_peer_list_new(pool, BTP_TORRENT_MAX_PEERS);
        
    return rv;
}

apr_status_t btp_torrent_peer_check_input(btp_torrent* t, btp_peer* p) {
    int ret;
    
    if(p->state & BTP_PEER_STATE_IN_COMMAND) {
        if(p->cmdbuf_expected == p->cmdbuf_len)
            ret = btp_run_command(p->command, t, p, NULL);
        else
            ret = btp_peer_recv_in_cmd(p);
    } else {
        ret = btp_peer_recv_no_cmd(p);
    }
    
    return ret;
}

apr_status_t btp_torrent_peer_run(btp_torrent* t, btp_peer* p) {
    return btp_torrent_peer_check_input(t, p);
}

apr_status_t btp_torrent_peers_run(btp_torrent* t) {
    apr_status_t rv = APR_SUCCESS;
    bt_llist_e* c;
    
    for(c=t->peers->first; c; c=c->next)
        if((rv = btp_torrent_peer_run(t, (btp_peer*)c->d)) != APR_SUCCESS)
            return rv;
        
    return rv;
}

apr_status_t btp_torrent_run(btp_torrent* t) {
    time_t now;
    
    /* do nothing */
    if(t->status & (BTP_TORRENT_STATUS_STOPPED | BTP_TORRENT_STATUS_QUIT))
        return APR_SUCCESS;
    
    /* check metainfo */
    if(t->status & BTP_TORRENT_STATUS_NEW) {
        t->status =
            (t->status & ~BTP_TORRENT_STATUS_NEW) | BTP_TORRENT_STATUS_CHECK;
        
        return btp_metainfo_check(t, NULL);
    }
    
    /* shutdown due to error */
    if(t->status & BTP_TORRENT_STATUS_ERROR) {
        t->status = t->status | BTP_TORRENT_STATUS_QUIT;
        return APR_SUCCESS;
    } 
    
    /* activate torrent */
    t->status |= BTP_TORRENT_STATUS_ACTIVE;
    now = time(NULL);
    
    /* send announce request */
    if((!(t->status & BTP_TORRENT_STATUS_ANNOUNCED)) || now <= t->n_announce_t)
        return btp_torrent_announce(t);
    
    
    
    return APR_SUCCESS;
}

uint64_t btp_torrent_bytes_left(btp_torrent* torrent) {
    uint64_t left;
    
    left =
        (uint64_t) (torrent->block_count - torrent->pieces->block_count) *
        (uint64_t) torrent->block_size;
    
    if(
        !btp_bitfield_has(torrent->pieces->block_bits, torrent->block_count-1)
        &&
        torrent->info->total_size % torrent->block_size
    ) {
        left += torrent->info->total_size % torrent->block_size;
        left -= torrent->block_size;
    }
    
    return left;
}

int btp_torrent_piece_count_blocks(btp_torrent* tor, int piece) {
    if(
        piece < tor->info->piece_count - 1 ||
        !(tor->block_count % (tor->info->piece_size / tor->block_size))
    )
        return tor->info->piece_size / tor->block_size;
    else
        return tor->block_count % (tor->info->piece_size / tor->block_size);
}

apr_status_t btp_torrent_receive_peerlist_peer(
    btp_torrent* t, bt_bcode* peer, apr_pool_t* temp
) {
    bt_bcode* bip = bt_bcode_find(peer, "ip");
    bt_bcode* bport = bt_bcode_find(peer, "port");
    apr_sockaddr_t* addr;
    apr_port_t port;
    char* ip;
    char* key;
    apr_status_t ret;
    
    if(!(bip && bport))
        return APR_EBADIP;
    
    port = bport->val.i;
    
    if((ret = apr_sockaddr_info_get(
        &addr, bip->val.s.s, APR_UNSPEC, port, APR_IPV4_ADDR_OK, temp
    )) != APR_SUCCESS)
        return ret;
    
    if((ret = apr_sockaddr_ip_get(&ip, addr)) != APR_SUCCESS)
        return ret;
    
    key = apr_psprintf(temp, "%s:%i", ip, port);
    if(!apr_table_get(t->known_peers, key))
        apr_table_set(t->known_peers, key, "");
    
    return APR_SUCCESS;
}

apr_status_t btp_torrent_receive_peerlist(btp_torrent* t, bt_bcode* peers) {
    int i;
    apr_status_t ret = APR_SUCCESS;
    apr_pool_t* temp;
    
    apr_pool_create(&temp, t->pool);
    
    for(i=0; (ret == APR_SUCCESS) && (i<peers->val.l.count); i++)
        ret = btp_torrent_receive_peerlist_peer(
            t, &(peers->val.l.vals[i]), temp
        );
    
    apr_pool_destroy(temp);
   
    return ret;
}
