/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* GNOME Volume Applet
 * Copyright (C) 2004 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * applet.c: the main applet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* this is for lrint */
#define _ISOC99_SOURCE
#include <math.h>
#include <string.h>

#include <glib-object.h>
#include <gdk/gdkkeysyms.h>

#include <gtk/gtkaboutdialog.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkicontheme.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkrange.h>
#include <gtk/gtkwidget.h>

#include <gconf/gconf-client.h>
#include "moblin.h"
#include "applet.h"
#include "volume.h"

#define SLIDER_WIDTH 42
#define pse get_primary_sound_element_index

GType 		moblin_volume_applet_get_type 	(void);
static void	moblin_volume_applet_class_init	(MoblinVolumeAppletClass *klass);
static void	moblin_volume_applet_init	(MoblinVolumeApplet *applet);
G_DEFINE_TYPE (MoblinVolumeApplet, moblin_volume_applet, GTK_TYPE_BUTTON)

static gboolean	moblin_volume_applet_scroll	(MoblinVolumeApplet *applet, GdkEventScroll *event);
static gboolean	moblin_volume_applet_button	(MoblinVolumeApplet *applet, GdkEventButton *event);
static void	moblin_volume_applet_refresh	(MoblinVolumeApplet *applet, gboolean force_refresh);
static void	moblin_volume_applet_dispose	(MoblinVolumeApplet *applet);
static void     moblin_volume_applet_size_allocate (MoblinVolumeApplet *applet, GtkAllocation *allocation);
static void	moblin_volume_applet_popup_dock	(MoblinVolumeApplet *applet);
static void	moblin_volume_applet_popdown_dock (MoblinVolumeApplet *applet);
static void	cb_volume			(GtkAdjustment *adj, gpointer data);
static void	cb_theme_change			(GtkIconTheme *icon_theme, gpointer data);
static void	cb_stop_scroll_events		(GtkWidget *widget, GdkEvent  *event);
static int	set_alsa_volume			(MoblinVolumeApplet *applet, int value);
static int	get_alsa_volume			(MoblinVolumeApplet *applet);
static int vmin, vmax;

static int
set_alsa_volume(MoblinVolumeApplet *applet, int value)
{

  if(applet->mixerfd < 0) return value;

  if (value > vmax) value = vmax;
  if (value < vmin) value = vmin;

  set_all_volume(applet->mixerfd, value);

  return value;
}

static int
get_alsa_volume(MoblinVolumeApplet *applet)
{
  int vol;

  if(applet->mixerfd < 0) return 0;

  get_volume(applet->mixerfd, 0, &vol);

  return vol;
}

static void
init_pixbufs (MoblinVolumeApplet *applet)
{
  static const gchar *pix_filenames[] = {
    "moblin-volumelevel0",
    "moblin-volumelevel1",
    "moblin-volumelevel2",
    "moblin-volumelevel3",
    "moblin-volumelevel4",
    NULL
  };
  gint n;
  
  for (n = 0; pix_filenames[n] != NULL; n++) {
    if (applet->pix[n])
      g_object_unref (applet->pix[n]);
    
    applet->pix[n] = gtk_icon_theme_load_icon (moblin_icon_theme_get_default (),
					       pix_filenames[n],
					       applet->panel_size - 2,
					       0,
					       NULL);
    if (gtk_widget_get_default_direction () == GTK_TEXT_DIR_RTL) {
      GdkPixbuf *temp;

      temp = gdk_pixbuf_flip (applet->pix[n], TRUE);
      g_object_unref (G_OBJECT (applet->pix[n]));
      applet->pix[n] = temp;
    }
  }
}

static void
moblin_volume_applet_class_init (MoblinVolumeAppletClass *klass)
{
	/* nothing to do here */
}

static void
moblin_volume_applet_init (MoblinVolumeApplet *applet)
{
  GtkWidget *dock;
  GtkWidget *image;
  AtkObject *ao;

  applet->timeout = 0;
  applet->elements = NULL;
  applet->client = gconf_client_get_default ();
  applet->lock = FALSE;
  applet->prefs = NULL;
  applet->dock = NULL;
  applet->panel_size = 50;

  /* init pixbufs */
  init_pixbufs (applet);

  /* icon (our main UI) */
  image = gtk_image_new ();
  applet->image = GTK_IMAGE (image);
  gtk_container_add (GTK_CONTAINER (applet), image);
  gtk_widget_show (image);

  /* dock window (expanded UI) */
  applet->pop = FALSE;

  /* tooltip over applet */
  gtk_widget_set_tooltip_text (GTK_WIDGET (applet), _("Volume Control"));

  g_signal_connect (G_OBJECT(applet), "dispose",
                    G_CALLBACK(moblin_volume_applet_dispose), NULL);

  g_signal_connect (G_OBJECT(applet), "size-allocate",
                    G_CALLBACK(moblin_volume_applet_size_allocate), NULL);

  g_signal_connect (G_OBJECT(applet), "scroll-event",
                    G_CALLBACK(moblin_volume_applet_scroll), NULL);

  g_signal_connect (G_OBJECT(applet), "button-press-event",
                    G_CALLBACK(moblin_volume_applet_button), NULL);

  /* prevent scroll events from reaching the tooltip */
  g_signal_connect (G_OBJECT (applet),
		    "event-after", G_CALLBACK (cb_stop_scroll_events),
		    NULL);

  /* handle icon theme changes */
  g_signal_connect (gtk_icon_theme_get_default (),
		    "changed", G_CALLBACK (cb_theme_change),
		    applet);

  /* i18n */
  ao = gtk_widget_get_accessible (GTK_WIDGET (applet));
  atk_object_set_name (ao, _("Volume Control"));

  dock = moblin_volume_applet_dock_new (SLIDER_WIDTH, vmin, vmax);
  /* parent, for signal forwarding */
  gtk_widget_set_parent (dock, GTK_WIDGET (applet));
  applet->dock = GNOME_VOLUME_APPLET_DOCK (dock);
}

gboolean
moblin_volume_applet_setup (MoblinVolumeApplet *applet)
{
  GtkObject *adj;

  applet->mixerfd = init_alsa_vars();
  if(applet->mixerfd >= 0)
  {
    applet->mixerfd = pse();
    get_volume_range(applet->mixerfd, &vmin, &vmax);
  }

  adj = gtk_adjustment_new (50, vmin, vmax, 4, 10, 0);
  gtk_adjustment_set_value (GTK_ADJUSTMENT (adj), get_alsa_volume (applet));
  moblin_volume_applet_dock_change (applet->dock, GTK_ADJUSTMENT (adj));
  g_signal_connect (adj, "value-changed", G_CALLBACK (cb_volume), applet);
  moblin_volume_applet_refresh (applet, TRUE);

  gtk_widget_show (GTK_WIDGET (applet));

  return TRUE;
}

static void
moblin_volume_applet_dispose (MoblinVolumeApplet *applet)
{
  gint n;

  moblin_volume_applet_popdown_dock (applet);

  if (applet->timeout) {
    g_source_remove (applet->timeout);
    applet->timeout = 0;
  }

  for (n = 0; n < 5; n++) {
    if (applet->pix[n] != NULL) {
      g_object_unref (G_OBJECT (applet->pix[n]));
      applet->pix[n] = NULL;
    }
  }
}

/*
 * popup (show) or popdown (hide) the dock.
 */

static void
moblin_volume_applet_popup_dock (MoblinVolumeApplet *applet)
{
  GtkWidget *widget = GTK_WIDGET (applet);
  gint x, y;

  /* need to know imperically the dock's allocation values */
  if((GTK_WIDGET(applet->dock)->allocation.x < 0) ||
     (GTK_WIDGET(applet->dock)->allocation.y < 0))
  {
    gtk_widget_realize (GTK_WIDGET (applet->dock));
  }

  /* we need to determine the x, y screen coords of the slider */
  /* first get the origin of the status bar relative to the screen origin */
  gdk_window_get_origin (widget->window, &x, &y);

  /* our applet's allocation.x,y coords are relative to the sbar origin */
  /* the slider should be aligned to the bottom center of the applet icon */
  x += widget->allocation.x + (widget->allocation.width/2) -
             (GTK_WIDGET (applet->dock)->allocation.width/2);
  y += widget->allocation.height + widget->allocation.y;

  gtk_window_move (GTK_WINDOW (applet->dock), x, y);
  gtk_widget_show_all (GTK_WIDGET (applet->dock));

  /* grab input */
  gtk_widget_grab_focus (GTK_WIDGET(applet));
  gtk_grab_add (widget);
  gdk_pointer_grab (widget->window, TRUE,
		    GDK_BUTTON_PRESS_MASK |
		    GDK_BUTTON_RELEASE_MASK |
		    GDK_POINTER_MOTION_MASK,
		    NULL, NULL, GDK_CURRENT_TIME);
  gdk_keyboard_grab (widget->window, TRUE, GDK_CURRENT_TIME);

  /* set menu item as active */
  gtk_widget_set_state (GTK_WIDGET (applet), GTK_STATE_SELECTED);
  moblin_volume_applet_refresh (applet, TRUE);

  /* keep state */
  applet->pop = TRUE;
}

static void
moblin_volume_applet_popdown_dock (MoblinVolumeApplet *applet)
{
  GtkWidget *widget = GTK_WIDGET (applet);

  if (!applet->pop)
    return;

  /* release input */
  gdk_keyboard_ungrab (GDK_CURRENT_TIME);
  gdk_pointer_ungrab (GDK_CURRENT_TIME);
  gtk_grab_remove (widget);

  /* hide */
  gtk_widget_hide_all (GTK_WIDGET (applet->dock));

  /* set menu item as active */
  gtk_widget_set_state (GTK_WIDGET (applet), GTK_STATE_NORMAL);
  moblin_volume_applet_refresh (applet, TRUE);

  /* keep state */
  applet->pop = FALSE;
}

static void
moblin_volume_applet_pop_dock (MoblinVolumeApplet *applet)
{
  if (applet->pop) {
    moblin_volume_applet_popdown_dock (applet);
  } else {
    moblin_volume_applet_popup_dock (applet);
  }
}

/*
 * Control events, change volume and so on.
 */

static gboolean
moblin_volume_applet_scroll (MoblinVolumeApplet *applet,
			    GdkEventScroll *event)
{
  if (event->type == GDK_SCROLL) {
    switch (event->direction) {
      case GDK_SCROLL_UP:
      case GDK_SCROLL_DOWN: {
        GtkAdjustment *adj = gtk_range_get_adjustment (applet->dock->scale);
        gdouble volume = adj->value;

        if (event->direction == GDK_SCROLL_UP) {
          volume += adj->step_increment;
          if (volume > adj->upper)
            volume = adj->upper;
        } else {
          volume -= adj->step_increment;
          if (volume < adj->lower)
            volume = adj->lower;
        }

        gtk_range_set_value (applet->dock->scale, volume);
        return TRUE;
      }
      default:
        break;
    }
  }

  return FALSE;
}

static gboolean
moblin_volume_applet_button (MoblinVolumeApplet *applet,
			    GdkEventButton *event)
{
  /* react only to left mouse button */
  if (event->button != 1) {
       return FALSE;
  }

  if(applet->mixerfd >= 0)
    moblin_volume_applet_pop_dock (applet);
  return TRUE;
}

void moblin_volume_applet_size_allocate (MoblinVolumeApplet *applet, 
					GtkAllocation *allocation)
{
  if (applet->panel_size == allocation->height)
    return;
  applet->panel_size = allocation->height;

  init_pixbufs (applet);
  moblin_volume_applet_refresh (applet, TRUE);
}

/*
 * Volume changed.
 */

static void
cb_volume (GtkAdjustment *adj,
	   gpointer data)
{
  MoblinVolumeApplet *applet = data;
  gdouble v;

  if (applet->lock)
    return;
  applet->lock = TRUE;

  v = gtk_adjustment_get_value (adj);

  set_alsa_volume(applet, (int)v);
  applet->lock = FALSE;
  applet->force_next_update = TRUE;

  moblin_volume_applet_refresh (GNOME_VOLUME_APPLET (data), FALSE);
}

/*
 * Automatic timer. Check for changes.
 */

#define STATE(vol,m) (((gint) vol << 1) | (m ? 1 : 0))

static void
moblin_volume_applet_refresh (MoblinVolumeApplet *applet,
			     gboolean           force_refresh)
{
  GdkPixbuf *pixbuf;
  gint n, volume;
  char tooltip_str[100];
  gboolean mute = FALSE, did_change;

  volume = get_alsa_volume (applet);
  if (volume <= 0) {
    mute = TRUE;
    n = 0;
  } else {
    /* select image */
    n = 4 * volume / vmax + 1;
    if (n < 1)
      n = 1;
    if (n > 4)
      n = 4;
  }

  did_change = (force_refresh || applet->force_next_update);
  applet->force_next_update = FALSE;

  if (did_change) {
    if (mute) {
      pixbuf = applet->pix[0];
    } else {
      pixbuf = applet->pix[n];
    }
    gtk_image_set_from_pixbuf (applet->image, pixbuf);
  } else {
	return;
  }

  if (mute) {
    sprintf(tooltip_str, "Volume: muted");
  } else {
    sprintf(tooltip_str, "Volume: %d%%", volume);
  }

  gtk_widget_set_tooltip_text (GTK_WIDGET (applet), tooltip_str);

  applet->lock = TRUE;
  gtk_range_set_value (applet->dock->scale, volume);
  applet->lock = FALSE;
}

static void
cb_theme_change (GtkIconTheme *icon_theme,
		 gpointer data)
{
  MoblinVolumeApplet *applet = GNOME_VOLUME_APPLET (data);

  init_pixbufs (applet);
  moblin_volume_applet_refresh (applet, TRUE);
}

/*
 * Block the tooltips event-after handler on scroll events.
 */

static void
cb_stop_scroll_events (GtkWidget *widget,
		       GdkEvent  *event)
{
  if (event->type == GDK_SCROLL)
    g_signal_stop_emission_by_name (widget, "event-after");
}

MoblinVolume* 
moblin_volume_new()
{
MoblinVolume *obj;

  if(!moblin_volume_applet_get_type())
  {
    g_warning("Failed to register type MoblinVolumeApplet\n");
    return NULL;
  }
  obj = g_new0(MoblinVolume, 1);
  obj->applet = (MoblinVolumeApplet*)g_object_new(moblin_volume_applet_get_type(), NULL);

  return obj;
}
