/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 *
 * GNOME Power Manager Brightness Applet
 * Copyright (C) 2006 Benjamin Canou <bookeldor@gmail.com>
 * Copyright (C) 2007 Richard Hughes <richard@hughsie.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtkbox.h>
#include <gdk/gdkkeysyms.h>
#include <glib-object.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <dbus/dbus-glib.h>
#include "libbrightnesscontrol/brightness.h"
#include <libhildondesktop/hildon-status-bar-item.h>
#include "moblin.h"

#define GPM_TYPE_BRIGHTNESS_APPLET		(moblin_brightness_applet_get_type ())
#define GPM_BRIGHTNESS_APPLET(o)		(G_TYPE_CHECK_INSTANCE_CAST ((o), GPM_TYPE_BRIGHTNESS_APPLET, MoblinBrightnessApplet))
#define GPM_BRIGHTNESS_APPLET_CLASS(k)		(G_TYPE_CHECK_CLASS_CAST((k), GPM_TYPE_BRIGHTNESS_APPLET, MoblinBrightnessAppletClass))
#define GPM_IS_BRIGHTNESS_APPLET(o)		(G_TYPE_CHECK_INSTANCE_TYPE ((o), GPM_TYPE_BRIGHTNESS_APPLET))
#define GPM_IS_BRIGHTNESS_APPLET_CLASS(k)	(G_TYPE_CHECK_CLASS_TYPE ((k), GPM_TYPE_BRIGHTNESS_APPLET))
#define GPM_BRIGHTNESS_APPLET_GET_CLASS(o)	(G_TYPE_INSTANCE_GET_CLASS ((o), GPM_TYPE_BRIGHTNESS_APPLET, MoblinBrightnessAppletClass))

#define SLIDER_WIDTH 42

typedef struct{
	GtkButton parent;
	GtkImage *image;
	/* applet state */
	gboolean call_worked; /* g-p-m refusing action */
	gboolean popped; /* the popup is shown */
	/* the popup and its widgets */
	GtkWidget *popup, *slider, *btn_plus, *btn_minus;
	/* the icon and a cache for size*/
	GdkPixbuf *icon[4];
	/* a cache for panel size */
	gint size;
	/* BrightnessControl members */
        DBusGProxy *proxy;
        DBusGConnection *connection;
        gchar *udi;
        guint min_level;
        guint max_level;
        guint level;
        void (*brightness_changed_cb)(DBusGProxy*, guint, struct _BrightnessControl*);
} MoblinBrightnessApplet;

typedef struct{
	GtkButtonClass	parent_class;
} MoblinBrightnessAppletClass;

typedef struct _MoblinBrightness {
  HildonStatusBarItem* item;
  GtkWidget* button;
  MoblinBrightnessApplet* applet;
} MoblinBrightness;

GType            moblin_brightness_applet_get_type   (void);
static void      moblin_brightness_applet_class_init (MoblinBrightnessAppletClass *klass);
static void      moblin_brightness_applet_init       (MoblinBrightnessApplet *applet);
G_DEFINE_TYPE (MoblinBrightnessApplet, moblin_brightness_applet, GTK_TYPE_BUTTON)

static gboolean  moblin_applet_scroll_cb             (MoblinBrightnessApplet *applet, GdkEventScroll *event);
static gboolean  moblin_applet_popup_cb              (MoblinBrightnessApplet *applet, GdkEventButton *event);
static void      init_pixbufs              (MoblinBrightnessApplet *applet);
static void      moblin_applet_size_allocate         (MoblinBrightnessApplet *applet, GtkAllocation *allocation);
static gboolean  moblin_applet_draw_cb               (MoblinBrightnessApplet *applet);
static void      moblin_applet_theme_change_cb       (GtkIconTheme *icon_theme, gpointer data);
static void      moblin_applet_stop_scroll_events_cb (GtkWidget *widget, GdkEvent  *event);
static gboolean  moblin_applet_destroy_popup_cb      (MoblinBrightnessApplet *applet);
static void      moblin_applet_update_tooltip        (MoblinBrightnessApplet *applet);
static void      moblin_applet_update_popup_level    (MoblinBrightnessApplet *applet);
static gboolean  moblin_applet_plus_cb               (GtkWidget *w, MoblinBrightnessApplet *applet);
static gboolean  moblin_applet_minus_cb              (GtkWidget *w, MoblinBrightnessApplet *applet);
static gboolean  moblin_applet_slide_cb              (GtkWidget *w, MoblinBrightnessApplet *applet);
static void      moblin_applet_create_popup          (MoblinBrightnessApplet *applet);
static void      moblin_applet_destroy_cb            (GtkObject *object);

#define GPM_BRIGHTNESS_APPLET_NAME		_("Power Manager Brightness Applet")
#define GPM_BRIGHTNESS_APPLET_DESC		_("Adjusts laptop panel brightness.")

/**
 * init_pixbufs:
 * @applet: Brightness applet instance
 *
 * retrieve an icon from stock with a size adapted to panel
 **/
static void
init_pixbufs (MoblinBrightnessApplet *applet)
{
  static const gchar *pix_filenames[] = {
    "moblin-displaybright1",
    "moblin-displaybright2",
    "moblin-displaybright3",
    "moblin-displaybright4",
    NULL
  };
  int i;

	for(i = 0; i < 4; i++) {
	    if (applet->icon[i])
		g_object_unref (applet->icon[i]);

	    applet->icon[i] = gtk_icon_theme_load_icon (moblin_icon_theme_get_default (),
		pix_filenames[i], applet->size - 2, 0, NULL);
	}
}

/**
 * moblin_applet_size_allocate:
 * @applet: Brightness applet instance
 *
 * check if panel size has changed and applet adapt size
 **/
static void
moblin_applet_size_allocate (MoblinBrightnessApplet *applet,
	GtkAllocation *allocation)
{
	if (applet->size != allocation->height)
	{
		applet->size = allocation->height;
		init_pixbufs (applet);
		moblin_applet_draw_cb(applet);
	}
}

/**
 * moblin_applet_draw_cb:
 * @applet: Brightness applet instance
 *
 * draws applet content (background + icon)
 **/
static gboolean
moblin_applet_draw_cb (MoblinBrightnessApplet *applet)
{
	gint n;

	if (applet->size <= 2) {
		return FALSE;
	}

	/* if no icon, then don't try to display */
	if (applet->icon[0] == NULL) {
		return FALSE;
	}

	n = (4*(applet->level))/((applet->max_level)-(applet->min_level)+1);
	if(n > 3) n = 3;
	if(n < 0) n = 0;

	gtk_image_set_from_pixbuf (applet->image, applet->icon[n]);
	return TRUE;
}

/**
 * moblin_applet_destroy_popup_cb:
 * @applet: Brightness applet instance
 *
 * destroys the popup (called if orientation has changed)
 **/
static gboolean
moblin_applet_destroy_popup_cb (MoblinBrightnessApplet *applet)
{
	if (applet->popup != NULL) {
		gtk_widget_set_parent (applet->popup, NULL);
		gtk_widget_destroy (applet->popup);
		applet->popup = NULL;
	}
	return TRUE;
}

/**
 * moblin_applet_update_tooltip:
 * @applet: Brightness applet instance
 *
 * sets tooltip's content (percentage or disabled)
 **/
static void
moblin_applet_update_tooltip (MoblinBrightnessApplet *applet)
{
	static gchar buf[101];
	if (applet->popped == FALSE) {
		snprintf (buf, 100, _("LCD brightness : %d%%"), 
			brightness_from_level(applet->level));
		gtk_widget_set_tooltip_text (GTK_WIDGET(applet), buf);
	} else {
		gtk_widget_set_tooltip_text (GTK_WIDGET(applet), NULL);
	}
}

/**
 * moblin_applet_update_popup_level:
 * @applet: Brightness applet instance
 * @get_hw: set UI value from HW value
 * @set_hw: set HW value from UI value
 *
 * updates popup and hardware level of brightness
 * FALSE FAlSE -> set UI from cached value
 * TRUE  FAlSE -> set UI from HW value
 * TRUE  FALSE -> set HW from UI value, then set UI from HW value
 * FALSE TRUE  -> set HW from UI value
 **/
static void
moblin_applet_update_popup_level (MoblinBrightnessApplet *applet)
{
	if (applet->popup != NULL) {
		gtk_widget_set_sensitive (applet->btn_plus, applet->level < applet->max_level);
		gtk_widget_set_sensitive (applet->btn_minus, applet->level > applet->min_level);
		gtk_range_set_value (GTK_RANGE(applet->slider), (guint) applet->level);
	}
	moblin_applet_update_tooltip (applet);
}

/**
 * moblin_applet_plus_cb:
 * @widget: The sending widget (plus button)
 * @applet: Brightness applet instance
 *
 * callback for the plus button
 **/
static gboolean
moblin_applet_plus_cb (GtkWidget *w, MoblinBrightnessApplet *applet)
{
	if (applet->level < applet->max_level) {
		applet->level++;
	}
	applet->call_worked = set_brightness ((BrightnessControl*)&(applet->proxy));
	moblin_applet_update_popup_level (applet);
	moblin_applet_draw_cb (applet);
	return TRUE;
}

/**
 * moblin_applet_minus_cb:
 * @widget: The sending widget (minus button)
 * @applet: Brightness applet instance
 *
 * callback for the minus button
 **/
static gboolean
moblin_applet_minus_cb (GtkWidget *w, MoblinBrightnessApplet *applet)
{
	if (applet->level > applet->min_level) {
		applet->level--;
	}
	applet->call_worked = set_brightness ((BrightnessControl*)&(applet->proxy));
	moblin_applet_update_popup_level (applet);
	moblin_applet_draw_cb (applet);
	return TRUE;
}

/**
 * moblin_applet_slide_cb:
 * @widget: The sending widget (slider)
 * @applet: Brightness applet instance
 *
 * callback for the slider
 **/
static gboolean
moblin_applet_slide_cb (GtkWidget *w, MoblinBrightnessApplet *applet)
{
	applet->level = gtk_range_get_value (GTK_RANGE(applet->slider));
	applet->call_worked = set_brightness ((BrightnessControl*)&(applet->proxy));
	moblin_applet_update_popup_level (applet);
	moblin_applet_draw_cb (applet);
	return TRUE;
}

/**
 * moblin_applet_scroll_cb:
 * @applet: Brightness applet instance
 * @event: The scroll event
 *
 * callback handling mouse scrolls, either when the applet
 * is not popped and the mouse is over the applet, or when
 * the applet is popped and no matter where the mouse is.
 **/
static gboolean
moblin_applet_scroll_cb (MoblinBrightnessApplet *applet, GdkEventScroll *event)
{
	int i;

	if (event->type == GDK_SCROLL) {
		if (event->direction == GDK_SCROLL_UP) {
			for (i = 0;i < 5;i++) {
				moblin_applet_plus_cb (NULL, applet);
			}
			
		} else {
			for (i = 0;i < 5;i++) {
				moblin_applet_minus_cb (NULL, applet);
			}
		}
		return TRUE;
	}

	return FALSE;
}

/**
 * moblin_applet_create_popup:
 * @applet: Brightness applet instance
 *
 * cretes a new popup according to orientation of panel
 **/
static void
moblin_applet_create_popup (MoblinBrightnessApplet *applet)
{
	static GtkWidget *box, *frame;

	moblin_applet_destroy_popup_cb (applet);

	/* slider */
	applet->slider = gtk_vscale_new_with_range (applet->min_level, 
		applet->max_level, 1);
	gtk_widget_set_size_request (applet->slider, SLIDER_WIDTH, 200);
	gtk_range_set_inverted (GTK_RANGE(applet->slider), TRUE);
	gtk_scale_set_draw_value (GTK_SCALE(applet->slider), FALSE);
	gtk_range_set_value (GTK_RANGE(applet->slider), applet->level);
	g_signal_connect (G_OBJECT(applet->slider), "value-changed", G_CALLBACK(moblin_applet_slide_cb), applet);

	/* minus button */
	applet->btn_minus = gtk_button_new_with_label ("\342\210\222"); /* U+2212 MINUS SIGN */
	gtk_button_set_relief (GTK_BUTTON(applet->btn_minus), GTK_RELIEF_NORMAL);
	gtk_widget_set_size_request (applet->btn_minus, SLIDER_WIDTH, SLIDER_WIDTH);
	g_signal_connect (G_OBJECT(applet->btn_minus), "pressed", G_CALLBACK(moblin_applet_minus_cb), applet);

	/* plus button */
	applet->btn_plus = gtk_button_new_with_label ("+");
	gtk_button_set_relief (GTK_BUTTON(applet->btn_plus), GTK_RELIEF_NORMAL);
	gtk_widget_set_size_request (applet->btn_plus, SLIDER_WIDTH, SLIDER_WIDTH);
	g_signal_connect (G_OBJECT(applet->btn_plus), "pressed", G_CALLBACK(moblin_applet_plus_cb), applet);

	/* box */
	box = gtk_vbox_new (FALSE, 1);
	gtk_box_pack_start (GTK_BOX(box), applet->btn_plus, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(box), applet->slider, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(box), applet->btn_minus, FALSE, FALSE, 0);

	/* frame */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER(frame), box);

	/* window */
	applet->popup = gtk_window_new (GTK_WINDOW_POPUP);
	GTK_WIDGET_UNSET_FLAGS (applet->popup, GTK_TOPLEVEL);
	gtk_widget_set_parent (applet->popup, GTK_WIDGET(applet));
	gtk_container_add (GTK_CONTAINER(applet->popup), frame);
}

/**
 * moblin_applet_popup_cb:
 * @applet: Brightness applet instance
 *
 * pops and unpops
 **/
static gboolean
moblin_applet_popup_cb (MoblinBrightnessApplet *applet, GdkEventButton *event)
{
	gint x, y;

	/* react only to left mouse button */
	if (event->button != 1) {
		return FALSE;
	}

	/* if yet popped, release focus and hide then redraw applet unselected */
	if (applet->popped) {
		gdk_keyboard_ungrab (GDK_CURRENT_TIME);
		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (GTK_WIDGET(applet));
		gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_NORMAL);
		gtk_widget_hide (applet->popup);
		applet->popped = FALSE;
		moblin_applet_draw_cb (applet);
		moblin_applet_update_tooltip (applet);
		return TRUE;
	}

	get_brightness ((BrightnessControl*)&(applet->proxy));
	/* update UI for current brightness */
	moblin_applet_update_popup_level (applet);

	/* otherwise pop */
	applet->popped = TRUE;
	
	moblin_applet_draw_cb (applet);

	/* create a new popup (initial or if panel parameters changed) */
	if (applet->popup == NULL) {
		moblin_applet_create_popup (applet);
	}

	/* update UI for current brightness */
	moblin_applet_update_popup_level (applet);

	gtk_widget_show_all (applet->popup);

	/* retrieve geometry parameters and move window appropriately */
	gdk_window_get_origin (GTK_WIDGET(applet)->window, &x, &y);
	x += GTK_WIDGET(applet)->allocation.x
		+ GTK_WIDGET(applet)->allocation.width/2;
	y += GTK_WIDGET(applet)->allocation.y
		+ GTK_WIDGET(applet)->allocation.height;
	x -= applet->popup->allocation.width/2;
	gtk_window_move (GTK_WINDOW (applet->popup), x, y);

	/* grab input */
	gtk_widget_grab_focus (GTK_WIDGET(applet));
	gtk_grab_add (GTK_WIDGET(applet));
	gdk_pointer_grab (GTK_WIDGET(applet)->window, TRUE,
			  GDK_BUTTON_PRESS_MASK |
			  GDK_BUTTON_RELEASE_MASK |
			  GDK_POINTER_MOTION_MASK,
			  NULL, NULL, GDK_CURRENT_TIME);
	gdk_keyboard_grab (GTK_WIDGET(applet)->window,
			   TRUE, GDK_CURRENT_TIME);
	gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_SELECTED);

	return TRUE;
}

/**
 * moblin_applet_theme_change_cb:
 *
 * Updtes icon when theme changes
 **/
static void
moblin_applet_theme_change_cb (GtkIconTheme *icon_theme, gpointer data)
{
	MoblinBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET (data);
	init_pixbufs (applet);
}

/**
 * moblin_applet_stop_scroll_events_cb:
 *
 * Prevents scroll events from reaching the tooltip
 **/
static void
moblin_applet_stop_scroll_events_cb (GtkWidget *widget, GdkEvent  *event)
{
	if (event->type == GDK_SCROLL)
		g_signal_stop_emission_by_name (widget, "event-after");
}

/**
 * moblin_applet_destroy_cb:
 * @object: Class instance to destroy
 **/
static void
moblin_applet_destroy_cb (GtkObject *object)
                                 
{
int i;

	MoblinBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET(object);

	for(i = 0; i < 4; i++) {
	    if (applet->icon[i] != NULL) {
		gdk_pixbuf_unref (applet->icon[i]);
	    }
	}
}

/**
 * moblin_brightness_applet_class_init:
 * @klass: Class instance
 **/
static void
moblin_brightness_applet_class_init (MoblinBrightnessAppletClass *class)
{
	/* nothing to do here */
}

static void
brightness_changed_cb (DBusGProxy          *proxy,
		       guint	            brightness,
		       BrightnessControl   *bc)
{
	bc->level = brightness;
}

/**
 * moblin_brightness_applet_init:
 * @applet: Brightness applet instance
 **/
static void
moblin_brightness_applet_init (MoblinBrightnessApplet *applet)
{
	GtkWidget *image;

	/* initialize fields */
	applet->size = 50;
	applet->call_worked = TRUE;
	applet->popped = FALSE;
	applet->popup = NULL;
	applet->connection = NULL;
	applet->proxy = NULL;
	applet->brightness_changed_cb = brightness_changed_cb;

	init_pixbufs (applet);

	/* icon (our main UI) */
	image = gtk_image_new ();
	applet->image = GTK_IMAGE (image);
	gtk_container_add (GTK_CONTAINER (applet), image);
	gtk_widget_show (image);

	if(!brightness_dbus_connect((BrightnessControl*)&(applet->proxy)))
		return;

	/* coldplug */
	applet->call_worked = get_brightness ((BrightnessControl*)&(applet->proxy));
	moblin_applet_update_popup_level (applet);

	/* connect */
	g_signal_connect (G_OBJECT(applet), "button-press-event",
			  G_CALLBACK(moblin_applet_popup_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "scroll-event",
			  G_CALLBACK(moblin_applet_scroll_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "destroy",
			  G_CALLBACK(moblin_applet_destroy_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "size-allocate",
			  G_CALLBACK(moblin_applet_size_allocate), NULL);

	/* prevent scroll events from reaching the tooltip */
	g_signal_connect (G_OBJECT (applet), "event-after", G_CALLBACK (moblin_applet_stop_scroll_events_cb), NULL);

	g_signal_connect (gtk_icon_theme_get_default (), "changed", G_CALLBACK (moblin_applet_theme_change_cb), applet);
}

MoblinBrightness*
moblin_brightness_new()
{
MoblinBrightness *obj;

  if(!moblin_brightness_applet_get_type())
  {
    g_warning("Failed to register type MoblinBrightnessApplet\n");
    return NULL;
  }
  obj = g_new0(MoblinBrightness, 1);
  obj->applet = (MoblinBrightnessApplet*)g_object_new(moblin_brightness_applet_get_type(), NULL);

  return obj;
}

void *brightness_initialize(HildonStatusBarItem *item, GtkWidget **button)
{
  MoblinBrightness *obj = moblin_brightness_new();

  if(!obj)
  {
    g_warning("Failed to initialize the brightness applet\n");
    return NULL;
  }

  obj->item = item;
  *button = obj->button = GTK_WIDGET(obj->applet);

  moblin_applet_draw_cb (obj->applet);

  /* show */
  gtk_widget_show_all (GTK_WIDGET(obj->applet));

  return obj;
}

void brightness_update(void *data, gint value1, gint value2, const gchar *str)
{

}

void brightness_destroy(void *data)
{
}

gint brightness_get_priority(void *data)
{
	return 1;
}

void brightness_entry (HildonStatusBarPluginFn_st *fn)
{
	if (fn == NULL)
	{
		g_warning("bad setup call\n");
		return;
	}

	fn->initialize = brightness_initialize;
	fn->destroy = brightness_destroy;
	fn->update = brightness_update;
	fn->get_priority = brightness_get_priority;
}
