#include <config.h>
#include <string.h>
#include <stdio.h>
#include <locale.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkx.h>
#include <X11/extensions/Xrandr.h>

#include "moblin-settings-daemon.h"
#include "moblin-settings-xsettings.h"
#include "xsettings-manager.h"

extern XSettingsManager **managers;

typedef struct _TranslationEntry TranslationEntry;
typedef void (* TranslationFunc) (TranslationEntry *trans,
                                  GConfValue       *value);
struct _TranslationEntry
{
  const char *gconf_key;
  const char *xsetting_name;
  
  GConfValueType gconf_type;
  TranslationFunc translate;
};

static struct {
        Rotation  rotation;
        gchar const    * name;
} const rotations[] = {
        {RR_Rotate_0,   N_("normal")},
        {RR_Rotate_90,  N_("left")},
        {RR_Rotate_180, N_("inverted")},
        {RR_Rotate_270, N_("right")}
};

static void
translate_bool_int (TranslationEntry *trans,
		    GConfValue       *value)
{
  int i;

  g_assert (value->type == trans->gconf_type);
  
  for (i = 0; managers [i]; i++)  
    xsettings_manager_set_int (managers [i], trans->xsetting_name,
                               gconf_value_get_bool (value));
}

static void
translate_int_int (TranslationEntry *trans,
                   GConfValue       *value)
{
  int i;

  g_assert (value->type == trans->gconf_type);

  for (i = 0; managers [i]; i++)  
    xsettings_manager_set_int (managers [i], trans->xsetting_name,
                               gconf_value_get_int (value));
}

static void
translate_string_string (TranslationEntry *trans,
                         GConfValue       *value)
{
  int i;

  g_assert (value->type == trans->gconf_type);

  for (i = 0; managers [i]; i++)  
    xsettings_manager_set_string (managers [i],
                                  trans->xsetting_name,
                                  gconf_value_get_string (value));
}

static void
translate_string_string_toolbar (TranslationEntry *trans,
				 GConfValue       *value)
{
  int i;
  const char *tmp;
  
  g_assert (value->type == trans->gconf_type);

  /* This is kind of a workaround since GNOME expects the key value to be
   * "both_horiz" and gtk+ wants the XSetting to be "both-horiz".
   */
  tmp = gconf_value_get_string (value);
  if (tmp && strcmp (tmp, "both_horiz") == 0)
	  tmp = "both-horiz";

  for (i = 0; managers [i]; i++) 
    xsettings_manager_set_string (managers [i],
                                  trans->xsetting_name,
                                  tmp);
}

static TranslationEntry translations [] = {
  { "/desktop/gnome/peripherals/mouse/double_click",	"Net/DoubleClickTime",
      GCONF_VALUE_INT,		translate_int_int },
  { "/desktop/gnome/peripherals/mouse/drag_threshold",  "Net/DndDragThreshold",
      GCONF_VALUE_INT,          translate_int_int },
  { "/desktop/gnome/gtk-color-palette",			"Gtk/ColorPalette",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/font_name",		"Gtk/FontName",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/gtk_key_theme",		"Gtk/KeyThemeName",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/toolbar_style",			"Gtk/ToolbarStyle",
      GCONF_VALUE_STRING,	translate_string_string_toolbar },
  { "/desktop/gnome/interface/toolbar_icon_size",		"Gtk/ToolbarIconSize",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/can_change_accels",		"Gtk/CanChangeAccels",
      GCONF_VALUE_BOOL,		translate_bool_int },
  { "/desktop/gnome/interface/cursor_blink",		"Net/CursorBlink",
      GCONF_VALUE_BOOL,		translate_bool_int },
  { "/desktop/gnome/interface/cursor_blink_time",	"Net/CursorBlinkTime",
      GCONF_VALUE_INT,		translate_int_int },
  { "/desktop/gnome/interface/gtk_theme",		"Net/ThemeName",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/gtk_color_scheme",	"Gtk/ColorScheme",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/gtk-im-preedit-style",	"Gtk/IMPreeditStyle",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/gtk-im-status-style",	"Gtk/IMStatusStyle",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/icon_theme",		"Net/IconThemeName",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/file_chooser_backend",	"Gtk/FileChooserBackend",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/interface/menus_have_icons",	"Gtk/MenuImages",
      GCONF_VALUE_BOOL,		translate_bool_int },
  { "/desktop/gnome/interface/menubar_accel",  "Gtk/MenuBarAccel",
      GCONF_VALUE_STRING,      translate_string_string },
  { "/desktop/gnome/peripherals/mouse/cursor_theme",	"Gtk/CursorThemeName",
      GCONF_VALUE_STRING,	translate_string_string },
  { "/desktop/gnome/peripherals/mouse/cursor_size",	"Gtk/CursorThemeSize",
      GCONF_VALUE_INT,		translate_int_int },
  { "/desktop/gnome/interface/show_input_method_menu",  "Gtk/ShowInputMethodMenu",
      GCONF_VALUE_BOOL,		translate_bool_int },
  { "/desktop/gnome/interface/show_unicode_menu",  "Gtk/ShowUnicodeMenu",
      GCONF_VALUE_BOOL,		translate_bool_int },
};

static TranslationEntry*
find_translation_entry (const char *gconf_key)
{
  int i;

  i = 0;
  while (i < G_N_ELEMENTS (translations))
    {
      if (strcmp (translations[i].gconf_key, gconf_key) == 0)
        return &translations[i];

      ++i;
    }

  return NULL;
}

static const gchar* 
type_to_string (GConfValueType type)
{
  switch (type)
    {
    case GCONF_VALUE_INT:
      return "int";
    case GCONF_VALUE_STRING:
      return "string";
    case GCONF_VALUE_FLOAT:
      return "float";
    case GCONF_VALUE_BOOL:
      return "bool";
    case GCONF_VALUE_SCHEMA:
      return "schema";
    case GCONF_VALUE_LIST:
      return "list";
    case GCONF_VALUE_PAIR:
      return "pair";
    case GCONF_VALUE_INVALID:
      return "*invalid*";
    default:
      g_assert_not_reached();
      return NULL; /* for warnings */
    }
}

static void
process_value (TranslationEntry *trans,
               GConfValue       *val)
{  
  if (val == NULL)
    {
      int i;

      for (i = 0; managers [i]; i++)  
        xsettings_manager_delete_setting (managers [i], trans->xsetting_name);
    }
  else
    {
      if (val->type == trans->gconf_type)
        {
          (* trans->translate) (trans, val);
        }
      else
        {
          g_warning (_("GConf key %s set to type %s but its expected type was %s\n"),
                     trans->gconf_key,
                     type_to_string (val->type),
                     type_to_string (trans->gconf_type));
        }
    }
}

static void
xsettings_callback (GConfEntry *entry)
{
  TranslationEntry *trans;
  int i;

  trans = find_translation_entry (entry->key);
  if (trans == NULL)
    return;

  process_value (trans, entry->value);
  
  for (i = 0; managers [i]; i++)
    xsettings_manager_set_string (managers [i], "Net/FallbackIconTheme",
				  "gnome");

  for (i = 0; managers [i]; i++)  
    xsettings_manager_notify (managers [i]);
}

void
moblin_settings_xsettings_init (GConfClient *client)
{
  moblin_settings_daemon_register_callback ("/desktop/gtk", xsettings_callback);
  moblin_settings_daemon_register_callback ("/desktop/gnome/interface", xsettings_callback);
}

void
moblin_settings_xsettings_load (GConfClient *client)
{
char *res, *rot;
int i, w, h, num, new_res;
int res_changed=0, rot_changed=0;
GdkDisplay *display;
Display *xdisplay;
GdkScreen *screen;
GdkWindow *root_window;
XRRScreenConfiguration *config;
XRRScreenSize *sizes;
Rotation new_rot;

    res = gconf_client_get_string (client, 
	    "/desktop/moblin/screen/default/0/resolution", NULL);
    rot = gconf_client_get_string (client, 
	    "/desktop/moblin/screen/default/0/rotation", NULL);
    if(res||rot)
    {
	display = gdk_display_get_default ();
	xdisplay = gdk_x11_display_get_xdisplay (display);
	screen = gdk_display_get_screen (display, 0);
	root_window = gdk_screen_get_root_window (screen);
	config =
	    XRRGetScreenInfo (gdk_x11_display_get_xdisplay (display),
	    gdk_x11_drawable_get_xid (GDK_DRAWABLE (root_window)));
	sizes = XRRConfigSizes (config, &num);
	new_res = XRRConfigCurrentConfiguration (config, &new_rot);

	if(res)
	{
	    if(sscanf(res, "%dx%d", &w, &h) == 2)
	    {
		for(i = 0; (i < num)&&(sizes[i].width != w)&&
		    (sizes[i].height != h); i++);
		if((i < num)&&(new_res != i)) 
		{
		    new_res = i;
		    res_changed = 1;
		}
	    }
	    g_free(res);
	}
	if(rot)
	{
	    for(i = 0; (i < G_N_ELEMENTS(rotations))&&
		strcmp(rot, _(rotations[i].name)); i++);
	    if((i < G_N_ELEMENTS(rotations))&&
		(new_rot != rotations[i].rotation))
	    {
		new_rot = rotations[i].rotation;
		rot_changed = 1;
	    }
	    g_free(rot);
	}
	if(res_changed||rot_changed)
	{
	    XRRSetScreenConfig (xdisplay, config,
		gdk_x11_drawable_get_xid (GDK_DRAWABLE (root_window)),
		new_res, new_rot, GDK_CURRENT_TIME);
	}
    }
    for (i = 0; i < G_N_ELEMENTS (translations); i++)
    {
	GConfValue *val;
	GError *err;

	err = NULL;
	val = gconf_client_get (client,
                              translations[i].gconf_key,
                              &err);

	if (err != NULL)
	{
	    g_warning ("Error getting value for %s: %s\n",
		    translations[i].gconf_key, err->message);
	    g_error_free (err);
	}
	else
	{
	    if((val != NULL)&&(val->type == GCONF_VALUE_STRING))
		g_debug("%s = %s", translations[i].gconf_key, gconf_value_get_string(val));
	    process_value (&translations[i], val);
	    if (val != NULL)
		gconf_value_free (val);
	}
    }

    for (i = 0; managers [i]; i++)
	xsettings_manager_set_string (managers [i], "Net/FallbackIconTheme",
				  "gnome");

    for (i = 0; managers [i]; i++)  
	xsettings_manager_notify (managers [i]);
}
