/*
 * moblin-system-dbus.c
 *
 * Copyright (C) 2007 Jan Arne Petersen <jap@gnome.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "moblin-system-dbus.h"
#include "moblin-system-marshal.h"
#include "calibrator.h"
#include "datetime.h"

static GObjectClass *parent_class = NULL;

static gboolean
system_daemon_apply_calibration (GObject * object, 
	const gint xmin, const gint xmax, const gint ymin, 
	const gint ymax, const guint flags, GError **error)
{
	return apply_calibration(xmin, xmax, ymin, ymax, flags);
}

static gboolean
system_daemon_read_calibration (GObject * object, 
	gint *xmin, gint *xmax, gint *ymin, 
	gint *ymax, guint *flags, GError **error)
{
	return read_calibration(xmin, xmax, ymin, ymax, flags);
}

static gboolean
system_daemon_cancel_touch (GObject * object, GError **error)
{
	return cancel_touch();
}

static gboolean
system_daemon_capture_touch (GObject * object, GError **error)
{
	return capture_touch();
}

gboolean
system_daemon_set_time(MoblinSystemServer * server,
	const gint year, const gint month, const gint day,
	const gint hour, const gint minute, const gint second,
	GError **error)
{
    return set_time(server, year, month, day, hour, minute, second);
}

gboolean
system_daemon_get_time(MoblinSystemServer * server, 
	gint *year, gint *month, gint *day, 
	gint *hour, gint *minute, gint *second,
	GError **error)
{
    return get_time(server, year, month, day, hour, minute, second);
}

gboolean
system_daemon_set_timezone(MoblinSystemServer * server, 
	const gchar *tz, GError **error)
{
    return set_timezone (server, tz);
}

gboolean
system_daemon_get_timezone(MoblinSystemServer * server, 
	gchar **tz, GError **error)
{
    return get_timezone (server, tz);
}

G_DEFINE_TYPE (MoblinSystemServer, moblin_system_server, G_TYPE_OBJECT)
#include "moblin-system-server.h"

GObject *
moblin_system_server_get (void)
{
	return g_object_new (MOBLIN_SYSTEM_TYPE_SERVER, NULL);
}

static GObject*
moblin_system_server_constructor (GType                  type,
				  guint                  n_construct_params,
				  GObjectConstructParam *construct_params)
{
	static GObject *singleton = NULL;
	GObject *object;

	if (!singleton) {
		singleton = G_OBJECT_CLASS (parent_class)->constructor (type, n_construct_params, construct_params);
		object = singleton;
	} else {
		object = g_object_ref (singleton);
	}

	return object;
}


static void
moblin_system_server_class_init (MoblinSystemServerClass * klass)
{
	GError *error = NULL;
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->constructor = moblin_system_server_constructor;

	/* Init the DBus connection, per-klass */
	klass->connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if (klass->connection == NULL) {
		PRINTF("Unable to connect to dbus: %s",
			   error->message);
		g_error_free (error);
		return;
	}

	/* &dbus_glib__object_info is provided in the server-bindings.h file */
	dbus_g_object_type_install_info (MOBLIN_SYSTEM_TYPE_SERVER,
					 &dbus_glib_system_daemon_object_info);
}

static gboolean
oobs_connect(MoblinSystemServer * server)
{
OobsResult result;

	server->session = oobs_session_get();
	if(server->session == NULL)
	{
	    PRINTF("oobs_connect: oobs_session_get failed\n");
	    return FALSE;
	}
	result = oobs_session_get_platform (server->session, NULL);
	if(result != OOBS_RESULT_OK)
	{
	    PRINTF("oobs_connect: oobs_session_get_platform failed, res = %d\n",
		result);
	    g_object_unref(server->session);
	    server->session = NULL;
	    return FALSE;
	}

	PRINTF("OOBS CONNECT\n");
	return TRUE;
}

static gboolean
time_connect(MoblinSystemServer * server)
{
	server->time_config = oobs_time_config_get();
	if(server->time_config == NULL)
	{
	    PRINTF("time_connect: oobs_time_config_get failed\n");
	    return FALSE;
	}

	PRINTF("TIME CONNECT\n");
	return TRUE;
}

static void
moblin_system_server_init (MoblinSystemServer * server)
{
	GError *error = NULL;
	DBusGProxy *driver_proxy;
	MoblinSystemServerClass *klass =
	    MOBLIN_SYSTEM_SERVER_GET_CLASS (server);
	unsigned request_ret;

	/* Register DBUS path */
	dbus_g_connection_register_g_object (klass->connection,
					     "/org/moblin/SystemDaemon",
					     G_OBJECT (server));

	/* Register the service name, the constant here are defined in dbus-glib-bindings.h */
	driver_proxy = dbus_g_proxy_new_for_name (klass->connection,
						  DBUS_SERVICE_DBUS,
						  DBUS_PATH_DBUS,
						  DBUS_INTERFACE_DBUS);

	if (!org_freedesktop_DBus_request_name
	    (driver_proxy, "org.moblin.SystemDaemon", 0, &request_ret,
	     &error)) {
		PRINTF("Unable to register service: %s", error->message);
		g_error_free (error);
	}

	g_object_unref (driver_proxy);
	if(oobs_connect(server))
	{
		time_connect(server);
	}
}

extern MoblinSystemServer *dbusserver;

void
capture_touch_reply(gint x, gint y)
{
DBusMessage *message;
DBusConnection *cnct;
MoblinSystemServerClass *klass;

	klass = MOBLIN_SYSTEM_SERVER_GET_CLASS (dbusserver);
	cnct = dbus_g_connection_get_connection(klass->connection);
	message = dbus_message_new_signal(
	    "/org/moblin/SystemDaemon",
	    "org.moblin.SystemDaemon",
	    "TouchScreenData");
	dbus_message_append_args(message,
	    DBUS_TYPE_INT32, &x,
	    DBUS_TYPE_INT32, &y,
	    DBUS_TYPE_INVALID);
	dbus_connection_send(cnct, message, NULL);
	dbus_connection_flush(cnct);
	dbus_message_unref(message);
	PRINTF("CaptureTouchReply x=%d, y=%d\n", x, y);
}
