/* volume.c
 *
 * Written by Todd Brandt <todd.e.brandt@intel.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib-object.h>
#include <math.h>
#include <fcntl.h>
#include <sys/types.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <alsa/asoundlib.h>
#include <sys/poll.h>
#include "volume.h"

#define ALSADEVFILE "/dev/snd/seq"

enum {
	ELEM_NONE=0,
	ELEM_COMMON,
	ELEM_PLAYBACK,
	ELEM_CAPTURE
};

enum {
	DUP_NAMES_IGNORED,
	DUP_NAMES_ALLOWED
};

struct ptrack {
        /* static info */
        gchar *name;
        snd_mixer_selem_id_t *sid;
        snd_mixer_elem_t *elem;
        char has_vol_ctl;
        char has_switch_ctl;
        long vmin;
        long vmax;
	int num_channels;
	int *channels;
};

static struct snd_mixer_selem_regopt smixer_options;
static char card[64] = "default";

/* Master list of all available playback tracks from the sound card */
static struct ptrack *ptlist = NULL;
static int ptnum = 0;
static int primary_sound_element = -1;
static int init = 0;

int num_sound_elements()
{
	return ptnum;
}

int num_channels(int e)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	  ((t=&ptlist[e]) == NULL))
		return -1;

	return t->num_channels;
}

int get_primary_sound_element_index() 
{
        return primary_sound_element;
}

void set_primary_sound_element_index(int e) 
{
	if((e >= 0)&&(e < ptnum))
        	primary_sound_element = e;
}

gchar *name_from_index(int e)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	  ((t=&ptlist[e]) == NULL))
		return NULL;

	return t->name;
}

const char *name_from_channel(int e, int c)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	    ((t=&ptlist[e]) == NULL)||(c < 0)||
	    (c >= t->num_channels))
		return NULL;

	return snd_mixer_selem_channel_name(t->channels[c]);
}

int index_from_name(char *name)
{
	int i;
	struct ptrack *t;

	if(!ptlist || !init)
		return -1;

	for(i = 0; i < ptnum; i++) {
		if((t = &ptlist[i]) == NULL) 
			continue;
		if(!strcmp(name, t->name)) 
			return i;
	}

	return -1;
}

int get_volume_range(int e, int *min, int *max)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	  ((t=&ptlist[e]) == NULL)||!t->has_vol_ctl)
		return -1;

	if(min) *min = t->vmin;
	if(max) *max = t->vmax;

	return t->vmax - t->vmin;
}

int get_pse_volume_range()
{
	struct ptrack *t;

	if((primary_sound_element < 0)||
	  (primary_sound_element >= ptnum)||!ptlist||!init||
	  ((t=&ptlist[primary_sound_element]) == NULL)||!t->has_vol_ctl)
		return -1;

	return t->vmax - t->vmin;
}

int get_support(int e, int *vol, int *sw)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	  ((t=&ptlist[e]) == NULL))
		return -1;

	if(vol) *vol = t->has_vol_ctl;
	if(sw) *sw = t->has_switch_ctl;

	return 0;
}

int get_volume(int e, int c, int *val)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	    ((t=&ptlist[e]) == NULL)||(c < 0)||
	    !t->has_vol_ctl||(c >= t->num_channels))
		return -1;

	if(t->has_vol_ctl == ELEM_CAPTURE) {
	    snd_mixer_selem_get_capture_volume(t->elem,
		t->channels[c], (long *)val);
	} else {
	    snd_mixer_selem_get_playback_volume(t->elem,
		t->channels[c], (long *)val);
	}
#if 0
	g_debug("get vol %s: %s=%d\n", 
		t->name,
		snd_mixer_selem_channel_name(t->channels[c]),
		*val);
#endif

	return 0;
}

int set_volume(int e, int c, int val)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	    ((t=&ptlist[e]) == NULL)||(c < 0)||
	    !t->has_vol_ctl||(c >= t->num_channels))
		return -1;

	if(t->has_vol_ctl == ELEM_CAPTURE) {
	    snd_mixer_selem_set_capture_volume(t->elem,
		t->channels[c], val);
	} else {
	    snd_mixer_selem_set_playback_volume(t->elem,
		t->channels[c], val);
	}
#if 0
	g_debug("set vol %s: %s=%d\n", 
		t->name,
		snd_mixer_selem_channel_name(t->channels[c]),
		val);
#endif

	return 0;
}

int set_all_volume(int e, int val)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	    ((t=&ptlist[e]) == NULL)||!t->has_vol_ctl)
		return -1;

	if(t->has_vol_ctl == ELEM_CAPTURE) {
	    snd_mixer_selem_set_capture_volume_all(t->elem, val);
	} else {
	    snd_mixer_selem_set_playback_volume_all(t->elem, val);
	}
#if 0
	g_debug("set all vol %s: %d\n", 
		t->name,
		val);
#endif

	return 0;
}

int get_switch(int e, int c, int *val)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	    ((t=&ptlist[e]) == NULL)||(c < 0)||
	    !t->has_switch_ctl||(c >= t->num_channels))
		return -1;

	if(t->has_switch_ctl == ELEM_CAPTURE) {
	    snd_mixer_selem_get_capture_switch(t->elem,
		t->channels[c], val);
	} else {
	    snd_mixer_selem_get_playback_switch(t->elem,
		t->channels[c], val);
	}
#if 0
	g_debug("get switch %s: %s=%d\n", 
		t->name,
		snd_mixer_selem_channel_name(t->channels[c]),
		*val);
#endif

	return 0;
}

int set_switch(int e, int c, int val)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	    ((t=&ptlist[e]) == NULL)||(c < 0)||
	    !t->has_switch_ctl||(c >= t->num_channels))
		return -1;

	if(t->has_switch_ctl == ELEM_CAPTURE) {
	    snd_mixer_selem_set_capture_switch(t->elem,
		t->channels[c], val);
	} else {
	    snd_mixer_selem_set_playback_switch(t->elem,
		t->channels[c], val);
	}
#if 0
	g_debug("set switch %s: %s=%d\n", 
		t->name,
		snd_mixer_selem_channel_name(t->channels[c]),
		val);
#endif

	return 0;
}

int set_all_switch(int e, int val)
{
	struct ptrack *t;

	if((e < 0)||(e >= ptnum)||!ptlist||!init||
	    ((t=&ptlist[e]) == NULL)||!t->has_switch_ctl)
		return -1;

	if(t->has_switch_ctl == ELEM_CAPTURE) {
	    snd_mixer_selem_set_capture_switch_all(t->elem, val);
	} else {
	    snd_mixer_selem_set_playback_switch_all(t->elem, val);
	}
#if 0
	g_debug("set all switch %s: %d\n", 
		t->name,
		val);
#endif

	return 0;
}

int set_all_playback_switches_all_channels(int val)
{
	struct ptrack *t;
	int e;

	if(!ptlist||!init) return -1;

	for(e = 0; e < ptnum; e++)
	{
	    if(((t = &ptlist[e]) == NULL)||!t->has_switch_ctl)
		continue;
	    if((t->has_switch_ctl == ELEM_PLAYBACK)||
		(t->has_switch_ctl == ELEM_COMMON)) {
		snd_mixer_selem_set_playback_switch_all(t->elem, val);
		g_debug("%s %s\n", (val)?"Unmute":"Mute", t->name);
	    }
	}
	return 0;
}

void process_elements(snd_mixer_t *handle, int etype, int allow_dups)
{
	int sw, i, playback=(etype==ELEM_PLAYBACK);
	long vol;
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;
	char *t, message[1000], name[100];
	const char *hwname;
	int has_vol_ctl, has_switch_ctl, index;
	int secondary_sound_element = -1;

	snd_mixer_selem_id_alloca(&sid);

	/* we test every element on the sound card */
	for (elem = snd_mixer_first_elem(handle); elem; 
		elem = snd_mixer_elem_next(elem))
	{
		snd_mixer_selem_get_id(elem, sid);
		if (!snd_mixer_selem_is_active(elem))
			continue;
		/* we need to see what volume and swtich controls */
		/* each element has, because we need at least one */
		has_vol_ctl = ELEM_NONE;
		has_switch_ctl = ELEM_NONE;

		if(playback) {
		    if (snd_mixer_selem_has_common_volume(elem)) {
			has_vol_ctl = ELEM_COMMON;
		    } else if (snd_mixer_selem_has_playback_volume(elem)) {
			has_vol_ctl = ELEM_PLAYBACK;
		    }
		    if (snd_mixer_selem_has_common_switch(elem)) {
			has_switch_ctl = ELEM_COMMON;
		    } else if (snd_mixer_selem_has_playback_switch(elem)) {
			has_switch_ctl = ELEM_PLAYBACK;
		    }
		} else {
		    if (!snd_mixer_selem_has_common_volume(elem)&&
			snd_mixer_selem_has_capture_volume(elem))
		    {
			has_vol_ctl = ELEM_CAPTURE;
		    }
		    if (!snd_mixer_selem_has_common_switch(elem)&&
			snd_mixer_selem_has_capture_switch(elem))
		    {
			has_switch_ctl = ELEM_CAPTURE;
                    }
		}

		/* if the elem has no volume or switch capability, ignore it */
		if((has_vol_ctl == ELEM_NONE)&&(has_switch_ctl == ELEM_NONE))
			continue;

		/* element name from hardware */
		hwname = snd_mixer_selem_id_get_name(sid);
		/* element index */
		index = snd_mixer_selem_id_get_index(sid);

		/* if this is not the first index, it's a dup */
		if(index&&!allow_dups)
			continue;

		if(index) {
			sprintf(name, "%s%i", hwname, index);
		} else {
			strcpy(name, hwname);
		} 

		/* at this point we want the element, so add a space for */
		/* it in the list */
		if(ptlist == NULL) {
			ptlist = (struct ptrack*)malloc(
				(++ptnum)*sizeof(struct ptrack));
		} else {
			ptlist = (struct ptrack*)realloc(ptlist, 
				(++ptnum)*sizeof(struct ptrack));
		}
		memset(&ptlist[ptnum-1], 0, sizeof(struct ptrack));

		/* initialize what we already know about this element */
		ptlist[ptnum-1].name = g_strdup(name);
		ptlist[ptnum-1].sid = sid;
		ptlist[ptnum-1].elem = elem;
		ptlist[ptnum-1].has_vol_ctl = has_vol_ctl;
		ptlist[ptnum-1].has_switch_ctl = has_switch_ctl;

		/* determine the element's volume range */
		if(has_vol_ctl)
		{
		    if(playback) {
			snd_mixer_selem_get_playback_volume_range(elem, 
				&ptlist[ptnum-1].vmin, &ptlist[ptnum-1].vmax);
		    } else {
			snd_mixer_selem_get_capture_volume_range(elem, 
				&ptlist[ptnum-1].vmin, &ptlist[ptnum-1].vmax);
		    }
		}

		ptlist[ptnum-1].num_channels = 0;
		ptlist[ptnum-1].channels = NULL;
		for(i = 0; i < SND_MIXER_SCHN_LAST; i++)
		{
		    if((playback&&snd_mixer_selem_has_playback_channel(elem, i))||
		       (!playback&&snd_mixer_selem_has_capture_channel(elem, i)))
		    {
			if(ptlist[ptnum-1].channels == NULL) {
			    ptlist[ptnum-1].channels = 
				(int*)malloc(
				(++(ptlist[ptnum-1].num_channels))*sizeof(int));
			} else {
			    ptlist[ptnum-1].channels = 
				(int*)realloc(ptlist[ptnum-1].channels, 
				(++(ptlist[ptnum-1].num_channels))*sizeof(int));
			}
			ptlist[ptnum-1].channels[ptlist[ptnum-1].num_channels-1] = i;
		    }
		}

		sprintf(message, "Found Element %s:", name);
		t = message+strlen(message);

		for(i = 0; i < ptlist[ptnum-1].num_channels; i++)
		{
		    sprintf(t, " %s",
			snd_mixer_selem_channel_name(ptlist[ptnum-1].channels[i]));
		    t += strlen(t);
		    if(has_vol_ctl)
		    {
			if(playback) {
			    snd_mixer_selem_get_playback_volume(elem, 
				ptlist[ptnum-1].channels[i], &vol);
			} else {
			    snd_mixer_selem_get_capture_volume(elem, 
				ptlist[ptnum-1].channels[i], &vol);
			}
			sprintf(t, "-V%d", (int)vol);
			t += strlen(t);
		    }
		    if(has_switch_ctl)
		    {
			if(playback) {
			    snd_mixer_selem_get_playback_switch(elem, 
				ptlist[ptnum-1].channels[i], &sw);
			} else {
			    snd_mixer_selem_get_capture_switch(elem, 
				ptlist[ptnum-1].channels[i], &sw);
			}
			sprintf(t, "-S%d", sw);
			t += strlen(t);
		    }
		}

		g_debug("%s\n", message);

		/* get a pse with both vol and sw */
		if((primary_sound_element < 0)&&playback&&has_vol_ctl&&has_switch_ctl)
		{
			primary_sound_element = ptnum-1;
			g_debug("Primary Sound Element = %s\n", ptlist[primary_sound_element].name);
		}

		/* get a sse with vol */
		if((secondary_sound_element < 0)&&playback&&has_vol_ctl)
		{
			secondary_sound_element = ptnum-1;
		}
	}
	if(primary_sound_element < 0) primary_sound_element = secondary_sound_element;
}

int init_alsa_vars()
{
	int err;
	snd_mixer_t *handle = NULL;

	if(init) return 0;

	if(ptlist != NULL) {
		free(ptlist);
		ptlist = NULL;
		ptnum = 0;
	}

	if(access(ALSADEVFILE, F_OK)) {
		g_warning("Can not access the Alsa device node!");
		return -1;
	}

	smixer_options.device = card;

 	if ((err = snd_mixer_open(&handle, 0)) < 0) {
		g_warning("Mixer %s open error: %s", card, snd_strerror(err));
		handle = NULL;
		return -1;
	}

	if ((err = snd_mixer_attach(handle, card)) < 0) {
		g_warning("Mixer attach %s error: %s", card, snd_strerror(err));
		snd_mixer_close(handle);
		handle = NULL;
		return -1;
	}
	if ((err = snd_mixer_selem_register(handle, NULL, NULL)) < 0)
	{
		g_warning("Mixer register error: %s", snd_strerror(err));
		snd_mixer_close(handle);
		handle = NULL;
		return -1;
	}
	err = snd_mixer_load(handle);
	if (err < 0) {
		g_warning("Mixer %s load error: %s", card, snd_strerror(err));
		snd_mixer_close(handle);
		handle = NULL;
		return -1;
	}

	/* playback & common elements */
	process_elements(handle, ELEM_PLAYBACK, DUP_NAMES_ALLOWED);
	/* capture elements */
	process_elements(handle, ELEM_CAPTURE, DUP_NAMES_IGNORED);

	if(ptnum < 1) {
		g_warning("No usable sound elements found\n");
		return -1;
	}
	init = 1;
	return 0;
}
