/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 *
 * Author: Mike Butler, mgb@mitre.org
 *
 * $Id: UtTime.h,v 1.11 1999/08/29 06:46:28 mgb Exp $
 *
 * 04-Nov-97 [mgb] Added StrGlob, simple glob-style matching
 * ToDo:
 *    [ ] Make Parse() report errors some how!
 *    [ ] ?Add Relative() predicate?  (False if time is a full date?)
 */
#ifndef UtTime_h
#define UtTime_h

#include <sys/time.h>
#include <UtString.h>
#include <stdio.h>		// for sprintf()...
#include <math.h>		// for HUGE_VAL  (+Inf)
#include <limits.h>		// for LONG_MAX
// class Time
/* Time class...
 *
 * Maintains a high-precision time in seconds and milliseconds.
 * Assuming that an unsigned long is 32-bits, this class can represent
 * intervals upto 136 years in a precision to milliseconds.  Also
 * provided are routines for input and output...
 *
 * Bugs: Doh! Should have used struct timeval!
 */
class Time {
  friend inline ostream &operator<<(ostream &, const Time&);
private:
  double cSeconds;
public:
  enum Format { tSmart, tMicro, tMilli, tDelta, 
		tTime, tHHMMSS, tHHMM, tSeconds,
		tDate, tDatetime, tFileName, };

  explicit Time(double s)  : cSeconds(s) { }
  explicit Time(int s = 0) : cSeconds(s) { }
  explicit Time(long s)    : cSeconds(s) { }
  // #### Need error return for these, but how?
  explicit Time(const char *v) { Parse(v); }
  explicit Time(const String &s) { Parse(s.c_str()); }

  // Convert to and from  Unix timeval struct...
  // Special handling of "Eternity"...
  operator timeval() const {
    timeval tv;
    if(cSeconds == HUGE_VAL) {
      tv.tv_sec  = LONG_MAX;
      tv.tv_usec = LONG_MAX;
    } else {
      tv.tv_sec  = long(cSeconds);
      tv.tv_usec = long((cSeconds - tv.tv_sec) * 1000000.0 + 0.5);
    }
    return(tv);
  }
  Time(const timeval &tv) { 
    if(tv.tv_sec == LONG_MAX) {
      cSeconds = HUGE_VAL;
    } else {
      cSeconds = tv.tv_sec + double(tv.tv_usec)/1000000.0; 
    }
  }
  // Flag for a time that can never happen...
  static Time Eternity() { return(Time(HUGE_VAL)); }

  static Time Midnight() {
    time_t now = time(0);
    struct tm local = (*localtime(&now));
    local.tm_sec = 0;
    local.tm_min = 0;
    local.tm_hour = 0;
    double mid = mktime(&local);
    return(Time(mid));
  }

  static Time OnHour() {
    time_t now = time(0);
    struct tm local = (*localtime(&now));
    local.tm_sec = 0;
    local.tm_min = 0;
    double mid = mktime(&local);
    return(Time(mid));
  }

  static Time Now() {
    double now;
    struct timeval tp;
    gettimeofday(&tp, 0);
    now = tp.tv_sec + float(tp.tv_usec) / 1000000.0;
    return(Time(now));
  }

  /* Make printable version of time... */
  String Str(Format fmt = tSmart) const {
    char s[100];
    long seconds = long(cSeconds);
    long milli = long((cSeconds - seconds) * 1000.0);
    struct tm l = (*localtime((time_t *)&seconds));
    struct tm g = (*gmtime((time_t *)&seconds));
    
    for(;;) {
      switch(fmt) {
      case tDatetime:
	sprintf(s, "%02d/%02d/%04d,%02d:%02d:%02d.%03ld", 
		l.tm_mon+1, l.tm_mday, l.tm_year+1900, l.tm_hour, l.tm_min, l.tm_sec, milli);
	return(s);

      case tDate:
	sprintf(s, "%02d/%02d/%04d", l.tm_mon+1, l.tm_mday, l.tm_year+1900);
	return(s);

      case tFileName:
	sprintf(s, "%04d-%02d-%02d-%02d%02d%02d", 
		l.tm_year+1900, l.tm_mon+1, l.tm_mday, 
		l.tm_hour, l.tm_min, l.tm_sec);
	return(s);

      case tTime:
	sprintf(s, "%02d:%02d:%02d.%03ld", l.tm_hour, l.tm_min, l.tm_sec, milli);
	return(s);

      case tHHMMSS:
	sprintf(s, "%02d:%02d:%02d", l.tm_hour, l.tm_min, l.tm_sec);
	return(s);

      case tHHMM:
	sprintf(s, "%02d:%02d", l.tm_hour, l.tm_min);
	return(s);

      case tSeconds:
	sprintf(s, "%ld.%03ld", seconds, milli);
	return(s);

      case tDelta:
	if(g.tm_yday) sprintf(s, "%d:%02d:%02d:%02d.%03ld", 
			      g.tm_yday, g.tm_hour, g.tm_min, g.tm_sec, milli);
	else if(g.tm_hour) sprintf(s, "%02d:%02d:%02d.%03ld", g.tm_hour, g.tm_min, g.tm_sec, milli);
	else if(g.tm_min)  sprintf(s, "%02d:%02d.%03ld", g.tm_min, g.tm_sec, milli);
	else sprintf(s, "0:%02d.%03ld", g.tm_sec, milli);	
	return(s);

      case tMilli:
	sprintf(s, "%.3fmS", 1000.0*cSeconds);	
	return(s);

      case tMicro:
	sprintf(s, "%.3fuS", 1000000.0*cSeconds);	
	return(s);

      case tSmart:
      default:
	if(cSeconds == HUGE_VAL) return("+Inf");
	else if(cSeconds > (60*60*24*365)) fmt = tDatetime;
	else if(cSeconds < 0.001) fmt = tMicro;
	else if(cSeconds < 1) fmt = tMilli;
	else fmt = tDelta;
	continue;
      }
      /* Should never get here! */
      return("??");
    }
  }

  operator double() const { return(cSeconds); }

  Time operator+(double s) const { return(Time(cSeconds + s)); }
  Time operator-(double s) const { 
    return(Time((s > cSeconds) ? 0 : cSeconds - s)); 
  }

  bool Parse(const char *v) {
    time_t tmp = 0;
    struct tm t = (*localtime(&tmp));
    long dd, hh, mm;
    double seconds;
    cSeconds = 0;		// Default value...
    if(v) {
      if (6 == sscanf(v, "%d/%d/%d,%d:%d:%lf", &t.tm_mon, &t.tm_mday, &t.tm_year, &t.tm_hour, &t.tm_min, &seconds)) { 
	t.tm_mon--; t.tm_hour--; t.tm_year -= 1900;
	tmp = mktime(&t);
	cSeconds = (double)tmp + seconds;
	return(true);
      }
      else if(4 == sscanf(v, "%ld:%ld:%ld:%lf", &dd, &hh, &mm, &seconds)) { }
      else if(3 == sscanf(v, "%ld:%ld:%lf", &hh, &mm, &seconds)) { dd = 0; }
      else if(2 == sscanf(v, "%ld:%lf", &mm, &seconds)) { dd = hh = 0; }
      else if(1 == sscanf(v, "%lf", &seconds)) { dd = hh = mm = 0; }
      else return(false);

      cSeconds = ((double)(((((dd * 24) + hh) * 60) + mm) * 60)) + seconds;
      return(true);
    }
    return(false);
  }
};

inline ostream &operator<<(ostream &os, const Time& time)
{
  os << time.Str(Time::tSmart);
  return(os);
}

#endif // UtTime_h
