/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * An Interface class for handling attributes associated with IP
 * interfaces.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id$
 *  
 */
#ifndef __UtInterface_h
#define __UtInterface_h

#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <list.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <net/ethernet.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <UtReport.h>
#include <UtString.h>
#include <UtUtil.h>

  
class Interface {

  String cName;
  unsigned int cAddr;
  unsigned int cNetmask;
  unsigned int cMetric;
  bool cHasBroadcastAddr;
  bool cHasDestAddr;
  bool cHasHwAddr;
  unsigned int cBroadcastAddr;
  unsigned int cDestAddr;
  struct sockaddr cHwAddr;

  void FileWrite(const char* fname, bool on) {
    String path = String("/proc/sys/net/ipv4/conf/") + cName + "/" + fname;
    String s = (on ? "1" : "0");
    s += "\n";
    int fd;
    if((fd = open(path.str(),O_WRONLY)) == -1) {
      Report::Error(String("Interface::Redirects() - could not open '") + path + "' :" + strerror(errno));
    }
    if(write(fd,s.str(),s.length()) != ((int)s.length())) {
      Report::Error(String("Interface::Redirects() - could not write '") + path + "' ");
    }
    close(fd);
  }

public:
  Interface(const String& name,
	    unsigned int addr,
	    unsigned int netmask,
	    unsigned int metric,
	    unsigned int* broadcastAddr,
	    unsigned int* destAddr,
	    struct sockaddr* hwAddr) :
    cName(name),
    cAddr(addr),
    cNetmask(netmask),
    cMetric(metric),
    cHasBroadcastAddr(broadcastAddr != 0),
    cHasDestAddr(destAddr != 0),
    cHasHwAddr(hwAddr != 0),
    cBroadcastAddr((broadcastAddr ? *broadcastAddr : 0)),
    cDestAddr((destAddr ? *destAddr : 0))
    {
       if(hwAddr) cHwAddr = *hwAddr;
    };

  Interface() : 
    cName("unknown"), 
    cAddr(0),
    cNetmask(0),
    cMetric(0),
    cHasBroadcastAddr(false),
    cHasDestAddr(false),
    cHasHwAddr(false),
    cBroadcastAddr(0),
    cDestAddr(0)
  {};

  virtual ~Interface() {};
  String Name() const { return cName; };
  unsigned int Addr() const { return cAddr; };
  unsigned int Netmask() const { return cNetmask; };
  unsigned int Metric() const { return cMetric; };
  bool HasBroadcastAddr() const { return cHasBroadcastAddr; };
  bool HasDestAddr() const { return cHasDestAddr; };
  bool HasHwAddr() const { return cHasHwAddr; };
  unsigned int BroadcastAddr() const { return cBroadcastAddr; };
  unsigned int DestAddr() const { return cDestAddr; };
  struct sockaddr HwAddr() const { return cHwAddr; };

  unsigned int& Addr() { return cAddr; };
  unsigned int& Netmask() { return cNetmask; };
  unsigned int& Metric() { return cMetric; };
  bool& HasBroadcastAddr() { return cHasBroadcastAddr; };
  bool& HasDestAddr() { return cHasDestAddr; };
  bool& HasHwAddr() { return cHasHwAddr; };
  unsigned int& BroadcastAddr() { return cBroadcastAddr; };
  unsigned int& DestAddr() { return cDestAddr; };
  struct sockaddr& HwAddr() { return cHwAddr; };

  String Dump() const {
    String s;
    s += String("<Interface> ");
    s += cName + " ";
    s += DumpAddr(cAddr) + " / ";
    s += DumpAddr(cNetmask) + " ";
    s += String("m=") + String(cMetric) + " ";
    if(cHasBroadcastAddr) s += String("bcast=") + DumpAddr(cBroadcastAddr) + " ";
    if(cHasDestAddr) s += String("dest=") + DumpAddr(cDestAddr) + " ";
    if(cHasHwAddr) s += String("hw=") + DumpHwAddr(cHwAddr) + " ";
    // s += "\n";
    return(s);
  }


  /* Enable/disable redirects on the interface */
  void Redirects(bool on) {
    FileWrite("accept_redirects",on);
    FileWrite("send_redirects",on);
  }

  /* Enable/disable the rp filter on an interface */
  void RpFilter(bool on) {
    FileWrite("rp_filter",on);
  }

  /* Discovers all IP interfaces which are UP and are not LOOPBACK
   * and returns them as a list.
   */
  static list<Interface> Discover()
    {
      char* buf;
      struct ifconf ifc;
      struct ifreq* ifr;
      int sock;
      list<Interface> ifList;

      /* Get a socket so we can issue the ioctl */
      if( (sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
	Report::Error("Interaface::Discover() - socket() failed.");
      }
      
      /* We don't know how big a buffer we need for the SIOCGIFCONF ioctl.
       * Also, ioctl will return successfully if our buffer is too small and
       * simply truncate the result.
       * [Stevens] p 434 shows a technique by which we can issue the ioctl with
       * twice, the second time with a larger buffer. If the size of the result
       * does not change, then we know we got the whole result.
       */
      int len = 100 * sizeof(struct ifreq);
      int lastlen = 0;
      
      for(;;) {
	if((buf = (char*)malloc(len)) == 0) {
	  Report::Error("Interface::Discover() - malloc() failed.");
	}
	
	ifc.ifc_len = len;
	ifc.ifc_buf = buf;
	
	if(ioctl(sock, SIOCGIFCONF, &ifc) < 0) {
	  if((errno != EINVAL) || (lastlen != 0)) { 
	    Report::Error("Interface::Discover() - ioctl(SIOCGIFCONF) failed.");
	  }
	}
	else {
	  if(ifc.ifc_len == lastlen) break; // We got the whole result
	  lastlen = ifc.ifc_len;
	}
	len += 100 * sizeof(struct ifreq);
	free(buf);
      }
      
      // Here if we have a successful result from the ioctl(SIOCGIFCONF)
      
      
      // Iterate across the result buffer and yank out interface info
      String lastname;
      char* end = buf + ifc.ifc_len;
      for(char* p = buf; p < end; ) {
	ifr = (struct ifreq*)p;
	p += sizeof(ifr->ifr_name) + sizeof(struct sockaddr);

	// We only care about IP interfaces!
	if(ifr->ifr_addr.sa_family != AF_INET) continue; 
	
	// Avoid aliases by making sure the name of this interface is different
	// than the last one we looked at
	int i;
	for(i=0; i < IFNAMSIZ; i++) {
	  if(ifr->ifr_name[i] == 0) break;
	}
	String name(ifr->ifr_name,0,i);
	if(name == lastname) continue; 
	lastname = name;
	
	unsigned int addr = ntohl(((struct sockaddr_in *)&(ifr->ifr_addr))->sin_addr.s_addr);

	// Make sure this interface is UP
	// Note: Make a copy of the ifreq because we don't want to pollute the
	// remaining interfaces in the result buffer
	struct ifreq ifrcopy = *ifr;
	if(ioctl(sock, SIOCGIFFLAGS, &ifrcopy) < 0) {
	  Report::Error("Interface::Discover() - ioctl(SIOCGIFFLAGS) failed.");
	}
	if((ifrcopy.ifr_flags & IFF_UP) == 0) continue;
	
	// Ignore loopback interfaces
	if(ifrcopy.ifr_flags & IFF_LOOPBACK) continue;
	
	bool isBroadcast    = ifrcopy.ifr_flags & IFF_BROADCAST;
	bool isPointToPoint = ifrcopy.ifr_flags & IFF_POINTOPOINT;
	
	// Get the netmask for this interface
	ifrcopy = *ifr;
	if (ioctl (sock, SIOCGIFNETMASK, &ifrcopy) < 0) {
	  Report::Error("Interface::Discover() - ioctl(SIOCGIFNETMASK) failed.");
	}
	unsigned int netmask = ntohl(((struct sockaddr_in *)&ifrcopy.ifr_netmask)->sin_addr.s_addr);

	// Get the metric for this interface
	ifrcopy = *ifr;
	if (ioctl (sock, SIOCGIFMETRIC, &ifrcopy) < 0) {
	  Report::Error("Interface::Discover() - ioctl(SIOCGIFMETRIC) failed.");
	}
	unsigned int metric = ifrcopy.ifr_metric;


	// Get the broadcast address if this is a broadcast capable interface
	unsigned int broadcastAddr;
	if(isBroadcast) {
	  ifrcopy = *ifr;
	  if (ioctl (sock, SIOCGIFBRDADDR, &ifrcopy) < 0) {
	    Report::Error("Interface::Discover() - ioctl(SIOCGIFBRDADDR) failed.");
	  }
	  broadcastAddr = ntohl(((struct sockaddr_in *)&ifrcopy.ifr_broadaddr)->sin_addr.s_addr);
	}
	
	// Get the destination address if this is a point to point interface
	unsigned int destAddr;
	if(isPointToPoint) {
	  ifrcopy = *ifr;
	  if (ioctl (sock, SIOCGIFDSTADDR, &ifrcopy) < 0) {
	    Report::Error("Interface::Discover() - ioctl(SIOCGIFDSTADDR) failed.");
	  }
	  destAddr = ntohl(((struct sockaddr_in *)&ifrcopy.ifr_dstaddr)->sin_addr.s_addr);
	}

	// See if the interface has a hardware address
	ifrcopy = *ifr;
	struct sockaddr hwAddr;
	bool hasHwAddr = false;
	if (ioctl (sock, SIOCGIFHWADDR, &ifrcopy) == 0) {
	  //#### Should we only handle link address types of a certain type??
	  //#### For now we blindly accept all...
	  hasHwAddr = true;
	  hwAddr = *(struct sockaddr *)&ifrcopy.ifr_hwaddr;
	}

	// Finally, we can instantiate an Interface at the end of our list
	ifList.push_back(Interface(name,addr,netmask,metric,
				   (isBroadcast ? &broadcastAddr : 0),
				   (isPointToPoint ? &destAddr : 0),
				   (hasHwAddr ? &hwAddr : 0)));

      }
      return(ifList);
    };
};

#endif
