///////////////////////////////////////////////////////////////////////////////
// $Id: tilegroup.cpp,v 1.5 2004/10/03 19:30:45 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////

/*! \file    tilegroup.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the tile group/category class
*/

// Qt includes
#include <qbitmap.h>
#include <qdatetime.h>
#include <qimage.h>
#include <qfileinfo.h>
#include <qpainter.h>
#include <qpixmap.h>

// local includes
#include "indexfile.h"
#include "tilecache.h"
#include "tilegroup.h"

///////////////////////////////////////////////////////////////////////////////

TileGroup::TileGroup() : m_count(0)
{
	m_images.setAutoDelete(true);
	m_pixmaps.setAutoDelete(true);
}

///////////////////////////////////////////////////////////////////////////////

TileGroup::~TileGroup()
{
}

///////////////////////////////////////////////////////////////////////////////

bool TileGroup::load(const QString& filename)
{
	IndexFile file;

	if (!file.openFile(filename)) return false;

	QStringList files = file.files();
	m_images.clear();

	m_images.resize(files.count() * 2 + 1);

	QFileInfo fileinfo(filename);
	QString path = fileinfo.dirPath(true);

	int index = 0;
	QString tilefile(path + "/%0");
	for (QStringList::iterator it = files.begin(); it != files.end(); ++it, ++index)
	{
		if ((*it).isEmpty()) continue; // no tile for this index

		QImage img;
		if (!img.load(tilefile.arg(*it)))
		{
			qWarning("Failed to load tile %s",
			         static_cast<const char*>( QFile::encodeName(tilefile.arg(*it)) ));
			continue;
		}

		// create a 32 bit depth image
		QImage* image = new QImage(img.convertDepth(32));
		m_images.insert(index, image);

		// create pixmaps for widgets
		QBitmap mask(image->width(), image->height());
		QPainter painter(&mask);

		for (int y = 0; y < image->height(); ++y)
		{
			for (int x = 0; x < image->width(); ++x)
			{
				QRgb col = image->pixel(x, y);
				if (qRed(col) == 0 && qGreen(col) == 0 && qBlue(col) == 0)
				{
					painter.setPen(Qt::color0);
				}
				else
				{
					painter.setPen(Qt::color1);
				}
				painter.drawPoint(x, y);
			}
		}
		QPixmap* pixmap = new QPixmap(*image);
		pixmap->setMask(mask);
		m_pixmaps.insert(index, pixmap);
	}

	m_count = files.count();

	return true;
}

///////////////////////////////////////////////////////////////////////////////

QPixmap* TileGroup::pixmap(uint index)
{
	if (!m_pixmaps.isEmpty())
		return m_pixmaps[index];
		
	return 0;
}

///////////////////////////////////////////////////////////////////////////////

TileCache* TileGroup::createPixmapCache(int alpha, int brightness)
{
	if (alpha < 0 || alpha > 255 || brightness < 0 || brightness > 255)
		return 0;
	
	TileCache* cache = new TileCache();
	
	qDebug("creating pixmap cache for group %s",
	       m_name.latin1());
	QTime time;
	time.start();
	QIntDictIterator<QImage> it(m_images);
	for(; it.current() != 0; ++it)
	{
		//QImage image = it.current()->copy();
		QImage image(*(it.current()));
		image.setAlphaBuffer(true);

		QBitmap mask(image.width(), image.height());
		QPainter painter;
		painter.begin(&mask);
		
		for (int x = 0; x < image.width(); ++x)
		{
			for(int y = 0; y < image.height(); ++y)
			{
				QRgb color = image.pixel(x, y);
				int red   = qRed(color);
				int green = qGreen(color);
				int blue  = qBlue(color); 
				if (red == 0 && green == 0 && blue == 0)
				{
					painter.setPen(Qt::color0);
					//image.setPixel(x, y, qRgba(red, green, blue, 255));
					
				}
				else
				{
					painter.setPen(Qt::color1);
					//image.setPixel(x, y, qRgba(red, green, blue, alpha));
				}
				painter.drawPoint(x, y);
			}
		}
		painter.end();
		
		QPixmap* pixmap = new QPixmap(image);
		pixmap->setMask(mask);
		cache->insertPixmap(it.currentKey(), pixmap);
	}
	qDebug("Elapsed time: %d ms", time.elapsed());
	
	return cache;
}

///////////////////////////////////////////////////////////////////////////////

void TileGroup::copy(const TileGroup& other)
{
	m_name         = other.m_name;
	m_count        = other.m_count;

	m_images.clear();
	QIntDictIterator<QImage> it(other.m_images);
	for(; it.current() != 0; ++it)
	{
		QImage* image = it.current();
		m_images.insert(it.currentKey(), new QImage(*image));
	}
}

// End of file

