#!/bin/sh

set -eu

if [ -e ./mmdebstrap -a -e ./taridshift -a -e ./tarfilter -a -e ./coverage.py ]; then
	TMPFILE=$(mktemp)
	perltidy < ./mmdebstrap > "$TMPFILE"
	ret=0
	diff -u ./mmdebstrap "$TMPFILE" || ret=$?
	if [ "$ret" -ne 0 ]; then
		echo "perltidy failed" >&2
		rm "$TMPFILE"
		exit 1
	fi
	rm "$TMPFILE"

	if [ $(sed -e '/^__END__$/,$d' ./mmdebstrap | wc --max-line-length) -gt 79 ]; then
		echo "exceeded maximum line length of 79 characters" >&2
		exit 1
	fi

	perlcritic --severity 4 --verbose 8 ./mmdebstrap

	black --check ./taridshift ./tarfilter ./coverage.py
fi

mirrordir="./shared/cache/debian"

if [ ! -e "$mirrordir" ]; then
	echo "run ./make_mirror.sh before running $0" >&2
	exit 1
fi

# we use -f because the file might not exist
rm -f shared/cover_db.img

: "${DEFAULT_DIST:=unstable}"
: "${HAVE_QEMU:=yes}"
: "${RUN_MA_SAME_TESTS:=yes}"

if [ "$HAVE_QEMU" = "yes" ]; then
	# prepare image for cover_db
	guestfish -N shared/cover_db.img=disk:64M -- mkfs vfat /dev/sda

	if [ ! -e "./shared/cache/debian-$DEFAULT_DIST.qcow" ]; then
		echo "./shared/cache/debian-$DEFAULT_DIST.qcow does not exist" >&2
		exit 1
	fi
fi

# check if all required debootstrap tarballs exist
notfound=0
for dist in oldstable stable testing unstable; do
	for variant in minbase buildd -; do
		if [ ! -e "shared/cache/debian-$dist-$variant.tar" ]; then
			echo "shared/cache/debian-$dist-$variant.tar does not exist" >&2
			notfound=1
		fi
	done
done
if [ "$notfound" -ne 0 ]; then
	echo "not all required debootstrap tarballs are present" >&2
	exit 1
fi

# only copy if necessary
if [ ! -e shared/mmdebstrap ] || [ mmdebstrap -nt shared/mmdebstrap ]; then
	if [ -e ./mmdebstrap ]; then
		cp -a mmdebstrap shared
	else
		cp -a /usr/bin/mmdebstrap shared
	fi
fi
if [ ! -e shared/taridshift ] || [ taridshift -nt shared/taridshift ]; then
	if [ -e ./taridshift ]; then
		cp -a ./taridshift shared
	else
		cp -a /usr/bin/mmtaridshift shared/taridshift
	fi
fi
if [ ! -e shared/tarfilter ] || [ tarfilter -nt shared/tarfilter ]; then
	if [ -e ./tarfilter ]; then
		cp -a tarfilter shared
	else
		cp -a /usr/bin/mmtarfilter shared/tarfilter
	fi
fi
if [ ! -e shared/proxysolver ] || [ proxysolver -nt shared/proxysolver ]; then
	if [ -e ./proxysolver ]; then
		cp -a proxysolver shared
	else
		cp -a /usr/lib/apt/solvers/mmdebstrap-dump-solution shared/proxysolver
	fi
fi
if [ ! -e shared/ldconfig.fakechroot ] || [ ldconfig.fakechroot -nt shared/ldconfig.fakechroot ]; then
	if [ -e ./ldconfig.fakechroot ]; then
		cp -a ldconfig.fakechroot shared
	else
		cp -a /usr/libexec/mmdebstrap/ldconfig.fakechroot shared/ldconfig.fakechroot
	fi
fi
mkdir -p shared/hooks/merged-usr
if [ ! -e shared/hooks/merged-usr/setup00.sh ] || [ hooks/merged-usr/setup00.sh -nt shared/hooks/merged-usr/setup00.sh ]; then
	if [ -e hooks/merged-usr/setup00.sh ]; then
		cp -a hooks/merged-usr/setup00.sh shared/hooks/merged-usr/
	else
		cp -a /usr/share/mmdebstrap/hooks/merged-usr/setup00.sh shared/hooks/merged-usr/
	fi
fi
mkdir -p shared/hooks/eatmydata
if [ ! -e shared/hooks/eatmydata/extract.sh ] || [ hooks/eatmydata/extract.sh -nt shared/hooks/eatmydata/extract.sh ]; then
	if [ -e hooks/eatmydata/extract.sh ]; then
		cp -a hooks/eatmydata/extract.sh shared/hooks/eatmydata/
	else
		cp -a /usr/share/mmdebstrap/hooks/eatmydata/extract.sh shared/hooks/eatmydata/
	fi
fi
if [ ! -e shared/hooks/eatmydata/customize.sh ] || [ hooks/eatmydata/customize.sh -nt shared/hooks/eatmydata/customize.sh ]; then
	if [ -e hooks/eatmydata/customize.sh ]; then
		cp -a hooks/eatmydata/customize.sh shared/hooks/eatmydata/
	else
		cp -a /usr/share/mmdebstrap/hooks/eatmydata/customize.sh shared/hooks/eatmydata/
	fi
fi
mkdir -p shared/hooks/file-mirror-automount
if [ ! -e shared/hooks/file-mirror-automount/setup00.sh ] || [ hooks/file-mirror-automount/setup00.sh -nt shared/hooks/file-mirror-automount/setup00.sh ]; then
	if [ -e hooks/file-mirror-automount/setup00.sh ]; then
		cp -a hooks/file-mirror-automount/setup00.sh shared/hooks/file-mirror-automount/
	else
		cp -a /usr/share/mmdebstrap/hooks/file-mirror-automount/setup00.sh shared/hooks/file-mirror-automount/
	fi
fi
if [ ! -e shared/hooks/file-mirror-automount/customize00.sh ] || [ hooks/file-mirror-automount/customize00.sh -nt shared/hooks/file-mirror-automount/customize00.sh ]; then
	if [ -e hooks/file-mirror-automount/customize00.sh ]; then
		cp -a hooks/file-mirror-automount/customize00.sh shared/hooks/file-mirror-automount/
	else
		cp -a /usr/share/mmdebstrap/hooks/file-mirror-automount/customize00.sh shared/hooks/file-mirror-automount/
	fi
fi

# choose the timestamp of the unstable Release file, so that we get
# reproducible results for the same mirror timestamp
SOURCE_DATE_EPOCH=$(date --date="$(grep-dctrl -s Date -n '' "$mirrordir/dists/$DEFAULT_DIST/Release")" +%s)

# for traditional sort order that uses native byte values
export LC_ALL=C.UTF-8

: "${HAVE_UNSHARE:=yes}"
: "${HAVE_PROOT:=yes}"
: "${HAVE_BINFMT:=yes}"

defaultmode="auto"
if [ "$HAVE_UNSHARE" != "yes" ]; then
	defaultmode="root"
fi

# by default, use the mmdebstrap executable in the current directory together
# with perl Devel::Cover but allow to overwrite this
: "${CMD:=perl -MDevel::Cover=-silent,-nogcov ./mmdebstrap}"
mirror="http://127.0.0.1/debian"

export HAVE_QEMU HAVE_UNSHARE HAVE_BINFMT RUN_MA_SAME_TESTS DEFAULT_DIST SOURCE_DATE_EPOCH CMD mirror

./coverage.py

if [ -e shared/cover_db.img ]; then
	# produce report inside the VM to make sure that the versions match or
	# otherwise we might get:
	# Can't read shared/cover_db/runs/1598213854.252.64287/cover.14 with Sereal: Sereal: Error: Bad Sereal header: Not a valid Sereal document. at offset 1 of input at srl_decoder.c line 600 at /usr/lib/x86_64-linux-gnu/perl5/5.30/Devel/Cover/DB/IO/Sereal.pm line 34, <$fh> chunk 1.
	cat << END > shared/test.sh
cover -nogcov -report html_basic cover_db >&2
mkdir -p report
for f in common.js coverage.html cover.css css.js mmdebstrap--branch.html mmdebstrap--condition.html mmdebstrap.html mmdebstrap--subroutine.html standardista-table-sorting.js; do
	cp -a cover_db/\$f report
done
cover -delete cover_db >&2
END
	if [ "$HAVE_QEMU" = "yes" ]; then
		./run_qemu.sh
	elif [ "$mode" = "root" ]; then
		./run_null.sh SUDO
	else
		./run_null.sh
	fi

	echo
	echo open file://$(pwd)/shared/report/coverage.html in a browser
	echo
fi

rm shared/test.sh shared/tar1.txt shared/tar2.txt shared/pkglist.txt shared/doc-debian.tar.list shared/mmdebstrap shared/taridshift shared/tarfilter shared/proxysolver
