//
//  =================================================================
//
//    10.10.01   <--  Date of Last Modification.
//                   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//  -----------------------------------------------------------------
//
//  **** Module  :  MW_Class <implementation>
//       ~~~~~~~~~
//  **** Classes    :  CMWClass ( mwater functional class )
//       ~~~~~~~~~~~~  
//
//  (C) E.Krissinel'  2001
//
//  =================================================================
//


#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MW_Class__
#include "mw_class.h"
#endif

#define __CPlusPlus

#ifndef  __CCPLib__
#include "ccplib.h"
#endif

#ifndef  __CParser__
#include "cparser.h"
#endif


//  =========================   CMWClass   ==========================


CMWClass::CMWClass() : CMMDBManager()  {
  initMWClass();
}

CMWClass::~CMWClass() {
  freeContacts   ();
  freeSymOpTitles();
}

void CMWClass::initMWClass() {

  selMolecule = -1;
  selWater    = -1;

  minDist     = 0.73;
  maxDist     = 4.00;

  sortmode    = CNSORT_OFF;
  ncells      = 2;

  strcpy ( wChainID,"AUTO" );
  strcpy ( wResName,"HOH"  );

  Molecule    = NULL;
  Water       = NULL;
  nMolAtoms   = 0;
  nWatAtoms   = 0;

  contact     = NULL;
  ncontacts   = 0;
  contstat    = NULL;
  
  symOpTitle  = NULL;
  nSymOps     = 0;

}

int CMWClass::Job ( int argc, pstr * argv )  {

  //  1. Check that input/output file names are there
  if (!getenv("XYZIN"))  {
    ccperror ( 1,"Input file (XYZIN) is not specified." );
    return 1;
  }
  if (!getenv("XYZOUT"))  {
    ccperror ( 1,"Output file (XYZOUT) is not specified." );
    return 2;
  }

  //  2. Read coordinate file.
  if (readCoorFile())  return 3;

  //  3. Parse input cards.
  if (parseInput())    return 4;

  //  4. Check input for consistency.
  if (checkInput())    return 5;

  //  5. Print the transformation matrices.
  printTMatrices();

  //  6. Find contacts between protein and water.
  findContacts();

  //  7. Print contacts.
  printContacts();

  //  8. Generate new water molecules.
  generateWaters();

  //  9. Arrange water molecules into chains.
  arrangeWaters();

  //  10. Write out the resulting file.
  writeCoorFile();
  

  freeContacts   ();
  freeSymOpTitles();

  return 0;

}


static char separator[] =
 "--------------------------------------------------------------------------";

static char selectNotSolvent[] =
   "(!ADE,CYT,GUA,INO,THY,URA,WAT,HOH,TIP,H2O,DOD,MOH)";
static char selectSolvent[] =
   "(ADE,CYT,GUA,INO,THY,URA,WAT,HOH,TIP,H2O,DOD,MOH)";


int CMWClass::readCoorFile()  {
char S[500];
int  RC,lcount;

  //  1. Set all necessary read flags -- check with the top of
  //  file  mmdb_file.h  as needed

  printf ( "\n %s\n\n",separator );

  SetFlag ( MMDBF_PrintCIFWarnings );

  //  2. Read coordinate file by its logical name
  RC = ReadCoorFile1 ( "XYZIN" );

  //  3. Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  now quit
    return 1;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (GetFileType())  {
      case MMDB_FILE_PDB    : printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    : printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary : printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",getenv("XYZIN") );
  }

  return 0;

}

int  GetSelectionKey ( int ntok, PARSERTOKEN * token, int & selKey )  {
  selKey = SKEY_OR;
  if (ntok>3)  {
    if (keymatch("|",token[2].word)) {
      if (ntok>4)  {
        if (keymatch("OR" ,token[3].word))  selKey = SKEY_OR;
        else if (keymatch("XOR",token[3].word))  selKey = SKEY_XOR;
        else if (keymatch("AND",token[3].word))  selKey = SKEY_AND;
        else  {
          ccperror ( 1,"wrong selection key" );
          return -98;
        }
      } else  {
        ccperror ( 1,"no selection key" );
        return -98;
      }
    } else  {
      ccperror ( 1,"wrong syntax of the selection statement" );
      return -98;
    }
  }
  return 0;
}

int CMWClass::parseInput()  {
// Interprete cards from standard input stream
Boolean       molSel,watSel;
int           RC,selKey,i;
char          symGroup[500];
// input parser parameters
int           ntok=0;
char          line[201],*key;
PARSERTOKEN * token=NULL;
PARSERARRAY * parser;

  printf ( "\n %s\n  Input cards\n\n",separator );

  //  1. We will select atoms in the course of reading the input
  //     cards. Each _new_ selection starts with creation of
  //     the selection handle (a handle may be used in several
  //     selections)
  selMolecule = NewSelection();
  selWater    = NewSelection();


  //  2. Assign default values for minimal and maximal contact
  //     distances and the sequence distance

  minDist  = 0.73;
  maxDist  = 4.00;
  sortmode = CNSORT_OFF;
  ncells   = 2;

  molSel   = False;
  watSel   = False;

  strcpy ( wChainID,"AUTO" );
  strcpy ( wResName,"HOH"  );

  // 3. Initialise a parser array used by cparser
  //    This is used to return the tokens and associated info
  //    Set maximum number of tokens per line to 20
  parser = (PARSERARRAY *) cparse_start(20);

  if (!parser)  ccperror ( 1,"Couldn't create parser array" );

  // 4. Set some convenient pointers to members of the parser array
  key   = parser->keyword;
  token = parser->token;

  // 5. Read lines from stdin until END/end keyword is entered or
  //    EOF is reached
  RC   = 0;

  while (!RC) {

    // Always blank the line before calling cparser
    // to force reading from stdin
    line[0] = '\0';

    // Call cparser to read input line and break into tokens.
    // Returns the number of tokens, or zero for eof
    ntok = cparser(line,200,parser,1);

    if (ntok < 1) {

      RC = 111;  // End of file encountered

    } else {      

      // Keyword interpretation starts here

      // Traditionally CCP4 keywords are case-independent and
      // and only use the first four characters.

      if (keymatch("MOLECULE",key))  {
	if (ntok<2) {
	  ccperror ( 1,"selection statement not found" );
          RC = -99;
        } else  {
          RC = GetSelectionKey ( ntok,token,selKey );
          if (!RC)  {
            RC = Select ( selMolecule,STYPE_ATOM,
                          token[1].fullstring,selKey );
            molSel = True;
          }
        }
      } else if (keymatch("WATER",key))  {
	if (ntok<2) {
	  ccperror ( 1,"selection statement not found" );
          RC = -99;
        } else  {
          RC = GetSelectionKey ( ntok,token,selKey );
          if (!RC)  {
            RC = Select ( selWater,STYPE_ATOM,
                          token[1].fullstring,selKey );
            watSel = True;
          }
        }
      } else if (keymatch("WCHAIN",key))  {
        strcpy ( wChainID,token[1].fullstring );
	if (strncasecmp(wChainID,"AUTO",4)   && 
            strncasecmp(wChainID,"REARRANGE",4) && wChainID[1]) {
	  ccperror ( 1,"fail to interpret argument in WCHAIN" );
          RC = -99;
        }
      } else if (keymatch("WRESIDUE",key))  {
        strcpy ( wResName,token[1].fullstring );
	if (strlen(wResName)!=3)  {
	  ccperror ( 1,"improper PDB residue name in WRESIDUE" );
          RC = -99;
        }
      } else if (keymatch("MINDIST",key))  {
	if (ntok != 2) {
	  ccperror ( 1,"MINDIST requires a single numerical argument" );
	  RC = -100;
	} else if (!token[1].isnumber) {
	  ccperror ( 1,"MINDIST argument must be numerical" );
	  RC = -101;
	} else {
	  minDist = token[1].value;
	  if (minDist<0.0) {
	    ccperror ( 1,"MINDIST argument must be non-negative" );
	    RC = -102;
	  }
	}
      } else if (keymatch("MAXDIST",key))  {
	if (ntok != 2) {
	  ccperror ( 1,"MAXDIST requires a single numerical argument" );
	  RC = -103;
	} else if (!token[1].isnumber) {
	  ccperror ( 1,"MAXDIST argument must be numerical" );
	  RC = -104;
	} else {
	  maxDist = token[1].value;
	  if (maxDist<=0.0) {
	    ccperror ( 1,"MAXDIST argument must be positive" );
	    RC = -105;
	  }
	}
      } else if (keymatch("SORT",key))  {
	if (ntok<2) {
	  ccperror ( 1,"SORT requires 2nd argument - "
                       "OFF | TARGET | SOURCE | DISTANCE" );
	  RC = -109;
        } else  {
          if (keymatch("OFF",token[1].word))  sortmode = CNSORT_OFF;
          else  {
            sortmode = CNSORT_1INC;
            if (keymatch("TARGET",token[1].word))
                sortmode = CNSORT_2INC;
            else if (keymatch("DISTANCE",token[1].word))
                sortmode = CNSORT_DINC;
            if (ntok>2)  {
              if (keymatch("DEC" ,token[2].word))  {
                if (sortmode==CNSORT_1INC)       sortmode = CNSORT_1DEC;
                else if (sortmode==CNSORT_2INC)  sortmode = CNSORT_2DEC;
                                           else  sortmode = CNSORT_DDEC;
              }
            }
          }
        }
      } else if (keymatch("CELLS",key))  {
	if (ntok<2) {
	  ccperror ( 1,"CELLS requires 2nd argument - OFF | 0 | 1 | 2" );
	  RC = -110;
        } else if (keymatch("OFF",token[1].word))  ncells = -1;
        else if (keymatch("0",token[1].word))      ncells = 0;
        else if (keymatch("1",token[1].word))      ncells = 1;
        else if (keymatch("2",token[1].word))      ncells = 2;
        else if (keymatch("3",token[1].word))      ncells = 3;
        else  {
	  ccperror ( 1,"Argument of CELLS not understood" );
	  RC = -111;
        }
      } else if (keymatch("SYMMETRY",key)) {
	if (ntok<2) {
          ccperror ( 1,"SYMMETRY requires 2nd argument, e.g. P 21 21 21" );
          RC = -112;
        } else {
          i = token[ntok-1].iend - token[1].ibeg + 1;
          strcpy_ncss ( symGroup,&line[token[1].ibeg],i );
          symGroup[i] = char(0);
          i = SetSpaceGroup ( UpperCase(symGroup) );
          switch (i)  {
            case SYMOP_NoLibFile         : 
                    ccperror ( 1,"symop.lib file not found" );
                    RC = -113;
                  break;
            case SYMOP_UnknownSpaceGroup :
                    ccperror ( 1,"Unknown space symmetry group" );
                    RC = -114;
                  break;
            case SYMOP_NoSymOps          :
                    ccperror ( 1,"no symmetry operations found found" );
                    RC = -115;
                  break;
            default : ;
          }
        }
      } else if (keymatch("GEOMETRY",key)) {
	if (ntok!= 7) {
	  ccperror ( 1,"GEOMETRY requires 6 parameters:"
                       " a b c alpha beta gamma" );
	  RC = -116;
	} else {
          i = 1;
          while ((i<ntok) && token[i].isnumber && (token[i].value>0.0)) i++;
	  if (i<ntok) {
	    printf ( "%ith GEOMETRY parameter is not numerical "
                     "or positive",i );
	    RC = -117;
	  } else
            SetCell ( token[1].value,token[2].value,token[3].value,
                      token[4].value,token[5].value,token[6].value,0 );
	}
      }	else  {
	printf ( "Unrecognised keyword \"%s\"\n",token[0].fullstring );
	RC = -118;
      }
    }
  }

  if (RC==111)  RC = 0;  // normal return from the parser loop

  //  Clean up parser array
  cparse_end ( parser );

  if (!molSel)
    Select ( selMolecule,STYPE_ATOM,selectNotSolvent,SKEY_NEW );
  if (!watSel)
    Select ( selWater,STYPE_ATOM,selectSolvent,SKEY_NEW );

  return RC;

}


int CMWClass::checkInput()  {

  //  1. Check completeness of crystallographic information

  if (ncells>=0)  {
    if (!isCrystInfo())  {
      ccperror ( 1,"Missing cell parameters (a,b,c,alpha,beta,gamma)" );
      return 1;
    } else if (!isSpaceGroup())  {
      ccperror ( 1,"Missing space group of symmetry" );
      return 2;
    } else if (!isTransfMatrix())  {
      ccperror ( 1,"Orthogonalization/fractionalization is impossible" );
      return 3;
    } else  {
      nSymOps = GetNumberOfSymOps();
      if (nSymOps==0)  {
        ccperror ( 1,"No symmetry operations found" );
        return 4;
      }
    }
  } else
    nSymOps = 0;


  //  2. Selected atoms may be now accessed through the selection
  //     index. Selection index is merely a vector of pointers
  //     on the selected atoms. Check the function and its
  //     parameters in file  mmdb_selmngr.h

  GetSelIndex ( selMolecule,Molecule,nMolAtoms );

  //  exclude hydrogens from water atoms
  Select ( selWater,STYPE_ATOM,0,"*",ANY_RES,"*",ANY_RES,"*",
           "*","*","H","*",SKEY_CLR );
  GetSelIndex ( selWater   ,Water   ,nWatAtoms );

  printf ( "\n %s\n\n"
           " Selected  %i  molecule atoms\n"
           " Selected  %i  water atoms (oxygens only)\n",
           separator,nMolAtoms,nWatAtoms );

  if (nMolAtoms<=0)  {
    ccperror ( 1,"NO MOLECULE ATOMS SELECTED" );
    return 5;
  }

  if (nWatAtoms<=0)  {
    ccperror ( 1,"NO WATER ATOMS SELECTED" );
    return 6;
  }

  return 0;

}

void CMWClass::printTMatrices()  {
mat44 TMatrix;
int   i,m;
  if (ncells>=0)  {
    printf ( "\n Unit cell numbering:  " );
    for (i=-ncells;i<=ncells;i++)
      printf ( "%2i",i+ncells+1 );
    printf ( "\n\n Symmetry transformation matrices in cell %i:\n",ncells+1 );
    for (m=0;m<nSymOps;m++)
      if (GetTMatrix(TMatrix,m,0,0,0)==SYMOP_Ok)
        printf (  "\n"
          "  Operation %s\n"
          "      [%10.4f %10.4f %10.4f]  [%10.4f]\n" 
          "  T = [%10.4f %10.4f %10.4f]  [%10.4f]\n"
          "      [%10.4f %10.4f %10.4f]  [%10.4f]\n",
          GetSymOp(m),
          TMatrix[0][0],TMatrix[0][1],TMatrix[0][2],TMatrix[0][3],
          TMatrix[1][0],TMatrix[1][1],TMatrix[1][2],TMatrix[1][3],
	  TMatrix[2][0],TMatrix[2][1],TMatrix[2][2],TMatrix[2][3] );
  }
}

void CMWClass::freeContacts()  {
  if (contact)  delete contact;
  contact   = NULL;
  ncontacts = 0;
  FreeVectorMemory ( contstat,0 );
}


long  CMWClass::makeGroupID ( int i, int j, int k, int m )  {
  return  (i+ncells) | ((j+ncells)<<8) | ((k+ncells)<<16) | (m<<24);
}

void  CMWClass::parseGroupID ( long group,
                               int & i, int & j, int & k, int & m )  {
  i = (int(group)     & 0x000000FF) + 1;
  j = (int(group>>8)  & 0x000000FF) + 1;
  k = (int(group>>16) & 0x000000FF) + 1;
  m =  int(group>>24) & 0x0000FFFF;
}

int  CMWClass::hesh4 ( int i, int j, int k, int m )  {
int  n=2*ncells+1;
  return  (i+ncells)*n*n*nSymOps +
          (j+ncells)*n*nSymOps +
          (k+ncells)*nSymOps +
          m;
}


void CMWClass::freeSymOpTitles()  {
int i,ns;
  if (symOpTitle)  {
    ns = hesh4 ( ncells,ncells,ncells,nSymOps );
    for (i=0;i<ns;i++)
      if (symOpTitle[i])  delete symOpTitle[i];
    delete symOpTitle;
  }
  symOpTitle = NULL;
}

void CMWClass::findContacts()  {
//    Find contacts between the selected molecule and water
//  atoms. Here we use dynamical allocation of contacts'
//  index. See details and the function description in file
//  mmdb_selmngr.h
CSymOp SymOp;
mat44  TMatrix;
int    i,j,k,m,ns,nh;

  freeContacts   ();
  freeSymOpTitles();

  if (ncells<0)  {
    // no unit cell modeling is required; find contacts between
    // atoms given in coordinate file only
    SeekContacts ( Molecule,nMolAtoms,Water,nWatAtoms,
                   minDist,maxDist,0,
                   contact,ncontacts,0,NULL );
  } else  {
    // construct all necessary cells for target atoms and run the
    // contact-looking procedure for each cell. In this implementation,
    // no new chains/residues/atoms are created; the contact-looking
    // procedure merely adjust coordinates by applying a transformation
    // matrix for symmetry mates and neighbouring unit cells.
    ns = hesh4 ( ncells,ncells,ncells,nSymOps );
    GetVectorMemory ( contstat,ns,0 );
    symOpTitle = new pstr[ns];
    for (i=0;i<ns;i++)
      symOpTitle[i] = NULL;
    ns = 0;
    for (i=-ncells;i<=ncells;i++)
      for (j=-ncells;j<=ncells;j++)
        for (k=-ncells;k<=ncells;k++)
          for (m=0;m<nSymOps;m++)  {
            //   GetMatrix(..) calculates the transformation matrix
            // for mth symmetry operation, which places atoms
            // into unit cell, shifted by i,j,k in a,b,c - directions,
            // respectively (in fractional space), from the principal
            // unit cell of the coordinate file. 
            if (GetTMatrix(TMatrix,m,i,j,k)!=SYMOP_Ok)  {
              printf (
                " *** unexpected error at calculating the transformation\n"
                "     matrix for symmetry operation %i in cell %i%i%i.\n",
                m,i+ncells+1,j+ncells+1,k+ncells+1 );
            } else  {
              //   SeekContacts(..) finds all contacts between the
              // source (Molecule) and target (Water) atoms, transforming
              // the target according to the (optional) transformation
              // matrix.
              //   makeGroupID(..) encodes the values of i,j,k,m
              // into single long integer to store it with the
              // contact found. These values are then used for
              // printout.
              SeekContacts ( Molecule,nMolAtoms,Water,nWatAtoms,
                             minDist,maxDist,0,
                             contact,ncontacts,0,&TMatrix,
                             makeGroupID(i,j,k,m) );
            }
            // store contact statistics by cell and operation
            contstat[ns++] = ncontacts;
            // compile the symmetry operation title
            nh = hesh4 ( i,j,k,m );
            if (!symOpTitle[nh])
              symOpTitle[nh] = new char[100];
            SymOp.SetSymOp   ( GetSymOp(m) );
            SymOp.GetTMatrix ( TMatrix );
            TMatrix[0][3] += i;
            TMatrix[1][3] += j;
            TMatrix[2][3] += k;
            SymOp.CompileOpTitle ( symOpTitle[nh],TMatrix,False );
          }
  }

}


void CMWClass::printContacts() {
//  Print contacts. Check with description of the structure
//  SContact in file mmdb_selmngr.h .
char S[300],S1[300];
int  i,j,k,m,ci,id,ns,nh;

  if (ncontacts<=0)  {

    printf ( "\n %s\n\n  NO CONTACTS FOUND.\n\n",separator );

  } else  {

    //  Sort contacts according to the requested sorting mode.
    SortContacts ( contact,ncontacts,sortmode );

    printf ( 
      "\n %s\n\n  %i contacts found:\n\n"
      "     MOLECULE ATOMS             SOLVENT ATOMS           DISTANCE",
      separator,ncontacts );

    if (ncells>=0)  printf ( " CELL   SYMMETRY" );
    printf ( "\n\n" );

    id = MinInt4;
    for (ci=0;ci<ncontacts;ci++)  {
      Water   [contact[ci].id2]->GetAtomIDfmt ( S1 );
      Molecule[contact[ci].id1]->GetAtomIDfmt ( S  );
      if (id!=contact[ci].id1)  id = contact[ci].id1;
                          else  memset ( S,' ',strlen(S) );
      if (ncells>=0)  {
        parseGroupID ( contact[ci].group,i,j,k,m );
        nh = hesh4 ( i-ncells-1,j-ncells-1,k-ncells-1,m );
        printf ( " %s %s %7.5g %1i%1i%1i %s\n",S,S1,contact[ci].dist, i,j,k,
                   symOpTitle[nh] );
      } else
        printf ( " %s %s %5.2f\n",S,S1,contact[ci].dist );
    }

  }

  if (ncells>=0)  {

    ns = hesh4 ( ncells,ncells,ncells,nSymOps );
    for (i=ns-1;i>0;i--)
      contstat[i] -= contstat[i-1];
    printf (
 "\n\n %s\n\n   Contact statistics:"
 "\n\n"
 "  CELL   FRACT COORD-S   SYM NO   SYM OPERATION              NUM OF CONT-S"
 "\n\n",separator );

    ns = 0;
    for (i=-ncells;i<=ncells;i++)
      for (j=-ncells;j<=ncells;j++)
        for (k=-ncells;k<=ncells;k++)
          for (m=0;m<nSymOps;m++)  {
            if (contstat[ns]>0)  {
              nh = hesh4 ( i,j,k,m );
              printf ( "  %1i%1i%1i      %2i %2i %2i    %5i     %-30s  %7i\n",
                       i+ncells+1,j+ncells+1,k+ncells+1, i,j,k,
                       m,symOpTitle[nh],contstat[ns] );
            }
            ns++;
          }

  }

  printf ( "\n\n  Total %i contacts\n\n %s\n\n",ncontacts,separator );

}


void  CMWClass::CopyAtom ( PCAtom atom, PCChain chain,
                           const ResName rname, mat44 * TMatrix )  {
PCAtom    a;
PCResidue r;
int       seqNum;
InsCode   insCode;
  a = new CAtom();
  a->Copy ( atom );
  if (TMatrix)
    a->Transform ( *TMatrix );
  seqNum = atom->GetSeqNum();
  strcpy ( insCode,atom->GetInsCode() );
  r = chain->GetResidue ( seqNum,insCode );
  if (!r)  {
    if (rname[0])  r = new CResidue ( chain,rname,seqNum,insCode );
             else  r = new CResidue ( chain,atom->GetResName(),
                                      seqNum,insCode );
  }
  r->AddAtom ( a );
}


void CMWClass::generateWaters()  {
//  Generate new water atoms in 'X,Y,Z' symmetry of primary cell
//  from their symmetry mates.
PCChain wChain;
mat44   TMatrix;
char    S[300];
int     ci,i,j,k,m;

  if (!strncasecmp(wChainID,"AUTO",4))
    GetNewChainID ( 1,wChainID,1 );
  else if (!strncasecmp(wChainID,"REARRANGE",4))  {
    GetNewChainID ( 1,wChainID,1 );
    strcat ( wChainID,"_rge" );
  }

  wChain = new CChain ( GetModel(1),wChainID );
  if ((ncells>=0) && (ncontacts>=0))
    for (ci=0;ci<ncontacts;ci++)  {
      parseGroupID ( contact[ci].group,i,j,k,m );
      i -= ncells+1;
      j -= ncells+1;
      k -= ncells+1;
      if ((i!=0) || (j!=0) || (k!=0) || (m!=0))  {
        // treat only water atoms that do not belong to 'X,Y,Z'
        // symmetry of primary cell
        if (GetTMatrix(TMatrix,m,i,j,k)!=SYMOP_Ok)
          printf ( " *** unexpected error at calculating the transformation\n"
                   "     matrix for symmetry operation %i in cell %i%i%i.\n",
                   m,i+ncells+1,j+ncells+1,k+ncells+1 );
        else
          CopyAtom ( Water[contact[ci].id2],wChain,wResName,&TMatrix );
      }
    }

  //  7.  Remove repeating water molecules.

  sprintf ( S,"/1/%s/*(%s).*/*[*]:*",wChainID,wResName );
  Select      ( selWater,STYPE_ATOM,S,SKEY_OR );
  GetSelIndex ( selWater,Water,nWatAtoms );  // now Water contains
                                             // *all* water molecules

  freeContacts();  // this is a _must_ for dynamically allocated index
                   // no contacts in the begining

  SeekContacts ( Water,nWatAtoms,Water,nWatAtoms,
                 0.0,minDist,0,contact,ncontacts,0,NULL );

  for (ci=0;ci<ncontacts;ci++)
    if (Water[contact[ci].id1] && Water[contact[ci].id2] &&
        (Water[contact[ci].id1]!=Water[contact[ci].id2]))  {
      if (!strcmp(Water[contact[ci].id1]->GetChainID(),wChainID))  {
        delete Water[contact[ci].id1];
        Water[contact[ci].id1] = NULL;  // important!
      } else if (!strcmp(Water[contact[ci].id2]->GetChainID(),wChainID))  {
        delete Water[contact[ci].id2];
        Water[contact[ci].id2] = NULL;  // important!
      }
    }

}


void CMWClass::arrangeWaters()  {
//  Arrange water molecules in chains by proximity to the protein,
//  if requested.
PPCChain chain;
PCChain  wChain;
PPCAtom  pChain;
ChainID  newWChainID;
char     S[500];
int      nChains,selChain,nChAtoms;
int      i,ci;

  if (!strstr(wChainID,"_rge"))  {

    // It should be a single water chain with new water molecules left.
    // Just resort its residues by incresing the residue sequence number.

    wChain = GetChain ( 1,wChainID );
    if (wChain)  {
      wChain->SortResidues();
      printf ( "  The newly found %i water molecules, contacting the protein,\n"
               "  are formed into chain '%s' and appended "
               "to the end of PDB file.\n\n %s\n\n",
               wChain->GetNumberOfAtoms(False),
               wChain->GetChainID(),separator );
    } else
      printf ( "  --- Unexpected error: can't find new water chain.\n" );

  } else  {

    // Rearrange all water chains by proximity to the protein
    // chains

    strcpy ( S,"|" );
    for (i=0;i<nWatAtoms;i++)
      if (Water[i])  {
        if (!strstr(S,Water[i]->GetChainID()))  {
          strcat ( S,Water[i]->GetChainID() );
          strcat ( S,"|" );
        }
      }

    GetChainTable ( 1,chain,nChains );
    selChain = NewSelection();
    for (i=0;(i<nChains) && (nWatAtoms>0);i++)
      if (chain[i])  {
        // check if this is a water chain
        if (!strstr(S,chain[i]->GetChainID()))  {
          Select ( selChain,STYPE_ATOM,chain[i]->GetChainID(),SKEY_NEW );
          GetSelIndex ( selChain,pChain,nChAtoms );
          if ((nWatAtoms>0) && (nChAtoms>0))  {
            // in spite of the fact that we remove atoms from selection
            // set 'Water', it is Ok to keep using it in SeekContacts(..)
            // without re-selection. SeekContacts(..) works properly
            // with any incomplete sets of atoms, as long as missing
            // atoms have NULL pointers.
            freeContacts();
            SeekContacts ( pChain,nChAtoms,Water,nWatAtoms,
                           minDist,maxDist,0,
                           contact,ncontacts,0,NULL );
            if (ncontacts>0)  {
              GetNewChainID  ( 1,newWChainID,1 );
              wChain = new CChain ( GetModel(1),newWChainID );
              for (ci=0;ci<ncontacts;ci++)
                if (Water[contact[ci].id2])  {
                  CopyAtom ( Water[contact[ci].id2],wChain,"",NULL );
                  delete Water[contact[ci].id2];
                  Water[contact[ci].id2] = NULL;  // important!
                }
              wChain->SortResidues();
              printf ( " Formed chain '%s' with %i solvent "
                       "molecules surrounding protein chain '%s'.\n",
                       wChain->GetChainID(),
		       wChain->GetNumberOfAtoms(False),
                       chain[i]->GetChainID() );
            }
          }
        }
      }

    if (nWatAtoms>0)  {
      GetNewChainID  ( 1,newWChainID,1 );
      wChain = new CChain ( GetModel(1),newWChainID );
      for (i=0;i<nWatAtoms;i++)
        if (Water[i])  {
          CopyAtom ( Water[i],wChain,"",NULL );
          delete Water[i];
          Water[i] = NULL;  // important!
        }
      wChain->SortResidues();
      printf ( " Formed chain '%s' with %i solvent "
               "molecules not contacting the protein.\n",
               wChain->GetChainID(),
               wChain->GetNumberOfAtoms(False) );
    }

    // remove temporary water chain
    DeleteSelection ( selChain );
    selChain = NewSelection();
    Select ( selChain,STYPE_CHAIN,S,SKEY_NEW );
    DeleteSelObjects ( selChain );

    printf ( "\n %s\n\n",separator );

  }

}

void CMWClass::writeCoorFile()  {
  PDBCleanup ( PDBCLEAN_INDEX | PDBCLEAN_SERIAL );
  WritePDBASCII1 ( "XYZOUT" );
  printf ( " PDB file %s has been written out.\n\n",getenv("XYZOUT") );
}


