# -------------------------------------------------------------------------
#     Copyright (C) 2005-2012 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program.
# -------------------------------------------------------------------------

# load libs
import math
import wx

# load modules
from ids import *
import mwx
import images
import config
import mspy
import mspy.plot


# FLOATING PANEL WITH MASS DEFECT PLOT
# ------------------------------------

class panelMassDefectPlot(wx.MiniFrame):
    """Mass defect plot tool."""
    
    def __init__(self, parent):
        wx.MiniFrame.__init__(self, parent, -1, 'Mass Defect Plot', size=(400, 300), style=wx.DEFAULT_FRAME_STYLE)
        
        self.parent = parent
        
        self.currentDocument = None
        self.currentPeaks = None
        self.currentNotations = None
        
        # make gui items
        self.makeGUI()
        wx.EVT_CLOSE(self, self.onClose)
    # ----
    
    
    def makeGUI(self):
        """Make panel gui."""
        
        # make toolbar
        toolbar = self.makeToolbar()
        controlbar = self.makeControlbar()
        
        # make panels
        self.makePlotCanvas()
        
        # pack elements
        self.mainSizer = wx.BoxSizer(wx.VERTICAL)
        self.mainSizer.Add(toolbar, 0, wx.EXPAND, 0)
        self.mainSizer.Add(controlbar, 0, wx.EXPAND, 0)
        self.mainSizer.Add(self.plotCanvas, 1, wx.EXPAND, 0)
        
        # fit layout
        self.mainSizer.Fit(self)
        self.SetSizer(self.mainSizer)
        self.SetMinSize(self.GetSize())
    # ----
    
    
    def makeToolbar(self):
        """Make toolbar."""
        
        # init toolbar
        panel = mwx.bgrPanel(self, -1, images.lib['bgrToolbar'], size=(-1, mwx.TOOLBAR_HEIGHT))
        
        # make tools
        xAxis_label = wx.StaticText(panel, -1, "X axis:")
        xAxis_label.SetFont(wx.SMALL_FONT)
        
        choices = ['m/z', 'Nominal Mass', 'Kendrick Mass', 'Kendrick Mass(F)']
        self.xAxis_choice = wx.Choice(panel, -1, choices=choices, size=(175, mwx.SMALL_CHOICE_HEIGHT))
        self.xAxis_choice.Select(0)
        if config.massDefectPlot['xAxis'] in choices:
            self.xAxis_choice.Select(choices.index(config.massDefectPlot['xAxis']))
        
        yAxis_label = wx.StaticText(panel, -1, "Y axis:")
        yAxis_label.SetFont(wx.SMALL_FONT)
        
        choices = ['Fractional Mass', 'Mass Defect', 'Relative Mass Defect', 'Kendrick Mass Defect', 'Kendrick Mass Defect (F)']
        self.yAxis_choice = wx.Choice(panel, -1, choices=choices, size=(200, mwx.SMALL_CHOICE_HEIGHT))
        self.yAxis_choice.Select(0)
        if config.massDefectPlot['yAxis'] in choices:
            self.yAxis_choice.Select(choices.index(config.massDefectPlot['yAxis']))
        self.yAxis_choice.Bind(wx.EVT_CHOICE, self.onYAxisChanged)
        
        groupFormula_label = wx.StaticText(panel, -1, "Group formula:")
        groupFormula_label.SetFont(wx.SMALL_FONT)
        self.groupFormula_value = mwx.formulaCtrl(panel, -1, "", size=(80, -1))
        
        self.onYAxisChanged()
        
        # make buttons
        self.plot_butt = wx.Button(panel, -1, "Plot", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.plot_butt.Bind(wx.EVT_BUTTON, self.onPlot)
        
        # pack elements
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.CONTROLBAR_LSPACE)
        sizer.Add(xAxis_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.xAxis_choice, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(yAxis_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.yAxis_choice, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(groupFormula_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.groupFormula_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddStretchSpacer()
        sizer.AddSpacer(20)
        sizer.Add(self.plot_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(mwx.TOOLBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(sizer, 1, wx.EXPAND)
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def makeControlbar(self):
        """Make controlbar."""
        
        # init toolbar
        panel = mwx.bgrPanel(self, -1, images.lib['bgrControlbar'], size=(-1, mwx.CONTROLBAR_HEIGHT))
        
        # make elements
        nominalMass_label = wx.StaticText(panel, -1, "Nominal mass:")
        nominalMass_label.SetFont(wx.SMALL_FONT)
        choices = ['Round', 'Floor', 'Ceil']
        self.nominalMass_choice = wx.Choice(panel, -1, choices=choices, size=(80, mwx.SMALL_CHOICE_HEIGHT))
        self.nominalMass_choice.Select(0)
        if config.massDefectPlot['nominalMass'].title() in choices:
            self.nominalMass_choice.Select(choices.index(config.massDefectPlot['nominalMass'].title()))
        
        relIntCutoff_label = wx.StaticText(panel, -1, "Rel. int. threshold:")
        relIntCutoff_label.SetFont(wx.SMALL_FONT)
        relIntCutoffUnits_label = wx.StaticText(panel, -1, "%")
        relIntCutoffUnits_label.SetFont(wx.SMALL_FONT)
        self.relIntCutoff_value = wx.TextCtrl(panel, -1, str(config.massDefectPlot['relIntCutoff']*100), size=(50, mwx.SMALL_TEXTCTRL_HEIGHT), validator=mwx.validator('floatPos'))
        self.relIntCutoff_value.SetFont(wx.SMALL_FONT)
        
        self.ignoreCharge_check = wx.CheckBox(panel, -1, "Ignore charge")
        self.ignoreCharge_check.SetFont(wx.SMALL_FONT)
        self.ignoreCharge_check.SetValue(config.massDefectPlot['ignoreCharge'])
        
        self.removeIsotopes_check = wx.CheckBox(panel, -1, "Remove isotopes")
        self.removeIsotopes_check.SetFont(wx.SMALL_FONT)
        self.removeIsotopes_check.SetValue(config.massDefectPlot['removeIsotopes'])
        
        self.showNotations_check = wx.CheckBox(panel, -1, "Highlight annotated")
        self.showNotations_check.SetFont(wx.SMALL_FONT)
        self.showNotations_check.SetValue(config.massDefectPlot['showNotations'])
        self.showNotations_check.Bind(wx.EVT_CHECKBOX, self.onShowNotations)
        
        # pack elements
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.CONTROLBAR_LSPACE)
        sizer.Add(nominalMass_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.nominalMass_choice, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(relIntCutoff_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.relIntCutoff_value, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(relIntCutoffUnits_label, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(self.ignoreCharge_check, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.AddSpacer(20)
        sizer.Add(self.removeIsotopes_check, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(self.showNotations_check, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(mwx.CONTROLBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(sizer, 1, wx.EXPAND)
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makePlotCanvas(self):
        """Make plot canvas and set defalt parameters."""
        
        # init canvas
        self.plotCanvas = mspy.plot.canvas(self, size=(650, 300), style=mwx.PLOTCANVAS_STYLE_PANEL)
        self.plotCanvas.draw(mspy.plot.container([]))
        
        # set default params
        self.plotCanvas.setProperties(xLabel='m/z')
        self.plotCanvas.setProperties(yLabel='mass defect')
        self.plotCanvas.setProperties(showLegend=False)
        self.plotCanvas.setProperties(showZero=True)
        self.plotCanvas.setProperties(showGrid=True)
        self.plotCanvas.setProperties(showMinorTicks=False)
        self.plotCanvas.setProperties(showPosBar=True)
        self.plotCanvas.setProperties(posBarHeight=6)
        self.plotCanvas.setProperties(showIntBar=True)
        self.plotCanvas.setProperties(intBarHeight=6)
        self.plotCanvas.setProperties(showGel=False)
        self.plotCanvas.setProperties(zoomAxis='xy')
        self.plotCanvas.setProperties(checkLimits=True)
        self.plotCanvas.setProperties(autoScaleY=False)
        self.plotCanvas.setProperties(xPosDigits=config.main['mzDigits'])
        self.plotCanvas.setProperties(yPosDigits=config.main['mzDigits'])
        self.plotCanvas.setProperties(reverseScrolling=config.main['reverseScrolling'])
        self.plotCanvas.setProperties(reverseDrawing=False)
        self.plotCanvas.setMFunction('cross')
        self.plotCanvas.setLMBFunction('xDistance')
        
        axisFont = wx.Font(config.spectrum['axisFontSize'], wx.SWISS, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL, 0)
        self.plotCanvas.setProperties(axisFont=axisFont)
        
        self.plotCanvas.canvas.Bind(wx.EVT_LEFT_UP, self.onPlotCanvasLMU)
        
        self.plotCanvas.draw(mspy.plot.container([]))
    # ----
    
    
    def onClose(self, evt):
        """Destroy this frame."""
        
        # close self
        self.Destroy()
    # ----
    
    
    def onPlot(self, evt):
        """Calculate and show mass defect plot."""
        
        # clear previous data
        self.currentPeaks = None
        self.currentNotations = None
        
        # check document
        if not self.currentDocument or not self.currentDocument.spectrum.haspeaks():
            wx.Bell()
            return
        
        # get params
        if not self.getParams():
            self.updatePlotCanvas()
            wx.Bell()
            return
        
        # calculate data points
        self.calcDataPoints()
        
        # update plot
        self.updatePlotCanvas()
    # ----
    
    
    def onPlotCanvasLMU(self, evt):
        """Highlight selected point in spectrum."""
        
        # check document
        if self.currentDocument == None:
            evt.Skip()
            return
        
        # get cursor positions
        position = self.plotCanvas.getCursorPosition()
        distance = self.plotCanvas.getDistance()
        
        # sent event back to canvas
        self.plotCanvas.onLMU(evt)
        
        # highlight selected point in spectrum
        if position and distance and distance[0] == 0:
            self.parent.updateMassPoints([position[0]])
        
        evt.Skip()
    # ----
    
    
    def onYAxisChanged(self, evt=None):
        """Update unit formula value."""
        
        # get current selection
        selected = self.yAxis_choice.GetStringSelection()
        
        # set Kendrick formula
        if selected == 'Kendrick Mass Defect':
            self.groupFormula_value.SetValue('CH2')
        else:
            self.groupFormula_value.SetValue('')
        
        # enable custom formula
        if selected == 'Kendrick Mass Defect (F)':
            self.groupFormula_value.Enable()
            self.groupFormula_value.SetValue(config.massDefectPlot['groupFormula'])
        else:
            self.groupFormula_value.Disable()
    # ----
    
    
    def onShowNotations(self, evt):
        """Show / hide annotated points."""
        
        # get value
        config.massDefectPlot['showNotations'] = self.showNotations_check.GetValue()
        
        # update plot
        self.updatePlotCanvas()
    # ----
    
    
    def setData(self, document):
        """Set current document."""
        
        # set new document
        self.currentDocument = document
        
        # clear previous data
        self.currentPeaks = None
        self.currentNotations = None
        
        # update plot
        self.updatePlotCanvas()
    # ----
    
    
    def getParams(self):
        """Get all params from dialog."""
        
        # try to get values
        try:
            
            config.massDefectPlot['xAxis'] = self.xAxis_choice.GetStringSelection()
            config.massDefectPlot['yAxis'] = self.yAxis_choice.GetStringSelection()
            config.massDefectPlot['nominalMass'] = self.nominalMass_choice.GetStringSelection().lower()
            config.massDefectPlot['relIntCutoff'] = float(self.relIntCutoff_value.GetValue())/100.
            config.massDefectPlot['ignoreCharge'] = self.ignoreCharge_check.GetValue()
            config.massDefectPlot['removeIsotopes'] = self.removeIsotopes_check.GetValue()
            config.massDefectPlot['showNotations'] = self.showNotations_check.GetValue()
            
            formula = self.groupFormula_value.GetValue()
            cmpd = mspy.compound(formula)
            config.massDefectPlot['groupFormula'] = str(formula)
            
            return True
        
        except:
            wx.Bell()
            return False
    # ----
    
    
    def updatePlotCanvas(self):
        """Update plot canvas."""
        
        # make container
        container = mspy.plot.container([])
        
        # set axis labels
        xLabel=config.massDefectPlot['xAxis']
        yLabel = config.massDefectPlot['yAxis']
        
        if config.massDefectPlot['yAxis'] == 'Custom Unit Mass Defect':
            yLabel = '%s (%s)' % (config.massDefectPlot['yAxis'], config.massDefectPlot['groupFormula'])
        
        self.plotCanvas.setProperties(xLabel=xLabel)
        self.plotCanvas.setProperties(yLabel=yLabel)
        
        # check data
        if self.currentPeaks == None and self.currentNotations == None:
            self.plotCanvas.draw(container)
            return
        
        # make points objects
        if self.currentPeaks != None:
            points = mspy.plot.points(self.currentPeaks, pointColour=(0,255,0), showPoints=True, showLines=False, legend='peak list')
            container.append(points)
        
        if self.currentNotations != None and config.massDefectPlot['showNotations']:
            points = mspy.plot.points(self.currentNotations, pointColour=(255,0,0), showPoints=True, showLines=False, legend='annotated')
            container.append(points)
        
        # draw container
        self.plotCanvas.draw(container)
    # ----
    
    
    def calcDataPoints(self):
        """Calculate mass defect plot data."""
        
        # clear previous data
        self.currentPeaks = []
        self.currentNotations = []
        
        # get spectrum polarity
        polarity = self.currentDocument.spectrum.polarity
        if not polarity:
            polarity = 1
        
        # calculate mass defect for peaks
        buff = []
        for peak in self.currentDocument.spectrum.peaklist:
            if config.massDefectPlot['removeIsotopes'] and not peak.isotope in (0, None):
                continue
            if peak.ri >= config.massDefectPlot['relIntCutoff']:
                buff.append([peak.mz, peak.charge])
        
        self.currentPeaks = self.calcMassDefect(buff, polarity)
        
        # calculate mass defect for notations
        items = []
        for item in self.currentDocument.annotations:
            items.append(item)
        for sequence in self.currentDocument.sequences:
            for item in sequence.matches:
                items.append(item)
        
        basepeak = self.currentDocument.spectrum.peaklist.basepeak
        if basepeak:
            basepeak = basepeak.intensity
        
        buff = []
        for item in items:
            if not basepeak or (item.ai-item.base)/basepeak >= config.massDefectPlot['relIntCutoff']:
                buff.append([item.mz, item.charge])
        
        self.currentNotations = self.calcMassDefect(buff, polarity)
    # ----
    
    
    def calcMassDefect(self, peaks, polarity=1):
        """Calculate mass defect plot data."""
        
        buff = []
        
        # init rounding fn
        if config.massDefectPlot['nominalMass'] == 'floor':
            nominalMass = math.floor
        elif config.massDefectPlot['nominalMass'] == 'ceil':
            nominalMass = math.ceil
        else:
            nominalMass = round
        
        # init Kendrick factor
        kendrickF = 1.
        if config.massDefectPlot['groupFormula']:
            group = mspy.compound(config.massDefectPlot['groupFormula'])
            kendrickF = group.nominalmass(0)/group.mass(0)
        
        # calculate data points
        for peak in peaks:
            
            mz = peak[0]
            mzDecon = peak[0]
            if not config.massDefectPlot['ignoreCharge'] and peak[1]:
                mzDecon = mspy.mz(peak[0], 1*polarity, peak[1], agentFormula='H', agentCharge=1)
            
            # init point
            point = [mz, 0.0]
            
            # make X axis
            if config.massDefectPlot['xAxis'] == 'm/z':
                pass
            
            elif config.massDefectPlot['xAxis'] == 'Nominal Mass':
                point[0] = nominalMass(mz)
            
            elif config.massDefectPlot['xAxis'] == 'Kendrick Mass':
                point[0] = mz * kendrickF
            
            elif config.massDefectPlot['xAxis'] == 'Kendrick Mass (F)':
                point[0] = mz * kendrickF
            
            # make Y axis
            if config.massDefectPlot['yAxis'] == 'Fractional Mass':
                point[1] = math.modf(mz)[0]
            
            elif config.massDefectPlot['yAxis'] == 'Mass Defect':
                point[1] = nominalMass(mzDecon) - mzDecon
            
            elif config.massDefectPlot['yAxis'] == 'Relative Mass Defect':
                point[1] = ((nominalMass(mzDecon) - mzDecon) / mzDecon) * 1e6
            
            elif config.massDefectPlot['yAxis'] == 'Kendrick Mass Defect':
                point[1] = nominalMass(mzDecon * kendrickF) - (mzDecon * kendrickF)
            
            elif config.massDefectPlot['yAxis'] == 'Kendrick Mass Defect (F)':
                point[1] = nominalMass(mzDecon * kendrickF) - (mzDecon * kendrickF)
            
            # append point
            buff.append(point)
        
        return buff
    # ----
    
    

