// -------------------------------------------------------------------------
//     Copyright (C) 2005-2012 Martin Strohalm <www.mmass.org>
//
//     This program is free software; you can redistribute it and/or modify
//     it under the terms of the GNU General Public License as published by
//     the Free Software Foundation; either version 3 of the License, or
//     (at your option) any later version.
//
//     This program is distributed in the hope that it will be useful,
//     but WITHOUT ANY WARRANTY; without even the implied warranty of
//     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
//     GNU General Public License for more details.
//
//     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
//     main directory of the program.
// -------------------------------------------------------------------------

#include <stdlib.h>
#include <math.h>
#include "Python.h"
#include "arrayobject.h"


#define ELEM_SWAP(a,b) { register double t=(a);(a)=(b);(b)=t; }


// Calculate 1D array median.

double signal_median(double inarr[], int len)
{
    int low, high;
    int median;
    int middle, ll, hh;
    
    low = 0 ; high = len-1 ; median = (low + high) / 2;
    for ( ; ; ) {
        
        if ( high <= low ) {
            return inarr[median];
        }
        else if ( high == low + 1 ) {
            if ( inarr[low] > inarr[high] ) {
                ELEM_SWAP(inarr[low], inarr[high]);
            }
            return inarr[median];
        }
        
        middle = (low + high) / 2;
        if ( inarr[middle] > inarr[high] )    ELEM_SWAP(inarr[middle], inarr[high]);
        if ( inarr[low] > inarr[high] )       ELEM_SWAP(inarr[low], inarr[high]);
        if ( inarr[middle] > inarr[low] )     ELEM_SWAP(inarr[middle], inarr[low]);
        ELEM_SWAP(inarr[middle], inarr[low+1]) ;
        
        ll = low + 1;
        hh = high;
        for ( ; ; ) {
            do ll++; while ( inarr[low] > inarr[ll] );
            do hh--; while ( inarr[hh] > inarr[low] );
            if ( hh < ll ) {
                break;
            }
            ELEM_SWAP(inarr[ll], inarr[hh]);
        }
        
        ELEM_SWAP(inarr[low], inarr[hh]);
        
        if (hh <= median) {
            low = ll;
        }
        if (hh >= median) {
            high = hh - 1;
        }
    }
}


// Find index of the nearest higher point in signal. Return 0 or len if out.

int signal_locate_x( double inarr[], int len, int cell, double x )
{
    int lo = 0;
    int hi = len;
    int mid;
    
    while ( lo < hi ) {
        mid = ( lo + hi ) / 2;
        if ( x < inarr[mid*cell] )
            hi = mid;
        else
            lo = mid + 1;
    }
    
    return lo;
}

static PyObject *_wrap_signal_locate_x( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    double x;
    int len, dim, cell, result;
    double *cpoints;
    
    if ( !PyArg_ParseTuple(args, "Od", &points, &x) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    dim = cell = (int) PyArray_NDIM(points);
    if ( dim == 2 )
        cell = (int) PyArray_DIM(points, 1);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // find index
    result = signal_locate_x( cpoints, len, cell, x );
    
    return Py_BuildValue("i", result);
}


// Find index of the highest y-value in signal.

int signal_locate_max_y( double inarr[], int len, int cell )
{
    int i, imax;
    double value;
    
    // init starting values
    imax = 0;
    value = inarr[cell-1];
    
    // find maximum
    for ( i = 0; i < len; ++i) {
        if ( inarr[i*cell+cell-1] > value ) {
            imax = i;
            value = inarr[i*cell+cell-1];
        }
    }
    
    return imax;
}

static PyObject *_wrap_signal_locate_max_y( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    int len, dim, cell, result;
    double *cpoints;
    
    if ( !PyArg_ParseTuple(args, "O", &points) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    dim = cell = (int) PyArray_NDIM(points);
    if ( dim == 2 )
        cell = (int) PyArray_DIM(points, 1);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // find index
    result = signal_locate_max_y( cpoints, len, cell );
    
    return Py_BuildValue("i", result);
}


// Calculate inner point (x-value) between two points by linear interpolation.

double signal_interpolate_x( double x1, double y1, double x2, double y2, double y )
{
    double m, b;
    
    // check points
    if ( x1 == x2 ) {
        return x1;
    }
    
    // get params
    m = (y2 - y1)/(x2 - x1);
    b = y1 - m * x1;
    
    return (y - b) / m;
}

static PyObject *_wrap_signal_interpolate_x( PyObject *self, PyObject *args )
{
    double x1, x2, y1, y2, y;
    double result;
    
    if ( !PyArg_ParseTuple(args, "ddddd", &x1, &y1, &x2, &y2, &y) ) {
        return NULL;
    }
    
    // interpolate point
    result = signal_interpolate_x( x1, y1, x2, y2, y );
    
    return Py_BuildValue("d", result);
}


// Calculate inner point (y-value) between two points by linear interpolation.

double signal_interpolate_y( double x1, double y1, double x2, double y2, double x )
{
    double m, b;
    
    // check points
    if ( y1 == y2 ) {
        return y1;
    }
    
    // get params
    m = (y2 - y1)/(x2 - x1);
    b = y1 - m * x1;
    
    return m * x + b;
}

static PyObject *_wrap_signal_interpolate_y( PyObject *self, PyObject *args )
{
    double x1, x2, y1, y2, x;
    double result;
    
    if ( !PyArg_ParseTuple(args, "ddddd", &x1, &y1, &x2, &y2, &x) ) {
        return NULL;
    }
    
    // interpolate point
    result = signal_interpolate_y( x1, y1, x2, y2, x );
    
    return Py_BuildValue("d", result);
}


// Calculate 2D signal minima and maxima in both axis.

void signal_boundaries( double inarr[], int len, double *p_minX, double *p_minY, double *p_maxX, double *p_maxY )
{
    int i;
    
    // get x limits
    *p_minX = inarr[0];
    *p_maxX = inarr[2*len-2];
    
    // get y limits
    *p_minY = inarr[1];
    *p_maxY = inarr[1];
    for ( i = 0; i < len; ++i) {
        if ( inarr[i*2+1] < *p_minY ) {
            *p_minY = inarr[i*2+1];
        }
        if ( inarr[i*2+1] > *p_maxY ) {
            *p_maxY = inarr[i*2+1];
        }
    }
}

static PyObject *_wrap_signal_boundaries( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    int len;
    double *cpoints;
    double minX, minY, maxX, maxY;
    
    if ( !PyArg_ParseTuple(args, "O", &points) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // calculates boundaries
    signal_boundaries( cpoints, len, &minX, &minY, &maxX, &maxY );
    
    return Py_BuildValue("dddd", minX, minY, maxX, maxY);
}


// Find local maxima in 2D signal.

void signal_maxima( double inarr[], double outarr[], int len, int *p_count )
{
    double currentX, currentY;
    int i, counter, rising;
    
    // init first
    currentX = inarr[0];
    currentY = inarr[1];
    
    // find maxima
    rising = 0;
    counter = 0;
    for ( i=0; i<len; ++i) {
        
        if ( inarr[2*i+1] > currentY ) {
            rising = 1;
        }
        else if ( inarr[2*i+1] < currentY && rising ) {
            outarr[2*counter] = currentX;
            outarr[2*counter+1] = currentY;
            counter++;
            rising = 0;
        }
        currentX = inarr[2*i];
        currentY = inarr[2*i+1];
    }
    
    // save number of points
    *p_count = counter;
}

static PyObject *_wrap_signal_maxima( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    double *cpoints, *cresults, *cbuff;
    int len, counter;
    int i;
    
    if ( !PyArg_ParseTuple(args, "O", &points) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // allocate memory for buffer
    if ( (cbuff = (double*) malloc( len*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // find local maxima
    signal_maxima( cpoints, cbuff, len, &counter );
    
    // make numpy array
    dim[0] = (npy_intp) counter;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // copy buffer data to numpy array
    cresults = (double *) results->data;
    for ( i=0; i<counter; i++ ) {
      cresults[i*2] = cbuff[i*2];
      cresults[i*2+1] = cbuff[i*2+1];
    }
    
    free(cbuff);
    
    return PyArray_Return(results);
}


// Calculate y-value for given x-value.

double signal_intensity( double inarr[], int len, double x )
{
    int idx;
    double y;
    
    // locate x-value
    idx = signal_locate_x( inarr, len, 2, x );
    if ( idx == 0 || idx == len ) {
        return 0.0;
    }
    
    // interpolate y-value
    y = signal_interpolate_y( inarr[2*idx-2], inarr[2*idx-1], inarr[2*idx], inarr[2*idx+1], x );
    
    return y;
}

static PyObject *_wrap_signal_intensity( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    double *cpoints;
    double x, result;
    int len;
    
    if ( !PyArg_ParseTuple(args, "Od", &points, &x ) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type arrays
    cpoints = (double *) points->data;
    
    // get intensity
    result = signal_intensity( cpoints, len, x );
    
    return Py_BuildValue("d", result);
}


// Calculate centroid for given x-value measured at y-value.

double signal_centroid( double inarr[], int len, double x, double height )
{
    int idx, ileft, iright;
    double xleft, xright;
    
    // locate x-value
    idx = signal_locate_x( inarr, len, 2, x );
    if ( idx == 0 || idx == len ) {
        return 0.0;
    }
    
    // get left index
    ileft = idx;
    while ( ( ileft > 0 ) && ( inarr[2*ileft+1] > height ) ) {
        ileft--;
    }
    
    // get right index
    iright = idx;
    while ( ( ileft < len-1 ) && ( inarr[2*iright+1] > height ) ) {
        iright++;
    }
    
    // check indexes
    if ( ileft == iright ) {
        return inarr[2*ileft];
    }
    
    // interpolate y-values
    xleft = signal_interpolate_x( inarr[2*ileft], inarr[2*ileft+1], inarr[2*ileft+2], inarr[2*ileft+3], height );
    xright = signal_interpolate_x( inarr[2*iright-2], inarr[2*iright-1], inarr[2*iright], inarr[2*iright+1], height );
    
    return ((xleft + xright) / 2);
}

static PyObject *_wrap_signal_centroid( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    double *cpoints;
    double x, height, result;
    int len;
    
    if ( !PyArg_ParseTuple(args, "Odd", &points, &x, &height ) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type arrays
    cpoints = (double *) points->data;
    
    // get width
    result = signal_centroid( cpoints, len, x, height );
    
    return Py_BuildValue("d", result);
}


// Calculate peak width for given x-value measured at y-value.

double signal_width( double inarr[], int len, double x, double height )
{
    int idx, ileft, iright;
    double xleft, xright;
    
    // locate x-value
    idx = signal_locate_x( inarr, len, 2, x );
    if ( idx == 0 || idx == len ) {
        return 0.0;
    }
    
    // get left index
    ileft = idx - 1;
    while ( ( ileft > 0 ) && ( inarr[2*ileft+1] > height ) ) {
        ileft--;
    }
    
    // get right index
    iright = idx;
    while ( ( iright < len-1 ) && ( inarr[2*iright+1] > height ) ) {
        iright++;
    }
    
    // check indexes
    if ( ileft == iright ) {
        return 0.0;
    }
    
    // interpolate y-values
    xleft = signal_interpolate_x( inarr[2*ileft], inarr[2*ileft+1], inarr[2*ileft+2], inarr[2*ileft+3], height );
    xright = signal_interpolate_x( inarr[2*iright-2], inarr[2*iright-1], inarr[2*iright], inarr[2*iright+1], height );
    
    return fabs(xright - xleft);
}

static PyObject *_wrap_signal_width( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    double *cpoints;
    double x, height, result;
    int len;
    
    if ( !PyArg_ParseTuple(args, "Odd", &points, &x, &height ) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type arrays
    cpoints = (double *) points->data;
    
    // get width
    result = signal_width( cpoints, len, x, height );
    
    return Py_BuildValue("d", result);
}


// Calculate 2D signal area under curve.

double signal_area(double inarr[], int len)
{
    double area;
    double x1, x2, y1, y2;
    int i;
    
    // check points
    if ( len < 2) {
        return 0.0;
    }
    
    // calculate area
    area = 0.0;
    for ( i = 1; i < len; ++i ) {
        x1 = inarr[i*2-2];
        y1 = inarr[i*2-1];
        x2 = inarr[i*2];
        y2 = inarr[i*2+1];
        area += (y1*(x2-x1)) + ((y2-y1)*(x2-x1)/2);
    }
    
    return area;
}

static PyObject *_wrap_signal_area( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    int len;
    double *cpoints;
    double area;
    
    if ( !PyArg_ParseTuple(args, "O", &points) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // calculate area
    area = signal_area( cpoints, len );
    
    return Py_BuildValue("d", area);
}


// Calculate 2D signal noise.

void signal_noise( double inarr[], int len, double *p_level, double *p_width )
{
    double *cbuff;
    double level, width;
    int i;
    
    // init default values
    *p_level = 0.0;
    *p_width = 1.0;
    
    // allocate memory for buffer
    if ( (cbuff = (double*) malloc( len*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return;
    }
    
    // duplicate y-values from inarr
    for ( i = 0; i < len; ++i ) {
        cbuff[i] = inarr[2*i+1];
    }
    
    // find noise level (median of y-values)
    level = signal_median( cbuff, len );
    
    // calculate abs deviations
    for ( i = 0; i < len; ++i ) {
        cbuff[i] = fabs(cbuff[i] - level);
    }
    
    // calculate noise width (median of abs deviations)
    width = signal_median( cbuff, len );
    width *= 2;
    
    // free buffer
    free(cbuff);
    
    // save resultsvalues
    *p_level = level;
    *p_width = width;
}

static PyObject *_wrap_signal_noise( PyObject *self, PyObject *args )
{
    PyArrayObject *points;
    int len;
    double *cpoints;
    double level, width;
    
    if ( !PyArg_ParseTuple(args, "O", &points) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // calculate noise
    signal_noise( cpoints, len, &level, &width );
    
    return Py_BuildValue("dd", level, width);
}


// Make gaussian shape peak.

void signal_gaussian( double outarr[], int len, double x, double minY, double maxY, double fwhm )
{
    double minX, maxX, step;
    double width, amplitude;
    double currentX;
    int i;
    
    // get x-range
    minX = x - (5*fwhm);
    maxX = x + (5*fwhm);
    
    // get step
    step = (maxX - minX) / len;
    
    // get gaussian width
    width = fwhm / 1.66;
    
    // get real amplitude
    amplitude = maxY - minY;
    
    // make data
    currentX = minX;
    for ( i = 0; i < len; ++i ) {
        outarr[2*i] = currentX;
        outarr[2*i+1] = amplitude * exp( -((currentX-x)*(currentX-x)) / (width*width) ) + minY;
        currentX += step;
    }
}

static PyObject *_wrap_signal_gaussian( PyObject *self, PyObject *args )
{
    PyArrayObject *results;
    npy_intp dim[2];
    double x, minY, maxY, fwhm;
    int points;
    double *cresults;
    
    if ( !PyArg_ParseTuple(args, "ddddi", &x, &minY, &maxY, &fwhm, &points) ) {
        return NULL;
    }
    
    // make numpy array for results
    dim[0] = (npy_intp) points;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // make gaussian
    signal_gaussian( cresults, points, x, minY, maxY, fwhm );
    
    return PyArray_Return(results);
}


// Crop 2D signal to given x-axis range.

void signal_crop( double inarr[], double outarr[], int len, double minX, double maxX, int *p_count )
{
    
    int i, i1, i2;
    int counter = 0;
    double leftY, rightY;
    
    // get indexes of limits
    i1 = signal_locate_x(inarr, len, 2, minX);
    i2 = signal_locate_x(inarr, len, 2, maxX);
    
    // interpolate left edge
    if (i1 > 0) {
        leftY = signal_interpolate_y( inarr[i1*2-2], inarr[i1*2-1], inarr[i1*2], inarr[i1*2+1], minX );
        outarr[counter*2] = minX;
        outarr[counter*2+1] = leftY;
        counter++;
    }
    
    // add inner points
    for ( i = i1; i < i2; ++i) {
        outarr[counter*2] = inarr[i*2];
        outarr[counter*2+1] = inarr[i*2+1];
        counter++;
    }
    
    // interpolate right edge
    if (i2 < len) {
        rightY = signal_interpolate_y( inarr[i2*2-2], inarr[i2*2-1], inarr[i2*2], inarr[i2*2+1], maxX );
        outarr[counter*2] = maxX;
        outarr[counter*2+1] = rightY;
        counter++;
    }
    
    // save number of points
    *p_count = counter;
}

static PyObject *_wrap_signal_crop( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    double minX, maxX;
    double *cpoints, *cresults, *cbuff;
    int len, counter;
    int i;
    
    if ( !PyArg_ParseTuple(args, "Odd", &points, &minX, &maxX) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // allocate memory for buffer
    if ( (cbuff = (double*) malloc( 2*len*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // crop data
    signal_crop( cpoints, cbuff, len , minX, maxX, &counter );
    
    // make numpy array
    dim[0] = (npy_intp) counter;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // copy buffer data to numpy array
    cresults = (double *) results->data;
    for ( i=0; i<counter; i++ ) {
      cresults[i*2] = cbuff[i*2];
      cresults[i*2+1] = cbuff[i*2+1];
    }
    
    free(cbuff);
    
    return PyArray_Return(results);
}


// Offset 2D signal y-values by number.

void signal_offset( double inarr[], double outarr[], int len, double x, double y )
{
    int i;
    
    // offset points
    for ( i = 0; i < len; ++i) {
        outarr[i*2] = inarr[i*2] + x;
        outarr[i*2+1] = inarr[i*2+1] + y;
    }
}

static PyObject *_wrap_signal_offset( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    int len;
    double x, y;
    double *cpoints, *cresults;
    
    if ( !PyArg_ParseTuple(args, "Odd", &points, &x, &y) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // make array for results
    dim[0] = (npy_intp) len;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // offset data
    signal_offset( cpoints, cresults, len, x, y );
    
    return PyArray_Return(results);
}


// Multiply 2D signal y-values by number.

void signal_multiply( double inarr[], double outarr[], int len, double x, double y )
{
    int i;
    
    // multiply points
    for ( i = 0; i < len; ++i) {
        outarr[i*2] = inarr[i*2] * x;
        outarr[i*2+1] = inarr[i*2+1] * y;
    }
}

static PyObject *_wrap_signal_multiply( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    int len;
    double x, y;
    double *cpoints, *cresults;
    
    if ( !PyArg_ParseTuple(args, "Odd", &points, &x, &y) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // make array for results
    dim[0] = (npy_intp) len;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // multiply data
    signal_multiply( cpoints, cresults, len, x, y );
    
    return PyArray_Return(results);
}


// Scale and shift 2D signal - used for plots.

void signal_rescale( double inarr[], double outarr[], int len, double scaleX, double scaleY, double shiftX, double shiftY )
{
    int i;
    
    // multiply and offset points
    for ( i = 0; i < len; ++i) {
        outarr[i*2] = round(inarr[i*2]*scaleX+shiftX);
        outarr[i*2+1] = round(inarr[i*2+1]*scaleY+shiftY);
    }
}

static PyObject *_wrap_signal_rescale( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    int len;
    double scaleX, scaleY, shiftX, shiftY;
    double *cpoints, *cresults;
    
    if ( !PyArg_ParseTuple(args, "Odddd", &points, &scaleX, &scaleY, &shiftX, &shiftY) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // make array for results
    dim[0] = (npy_intp) len;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // scale and shift data
    signal_rescale( cpoints, cresults, len, scaleX, scaleY, shiftX, shiftY );
    
    return PyArray_Return(results);
}


// Normalize 2D signal y-values to max 1.

void signal_normalize( double inarr[], double outarr[], int len )
{
    double maxY;
    int i;
    
    // get max Y
    maxY = inarr[1];
    for ( i = 0; i < len; ++i) {
        if ( inarr[i*2+1] > maxY ) {
            maxY = inarr[i*2+1];
        }
    }
    
    // normalize points
    for ( i = 0; i < len; ++i) {
        outarr[i*2] = inarr[i*2];
        outarr[i*2+1] = inarr[i*2+1] / maxY;
    }
}

static PyObject *_wrap_signal_normalize( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    int len;
    double *cpoints, *cresults;
    
    if ( !PyArg_ParseTuple(args, "O", &points) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // make numpy array for results
    dim[0] = (npy_intp) len;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // normalize data
    signal_normalize( cpoints, cresults, len );
    
    return PyArray_Return(results);
}


// Smooth 2D signal by moving average filter.

void signal_smooth_ma( double inarr[], double outarr[], int len, int window, int cycles )
{
    int c, i, j, idx, ksize;
    double average, ksum;
    
    // check window size
    if ( window > len ) {
        window = len;
    }
    
    // make window even
    if ( window % 2 != 0) {
        window -= 1;
    }
    
    // make kernel
    ksize = window + 1;
    ksum = window + 1;
    double kernel[ksize];
    for ( i = 0; i <= ksize; ++i ) {
        kernel[i] = 1/ksum;
    }
    
    // smooth cycles
    for ( c = 0; c < cycles; ++c ) {
        
        // get average in window for every point
        for ( i = 0; i < len; ++i ) {
            
            average = 0.0;
            for ( j = 0; j <= window; ++j ) {
                idx = (int) fabs(i+j-window/2);
                if ( idx >= len) {
                    idx -= 2*(idx - len + 1);
                }
                average += kernel[j] * inarr[2*idx+1];
            }
            
            outarr[2*i] = inarr[2*i];
            outarr[2*i+1] = average;
        }
    }
}

static PyObject *_wrap_signal_smooth_ma( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    int window, cycles;
    int len;
    double *cpoints, *cresults;
    
    if ( !PyArg_ParseTuple(args, "Oii", &points, &window, &cycles) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // make numpy array for results
    dim[0] = (npy_intp) len;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // smooth data
    signal_smooth_ma( cpoints, cresults, len, window, cycles );
    
    return PyArray_Return(results);
}


// Smooth 2D signal by gaussian filter.

void signal_smooth_ga( double inarr[], double outarr[], int len, int window, int cycles )
{
    int c, i, j, idx, ksize;
    double average, ksum, r, k;
    
    // check window size
    if ( window > len ) {
        window = len;
    }
    
    // make window even
    if ( window % 2 != 0) {
        window -= 1;
    }
    
    // make kernel
    ksize = window + 1;
    ksum = 0.0;
    double kernel[ksize];
    for ( i = 0; i <= ksize; ++i ) {
        r = (i - (ksize-1)/2.0);
        k = exp(-(r*r/(ksize*ksize/16.0)));
        kernel[i] = k;
        ksum += k;
    }
    for ( i = 0; i <= ksize; ++i ) {
        kernel[i] /= ksum;
    }
    
    // smooth cycles
    for ( c = 0; c < cycles; ++c ) {
        
        // get average in window for every point
        for ( i = 0; i < len; ++i ) {
            
            average = 0.0;
            for ( j = 0; j <= window; ++j ) {
                idx = (int) fabs(i+j-window/2);
                if ( idx >= len) {
                    idx -= 2*(idx - len + 1);
                }
                average += kernel[j] * inarr[2*idx+1];
            }
            
            outarr[2*i] = inarr[2*i];
            outarr[2*i+1] = average;
        }
    }
}

static PyObject *_wrap_signal_smooth_ga( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    int window, cycles;
    int len;
    double *cpoints, *cresults;
    
    if ( !PyArg_ParseTuple(args, "Oii", &points, &window, &cycles) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type array
    cpoints = (double *) points->data;
    
    // make numpy array for results
    dim[0] = (npy_intp) len;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // smooth data
    signal_smooth_ga( cpoints, cresults, len, window, cycles );
    
    return PyArray_Return(results);
}


// Filter 2D signal points according to resolution - used for plots.

void signal_filter( double inarr[], double outarr[], int len, double resolution, int *p_count )
{
    double currentX, currentY, lastX, lastY, previousX, previousY, minY, maxY;
    int i, counter;
    
    // add first point
    outarr[0] = inarr[0];
    outarr[1] = inarr[1];
    lastX = previousX = inarr[0];
    lastY = minY = maxY = previousY = inarr[1];
    counter = 1;
    
    // filter points
    for ( i=1; i<len; i++ ) {
        currentX = inarr[2*i];
        currentY = inarr[2*i+1];
        
        // difference between current and previous x-values is higher than resolution
        // save previous point and its minimum and maximum
        if ( (currentX-lastX) >= resolution ) {
            
            // add base point
            if ( outarr[2*counter-2] != lastX || outarr[2*counter-1] != minY ) {
                outarr[2*counter] = lastX;
                outarr[2*counter+1] = minY;
                counter++;
            }
            
            // add previous minimum
            if ( maxY != minY ) {
                outarr[2*counter] = lastX;
                outarr[2*counter+1] = maxY;
                counter++;
            }
            
            // add previous maximum
            if ( previousY != maxY) {
                outarr[2*counter] = previousX;
                outarr[2*counter+1] = previousY;
                counter++;
            }
            
            // add curent point
            outarr[2*counter] = currentX;
            outarr[2*counter+1] = currentY;
            lastX = previousX = currentX;
            lastY = maxY = minY = previousY = currentY;
            counter++;
        }
        
        // difference between current and previous x-values is lower than resolution
        // remember minimum and maximum
        else {
            minY = ( currentY < minY ) ? currentY:minY;
            maxY = ( currentY > maxY ) ? currentY:maxY;
            previousX = currentX;
            previousY = currentY;
        }
    }
    
    // save number of points
    *p_count = counter;
}

static PyObject *_wrap_signal_filter( PyObject *self, PyObject *args )
{
    PyArrayObject *points, *results;
    npy_intp dim[2];
    double *cpoints, *cresults, *cbuff;
    double resolution;
    int len, counter;
    int i;
    
    if ( !PyArg_ParseTuple(args, "Od", &points, &resolution) ) {
        return NULL;
    }
    
    // get array size
    len = (int) PyArray_DIM(points, 0);
    
    // convert array to c-type arrays
    cpoints = (double *) points->data;
    
    // allocate memory for buffer
    if ( (cbuff = (double*) malloc( 4*2*len*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // filter data
    signal_filter( cpoints, cbuff, len, resolution, &counter );
    
    // make numpy array
    dim[0] = (npy_intp) counter;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // copy buffer data to numpy array
    cresults = (double *) results->data;
    for ( i=0; i<counter; i++ ) {
      cresults[i*2] = cbuff[i*2];
      cresults[i*2+1] = cbuff[i*2+1];
    }
    
    free(cbuff);
    
    return PyArray_Return(results);
}


// It unifies x-raster and combines 2D signals (y = yA + yB)

void signal_combine( double inarrA[], int lenA, double inarrB[], int lenB, double outarr[], int *p_count )
{
    int i, j, counter;
    double y;
    
    // init counters
    i = 0;
    j = 0;
    counter = 0;
    
    while ( i < lenA || j < lenB ) {
        
        // process points within both arrays
        if ( i < lenA && j < lenB ) {
            if ( inarrA[i*2] < inarrB[j*2] ) {
                outarr[counter*2] = inarrA[i*2];
                outarr[counter*2+1] = inarrA[i*2+1];
                if ( j > 0) {
                    y = signal_interpolate_y( inarrB[j*2-2], inarrB[j*2-1], inarrB[j*2], inarrB[j*2+1], inarrA[i*2]);
                    outarr[counter*2+1] += y;
                }
                i++;
            }
            else if ( inarrA[i*2] > inarrB[j*2] ) {
                outarr[counter*2] = inarrB[j*2];
                outarr[counter*2+1] = inarrB[j*2+1];
                if ( i > 0) {
                    y = signal_interpolate_y( inarrA[i*2-2], inarrA[i*2-1], inarrA[i*2], inarrA[i*2+1], inarrB[j*2]);
                    outarr[counter*2+1] += y;
                }
                j++;
            }
            else {
                outarr[counter*2] = inarrA[i*2];
                outarr[counter*2+1] = inarrA[i*2+1] + inarrB[j*2+1];
                i++;
                j++;
            }
        }
        
        // process additional points from array A
        else if ( i < lenA ) {
            outarr[counter*2] = inarrA[i*2];
            outarr[counter*2+1] = inarrA[i*2+1];
            i++;
        }
        
        // process additional points from array B
        else if ( j < lenB ) {
            outarr[counter*2] = inarrB[j*2];
            outarr[counter*2+1] = inarrB[j*2+1];
            j++;
        }
        
        counter++;
    }
    
    // save number of points
    *p_count = counter;
}

static PyObject *_wrap_signal_combine( PyObject *self, PyObject *args )
{
    PyArrayObject *pointsA, *pointsB, *results;
    npy_intp dim[2];
    double *cpointsA, *cpointsB, *cresults, *cbuff;
    int lenA, lenB, counter;
    int i;
    
    if ( !PyArg_ParseTuple(args, "OO", &pointsA, &pointsB) ) {
        return NULL;
    }
    
    // get array sizes
    lenA = (int) PyArray_DIM(pointsA, 0);
    lenB = (int) PyArray_DIM(pointsB, 0);
    
    // convert arrays to c-type arrays
    cpointsA = (double *) pointsA->data;
    cpointsB = (double *) pointsB->data;
    
    // allocate memory for buffer
    // use size of lenA + lenB to ensure sufficient size for new x-raster
    if ( (cbuff = (double*) malloc( 2*(lenA+lenB)*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // combine arrays
    signal_combine( cpointsA, lenA, cpointsB, lenB , cbuff, &counter );
    
    // make numpy array
    dim[0] = (npy_intp) counter;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // copy buffer data to numpy array
    cresults = (double *) results->data;
    for ( i=0; i<counter; i++ ) {
      cresults[i*2] = cbuff[i*2];
      cresults[i*2+1] = cbuff[i*2+1];
    }
    
    free(cbuff);
    
    return PyArray_Return(results);
}


// It unifies x-raster and overlay 2D signals (y = max(yA, yB))

void signal_overlay( double inarrA[], int lenA, double inarrB[], int lenB, double outarr[], int *p_count )
{
    int i, j, counter;
    double y;
    
    // init counters
    i = 0;
    j = 0;
    counter = 0;
    
    while ( i < lenA || j < lenB ) {
        
        // process points within both arrays
        if ( i < lenA && j < lenB ) {
            if ( inarrA[i*2] < inarrB[j*2] ) {
                outarr[counter*2] = inarrA[i*2];
                outarr[counter*2+1] = inarrA[i*2+1];
                if ( j > 0) {
                    y = signal_interpolate_y( inarrB[j*2-2], inarrB[j*2-1], inarrB[j*2], inarrB[j*2+1], inarrA[i*2]);
                    outarr[counter*2+1] = ( inarrA[i*2+1] > y ) ? inarrA[i*2+1] : y;
                }
                i++;
            }
            else if ( inarrA[i*2] > inarrB[j*2] ) {
                outarr[counter*2] = inarrB[j*2];
                outarr[counter*2+1] = inarrB[j*2+1];
                if ( i > 0) {
                    y = signal_interpolate_y( inarrA[i*2-2], inarrA[i*2-1], inarrA[i*2], inarrA[i*2+1], inarrB[j*2]);
                    outarr[counter*2+1] = ( inarrB[j*2+1] > y ) ? inarrB[j*2+1] : y;
                }
                j++;
            }
            else {
                outarr[counter*2] = inarrA[i*2];
                outarr[counter*2+1] = ( inarrA[i*2+1] > inarrB[j*2+1]) ? inarrA[i*2+1] : inarrB[j*2+1];
                i++;
                j++;
            }
        }
        
        // process additional points from array A
        else if ( i < lenA ) {
            outarr[counter*2] = inarrA[i*2];
            outarr[counter*2+1] = inarrA[i*2+1];
            i++;
        }
        
        // process additional points from array B
        else if ( j < lenB ) {
            outarr[counter*2] = inarrB[j*2];
            outarr[counter*2+1] = inarrB[j*2+1];
            j++;
        }
        
        counter++;
    }
    
    // save number of points
    *p_count = counter;
}

static PyObject *_wrap_signal_overlay( PyObject *self, PyObject *args )
{
    PyArrayObject *pointsA, *pointsB, *results;
    npy_intp dim[2];
    double *cpointsA, *cpointsB, *cresults, *cbuff;
    int lenA, lenB, counter;
    int i;
    
    if ( !PyArg_ParseTuple(args, "OO", &pointsA, &pointsB) ) {
        return NULL;
    }
    
    // get array sizes
    lenA = (int) PyArray_DIM(pointsA, 0);
    lenB = (int) PyArray_DIM(pointsB, 0);
    
    // convert arrays to c-type arrays
    cpointsA = (double *) pointsA->data;
    cpointsB = (double *) pointsB->data;
    
    // allocate memory for buffer
    // use size of lenA + lenB to ensure sufficient size for new x-raster
    if ( (cbuff = (double*) malloc( 2*(lenA+lenB)*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // overlay arrays
    signal_overlay( cpointsA, lenA, cpointsB, lenB , cbuff, &counter );
    
    // make numpy array
    dim[0] = (npy_intp) counter;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // copy buffer data to numpy array
    cresults = (double *) results->data;
    for ( i=0; i<counter; i++ ) {
      cresults[i*2] = cbuff[i*2];
      cresults[i*2+1] = cbuff[i*2+1];
    }
    
    free(cbuff);
    
    return PyArray_Return(results);
}


// It unifies x-raster and subtract 2D signals (y = yA - yB)

void signal_subtract( double inarrA[], int lenA, double inarrB[], int lenB, double outarr[], int *p_count )
{
    int i, j, counter;
    double y;
    
    // init counters
    i = 0;
    j = 0;
    counter = 0;
    
    while ( i < lenA || j < lenB ) {
        
        // process points within both arrays
        if ( i < lenA && j < lenB ) {
            if ( inarrA[i*2] < inarrB[j*2] ) {
                outarr[counter*2] = inarrA[i*2];
                outarr[counter*2+1] = inarrA[i*2+1];
                if ( j > 0) {
                    y = signal_interpolate_y( inarrB[j*2-2], inarrB[j*2-1], inarrB[j*2], inarrB[j*2+1], inarrA[i*2]);
                    outarr[counter*2+1] -= y;
                }
                i++;
            }
            else if ( inarrA[i*2] > inarrB[j*2] ) {
                outarr[counter*2] = inarrB[j*2];
                outarr[counter*2+1] = -inarrB[j*2+1];
                if ( i > 0) {
                    y = signal_interpolate_y( inarrA[i*2-2], inarrA[i*2-1], inarrA[i*2], inarrA[i*2+1], inarrB[j*2]);
                    outarr[counter*2+1] += y;
                }
                j++;
            }
            else {
                outarr[counter*2] = inarrA[i*2];
                outarr[counter*2+1] = inarrA[i*2+1] - inarrB[j*2+1];
                i++;
                j++;
            }
        }
        
        // process additional points from array A
        else if ( i < lenA ) {
            outarr[counter*2] = inarrA[i*2];
            outarr[counter*2+1] = inarrA[i*2+1];
            i++;
        }
        
        // process additional points from array B
        else if ( j < lenB ) {
            outarr[counter*2] = inarrB[j*2];
            outarr[counter*2+1] = -inarrB[j*2+1];
            j++;
        }
        
        counter++;
    }
    
    // save number of points
    *p_count = counter;
}

static PyObject *_wrap_signal_subtract( PyObject *self, PyObject *args )
{
    PyArrayObject *pointsA, *pointsB, *results;
    npy_intp dim[2];
    double *cpointsA, *cpointsB, *cresults, *cbuff;
    int lenA, lenB, counter;
    int i;
    
    if ( !PyArg_ParseTuple(args, "OO", &pointsA, &pointsB) ) {
        return NULL;
    }
    
    // get array sizes
    lenA = (int) PyArray_DIM(pointsA, 0);
    lenB = (int) PyArray_DIM(pointsB, 0);
    
    // convert arrays to c-type arrays
    cpointsA = (double *) pointsA->data;
    cpointsB = (double *) pointsB->data;
    
    // allocate memory for buffer
    // use size of lenA + lenB to ensure sufficient size for new x-raster
    if ( (cbuff = (double*) malloc( 2*(lenA+lenB)*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // subtract arrays
    signal_subtract( cpointsA, lenA, cpointsB, lenB , cbuff, &counter );
    
    // make numpy array
    dim[0] = (npy_intp) counter;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // copy buffer data to numpy array
    cresults = (double *) results->data;
    for ( i=0; i<counter; i++ ) {
      cresults[i*2] = cbuff[i*2];
      cresults[i*2+1] = cbuff[i*2+1];
    }
    
    free(cbuff);
    
    return PyArray_Return(results);
}


// Subtract baseline. X-raster remains the same.

void signal_subbase( double inarrA[], int lenA, double inarrB[], int lenB, double outarr[] )
{
    double shift, m, b;
    int i, j;
    
    // copy current data
    for ( i = 0; i < lenA; ++i ) {
        outarr[2*i] = inarrA[2*i];
        outarr[2*i+1] = inarrA[2*i+1];
    }
    
    // single point baseline
    if ( lenB == 1 ) {
        shift = inarrB[1];
        for ( i = 0; i < lenA; ++i ) {
            outarr[2*i+1] -= shift;
        }
    }
    
    // multiple point baseline
    else if ( lenB > 1 ) {
        
        // get first baseline segment
        j = 1;
        m = (inarrB[2*j+1] - inarrB[2*j-2+1]) / (inarrB[2*j] - inarrB[2*j-2]);
        b = inarrB[2*j-2+1] - m * inarrB[2*j-2];
        
        // shift data
        for ( i = 0; i < lenA; ++i ) {
            
            // get new baseline segment
            if ( (inarrA[2*i] > inarrB[2*j]) && (j < lenB-1) ) {
                j++;
                m = (inarrB[2*j+1] - inarrB[2*j-2+1]) / (inarrB[2*j] - inarrB[2*j-2]);
                b = inarrB[2*j-2+1] - m * inarrB[2*j-2];
            }
            
            shift = m * inarrA[2*i] + b;
            outarr[2*i+1] -= shift;
        }
    }
    
    // clip negative y-values
    for ( i = 0; i < lenA; ++i ) {
        outarr[2*i+1] = (outarr[2*i+1] < 0.0) ? 0.0 : outarr[2*i+1];
    }
}

static PyObject *_wrap_signal_subbase( PyObject *self, PyObject *args )
{
    PyArrayObject *pointsA, *pointsB, *results;
    npy_intp dim[2];
    double *cpointsA, *cpointsB, *cresults;
    int lenA, lenB;
    
    if ( !PyArg_ParseTuple(args, "OO", &pointsA, &pointsB) ) {
        return NULL;
    }
    
    // get array sizes
    lenA = (int) PyArray_DIM(pointsA, 0);
    lenB = (int) PyArray_DIM(pointsB, 0);
    
    // convert arrays to c-type arrays
    cpointsA = (double *) pointsA->data;
    cpointsB = (double *) pointsB->data;
    
    // allocate memory for results
    dim[0] = (npy_intp) lenA;
    dim[1] = 2;
    results = (PyArrayObject *) PyArray_SimpleNew(2, dim, PyArray_DOUBLE);
    if ( results == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    cresults = (double *) results->data;
    
    // subtract arrays
    signal_subbase( cpointsA, lenA, cpointsB, lenB , cresults );
    
    return PyArray_Return(results);
}


// Generates composition variants within given atom count and mass limits.

void formula_generate_composition( int elements, int minimum[], int maximum[], double masses[], double loMass, double hiMass, int outarr[], int limit, int *p_count, int pos)
{
    double mass = 0;
    int *current;
    int i;
    
    // calculate current mass
    for ( i = 0; i < elements; ++i) {
        mass += minimum[i]*masses[i];
    }
    
    // recursion end reached
    if ( pos == elements ) {
        
        // check mass tolerance and store current composition
        if ( (mass >= loMass) && (mass <= hiMass) && (*p_count < limit) ) {
            for ( i = 0; i < elements; ++i) {
                outarr[(*p_count)*elements+i] = minimum[i];
            }
            (*p_count)++;
        }
        return;
    }
    
    // duplicate current minima as a new item to allow changes
    if ( (current = (int*) malloc( elements*sizeof(int)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return;
    }
    for ( i = 0; i < elements; ++i) {
        current[i] = minimum[i];
    }
    
    // manin recursion loop
    // increment current position and send data to next recursion
    while (current[pos] <= maximum[pos]) {
        
        // check high mass and stored items limits
        if ( (mass > hiMass) || (*p_count >= limit) ) {
            break;
        }
        
        // do next recursion
        formula_generate_composition( elements, current, maximum, masses, loMass, hiMass, outarr, limit, p_count, pos+1);
        current[pos]++;
        mass += masses[pos];
    }
    
    // free current item
    free(current);
}

static PyObject *_wrap_formula_generate_composition( PyObject *self, PyObject *args )
{
    PyObject *minimum, *maximum, *masses;
    double loMass, hiMass;
    int limit;
    
    PyObject *results, *composition, *item;
    int *cminimum, *cmaximum, *cbuff;
    double *cmasses;
    int elements, counter;
    int i, j;
    
    if ( !PyArg_ParseTuple(args, "OOOddi", &minimum, &maximum, &masses, &loMass, &hiMass, &limit) ) {
        return NULL;
    }
    
    // get number of elements
    elements = (int) PyTuple_Size(minimum);
    
    // allocate memory for c-type arrays of input params
    if ( (cminimum = (int*) malloc( elements*sizeof(int)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    if ( (cmaximum = (int*) malloc( elements*sizeof(int)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    if ( (cmasses = (double*) malloc( elements*sizeof(double)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // duplicate values from input params
    for ( i = 0; i < elements; ++i) {
        cminimum[i] = (int) PyLong_AsLong(PyTuple_GetItem(minimum, i));
        cmaximum[i] = (int) PyLong_AsLong(PyTuple_GetItem(maximum, i));
        cmasses[i] = (double) PyFloat_AsDouble(PyTuple_GetItem(masses, i));
    }
    
    // allocate memory for buffer
    if ( (cbuff = (int*) malloc( limit*elements*sizeof(int)) ) == NULL ) {
        PyErr_Format(PyExc_MemoryError, "Insufficient memory");
        return NULL;
    }
    
    // generate compositions
    counter = 0;
    formula_generate_composition(elements, cminimum, cmaximum, cmasses, loMass, hiMass, cbuff, limit, &counter, 0);
    
    // convert results to python 2D list
    results = PyList_New(0);
    for ( i = 0; i < counter; ++i ) {
        composition = PyList_New(elements);
        for ( j = 0; j < elements; ++j ) {
            item = PyInt_FromLong( cbuff[i*elements+j] );
            PyList_SetItem(composition, j, item);
        }
        PyList_Append(results, composition);
    }
    
    // free memory
    free(cminimum);
    free(cmaximum);
    free(cmasses);
    free(cbuff);
    
    return results;
}



// SET METHODS TABLE
// -----------------

static PyMethodDef calculations_methods[] = {
   {"signal_locate_x", _wrap_signal_locate_x, METH_VARARGS, "signal_locate_x( PyArray, double )"},
   {"signal_locate_max_y", _wrap_signal_locate_max_y, METH_VARARGS, "signal_locate_max_y( PyArray )"},
   {"signal_interpolate_x", _wrap_signal_interpolate_x, METH_VARARGS, "signal_interpolate_x( double, double, double, double, double )"},
   {"signal_interpolate_y", _wrap_signal_interpolate_y, METH_VARARGS, "signal_interpolate_y( double, double, double, double, double )"},
   {"signal_boundaries", _wrap_signal_boundaries, METH_VARARGS, "signal_boundaries( PyArray )"},
   {"signal_maxima", _wrap_signal_maxima, METH_VARARGS, "signal_maxima( PyArray )"},
   {"signal_intensity", _wrap_signal_intensity, METH_VARARGS, "signal_intensity( PyArray, double )"},
   {"signal_centroid", _wrap_signal_centroid, METH_VARARGS, "signal_centroid( PyArray, double, double )"},
   {"signal_width", _wrap_signal_width, METH_VARARGS, "signal_width( PyArray, double, double )"},
   {"signal_area", _wrap_signal_area, METH_VARARGS, "signal_area( PyArray )"},
   {"signal_noise", _wrap_signal_noise, METH_VARARGS, "signal_noise( PyArray )"},
   {"signal_gaussian", _wrap_signal_gaussian, METH_VARARGS, "signal_gaussian( double, double, double, double, int )"},
   {"signal_crop", _wrap_signal_crop, METH_VARARGS, "signal_crop( PyArray, double, double )"},
   {"signal_offset", _wrap_signal_offset, METH_VARARGS, "signal_offset( PyArray, double, double )"},
   {"signal_multiply", _wrap_signal_multiply, METH_VARARGS, "signal_multiply( PyArray, double, double )"},
   {"signal_rescale", _wrap_signal_rescale, METH_VARARGS, "signal_rescale( PyArray, double, double, double, double )"},
   {"signal_normalize", _wrap_signal_normalize, METH_VARARGS, "signal_normalize( PyArray )"},
   {"signal_smooth_ma", _wrap_signal_smooth_ma, METH_VARARGS, "signal_smooth_ma( PyArray, int, int )"},
   {"signal_smooth_ga", _wrap_signal_smooth_ga, METH_VARARGS, "signal_smooth_ga( PyArray, int, int )"},
   {"signal_filter", _wrap_signal_filter, METH_VARARGS, "signal_filter( PyArray, double )"},
   {"signal_combine", _wrap_signal_combine, METH_VARARGS, "signal_combine( PyArray, PyArray )"},
   {"signal_overlay", _wrap_signal_overlay, METH_VARARGS, "signal_overlay( PyArray, PyArray )"},
   {"signal_subtract", _wrap_signal_subtract, METH_VARARGS, "signal_subtract( PyArray, PyArray )"},
   {"signal_subbase", _wrap_signal_subbase, METH_VARARGS, "signal_subbase( PyArray, PyArray )"},
   
   {"formula_generate_composition", _wrap_formula_generate_composition, METH_VARARGS, "formula_generate_composition( PyTupleObject, PyTupleObject, PyTupleObject, double, double, int )"},
   
   {NULL, NULL, 0, NULL}
};

PyMODINIT_FUNC initcalculations(void) {
    Py_InitModule3("calculations", calculations_methods,"");
    import_array();
}
