# -------------------------------------------------------------------------
#     Copyright (C) 2005-2011 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

# load libs
import threading
import numpy
import wx
import copy

# load modules
from ids import *
import mwx
import images
import config
import libs
import mspy


# FLOATING PANEL WITH PROCESSING TOOLS
# ------------------------------------

class panelProcessing(wx.MiniFrame):
    """Data processing tools."""
    
    def __init__(self, parent, tool='peakpicking'):
        wx.MiniFrame.__init__(self, parent, -1, 'Processing', size=(300, -1), style=wx.DEFAULT_FRAME_STYLE & ~ (wx.RESIZE_BORDER | wx.RESIZE_BOX | wx.MAXIMIZE_BOX))
        
        self.parent = parent
        
        self.processing = None
        
        self.currentTool = tool
        self.currentDocument = None
        self.previewData = None
        
        # make gui items
        self.makeGUI()
        wx.EVT_CLOSE(self, self.onClose)
        
        # select default tool
        self.onToolSelected(tool=self.currentTool)
    # ----
    
    
    def makeGUI(self):
        """Make panel gui."""
        
        # make toolbar
        toolbar = self.makeToolbar()
        
        # make panels
        math = self.makeMathPanel()
        crop = self.makeCropPanel()
        baseline = self.makeBaselinePanel()
        smoothing = self.makeSmoothingPanel()
        peakpicking = self.makePeakpickingPanel()
        deisotoping = self.makeDeisotopingPanel()
        deconvolution = self.makeDeconvolutionPanel()
        gauge = self.makeGaugePanel()
        
        # pack element
        self.mainSizer = wx.BoxSizer(wx.VERTICAL)
        self.mainSizer.Add(toolbar, 0, wx.EXPAND, 0)
        self.mainSizer.Add(math, 1, wx.EXPAND, 0)
        self.mainSizer.Add(crop, 1, wx.EXPAND, 0)
        self.mainSizer.Add(baseline, 1, wx.EXPAND, 0)
        self.mainSizer.Add(smoothing, 1, wx.EXPAND, 0)
        self.mainSizer.Add(peakpicking, 1, wx.EXPAND, 0)
        self.mainSizer.Add(deisotoping, 1, wx.EXPAND, 0)
        self.mainSizer.Add(deconvolution, 1, wx.EXPAND, 0)
        self.mainSizer.Add(gauge, 0, wx.EXPAND, 0)
        
        # hide panels
        self.mainSizer.Hide(1)
        self.mainSizer.Hide(2)
        self.mainSizer.Hide(3)
        self.mainSizer.Hide(4)
        self.mainSizer.Hide(5)
        self.mainSizer.Hide(6)
        self.mainSizer.Hide(7)
        self.mainSizer.Hide(8)
        
        # fit layout
        self.mainSizer.Fit(self)
        self.SetSizer(self.mainSizer)
    # ----
    
    
    def makeToolbar(self):
        """Make toolbar."""
        
        # init toolbar
        panel = mwx.bgrPanel(self, -1, images.lib['bgrToolbar'], size=(-1, mwx.TOOLBAR_HEIGHT))
        
        # make tools
        self.math_butt = wx.BitmapButton(panel, ID_processingMath, images.lib['processingMathOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.math_butt.SetToolTip(wx.ToolTip("Math operations"))
        self.math_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.crop_butt = wx.BitmapButton(panel, ID_processingCrop, images.lib['processingCropOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.crop_butt.SetToolTip(wx.ToolTip("Crop data"))
        self.crop_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.baseline_butt = wx.BitmapButton(panel, ID_processingBaseline, images.lib['processingBaselineOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.baseline_butt.SetToolTip(wx.ToolTip("Baseline correction"))
        self.baseline_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.smoothing_butt = wx.BitmapButton(panel, ID_processingSmoothing, images.lib['processingSmoothingOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.smoothing_butt.SetToolTip(wx.ToolTip("Smoothing"))
        self.smoothing_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.peakpicking_butt = wx.BitmapButton(panel, ID_processingPeakpicking, images.lib['processingPeakpickingOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.peakpicking_butt.SetToolTip(wx.ToolTip("Peak picking"))
        self.peakpicking_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.deisotoping_butt = wx.BitmapButton(panel, ID_processingDeisotoping, images.lib['processingDeisotopingOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.deisotoping_butt.SetToolTip(wx.ToolTip("Deisotoping"))
        self.deisotoping_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.deconvolution_butt = wx.BitmapButton(panel, ID_processingDeconvolution, images.lib['processingDeconvolutionOff'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.deconvolution_butt.SetToolTip(wx.ToolTip("Deconvolution"))
        self.deconvolution_butt.Bind(wx.EVT_BUTTON, self.onToolSelected)
        
        self.presets_butt = wx.BitmapButton(panel, -1, images.lib['toolsPresets'], size=(mwx.TOOLBAR_TOOLSIZE), style=wx.BORDER_NONE)
        self.presets_butt.SetToolTip(wx.ToolTip("Processing presets"))
        self.presets_butt.Bind(wx.EVT_BUTTON, self.onPresets)
        
        self.preview_butt = wx.Button(panel, -1, "Preview", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.preview_butt.SetFont(wx.SMALL_FONT)
        self.preview_butt.Bind(wx.EVT_BUTTON, self.onPreview)
        
        self.apply_butt = wx.Button(panel, -1, "Apply", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.apply_butt.SetFont(wx.SMALL_FONT)
        self.apply_butt.Bind(wx.EVT_BUTTON, self.onApply)
        
        self.toolbar = wx.BoxSizer(wx.HORIZONTAL)
        self.toolbar.AddSpacer(mwx.TOOLBAR_LSPACE)
        self.toolbar.Add(self.math_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        self.toolbar.Add(self.crop_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.Add(self.baseline_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.Add(self.smoothing_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.Add(self.peakpicking_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.Add(self.deisotoping_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.Add(self.deconvolution_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT, mwx.BUTTON_SIZE_CORRECTION)
        self.toolbar.AddSpacer(20)
        self.toolbar.Add(self.presets_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        self.toolbar.AddStretchSpacer()
        self.toolbar.AddSpacer(20)
        self.toolbar.Add(self.preview_butt, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 10)
        self.toolbar.Add(self.apply_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        self.toolbar.AddSpacer(mwx.TOOLBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(self.toolbar, 1, wx.EXPAND)
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def makeMathPanel(self):
        """Make controls for math operations."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        mathSpectrumB_label = wx.StaticText(panel, -1, "Spectrum B:")
        self.mathSpectrumB_combo = wx.ComboBox(panel, -1, choices=[], size=(200, mwx.COMBO_HEIGHT), style=wx.CB_READONLY)
        self.updateAvailableDocuments()
        
        mathOperation_label = wx.StaticText(panel, -1, "Operation:")
        self.mathOperationAdd_radio = wx.RadioButton(panel, -1, "A + B   ", style=wx.RB_GROUP)
        self.mathOperationSub_radio = wx.RadioButton(panel, -1, "A - B   ")
        self.mathOperationMul_radio = wx.RadioButton(panel, -1, "A * X")
        self.mathOperationNorm_radio = wx.RadioButton(panel, -1, "Normalize")
        self.mathOperationAdd_radio.SetValue(True)
        
        self.mathOperationAdd_radio.Bind(wx.EVT_RADIOBUTTON, self.onMathChanged)
        self.mathOperationSub_radio.Bind(wx.EVT_RADIOBUTTON, self.onMathChanged)
        self.mathOperationMul_radio.Bind(wx.EVT_RADIOBUTTON, self.onMathChanged)
        self.mathOperationNorm_radio.Bind(wx.EVT_RADIOBUTTON, self.onMathChanged)
        
        mathMul_label = wx.StaticText(panel, -1, "Multiplier X:")
        self.mathMul_value = wx.TextCtrl(panel, -1, '1', size=(70, -1), validator=mwx.validator('floatPos'))
        self.mathMul_value.Bind(wx.EVT_TEXT, self.onMathChanged)
        self.mathMul_value.Disable()
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(mathOperation_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.mathOperationAdd_radio, (0,1))
        grid.Add(self.mathOperationSub_radio, (1,1))
        grid.Add(self.mathOperationMul_radio, (2,1))
        grid.Add(self.mathOperationNorm_radio, (3,1))
        grid.Add(mathSpectrumB_label, (4,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.mathSpectrumB_combo, (4,1), (1,3))
        grid.Add(mathMul_label, (5,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.mathMul_value, (5,1), (1,3))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeCropPanel(self):
        """Make controls for crop."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        cropLowMass_label = wx.StaticText(panel, -1, "Low mass:")
        self.cropLowMass_value = wx.TextCtrl(panel, -1, str(config.processing['crop']['lowMass']), size=(70, -1), validator=mwx.validator('floatPos'))
        cropLowMassUnits_label = wx.StaticText(panel, -1, "m/z")
        
        cropHighMass_label = wx.StaticText(panel, -1, "High mass:")
        self.cropHighMass_value = wx.TextCtrl(panel, -1, str(config.processing['crop']['highMass']), size=(70, -1), validator=mwx.validator('floatPos'))
        cropHighMassUnits_label = wx.StaticText(panel, -1, "m/z")
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(cropLowMass_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.cropLowMass_value, (0,1))
        grid.Add(cropLowMassUnits_label, (0,2), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(cropHighMass_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.cropHighMass_value, (1,1))
        grid.Add(cropHighMassUnits_label, (1,2), flag=wx.ALIGN_CENTER_VERTICAL)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeBaselinePanel(self):
        """Make controls for baseline subtraction."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        baselinePrecision_label = wx.StaticText(panel, -1, "Precision:")
        self.baselinePrecision_slider = wx.Slider(panel, -1, config.processing['baseline']['precision'], 1, 100, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.baselinePrecision_slider.SetTickFreq(10,1)
        self.baselinePrecision_slider.Bind(wx.EVT_SCROLL, self.onBaselineChanged)
        
        baselineOffset_label = wx.StaticText(panel, -1, "Relative offset:")
        self.baselineOffset_slider = wx.Slider(panel, -1, config.processing['baseline']['offset']*100, 0, 100, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.baselineOffset_slider.SetTickFreq(10,1)
        self.baselineOffset_slider.Bind(wx.EVT_SCROLL, self.onBaselineChanged)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(baselinePrecision_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.baselinePrecision_slider, (0,1))
        grid.Add(baselineOffset_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.baselineOffset_slider, (1,1))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeSmoothingPanel(self):
        """Make controls for smoothing."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        smoothingMethod_label = wx.StaticText(panel, -1, "Method:")
        self.smoothingMethod_combo = wx.ComboBox(panel, -1, choices=['Moving Average', 'Savitzky-Golay'], size=(150, mwx.COMBO_HEIGHT), style=wx.CB_READONLY)
        self.smoothingMethod_combo.Select(0)
        if config.processing['smoothing']['method']=='SG':
            self.smoothingMethod_combo.Select(1)
        self.smoothingMethod_combo.Bind(wx.EVT_COMBOBOX, self.onSmoothingChanged)
        
        smoothingWindow_label = wx.StaticText(panel, -1, "Window size:")
        self.smoothingWindow_value = wx.TextCtrl(panel, -1, str(config.processing['smoothing']['windowSize']), size=(90, -1), validator=mwx.validator('floatPos'))
        smoothingWindowUnits_label = wx.StaticText(panel, -1, "m/z")
        self.smoothingWindow_value.Bind(wx.EVT_TEXT, self.onSmoothingChanged)
        
        smoothingCycles_label = wx.StaticText(panel, -1, "Cycles:")
        self.smoothingCycles_slider = wx.Slider(panel, -1, config.processing['smoothing']['cycles'], 1, 5, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.smoothingCycles_slider.SetTickFreq(1,1)
        self.smoothingCycles_slider.Bind(wx.EVT_SCROLL, self.onSmoothingChanged)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(smoothingMethod_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.smoothingMethod_combo, (0,1), (1,2))
        grid.Add(smoothingWindow_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.smoothingWindow_value, (1,1))
        grid.Add(smoothingWindowUnits_label, (1,2), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(smoothingCycles_label, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.smoothingCycles_slider, (2,1), (1,2))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makePeakpickingPanel(self):
        """Make controls for peak picking."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        peakpickingSNThreshold_label = wx.StaticText(panel, -1, "S/N threshold:")
        self.peakpickingSNThreshold_value = mwx.scrollTextCtrl(panel, -1, str(config.processing['peakpicking']['snThreshold']), multiplier=0.1, limits=(1,100), digits=1, size=(70, -1), validator=mwx.validator('floatPos'))
        self.peakpickingSNThreshold_value.Bind(wx.EVT_TEXT, self.onPeakpickingChanged)
        
        peakpickingAbsIntThreshold_label = wx.StaticText(panel, -1, "Abs. intensity threshold:")
        self.peakpickingAbsIntThreshold_value = wx.TextCtrl(panel, -1, str(config.processing['peakpicking']['absIntThreshold']), size=(70, -1), validator=mwx.validator('floatPos'))
        self.peakpickingAbsIntThreshold_value.Bind(wx.EVT_TEXT, self.onPeakpickingChanged)
        
        peakpickingRelIntThreshold_label = wx.StaticText(panel, -1, "Rel. intensity threshold:")
        self.peakpickingRelIntThreshold_value = mwx.scrollTextCtrl(panel, -1, str(config.processing['peakpicking']['relIntThreshold']*100), multiplier=0.1, limits=(0.01,100), digits=3, size=(70, -1), validator=mwx.validator('floatPos'))
        self.peakpickingRelIntThreshold_value.Bind(wx.EVT_TEXT, self.onPeakpickingChanged)
        peakpickingRelIntThresholdUnits_label = wx.StaticText(panel, -1, "%")
        
        peakpickingHeight_label = wx.StaticText(panel, -1, "Picking height:")
        self.peakpickingHeight_slider = wx.Slider(panel, -1, config.processing['peakpicking']['pickingHeight']*100, 1, 100, size=(150, -1), style=mwx.SLIDER_STYLE)
        self.peakpickingHeight_slider.SetTickFreq(10,1)
        self.peakpickingHeight_slider.Bind(wx.EVT_SCROLL, self.onPeakpickingChanged)
        
        peakpickingBaseline_label = wx.StaticText(panel, -1, "Apply baseline:")
        self.peakpickingBaseline_check = wx.CheckBox(panel, -1, " See baseline panel")
        self.peakpickingBaseline_check.SetFont(wx.SMALL_FONT)
        self.peakpickingBaseline_check.SetValue(bool(config.processing['peakpicking']['baseline']))
        self.peakpickingBaseline_check.Bind(wx.EVT_CHECKBOX, self.onPeakpickingChanged)
        
        peakpickingSmoothing_label = wx.StaticText(panel, -1, "Apply smoothing:")
        self.peakpickingSmoothing_check = wx.CheckBox(panel, -1, " See smoothing panel")
        self.peakpickingSmoothing_check.SetFont(wx.SMALL_FONT)
        self.peakpickingSmoothing_check.SetValue(bool(config.processing['peakpicking']['smoothing']))
        
        peakpickingDeisotoping_label = wx.StaticText(panel, -1, "Apply deisotoping:")
        self.peakpickingDeisotoping_check = wx.CheckBox(panel, -1, " See deisotoping panel")
        self.peakpickingDeisotoping_check.SetFont(wx.SMALL_FONT)
        self.peakpickingDeisotoping_check.SetValue(bool(config.processing['peakpicking']['deisotoping']))
        
        peakpickingRemoveShoulders_label = wx.StaticText(panel, -1, "Remove shoulder peaks:")
        peakpickingRemoveShoulders_label.SetToolTip(wx.ToolTip("For FTMS data only."))
        self.peakpickingRemoveShoulders_check = wx.CheckBox(panel, -1, "")
        self.peakpickingRemoveShoulders_check.SetValue(bool(config.processing['peakpicking']['removeShoulders']))
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(peakpickingSNThreshold_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingSNThreshold_value, (0,1))
        grid.Add(peakpickingAbsIntThreshold_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingAbsIntThreshold_value, (1,1))
        grid.Add(peakpickingRelIntThreshold_label, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingRelIntThreshold_value, (2,1))
        grid.Add(peakpickingRelIntThresholdUnits_label, (2,2), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(peakpickingHeight_label, (3,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingHeight_slider, (3,1), (1,2))
        grid.Add(peakpickingBaseline_label, (4,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingBaseline_check, (4,1), (1,2))
        grid.Add(peakpickingSmoothing_label, (5,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingSmoothing_check, (5,1), (1,2))
        grid.Add(peakpickingDeisotoping_label, (6,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingDeisotoping_check, (6,1), (1,2))
        grid.Add(peakpickingRemoveShoulders_label, (7,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.peakpickingRemoveShoulders_check, (7,1))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeDeisotopingPanel(self):
        """Make controls for charge calculations and deisotoping."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        deisotopingMaxCharge_label = wx.StaticText(panel, -1, "Maximum charge:")
        self.deisotopingMaxCharge_value = wx.TextCtrl(panel, -1, str(config.processing['deisotoping']['maxCharge']), size=(70, -1), validator=mwx.validator('int'))
        
        deisotopingMassTolerance_label = wx.StaticText(panel, -1, "Isotope mass tolerance:")
        self.deisotopingMassTolerance_value = wx.TextCtrl(panel, -1, str(config.processing['deisotoping']['massTolerance']), size=(70, -1), validator=mwx.validator('floatPos'))
        deisotopingMassToleranceUnits_label = wx.StaticText(panel, -1, "m/z")
        
        deisotopingIntTolerance_label = wx.StaticText(panel, -1, "Isotope intensity tolerance:")
        self.deisotopingIntTolerance_value = wx.TextCtrl(panel, -1, str(config.processing['deisotoping']['intTolerance']*100), size=(70, -1), validator=mwx.validator('floatPos'))
        deisotopingIntToleranceUnits_label = wx.StaticText(panel, -1, "%")
        
        deisotopingIsotopeShift_label = wx.StaticText(panel, -1, "Isotope mass shift:")
        self.deisotopingIsotopeShift_value = wx.TextCtrl(panel, -1, str(config.processing['deisotoping']['isotopeShift']), size=(70, -1), validator=mwx.validator('float'))
        self.deisotopingIsotopeShift_value.Bind(wx.EVT_TEXT, self.getParams)
        
        deisotopingRemoveIsotopes_label = wx.StaticText(panel, -1, "Remove isotopes:")
        self.deisotopingRemoveIsotopes_check = wx.CheckBox(panel, -1, "")
        self.deisotopingRemoveIsotopes_check.SetValue(bool(config.processing['deisotoping']['removeIsotopes']))
        
        deisotopingRemoveUnknown_label = wx.StaticText(panel, -1, "Remove unknown:")
        self.deisotopingRemoveUnknown_check = wx.CheckBox(panel, -1, "")
        self.deisotopingRemoveUnknown_check.SetValue(bool(config.processing['deisotoping']['removeUnknown']))
        
        deisotopingLabelEnvelopeTool_label = wx.StaticText(panel, -1, "Label envelope tool:")
        self.deisotopingLabelEnvelopeTool_combo = wx.ComboBox(panel, -1, choices=['Monoisotopic Mass', 'Envelope Centroid', 'All Isotopes'], size=(160, mwx.COMBO_HEIGHT), style=wx.CB_READONLY)
        self.deisotopingLabelEnvelopeTool_combo.Select(1)
        choices=['monoisotopic', 'centroid', 'isotopes']
        if config.processing['deisotoping']['labelEnvelope'] in choices:
            self.deisotopingLabelEnvelopeTool_combo.Select(choices.index(config.processing['deisotoping']['labelEnvelope']))
        self.deisotopingLabelEnvelopeTool_combo.Bind(wx.EVT_COMBOBOX, self.getParams)
        
        deisotopingEnvelopeIntensity_label = wx.StaticText(panel, -1, "Envelope intensity:")
        self.deisotopingEnvelopeIntensity_combo = wx.ComboBox(panel, -1, choices=['Envelope Maximum', 'Summed Isotopes', 'Averaged Isotopes'], size=(160, mwx.COMBO_HEIGHT), style=wx.CB_READONLY)
        self.deisotopingEnvelopeIntensity_combo.Select(0)
        choices=['maximum', 'sum', 'average']
        if config.processing['deisotoping']['envelopeIntensity'] in choices:
            self.deisotopingEnvelopeIntensity_combo.Select(choices.index(config.processing['deisotoping']['envelopeIntensity']))
        self.deisotopingEnvelopeIntensity_combo.Bind(wx.EVT_COMBOBOX, self.getParams)
        if config.processing['deisotoping']['labelEnvelope'] == 'isotopes':
            self.deisotopingEnvelopeIntensity_combo.Disable()
        
        deisotopingSetAsMonoisotopic_label = wx.StaticText(panel, -1, "Set labels as monoisotopes:")
        self.deisotopingSetAsMonoisotopic_check = wx.CheckBox(panel, -1, "")
        self.deisotopingSetAsMonoisotopic_check.SetValue(config.processing['deisotoping']['setAsMonoisotopic'])
        self.deisotopingSetAsMonoisotopic_check.Bind(wx.EVT_CHECKBOX, self.getParams)
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(deisotopingMaxCharge_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingMaxCharge_value, (0,1))
        grid.Add(deisotopingMassTolerance_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingMassTolerance_value, (1,1))
        grid.Add(deisotopingMassToleranceUnits_label, (1,2), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(deisotopingIntTolerance_label, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingIntTolerance_value, (2,1))
        grid.Add(deisotopingIntToleranceUnits_label, (2,2), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(deisotopingIsotopeShift_label, (3,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingIsotopeShift_value, (3,1))
        grid.Add(deisotopingRemoveIsotopes_label, (4,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingRemoveIsotopes_check, (4,1))
        grid.Add(deisotopingRemoveUnknown_label, (5,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingRemoveUnknown_check, (5,1))
        grid.Add(wx.StaticLine(panel), (6,0), (1,3), flag=wx.EXPAND|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(deisotopingLabelEnvelopeTool_label, (7,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingLabelEnvelopeTool_combo, (7,1), (1,2))
        grid.Add(deisotopingEnvelopeIntensity_label, (8,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingEnvelopeIntensity_combo, (8,1), (1,2))
        grid.Add(deisotopingSetAsMonoisotopic_label, (9,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deisotopingSetAsMonoisotopic_check, (9,1))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeDeconvolutionPanel(self):
        """Make controls for deconvolution."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        deconvolutionMassType_label = wx.StaticText(panel, -1, "Mass type:")
        self.deconvolutionMassType_combo = wx.ComboBox(panel, -1, choices=['Monoisotopic', 'Average'], size=(150, mwx.COMBO_HEIGHT), style=wx.CB_READONLY)
        self.deconvolutionMassType_combo.Select(config.processing['deconvolution']['massType'])
        
        deconvolutionGroupWindow_label = wx.StaticText(panel, -1, "Grouping window:")
        self.deconvolutionGroupWindow_value = wx.TextCtrl(panel, -1, str(config.processing['deconvolution']['groupWindow']), size=(90, -1), validator=mwx.validator('floatPos'))
        deconvolutionGroupWindowUnits_label = wx.StaticText(panel, -1, "m/z")
        
        deconvolutionGroupPeaks_label = wx.StaticText(panel, -1, "Group peaks:")
        self.deconvolutionGroupPeaks_check = wx.CheckBox(panel, -1, "")
        self.deconvolutionGroupPeaks_check.SetValue(config.processing['deconvolution']['groupPeaks'])
        
        deconvolutionForceGroupWindow_label = wx.StaticText(panel, -1, "Force grouping window:")
        self.deconvolutionForceGroupWindow_check = wx.CheckBox(panel, -1, "")
        self.deconvolutionForceGroupWindow_check.SetValue(config.processing['deconvolution']['forceGroupWindow'])
        
        # pack elements
        grid = wx.GridBagSizer(mwx.GRIDBAG_VSPACE, mwx.GRIDBAG_HSPACE)
        grid.Add(deconvolutionMassType_label, (0,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deconvolutionMassType_combo, (0,1), (1,2))
        grid.Add(deconvolutionGroupWindow_label, (1,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deconvolutionGroupWindow_value, (1,1))
        grid.Add(deconvolutionGroupWindowUnits_label, (1,2), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(deconvolutionGroupPeaks_label, (2,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deconvolutionGroupPeaks_check, (2,1))
        grid.Add(deconvolutionForceGroupWindow_label, (3,0), flag=wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.deconvolutionForceGroupWindow_check, (3,1))
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(grid, 0, wx.ALIGN_CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        # fit layout
        mainSizer.Fit(panel)
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeGaugePanel(self):
        """Make processing gauge."""
        
        panel = wx.Panel(self, -1)
        
        # make elements
        self.gauge = mwx.gauge(panel, -1)
        
        stop_butt = wx.BitmapButton(panel, -1, images.lib['stopper'], style=wx.BORDER_NONE)
        stop_butt.Bind(wx.EVT_BUTTON, self.onStop)
        
        # pack elements
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.Add(self.gauge, 1, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(10)
        sizer.Add(stop_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        
        # fit layout
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(sizer, 1, wx.EXPAND|wx.RIGHT|wx.LEFT|wx.BOTTOM, mwx.GAUGE_SPACE)
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def onClose(self, evt):
        """Close panel."""
        
        # check processing
        if self.processing != None:
            wx.Bell()
            return
        
        self.parent.updateTmpSpectrum(None)
        self.Destroy()
    # ----
    
    
    def onToolSelected(self, evt=None, tool=None):
        """Selected tool."""
        
        # check processing
        if self.processing != None:
            wx.Bell()
            return
        
        # get the tool
        if evt != None:
            tool = 'peakpicking'
            if evt.GetId() == ID_processingMath:
                tool = 'math'
            elif evt.GetId() == ID_processingCrop:
                tool = 'crop'
            elif evt.GetId() == ID_processingBaseline:
                tool = 'baseline'
            elif evt.GetId() == ID_processingSmoothing:
                tool = 'smoothing'
            elif evt.GetId() == ID_processingPeakpicking:
                tool = 'peakpicking'
            elif evt.GetId() == ID_processingDeisotoping:
                tool = 'deisotoping'
            elif evt.GetId() == ID_processingDeconvolution:
                tool = 'deconvolution'
        
        # set current tool
        self.currentTool = tool
        
        # hide panels
        self.mainSizer.Hide(1)
        self.mainSizer.Hide(2)
        self.mainSizer.Hide(3)
        self.mainSizer.Hide(4)
        self.mainSizer.Hide(5)
        self.mainSizer.Hide(6)
        self.mainSizer.Hide(7)
        
        # hide presets button
        self.toolbar.Hide(9)
        
        # set icons off
        self.math_butt.SetBitmapLabel(images.lib['processingMathOff'])
        self.crop_butt.SetBitmapLabel(images.lib['processingCropOff'])
        self.baseline_butt.SetBitmapLabel(images.lib['processingBaselineOff'])
        self.smoothing_butt.SetBitmapLabel(images.lib['processingSmoothingOff'])
        self.peakpicking_butt.SetBitmapLabel(images.lib['processingPeakpickingOff'])
        self.deisotoping_butt.SetBitmapLabel(images.lib['processingDeisotopingOff'])
        self.deconvolution_butt.SetBitmapLabel(images.lib['processingDeconvolutionOff'])
        
        # clear preview
        self.parent.updateTmpSpectrum(None)
        self.previewData = None
        
        # set panel
        if tool == 'math':
            self.SetTitle("Math Operations")
            self.mainSizer.Show(1)
            self.math_butt.SetBitmapLabel(images.lib['processingMathOn'])
            self.preview_butt.Enable(False)
            
        elif tool == 'crop':
            self.SetTitle("Crop")
            self.toolbar.Show(9)
            self.mainSizer.Show(2)
            self.crop_butt.SetBitmapLabel(images.lib['processingCropOn'])
            self.preview_butt.Enable(False)
            
        elif tool == 'baseline':
            self.SetTitle("Baseline Correction")
            self.toolbar.Show(9)
            self.mainSizer.Show(3)
            self.baseline_butt.SetBitmapLabel(images.lib['processingBaselineOn'])
            self.preview_butt.Enable(True)
            self.onBaselineChanged()
            
        elif tool == 'smoothing':
            self.SetTitle("Smoothing")
            self.toolbar.Show(9)
            self.mainSizer.Show(4)
            self.smoothing_butt.SetBitmapLabel(images.lib['processingSmoothingOn'])
            self.preview_butt.Enable(True)
            
        elif tool == 'peakpicking':
            self.SetTitle("Peak Picking")
            self.toolbar.Show(9)
            self.mainSizer.Show(5)
            self.peakpicking_butt.SetBitmapLabel(images.lib['processingPeakpickingOn'])
            self.preview_butt.Enable(False)
            self.onPeakpickingChanged()
            
        elif tool == 'deisotoping':
            self.SetTitle("Deisotoping")
            self.toolbar.Show(9)
            self.mainSizer.Show(6)
            self.deisotoping_butt.SetBitmapLabel(images.lib['processingDeisotopingOn'])
            self.preview_butt.Enable(False)
            
        elif tool == 'deconvolution':
            self.SetTitle("Deconvolution")
            self.toolbar.Show(9)
            self.mainSizer.Show(7)
            self.deconvolution_butt.SetBitmapLabel(images.lib['processingDeconvolutionOn'])
            self.preview_butt.Enable(False)
        
        # fit layout
        mwx.layout(self, self.mainSizer)
    # ----
    
    
    def onProcessing(self, status=True):
        """Show processing gauge."""
        
        self.gauge.SetValue(0)
        
        if status:
            self.MakeModal(True)
            self.mainSizer.Show(8)
        else:
            self.MakeModal(False)
            self.mainSizer.Hide(8)
            self.processing = None
            mspy.start()
        
        # fit layout
        self.Layout()
        self.mainSizer.Fit(self)
        try: wx.Yield()
        except: pass
    # ----
    
    
    def onStop(self, evt):
        """Cancel current processing."""
        
        if self.processing and self.processing.isAlive():
            mspy.stop()
        else:
            wx.Bell()
    # ----
    
    
    def onPresets(self, evt):
        """Show presets."""
        
        # get presets
        presets = libs.presets['processing'].keys()
        presets.sort()
        
        # make menu
        self.presets_popup = wx.Menu()
        for name in presets:
            item = self.presets_popup.Append(-1, name)
            self.presets_popup.Bind(wx.EVT_MENU, self.onPresetsSelected, item)
        
        self.presets_popup.AppendSeparator()
        item = self.presets_popup.Append(-1, "Save as Presets...")
        self.presets_popup.Bind(wx.EVT_MENU, self.onPresetsSave, item)
        
        # popup menu
        self.PopupMenu(self.presets_popup)
        self.presets_popup.Destroy()
    # ----
    
    
    def onPresetsSelected(self, evt):
        """Load selected presets."""
        
        # get presets
        item = self.presets_popup.FindItemById(evt.GetId())
        presets = libs.presets['processing'][item.GetText()]
        
        # set crop
        self.cropLowMass_value.SetValue(str(presets['crop']['lowMass']))
        self.cropHighMass_value.SetValue(str(presets['crop']['highMass']))
        
        # set baseline
        self.baselinePrecision_slider.SetValue(presets['baseline']['precision'])
        self.baselineOffset_slider.SetValue(presets['baseline']['offset']*100)
        
        # set smoothing
        if presets['smoothing']['method']=='MA':
            self.smoothingMethod_combo.Select(0)
        else:
            self.smoothingMethod_combo.Select(1)
        
        self.smoothingWindow_value.SetValue(str(presets['smoothing']['windowSize']))
        self.smoothingCycles_slider.SetValue(presets['smoothing']['cycles'])
        
        # set peakpicking
        self.peakpickingSNThreshold_value.SetValue(str(presets['peakpicking']['snThreshold']))
        self.peakpickingAbsIntThreshold_value.SetValue(str(presets['peakpicking']['absIntThreshold']))
        self.peakpickingRelIntThreshold_value.SetValue(str(presets['peakpicking']['relIntThreshold']*100))
        self.peakpickingHeight_slider.SetValue(presets['peakpicking']['pickingHeight']*100)
        self.peakpickingBaseline_check.SetValue(bool(presets['peakpicking']['baseline']))
        self.peakpickingSmoothing_check.SetValue(bool(presets['peakpicking']['smoothing']))
        self.peakpickingDeisotoping_check.SetValue(bool(presets['peakpicking']['deisotoping']))
        self.peakpickingRemoveShoulders_check.SetValue(bool(presets['peakpicking']['removeShoulders']))
        
        # set deisotoping
        self.deisotopingMaxCharge_value.SetValue(str(presets['deisotoping']['maxCharge']))
        self.deisotopingMassTolerance_value.SetValue(str(presets['deisotoping']['massTolerance']))
        self.deisotopingIntTolerance_value.SetValue(str(presets['deisotoping']['intTolerance']*100))
        self.deisotopingIsotopeShift_value.SetValue(str(presets['deisotoping']['isotopeShift']))
        self.deisotopingRemoveIsotopes_check.SetValue(bool(presets['deisotoping']['removeIsotopes']))
        self.deisotopingRemoveUnknown_check.SetValue(bool(presets['deisotoping']['removeUnknown']))
        self.deisotopingSetAsMonoisotopic_check.SetValue(bool(presets['deisotoping']['setAsMonoisotopic']))
        
        choices=['monoisotopic', 'centroid', 'isotopes']
        if presets['deisotoping']['labelEnvelope'] in choices:
            self.deisotopingLabelEnvelopeTool_combo.Select(choices.index(presets['deisotoping']['labelEnvelope']))
        else:
            self.deisotopingLabelEnvelopeTool_combo.Select(1)
        
        choices=['maximum', 'sum', 'average']
        if presets['deisotoping']['envelopeIntensity'] in choices:
            self.deisotopingEnvelopeIntensity_combo.Select(choices.index(presets['deisotoping']['envelopeIntensity']))
        else:
            self.deisotopingEnvelopeIntensity_combo.Select(0)
        
        # set deconvolution
        self.deconvolutionMassType_combo.Select(presets['deconvolution']['massType'])
        self.deconvolutionGroupWindow_value.SetValue(str(presets['deconvolution']['groupWindow']))
        self.deconvolutionGroupPeaks_check.SetValue(bool(presets['deconvolution']['groupPeaks']))
        self.deconvolutionForceGroupWindow_check.SetValue(bool(presets['deconvolution']['forceGroupWindow']))
        
        # readback all params
        self.getParams()
        
        # update guides
        self.onBaselineChanged()
        self.onPeakpickingChanged()
    # ----
    
    
    def onPresetsSave(self, evt):
        """Save current params as presets."""
        
        # check params
        if not self.getParams():
            wx.Bell()
            return
        
        # get presets name
        dlg = dlgPresetsName(self)
        if dlg.ShowModal() == wx.ID_OK:
            name = dlg.name
            dlg.Destroy()
        else:
            dlg.Destroy()
            return
        
        # save presets
        libs.presets['processing'][name] = copy.deepcopy(config.processing)
        libs.savePresets()
    # ----
    
    
    def onMathChanged(self, evt=None):
        """Disable / enable related items in math panel."""
        
        # check tool
        if self.currentTool != 'math':
            return
        
        # disable / enable items
        if self.mathOperationAdd_radio.GetValue():
            self.mathSpectrumB_combo.Enable()
            self.mathMul_value.Disable()
        elif self.mathOperationSub_radio.GetValue():
            self.mathSpectrumB_combo.Enable()
            self.mathMul_value.Disable()
        elif self.mathOperationMul_radio.GetValue():
            self.mathSpectrumB_combo.Disable()
            self.mathMul_value.Enable()
        elif self.mathOperationNorm_radio.GetValue():
            self.mathSpectrumB_combo.Disable()
            self.mathMul_value.Disable()
    # ----
    
    
    def onBaselineChanged(self, evt=None):
        """Show baseline while params are changing."""
        
        # check tool
        if self.currentTool != 'baseline':
            return
        
        # check current spectrum
        if not self.currentDocument or not len(self.currentDocument.spectrum.points):
            return
        
        # get params
        if not self.getParams():
            return
        
        # get baseline
        baseline = self.currentDocument.spectrum.baseline(\
            window=(1./config.processing['baseline']['precision']), \
            smooth=True, \
            offset=config.processing['baseline']['offset'] \
            )
        
        # make tmp spectrum
        points = []
        for x in baseline:
            points.append([x[0], x[1]])
        
        # send tmp spectrum to plot canvas
        self.parent.updateTmpSpectrum(points)
        self.previewData = None
    # ----
    
    
    def onSmoothingChanged(self, evt=None):
        """Clear smoothing preview while params changed."""
        
        # check tool
        if self.currentTool != 'smoothing':
            return
        
        # clear preview
        self.parent.updateTmpSpectrum(None)
        self.previewData = None
    # ----
    
    
    def onPeakpickingChanged(self, evt=None):
        """Show intensity threshold while params are changing."""
        
        # check tool
        if self.currentTool != 'peakpicking':
            return
        
        # check current spectrum
        if not self.currentDocument or not len(self.currentDocument.spectrum.points):
            return
        
        # get params
        if not self.getParams():
            return
        
        # get threshold line
        points = self.makeThresholdLine()
        
        # send tmp spectrum to plot canvas
        self.parent.updateTmpSpectrum(points)
    # ----
    
    
    def onPreview(self, evt=None):
        """Recalculate data and show preview."""
        
        # check processing
        if self.processing:
            return
        
        # clear preview
        self.parent.updateTmpSpectrum(None)
        self.previewData = None
        
        # check current spectrum
        if not self.currentDocument or not len(self.currentDocument.spectrum.points):
            wx.Bell()
            return
        
        # get all params
        if not self.getParams():
            return
        
        # show processing gauge
        self.onProcessing(True)
        self.preview_butt.Enable(False)
        self.apply_butt.Enable(False)
        
        # show preview
        if self.currentTool == 'baseline':
            self.processing = threading.Thread(target=self.runPreviewBaseline)
        elif self.currentTool == 'smoothing':
            self.processing = threading.Thread(target=self.runPreviewSmoothing)
        else:
            return
        
        # pulse gauge while working
        self.processing.start()
        while self.processing and self.processing.isAlive():
            self.gauge.pulse()
        
        # send tmp spectrum to plot canvas
        self.parent.updateTmpSpectrum(self.previewData)
        
        # hide processing gauge
        self.onProcessing(False)
        self.preview_butt.Enable(True)
        self.apply_butt.Enable(True)
    # ----
    
    
    def onApply(self, evt=None):
        """Apply processing to the data."""
        
        # check processing
        if self.processing:
            return
        
        # clear tmp spectrum
        self.parent.updateTmpSpectrum(None)
        
        # check current spectrum
        if not self.currentDocument:
            wx.Bell()
            return
        
        # check data
        if self.currentTool in ('baseline', 'smoothing', 'peakpicking') and not len(self.currentDocument.spectrum.points):
            wx.Bell()
            return
        elif self.currentTool in ('deisotoping', 'deconvolution') and not len(self.currentDocument.spectrum.peaklist):
            wx.Bell()
            return
        elif self.currentTool in ('deconvolution'):
            if not self.checkChargedPeaks():
                return
        
        # get all params
        if not self.getParams():
            return
        
        # check deisotopping tolerance
        if self.currentTool in ('peakpicking', 'deisotoping'):
            if not self.checkIsotopeMassTolerance():
                return
        
        # show processing gauge
        self.onProcessing(True)
        self.preview_butt.Enable(False)
        self.apply_butt.Enable(False)
        
        # process data
        if self.currentTool == 'math':
            self.processing = threading.Thread(target=self.runApplyMath)
        elif self.currentTool == 'crop':
            self.processing = threading.Thread(target=self.runApplyCrop)
        elif self.currentTool == 'baseline':
            self.processing = threading.Thread(target=self.runApplyBaseline)
        elif self.currentTool == 'smoothing':
            self.processing = threading.Thread(target=self.runApplySmoothing)
        elif self.currentTool == 'peakpicking':
            self.processing = threading.Thread(target=self.runApplyPeakpicking)
        elif self.currentTool == 'deisotoping':
            self.processing = threading.Thread(target=self.runApplyDeisotoping)
        elif self.currentTool == 'deconvolution':
            self.processing = threading.Thread(target=self.runApplyDeconvolution)
        else:
            return
        
        # pulse gauge while working
        self.processing.start()
        while self.processing and self.processing.isAlive():
            self.gauge.pulse()
        
        # update gui
        if self.currentTool != 'deconvolution':
            self.parent.onDocumentChanged(('spectrum', 'notations'))
        
        # hide processing gauge
        self.onProcessing(False)
        if self.currentTool in ('baseline', 'smoothing'):
            self.preview_butt.Enable(True)
        self.apply_butt.Enable(True)
        
        # clear preview
        self.previewData = None
        
        # update peak picking threshold line
        if self.currentTool == 'peakpicking':
            self.onPeakpickingChanged()
    # ----
    
    
    def setData(self, document):
        """Set current document."""
        
        self.currentDocument = document
        self.previewData = None
        self.onToolSelected(tool=self.currentTool)
    # ----
    
    
    def getParams(self, evt=None):
        """Get all params from dialog."""
        
        # try to get values
        try:
            
            # math operations
            if self.mathOperationAdd_radio.GetValue():
                config.processing['math']['operation'] = 'add'
                config.processing['math']['spectrumB'] = self.mathSpectrumB_combo.GetValue()
            elif self.mathOperationSub_radio.GetValue():
                config.processing['math']['operation'] = 'sub'
                config.processing['math']['spectrumB'] = self.mathSpectrumB_combo.GetValue()
            elif self.mathOperationMul_radio.GetValue():
                config.processing['math']['operation'] = 'mul'
                config.processing['math']['multiplier'] = float(self.mathMul_value.GetValue())
            elif self.mathOperationNorm_radio.GetValue():
                config.processing['math']['operation'] = 'norm'
            
            # crop
            config.processing['crop']['lowMass'] = float(self.cropLowMass_value.GetValue())
            config.processing['crop']['highMass'] = float(self.cropHighMass_value.GetValue())
            
            # baseline
            config.processing['baseline']['precision'] = float(self.baselinePrecision_slider.GetValue())
            config.processing['baseline']['offset'] = float(self.baselineOffset_slider.GetValue())/100.
            
            # smoothing
            config.processing['smoothing']['windowSize'] = float(self.smoothingWindow_value.GetValue())
            config.processing['smoothing']['cycles'] = int(self.smoothingCycles_slider.GetValue())
            
            config.processing['smoothing']['method'] = 'MA'
            if self.smoothingMethod_combo.GetValue() == 'Savitzky-Golay':
                config.processing['smoothing']['method'] = 'SG'
            
            # peak picking
            config.processing['peakpicking']['snThreshold'] = float(self.peakpickingSNThreshold_value.GetValue())
            config.processing['peakpicking']['absIntThreshold'] = float(self.peakpickingAbsIntThreshold_value.GetValue())
            config.processing['peakpicking']['relIntThreshold'] = float(self.peakpickingRelIntThreshold_value.GetValue())/100.
            config.processing['peakpicking']['pickingHeight'] = float(self.peakpickingHeight_slider.GetValue())/100.
            config.processing['peakpicking']['baseline'] = bool(self.peakpickingBaseline_check.GetValue())
            config.processing['peakpicking']['smoothing'] = bool(self.peakpickingSmoothing_check.GetValue())
            config.processing['peakpicking']['deisotoping'] = bool(self.peakpickingDeisotoping_check.GetValue())
            config.processing['peakpicking']['removeShoulders'] = bool(self.peakpickingRemoveShoulders_check.GetValue())
            
            # deisotoping
            config.processing['deisotoping']['maxCharge'] = int(self.deisotopingMaxCharge_value.GetValue())
            config.processing['deisotoping']['massTolerance'] = float(self.deisotopingMassTolerance_value.GetValue())
            config.processing['deisotoping']['intTolerance'] = float(self.deisotopingIntTolerance_value.GetValue())/100.
            config.processing['deisotoping']['isotopeShift'] = float(self.deisotopingIsotopeShift_value.GetValue())
            config.processing['deisotoping']['removeIsotopes'] = bool(self.deisotopingRemoveIsotopes_check.GetValue())
            config.processing['deisotoping']['removeUnknown'] = bool(self.deisotopingRemoveUnknown_check.GetValue())
            config.processing['deisotoping']['setAsMonoisotopic'] = bool(self.deisotopingSetAsMonoisotopic_check.GetValue())
            
            labelEnvelope = self.deisotopingLabelEnvelopeTool_combo.GetValue()
            if labelEnvelope == 'Monoisotopic Mass':
                config.processing['deisotoping']['labelEnvelope'] = 'monoisotopic'
            elif labelEnvelope == 'Envelope Centroid':
                config.processing['deisotoping']['labelEnvelope'] = 'centroid'
            elif labelEnvelope == 'All Isotopes':
                config.processing['deisotoping']['labelEnvelope'] = 'isotopes'
            
            envelopeIntensity = self.deisotopingEnvelopeIntensity_combo.GetValue()
            if envelopeIntensity == 'Envelope Maximum':
                config.processing['deisotoping']['envelopeIntensity'] = 'maximum'
            elif envelopeIntensity == 'Summed Isotopes':
                config.processing['deisotoping']['envelopeIntensity'] = 'sum'
            elif envelopeIntensity == 'Averaged Isotopes':
                config.processing['deisotoping']['envelopeIntensity'] = 'average'
            
            if config.processing['deisotoping']['labelEnvelope'] == 'isotopes':
                self.deisotopingEnvelopeIntensity_combo.Disable()
            else:
                self.deisotopingEnvelopeIntensity_combo.Enable()
            
            if config.processing['deisotoping']['maxCharge'] == 0:
                wx.Bell()
                return False
            
            # deconvolution
            config.processing['deconvolution']['massType'] = 0
            if self.deconvolutionMassType_combo.GetValue() == 'Average':
                config.processing['deconvolution']['massType'] = 1
            
            config.processing['deconvolution']['groupWindow'] = float(self.deconvolutionGroupWindow_value.GetValue())
            config.processing['deconvolution']['groupPeaks'] = bool(self.deconvolutionGroupPeaks_check.GetValue())
            config.processing['deconvolution']['forceGroupWindow'] = bool(self.deconvolutionForceGroupWindow_check.GetValue())
            
        # ring error bell if error
        except:
            wx.Bell()
            return False
        
        # apply isotope shift to spectrum canvas
        distance = mspy.ISOTOPE_DISTANCE + config.processing['deisotoping']['isotopeShift']
        self.parent.spectrumPanel.spectrumCanvas.setProperties(isotopeDistance=distance)
        
        return True
    # ----
    
    
    def updateAvailableDocuments(self):
        """Update list of documents in math panel."""
        
        # update available documents
        self.mathSpectrumB_combo.Clear()
        self.mathSpectrumB_combo.Append('None')
        for x, document in enumerate(self.parent.documents):
            title = '#%d: %s' % (x+1, document.title)
            self.mathSpectrumB_combo.Append(title)
        self.mathSpectrumB_combo.Select(0)
    # ----
    
    
    def runPreviewBaseline(self):
        """Preview smoothing results."""
        
        # run task
        try:
            
            # correct baseline
            self.previewData = mspy.correctBaseline(\
                points=self.currentDocument.spectrum.points, \
                window=(1./config.processing['baseline']['precision']), \
                smooth=True, \
                offset=config.processing['baseline']['offset'] \
            )
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runPreviewSmoothing(self):
        """Preview smoothing results."""
        
        # run task
        try:
            
            # smooth data by MA
            if config.processing['smoothing']['method']=='MA':
            
                self.previewData = mspy.smoothMA(\
                    points=self.currentDocument.spectrum.points, \
                    window=config.processing['smoothing']['windowSize'], \
                    cycles=config.processing['smoothing']['cycles']\
                )
            
            # smooth data by SG
            elif config.processing['smoothing']['method']=='SG':
            
                self.previewData = mspy.smoothSG(\
                    points=self.currentDocument.spectrum.points, \
                    window=config.processing['smoothing']['windowSize'], \
                    cycles=config.processing['smoothing']['cycles']\
                )
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runApplyMath(self):
        """Math operations."""
        
        # run task
        try:
            
            # backup document
            self.currentDocument.backup(('spectrum', 'notations'))
            
            # get spectrum B
            if config.processing['math']['operation'] in ('add', 'sub'):
                title = config.processing['math']['spectrumB']
                if title != 'None':
                    index = int(title.split(':')[0][1:])
                    spectrumB = self.parent.documents[index-1].spectrum
                else:
                    wx.Bell()
                    return
            
            # process spectrum
            if config.processing['math']['operation'] == 'add':
                self.currentDocument.spectrum.concatenate(spectrumB)
            
            elif config.processing['math']['operation'] == 'sub':
                self.currentDocument.spectrum.subtract(spectrumB)
            
            elif config.processing['math']['operation'] == 'mul':
                self.currentDocument.spectrum.multiply(config.processing['math']['multiplier'])
            
            elif config.processing['math']['operation'] == 'norm':
                self.currentDocument.spectrum.normalize()
            
            # remove notations
            del self.currentDocument.annotations[:]
            for sequence in self.currentDocument.sequences:
                del sequence.matches[:]
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runApplyCrop(self):
        """Crop data."""
        
        # run task
        try:
            
            # backup document
            self.currentDocument.backup(('spectrum', 'notations'))
            
            # crop spectrum
            self.currentDocument.spectrum.crop(config.processing['crop']['lowMass'], config.processing['crop']['highMass'])
            
            # crop annotations
            indexes = []
            for x, annotation in enumerate(self.currentDocument.annotations):
                if annotation.mz < config.processing['crop']['lowMass'] or annotation.mz > config.processing['crop']['highMass']:
                    indexes.append(x)
            indexes.reverse()
            for x in indexes:
                del self.currentDocument.annotations[x]
            
            # crop matches
            for sequence in self.currentDocument.sequences:
                indexes = []
                for x, match in enumerate(sequence.matches):
                    if match.mz < config.processing['crop']['lowMass'] or match.mz > config.processing['crop']['highMass']:
                        indexes.append(x)
                indexes.reverse()
                for x in indexes:
                    del sequence.matches[x]
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runApplyBaseline(self):
        """Subtract baseline."""
        
        # run task
        try:
            
            # backup document
            self.currentDocument.backup(('spectrum', 'notations'))
            
            # correct baseline
            self.currentDocument.spectrum.correctBaseline(\
                window=(1./config.processing['baseline']['precision']), \
                smooth=True, \
                offset=config.processing['baseline']['offset'] \
            )
            
            # remove notations
            del self.currentDocument.annotations[:]
            for sequence in self.currentDocument.sequences:
                del sequence.matches[:]
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runApplySmoothing(self):
        """Smooth data."""
        
        # run task
        try:
            
            # backup document
            self.currentDocument.backup(('spectrum', 'notations'))
            
            # smooth spectrum
            self.currentDocument.spectrum.smooth(\
                method=config.processing['smoothing']['method'], \
                window=config.processing['smoothing']['windowSize'], \
                cycles=config.processing['smoothing']['cycles'] \
            )
            
            # remove notations
            del self.currentDocument.annotations[:]
            for sequence in self.currentDocument.sequences:
                del sequence.matches[:]
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runApplyPeakpicking(self):
        """Find peaks."""
        
        # run task
        try:
            
            # backup document
            self.currentDocument.backup(('spectrum', 'notations'))
            
            # get baseline window
            baselineWindow = 1.
            if config.processing['peakpicking']['baseline']:
                baselineWindow = 1./config.processing['baseline']['precision']
            
            # get smoothing method
            smoothMethod = None
            if config.processing['peakpicking']['smoothing']:
                smoothMethod = config.processing['smoothing']['method']
            
            # label spectrum
            self.currentDocument.spectrum.labelScan(\
                pickingHeight=config.processing['peakpicking']['pickingHeight'], \
                absThreshold=config.processing['peakpicking']['absIntThreshold'], \
                relThreshold=config.processing['peakpicking']['relIntThreshold'], \
                snThreshold=config.processing['peakpicking']['snThreshold'], \
                baselineWindow=baselineWindow, \
                baselineSmooth=True, \
                baselineOffset=config.processing['baseline']['offset'], \
                smoothMethod=smoothMethod, \
                smoothWindow=config.processing['smoothing']['windowSize'], \
                smoothCycles=config.processing['smoothing']['cycles'] \
                )
            
            # remove shoulder peaks
            if config.processing['peakpicking']['removeShoulders']:
                self.currentDocument.spectrum.removeShoulders(window=2.5, relThreshold=0.05, fwhm=0.01)
            
            # find isotopes and calculate charges
            if config.processing['peakpicking']['deisotoping']:
                self.currentDocument.spectrum.findIsotopes(\
                    maxCharge=config.processing['deisotoping']['maxCharge'], \
                    mzTolerance=config.processing['deisotoping']['massTolerance'], \
                    intTolerance=config.processing['deisotoping']['intTolerance'], \
                    isotopeShift=config.processing['deisotoping']['isotopeShift'] \
                )
                
                # remove isotopes
                if config.processing['deisotoping']['removeIsotopes']:
                    self.currentDocument.spectrum.removeIsotopes()
                
                # remove unknown
                if config.processing['deisotoping']['removeUnknown']:
                    self.currentDocument.spectrum.removeUnknown()
            
            # remove notations
            del self.currentDocument.annotations[:]
            for sequence in self.currentDocument.sequences:
                del sequence.matches[:]
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runApplyDeisotoping(self):
        """Calculate charges for peaks."""
        
        # run task
        try:
            
            # backup document
            self.currentDocument.backup(('spectrum', 'notations'))
            
            # find isotopes and calculate charges
            self.currentDocument.spectrum.findIsotopes(\
                maxCharge=config.processing['deisotoping']['maxCharge'], \
                mzTolerance=config.processing['deisotoping']['massTolerance'], \
                intTolerance=config.processing['deisotoping']['intTolerance'], \
                isotopeShift=config.processing['deisotoping']['isotopeShift'] \
            )
            
            # remove isotopes
            if config.processing['deisotoping']['removeIsotopes']:
                self.currentDocument.spectrum.removeIsotopes()
            
            # remove unknown
            if config.processing['deisotoping']['removeUnknown']:
                self.currentDocument.spectrum.removeUnknown()
            
            # remove notations
            del self.currentDocument.annotations[:]
            for sequence in self.currentDocument.sequences:
                del sequence.matches[:]
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def runApplyDeconvolution(self):
        """Recalculate peak list to singly-charged and make new document."""
        
        # run task
        try:
            
            # copy current document
            docData = copy.deepcopy(self.currentDocument)
            docData.title += ' - Deconvoluted'
            docData.format = 'mSD'
            docData.path = ''
            docData.dirty = True
            docData.backup(None)
            
            # remove notations
            del docData.annotations[:]
            for sequence in docData.sequences:
                del sequence.matches[:]
            
            # deconvolute peaklist
            massType = ['mo', 'av']
            docData.spectrum.deconvolute(massType=massType[config.processing['deconvolution']['massType']])
            
            # group peaks
            if config.processing['deconvolution']['groupPeaks']:
                docData.spectrum.groupPeaks(
                    window=config.processing['deconvolution']['groupWindow'], \
                    forceWindow=config.processing['deconvolution']['forceGroupWindow']
                )
            
            # append new document
            self.parent.onDocumentNew(document=docData, select=False)
            
        # task canceled
        except mspy.ForceQuit:
            return
    # ----
    
    
    def checkIsotopeMassTolerance(self):
        """Check isotope mass tolerance for cuurent max charge."""
        
        # get max tolerance
        z = float(abs(config.processing['deisotoping']['maxCharge']))
        if z == 1:
            return True
        else:
            maxTol = (1.00287+config.processing['deisotoping']['isotopeShift'])*(1/(z-1) - 1/z)
        
        # check max tolerance
        if config.processing['deisotoping']['massTolerance'] >= maxTol:
            wx.Bell()
            message = "For the specified charge your isotope mass tolerance must be\nlower than %.4f. Please correct the parameter in deisotoping panel." % maxTol
            dlg = mwx.dlgMessage(self, title="Isotope mass tolerance is too high.", message=message)
            dlg.ShowModal()
            dlg.Destroy()
            return False
        
        return True
    # ----
    
    
    def checkChargedPeaks(self):
        """Check if at least one peak in current peaklist has charge."""
        
        # check charge
        for peak in self.currentDocument.spectrum.peaklist:
            if peak.charge:
                return True
        
        # show message
        wx.Bell()
        message = "Only the peaks with specified charge can be deconvoluted.\nPlease use deisotoping tool or peak editor to set peak charges\nprior to deconvolution."
        dlg = mwx.dlgMessage(self, title="There are no charged peaks in your peak list.", message=message)
        dlg.ShowModal()
        dlg.Destroy()
        
        return False
    # ----
    
    
    def makeThresholdLine(self):
        """Make peakpicking threshold line."""
        
        # get baseline window
        window = 1.
        if config.processing['peakpicking']['baseline']:
            window = 1./config.processing['baseline']['precision']
        
        # get curent baseline
        baseline = self.currentDocument.spectrum.baseline(\
            window=window, \
            smooth=True, \
            offset=config.processing['baseline']['offset'] \
            )
        
        # get basepeak (approx. only)
        index = self.currentDocument.spectrum.points.argmax(axis=0)[1]
        basepeak = self.currentDocument.spectrum.points[index]
        
        # apply threshold
        points = []
        relThreshold = basepeak[1] * config.processing['peakpicking']['relIntThreshold']
        absThreshold = config.processing['peakpicking']['absIntThreshold']
        for x in range(len(baseline)):
            snThreshold = config.processing['peakpicking']['snThreshold'] * baseline[x][2]
            points.append([baseline[x][0], baseline[x][1] + max(snThreshold, relThreshold, absThreshold)])
        
        return points
    # ----
    
    


class dlgPresetsName(wx.Dialog):
    """Set presets name."""
    
    def __init__(self, parent):
        
        # initialize document frame
        wx.Dialog.__init__(self, parent, -1, "Method Name", style=wx.DEFAULT_DIALOG_STYLE)
        
        self.name = ''
        
        # make GUI
        sizer = self.makeGUI()
        
        # fit layout
        self.Layout()
        sizer.Fit(self)
        self.SetSizer(sizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----
    
    
    def makeGUI(self):
        """Make GUI elements."""
        
        staticSizer = wx.StaticBoxSizer(wx.StaticBox(self, -1, ""), wx.HORIZONTAL)
        
        # make elements
        self.name_value = wx.TextCtrl(self, -1, '', size=(300,-1), style=wx.TE_PROCESS_ENTER)
        self.name_value.Bind(wx.EVT_TEXT_ENTER, self.onOK)
        
        cancel_butt = wx.Button(self, wx.ID_CANCEL, "Cancel")
        ok_butt = wx.Button(self, wx.ID_OK, "Save")
        ok_butt.Bind(wx.EVT_BUTTON, self.onOK)
        
        # pack elements
        staticSizer.Add(self.name_value, 0, wx.ALL, 10)
        
        buttons = wx.BoxSizer(wx.HORIZONTAL)
        buttons.Add(cancel_butt, 0, wx.RIGHT, 15)
        buttons.Add(ok_butt, 0)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(staticSizer, 0, wx.CENTER|wx.TOP|wx.LEFT|wx.RIGHT, mwx.PANEL_SPACE_MAIN)
        mainSizer.Add(buttons, 0, wx.CENTER|wx.ALL, mwx.PANEL_SPACE_MAIN)
        
        return mainSizer
    # ----
    
    
    def onOK(self, evt):
        """Get name."""
        
        self.name = self.name_value.GetValue()
        if self.name:
            self.EndModal(wx.ID_OK)
        else:
            wx.Bell()
    # ----
    
