# -------------------------------------------------------------------------
#     Copyright (C) 2005-2011 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

# load libs
import wx
import wx.grid

# load modules
import config
import mwx
import images
import mspy


# FLOATING PANEL WITH PEAK DIFFERENCES TOOL
# -----------------------------------------

class panelDifferences(wx.MiniFrame):
    """Differences tools."""
    
    def __init__(self, parent):
        wx.MiniFrame.__init__(self, parent, -1, 'Peak Differences', size=(400, 300), style=wx.DEFAULT_FRAME_STYLE)
        
        self.parent = parent
        
        self.currentDocument = None
        self.currentDifferences = None
        self.currentMatches = None
        self.currentLimit = None
        
        self.difference = None
        self.dipeptides = None
        
        # make gui items
        self.makeGUI()
        wx.EVT_CLOSE(self, self.onClose)
    # ----
    
    
    def makeGUI(self):
        """Make panel gui."""
        
        # make toolbar
        toolbar = self.makeToolbar()
        
        # make panel
        differences = self.makeDifferencesPanel()
        
        # pack element
        self.mainSizer = wx.BoxSizer(wx.VERTICAL)
        self.mainSizer.Add(toolbar, 0, wx.EXPAND, 0)
        self.mainSizer.Add(differences, 1, wx.EXPAND, 0)
        
        # fit layout
        self.SetMinSize((100,100))
        self.Layout()
        self.mainSizer.Fit(self)
        self.SetSizer(self.mainSizer)
        self.SetMinSize(self.GetSize())
    # ----
    
    
    def makeToolbar(self):
        """Make toolbar."""
        
        # init toolbar
        panel = mwx.bgrPanel(self, -1, images.lib['bgrToolbarNoBorder'], size=(-1, mwx.TOOLBAR_HEIGHT))
        
        # make match fields
        difference_label = wx.StaticText(panel, -1, "Difference:")
        difference_label.SetFont(wx.SMALL_FONT)
        
        self.difference_value = wx.TextCtrl(panel, -1, '', size=(100, -1), style=wx.TE_PROCESS_ENTER, validator=mwx.validator('floatPos'))
        self.difference_value.Bind(wx.EVT_TEXT_ENTER, self.onSearch)
        
        self.aminoacids_check = wx.CheckBox(panel, -1, "Amino acids")
        self.aminoacids_check.SetFont(wx.SMALL_FONT)
        self.aminoacids_check.SetValue(config.differences['aminoacids'])
        
        self.dipeptides_check = wx.CheckBox(panel, -1, "Dipeptides")
        self.dipeptides_check.SetFont(wx.SMALL_FONT)
        self.dipeptides_check.SetValue(config.differences['dipeptides'])
        
        massType_label = wx.StaticText(panel, -1, "Mass:")
        massType_label.SetFont(wx.SMALL_FONT)
        
        self.massTypeMo_radio = wx.RadioButton(panel, -1, "Mo", style=wx.RB_GROUP)
        self.massTypeMo_radio.SetFont(wx.SMALL_FONT)
        self.massTypeMo_radio.SetValue(True)
        
        self.massTypeAv_radio = wx.RadioButton(panel, -1, "Av")
        self.massTypeAv_radio.SetFont(wx.SMALL_FONT)
        self.massTypeAv_radio.SetValue(config.differences['massType'])
        
        tolerance_label = wx.StaticText(panel, -1, "Tolerance:")
        tolerance_label.SetFont(wx.SMALL_FONT)
        
        self.tolerance_value = wx.TextCtrl(panel, -1, str(config.differences['tolerance']), size=(50, -1), validator=mwx.validator('floatPos'))
        
        toleranceUnits_label = wx.StaticText(panel, -1, "m/z")
        toleranceUnits_label.SetFont(wx.SMALL_FONT)
        
        self.search_butt = wx.Button(panel, -1, "Search", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.search_butt.SetFont(wx.SMALL_FONT)
        self.search_butt.Bind(wx.EVT_BUTTON, self.onSearch)
        
        # pack elements
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.CONTROLBAR_LSPACE)
        sizer.Add(difference_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.difference_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(self.aminoacids_check, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.dipeptides_check, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(massType_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.massTypeMo_radio, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.massTypeAv_radio, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(tolerance_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.tolerance_value, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(toleranceUnits_label, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddStretchSpacer()
        sizer.AddSpacer(20)
        sizer.Add(self.search_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(mwx.CONTROLBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(sizer, 1, wx.EXPAND)
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def makeDifferencesPanel(self):
        """Make differences panel."""
        
        panel = wx.Panel(self, -1)
        
        # make table
        self.makeDifferencesGrid(panel)
        self.makeMatchesList(panel)
        
        # pack main
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(self.differencesGrid, 1, wx.EXPAND)
        mainSizer.AddSpacer(mwx.SASH_SIZE)
        mainSizer.Add(self.matchesList, 0, wx.EXPAND|wx.ALL, mwx.LISTCTRL_NO_SPACE)
        
        # fit layout
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeDifferencesGrid(self, panel):
        """Make differences grid."""
        
        # make table
        self.differencesGrid = wx.grid.Grid(panel, -1, size=(600, 400), style=mwx.GRID_STYLE)
        self.differencesGrid.CreateGrid(0, 0)
        self.differencesGrid.DisableDragColSize()
        self.differencesGrid.DisableDragRowSize()
        self.differencesGrid.SetColLabelSize(19)
        self.differencesGrid.SetDefaultRowSize(19)
        self.differencesGrid.SetLabelFont(wx.SMALL_FONT)
        self.differencesGrid.SetDefaultCellFont(wx.SMALL_FONT)
        self.differencesGrid.SetDefaultCellAlignment(wx.ALIGN_RIGHT, wx.ALIGN_TOP)
        self.differencesGrid.SetDefaultCellBackgroundColour(wx.WHITE)
        
        self.differencesGrid.Bind(wx.grid.EVT_GRID_SELECT_CELL, self.onCellSelected)
    # ----
    
    
    def makeMatchesList(self, panel):
        """Make matches list."""
        
        # init peaklist
        self.matchesList = mwx.sortListCtrl(panel, -1, size=(171, 400), style=mwx.LISTCTRL_STYLE_MULTI)
        self.matchesList.SetFont(wx.SMALL_FONT)
        self.matchesList.setAltColour(mwx.LISTCTRL_ALTCOLOUR)
        
        # make columns
        self.matchesList.InsertColumn(0, "match", wx.LIST_FORMAT_LEFT)
        self.matchesList.InsertColumn(1, "error", wx.LIST_FORMAT_RIGHT)
        
        # set column widths
        for col, width in enumerate((75,75)):
            self.matchesList.SetColumnWidth(col, width)
    # ----
    
    
    def onClose(self, evt):
        """Destroy this frame."""
        self.Destroy()
    # ----
    
    
    def onSearch(self, evt):
        """Generate differences and search for specified mass(es)."""
        
        self.currentMatches = None
        
        # check document
        if not self.currentDocument:
            wx.Bell()
            return
        
        # get params
        if not self.getParams():
            wx.Bell()
            self.updateDifferencesGrid()
            self.updateMatchesList()
            return
        
        # calculate differences
        if not self.calcDifferences():
            wx.Bell()
            self.updateDifferencesGrid()
            self.updateMatchesList()
            return
        
        # search
        self.searchAll()
        
        # update gui
        self.updateDifferencesGrid()
        self.updateMatchesList()
    # ----
    
    
    def onCellSelected(self, evt):
        """Grid cell selected."""
        
        evt.Skip()
        
        # get cell
        col = evt.GetCol()
        row = evt.GetRow()
        
        # highlight selected cell
        self.differencesGrid.SelectBlock(row, col, row, col)
        
        # get peaklist
        peaklist = self.currentDocument.spectrum.peaklist
        if not peaklist:
            wx.Bell()
            return
        
        # get peaks and diff
        mz1 = peaklist[col].mz
        mz2 = peaklist[row].mz
        diff = abs(mz1-mz2)
        
        # highlight masses in plot
        self.parent.updateMassPoints([mz1,mz2])
        
        # search diff
        self.searchSelected(diff)
        self.updateMatchesList()
    # ----
    
    
    def setData(self, document):
        """Set data."""
        
        # set new document
        self.currentDocument = document
        self.currentDifferences = None
        self.currentMatches = None
        self.currentLimit = None
        
        # update gui
        self.updateDifferencesGrid()
        self.updateMatchesList()
    # ----
    
    
    def getParams(self):
        """Get all params from dialog."""
        
        # try to get values
        try:
            
            if self.difference_value.GetValue():
                self.difference = float(self.difference_value.GetValue())
            else:
                self.difference = None
            
            config.differences['aminoacids'] = int(self.aminoacids_check.GetValue())
            config.differences['dipeptides'] = int(self.dipeptides_check.GetValue())
            config.differences['tolerance'] = float(self.tolerance_value.GetValue())
            config.differences['massType'] = int(self.massTypeAv_radio.GetValue())
            
            return True
            
        except:
            wx.Bell()
            return False
    # ----
    
    
    def updateDifferencesGrid(self):
        """Update grid values."""
        
        # erase grid
        if self.differencesGrid.GetNumberRows():
            self.differencesGrid.DeleteCols(0, self.differencesGrid.GetNumberCols())
            self.differencesGrid.DeleteRows(0, self.differencesGrid.GetNumberRows())
        
        # check differences
        if not self.currentDifferences:
            return
        
        # get peaklist
        peaklist = self.currentDocument.spectrum.peaklist
        peaks = len(peaklist)
        
        # create new cells
        self.differencesGrid.AppendCols(peaks)
        self.differencesGrid.AppendRows(peaks)
        
        # create labels
        mzFormat = '%0.' + `config.main['mzDigits']` + 'f'
        cellAttr = wx.grid.GridCellAttr()
        cellAttr.SetReadOnly(True)
        for x in range(peaks):
            mass = mzFormat % peaklist[x].mz
            self.differencesGrid.SetColLabelValue(x, mass)
            self.differencesGrid.SetRowLabelValue(x, mass)
            self.differencesGrid.SetColAttr(x, cellAttr)
        
        # paste data
        mzFormat = '%0.' + `config.main['mzDigits']` + 'f'
        for x in range(peaks):
            for y in range(peaks):
                
                # get difference indexes
                if y==x:
                    self.differencesGrid.SetCellValue(x, y, '---')
                    continue
                elif y<x:
                    i=x
                    j=y
                elif y>x:
                    i=y
                    j=x
                
                # check limit
                if self.currentDifferences[i][j][0] > self.currentLimit:
                    continue
                
                # set value
                diff = mzFormat % self.currentDifferences[i][j][0]
                self.differencesGrid.SetCellValue(x, y, diff)
                
                # highlight matches
                if self.currentDifferences[i][j][1] == 'value':
                    self.differencesGrid.SetCellBackgroundColour(x, y, (100,255,100))
                elif self.currentDifferences[i][j][1] == 'amino':
                    self.differencesGrid.SetCellBackgroundColour(x, y, (0,200,255))
                elif self.currentDifferences[i][j][1] == 'dipep':
                    self.differencesGrid.SetCellBackgroundColour(x, y, (100,255,255))
    # ----
    
    
    def updateMatchesList(self):
        """Update list of matches."""
        
        # clear previous data and set new
        self.matchesList.DeleteAllItems()
        self.matchesList.setDataMap(self.currentMatches)
        
        # check data
        if not self.currentMatches:
            return
        
        # add new data
        mzFormat = '%0.' + `config.main['mzDigits']` + 'f'
        for row, item in enumerate(self.currentMatches):
            
            # format data
            error = mzFormat % (item[1])
            
            # add data
            self.matchesList.InsertStringItem(row, item[0])
            self.matchesList.SetStringItem(row, 1, error)
            self.matchesList.SetItemData(row, row)
        
        # sort data
        self.matchesList.sort()
        
        # scroll top
        self.matchesList.EnsureVisible(0)
    # ----
    
    
    def searchAll(self):
        """Search differences for specified value, aminoacids or dipeptides."""
        
        # get maxima
        aaMin = 0
        aaMax = 0
        dipMin = 0
        dipMax = 0
        if config.differences['aminoacids'] or config.differences['dipeptides']:
            masses = []
            for aa in mspy.aminoacids:
                masses.append(mspy.aminoacids[aa].mass[config.differences['massType']])
            aaMin = min(masses) - config.differences['tolerance']
            aaMax = max(masses) + config.differences['tolerance']
            dipMin = min(masses)*2 - config.differences['tolerance']
            dipMax = max(masses)*2 + config.differences['tolerance']
        
        # get current max
        if self.difference:
            self.currentLimit = max(aaMin, aaMax, dipMin, dipMax, self.difference+config.differences['tolerance'])
        else:
            self.currentLimit = max(aaMin, aaMax, dipMin, dipMax)
        
        # calculate dipeptides
        if not self.dipeptides and config.differences['dipeptides']:
            self.calcDipeptides()
        
        # search
        for x in range(len(self.currentDifferences)):
            for y in range(len(self.currentDifferences[x])):
                
                diff = self.currentDifferences[x][y][0]
                diffMin = diff - config.differences['tolerance']
                diffMax = diff + config.differences['tolerance']
                
                self.currentDifferences[x][y][1] = False
                matched = False
                
                # search for value
                if self.difference and diffMin <= self.difference <= diffMax:
                    self.currentDifferences[x][y][1] = 'value'
                    matched = True
                
                # search for aminoacids
                if not matched and config.differences['aminoacids'] and aaMin <= diff <= aaMax:
                    for aa in mspy.aminoacids:
                        if diffMin <= mspy.aminoacids[aa].mass[config.differences['massType']] <= diffMax:
                            self.currentDifferences[x][y][1] = 'amino'
                            matched = True
                
                # search for dipeptides
                if not matched and config.differences['dipeptides'] and dipMin <= diff <= dipMax:
                    for dip in self.dipeptides:
                        if diffMin <= self.dipeptides[dip][config.differences['massType']] <= diffMax:
                            self.currentDifferences[x][y][1] = 'dipep'
                            break
    # ----
    
    
    def searchSelected(self, diff):
        """Search difference for specified value, aminoacids or dipeptides."""
        
        self.currentMatches = []
        
        # search for value
        if self.difference:
            error = diff - self.difference
            if abs(error) <= config.differences['tolerance']:
                self.currentMatches.append([str(self.difference), error])
        
        # search for aminoacids
        if config.differences['aminoacids']:
            for aa in mspy.aminoacids:
                error = diff - mspy.aminoacids[aa].mass[config.differences['massType']]
                if abs(error) <= config.differences['tolerance']:
                    self.currentMatches.append([aa, error])
        
        # search for dipeptides
        if config.differences['dipeptides']:
            for dip in self.dipeptides:
                error = diff - self.dipeptides[dip][config.differences['massType']]
                if abs(error) <= config.differences['tolerance']:
                    self.currentMatches.append([dip, error])
    # ----
    
    
    def calcDifferences(self):
        """Calculate differences for current peaklist."""
        
        # get peaklist
        peaklist = self.currentDocument.spectrum.peaklist
        if not peaklist:
            return False
        
        # calc differences
        self.currentDifferences = []
        for x in range(len(peaklist)):
            buff = []
            for y in range(x+1):
                buff.append([peaklist[x].mz-peaklist[y].mz, False])
            self.currentDifferences.append(buff)
        
        return True
    # ----
    
    
    def calcDipeptides(self):
        """Calculate dipeptides masses."""
        
        self.dipeptides = {}
        symbols = mspy.aminoacids.keys()
        for x in range(len(symbols)):
            for y in range(x, len(symbols)):
                
                aX = symbols[x]
                aY = symbols[y]
                
                massX = mspy.aminoacids[aX].mass
                massY = mspy.aminoacids[aY].mass
                mass = (massX[0] + massY[0], massX[1] + massY[1])
                
                if aX != aY:
                    label = '%s%s/%s%s' % (aX,aY,aY,aX)
                else:
                    label = aX+aY
                
                self.dipeptides[label] = mass
    # ----
    

