/*
 * Copyright © 2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *   Kevin DuBois <kevin.dubois@canonical.com>
 */

#include "mir/graphics/android/mir_native_buffer.h"
#include "android_alloc_adaptor.h"
#include "android_format_conversion-inl.h"

#include <boost/throw_exception.hpp>
#include <stdexcept>

namespace mg=mir::graphics;
namespace mga=mir::graphics::android;
namespace geom=mir::geometry;

namespace
{
struct AndroidBufferHandleDeleter
{
    AndroidBufferHandleDeleter(std::shared_ptr<alloc_device_t> const& alloc_dev)
        : alloc_device(alloc_dev)
    {}

    void operator()(native_handle_t const* t)
    {
        alloc_device->free(alloc_device.get(), t);
    }
private:
    std::shared_ptr<alloc_device_t> const alloc_device;
};
}

mga::AndroidAllocAdaptor::AndroidAllocAdaptor(const std::shared_ptr<struct alloc_device_t>& alloc_device)
    : alloc_dev(alloc_device)
{
}

std::shared_ptr<ANativeWindowBuffer> mga::AndroidAllocAdaptor::alloc_buffer(
    geometry::Size size, geometry::PixelFormat pf, BufferUsage usage)
{
    buffer_handle_t buf_handle = NULL;
    auto stride = 0;
    auto format = mga::to_android_format(pf);
    auto width = static_cast<int>(size.width.as_uint32_t());
    auto height = static_cast<int>(size.height.as_uint32_t());
    auto usage_flag = convert_to_android_usage(usage);
    auto ret = alloc_dev->alloc(alloc_dev.get(), width, height,
                           format, usage_flag, &buf_handle, &stride);

    if (( ret ) || (buf_handle == NULL) || (stride == 0))
    {
        BOOST_THROW_EXCEPTION(std::runtime_error("buffer allocation failed\n"));
    }

    AndroidBufferHandleDeleter del1(alloc_dev);
    std::shared_ptr<native_handle_t> handle(buf_handle, del1);

    auto tmp = new mga::MirNativeBuffer(handle);
    std::shared_ptr<mga::MirNativeBuffer> buffer(tmp, [](MirNativeBuffer* buffer)
        {
            buffer->mir_dereference();
        });

    buffer->width = width;
    buffer->height = height;
    buffer->stride = stride;
    buffer->handle = buf_handle;
    buffer->format = format;
    buffer->usage = usage_flag;

    return buffer;
}

int mga::AndroidAllocAdaptor::convert_to_android_usage(BufferUsage usage)
{
    switch (usage)
    {
    case mga::BufferUsage::use_hardware:
        return (GRALLOC_USAGE_HW_TEXTURE | GRALLOC_USAGE_HW_RENDER);
    case mga::BufferUsage::use_framebuffer_gles:
        return (GRALLOC_USAGE_HW_RENDER | GRALLOC_USAGE_HW_COMPOSER | GRALLOC_USAGE_HW_FB);
    case mga::BufferUsage::use_software:
        return (GRALLOC_USAGE_SW_WRITE_OFTEN | GRALLOC_USAGE_SW_READ_OFTEN);
    default:
        return -1;
    }
}
