; $Id: formula.scm,v 1.79 2008/01/28 09:20:11 logik Exp $
; 7. Formulas and comprehension terms
; ===================================

; 7-1. Constructors and accessors
; ===============================

; A prime formula can have the form (atom r) with a term r of type
; boole, or else the form (predicate a r1 ... rn) with a predicate
; variable or constant a and terms r1 ... rn.  Formulas are built from
; prime formulas by

; - implication (imp formula1 formula2)
; - conjunction (and formula1 formula2)
; - tensor (tensor formula1 formula2)
; - all quantification  (all x formula)
; - existential quantification  (ex x formula)
; - all quantification without computational content (allnc x formula)
; - existential quantification without computational content (exnc x formula)
; - classical existential quantification (arithmetical version) 
;   (exca (x1 ... xn) formula)
; - classical existential quantification (logical version) 
;   (excl (x1 ... xn) formula)

; Here we allow that the quantified variable is formed without ^, i.e.
; ranges over total objects only.

(define (make-atomic-formula term) (list 'atom term))

(define atom-form-to-kernel cadr)

(define (atom-form? x)
  (and (list? x) (= 2 (length x)) (eq? 'atom (car x))))

(define (make-predicate-formula predicate . terms)
  (let* ((arity (predicate-to-arity predicate))
	 (types1 (arity-to-types arity))
	 (types2 (map term-to-type terms)))
    (if (not (= (length types1) (length types2)))
	(apply myerror
	       (append (list "make-predicate-formula" "types of arity:")
		       types1
		       (list "should be of the same length as types of terms:")
		       terms)))
    (if (equal? types1 types2)
	(cons 'predicate (cons predicate terms))
	(let ((coerce-ops (map types-to-coercion types2 types1)))
	  (cons 'predicate (cons predicate (map (lambda (x y) (x y))
						coerce-ops terms)))))))

(define predicate-form-to-predicate cadr)
(define predicate-form-to-args cddr)

(define (predicate-form? x)
  (and (pair? x) (eq? 'predicate (car x))))

(define (prime-form? x)
  (or (atom-form? x) (predicate-form? x)))

(define (make-=-term term1 term2)
  (let* ((type1 (term-to-type term1))
         (type2 (term-to-type term2))
         (=-term
          (if (equal? type1 type2)
	      (make-term-in-const-form
	       (finalg-to-=-const type1)) ;includes check for finalg
	      (myerror "make-=-term" "equal types expected" type1 type2))))
    (mk-term-in-app-form =-term term1 term2)))

(define (make-= term1 term2)
  (make-atomic-formula (make-=-term term1 term2)))

(define (make-eq term1 term2)
  (let* ((type1 (term-to-type term1))
         (type2 (term-to-type term2))
	 (tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (equal-predconst
	  (if (equal? type1 type2)
	      (make-predconst (make-arity tvar tvar)
			      (make-subst tvar type1)
			      -1 "Equal")
	      (myerror "make-eq" "equal types expected" type1 type2))))
    (make-predicate-formula equal-predconst term1 term2)))

(define (make-e term)
  (let* ((type (term-to-type term))
         (e-term (make-term-in-const-form
		  (finalg-to-e-const type)))) ;includes check for finalg
    (make-atomic-formula (mk-term-in-app-form e-term term))))

(define (make-total term)
  (let* ((type (term-to-type term))
	 (tvar (make-tvar -1 DEFAULT-TVAR-NAME))
         (total-predconst (make-predconst (make-arity tvar)
					  (make-subst tvar type)
					  -1 "Total")))
    (make-predicate-formula total-predconst term)))

(define (make-se term)
  (let* ((type (term-to-type term))
         (se-term (make-term-in-const-form
                   (sfinalg-to-se-const type)))) ;includes check for sfinalg
    (make-atomic-formula (mk-term-in-app-form se-term term))))

(define (make-stotal term)
  (let* ((type (term-to-type term))
	 (tvar (make-tvar -1 DEFAULT-TVAR-NAME))
         (stotal-predconst (make-predconst (make-arity tvar)
                                           (make-subst tvar type)
                                           -1 "STotal")))
    (make-predicate-formula stotal-predconst term)))

(define (make-stotal-or-se term)
  (let ((type (term-to-type term)))
    (if
     (sfinalg? type)
     (make-atomic-formula
      (mk-term-in-app-form
       (make-term-in-const-form
	(make-const (se-at type)
		    "SE" 'fixed-rules
		    (make-arrow type (make-alg "boole")) empty-subst
		    1 'prefix-op))
       term))
     (make-stotal term))))

(define (make-stotal-or-se-or-e term)
  (let ((type (term-to-type term)))
    (cond
     ((finalg? type)
      (make-atomic-formula
       (mk-term-in-app-form
	(make-term-in-const-form
	 (make-const (e-at type)
		     "E" 'fixed-rules
		     (make-arrow type (make-alg "boole")) empty-subst
		     1 'prefix-op))
	term)))
     ((sfinalg? type)
      (make-atomic-formula
       (mk-term-in-app-form
	(make-term-in-const-form
	 (make-const (se-at type)
		     "SE" 'fixed-rules
		     (make-arrow type (make-alg "boole")) empty-subst
		     1 'prefix-op))
	term)))
     (else (make-stotal term)))))

; Constructor and accessors for formulas:

(define (make-imp premise conclusion) (list 'imp premise conclusion))
(define imp-form-to-premise cadr)
(define imp-form-to-conclusion caddr)
(define (imp-form? x) (and (list? x) (= 3 (length x)) (eq? 'imp (car x))))

(define (make-and formula1 formula2) (list 'and formula1 formula2))
(define and-form-to-left cadr)
(define and-form-to-right caddr)
(define (and-form? x) (and (list? x) (= 3 (length x)) (eq? 'and (car x))))

(define (make-tensor formula1 formula2) (list 'tensor formula1 formula2))
(define tensor-form-to-left cadr)
(define tensor-form-to-right caddr)
(define (tensor-form? x)
  (and (list? x) (= 3 (length x)) (eq? 'tensor (car x))))

(define (make-all var kernel) (list 'all var kernel))
(define all-form-to-var cadr)
(define all-form-to-kernel caddr)
(define (all-form? x)
  (and (list? x) (= 3 (length x)) (eq? 'all (car x)) (var? (cadr x))))

(define (make-ex var kernel) (list 'ex var kernel))
(define ex-form-to-var cadr)
(define ex-form-to-kernel caddr)
(define (ex-form? x)
  (and (list? x) (= 3 (length x)) (eq? 'ex (car x)) (var? (cadr x))))

(define (make-allnc var kernel) (list 'allnc var kernel))
(define allnc-form-to-var cadr)
(define allnc-form-to-kernel caddr)
(define (allnc-form? x)
  (and (list? x) (= 3 (length x)) (eq? 'allnc (car x)) (var? (cadr x))))

(define (make-exnc var kernel) (list 'exnc var kernel))
(define exnc-form-to-var cadr)
(define exnc-form-to-kernel caddr)
(define (exnc-form? x)
  (and (list? x) (= 3 (length x)) (eq? 'exnc (car x)) (var? (cadr x))))

(define (make-exca vars kernel) (list 'exca vars kernel))
(define exca-form-to-vars cadr)
(define exca-form-to-kernel caddr)
(define (exca-form? x)
  (and (list? x) (= 3 (length x)) (eq? 'exca (car x))
       (list? (cadr x)) (pair? (cadr x)) (apply and-op (map var? (cadr x)))))

(define (make-excl vars kernel) (list 'excl vars kernel))
(define excl-form-to-vars cadr)
(define excl-form-to-kernel caddr)
(define (excl-form? x)
  (and (list? x) (= 3 (length x)) (eq? 'excl (car x))
       (list? (cadr x)) (pair? (cadr x)) (apply and-op (map var? (cadr x)))))

(define (formula-form? x)
  (and (pair? x)
       (memq (tag x) '(atom 
		       predicate
		       imp
		       and
		       tensor
		       all
		       ex
		       allnc
		       exnc
		       exca
		       excl))))

; For convenience we add

(define (mk-imp x . rest)
  (if (null? rest)
      x
      (make-imp x (apply mk-imp rest))))

; imp-form-to-premises computes the first (car x) premises of a formula.

(define (imp-form-to-premises formula . x)
  (cond
   ((null? x)
    (if (imp-form? formula)
	(cons (imp-form-to-premise formula)
	      (imp-form-to-premises (imp-form-to-conclusion formula)))
	'()))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (imp-form-to-conclusion rho))
	   (i 0 (+ 1 i))
	   (res '() (cons (imp-form-to-premise rho) res)))
	  ((or (= n i) (not (imp-form? rho)))
	   (if (= n i)
	       (reverse res)
	       (myerror "imp-form-to-premises:" n "premises expected in"
			formula))))))
   (else (myerror "imp-form-to-premises" "non-negative integer expected"
		  (car x)))))

; imp-form-to-final-conclusion computes the final conclusion (conclusion
; after removing the first (car x) premises) of a formula. 

(define (imp-form-to-final-conclusion formula . x)
  (cond
   ((null? x)
    (if (imp-form? formula)
	(imp-form-to-final-conclusion (imp-form-to-conclusion formula))
	formula))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (imp-form-to-conclusion rho))
	   (i 0 (+ 1 i))
	   (res formula (imp-form-to-conclusion res)))
	  ((or (= n i) (not (imp-form? rho)))
	   (if (= n i)
	       res
	       (myerror "imp-form-to-final-conclusion:"
			n "premises expected in" formula))))))
   (else (myerror "imp-form-to-final-conclusion"
		  "non-negative integer expected"
		  (car x)))))

(define (imp-form-to-premises-and-final-conclusion formula)
  (if (imp-form? formula)
      (let* ((rec-result (imp-form-to-premises-and-final-conclusion
			  (imp-form-to-conclusion formula)))
	     (formula-list (car rec-result))
	     (final-conclusion (cadr rec-result)))
	(list (cons (imp-form-to-premise formula) formula-list)
	      final-conclusion))
      (list '() formula)))

(define (imp-all-allnc-form-to-final-conclusion formula)
  (cond
   ((imp-form? formula)
    (imp-all-allnc-form-to-final-conclusion (imp-form-to-conclusion formula)))
   ((all-form? formula)
    (imp-all-allnc-form-to-final-conclusion (all-form-to-kernel formula)))
   ((allnc-form? formula)
    (imp-all-allnc-form-to-final-conclusion (allnc-form-to-kernel formula)))
   (else formula)))

(define (mk-and x . rest)
  (if (null? rest)
      x
      (apply mk-and (cons (make-and x (car rest)) (cdr rest)))))

(define (mk-tensor x . rest)
  (if (null? rest)
      x
      (make-tensor x (apply mk-tensor rest))))

(define (tensor-form-to-parts formula)
  (if (tensor-form? formula)
      (cons (tensor-form-to-left formula)
	    (tensor-form-to-parts (tensor-form-to-right formula)))
      (list formula)))

(define (mk-neg . x) (apply mk-imp (append x (list falsity))))
(define (mk-neg-log . x) (apply mk-imp (append x (list falsity-log))))

; (mk-all var1 ... formula) results from formula by first generalizing
; var1, then var2 etc.

(define (mk-all x . rest)
  (if (null? rest)
      x
      (make-all x (apply mk-all rest))))

; all-form-to-vars computes the first (car x) vars of a formula.

(define (all-form-to-vars formula . x)
  (cond
   ((null? x)
    (if (all-form? formula)
	(cons (all-form-to-var formula)
	      (all-form-to-vars (all-form-to-kernel formula)))
	'()))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (all-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res '() (cons (all-form-to-var rho) res)))
	  ((or (= n i) (not (all-form? rho)))
	   (if (= n i)
	       (reverse res)
	       (myerror "all-form-to-vars:" n "vars expected in"
			formula))))))
   (else (myerror "all-form-to-vars" "non-negative integer expected"
		  (car x)))))

; all-form-to-final-kernel computes the final kernel (kernel
; after removing the first (car x) vars) of a formula. 

(define (all-form-to-final-kernel formula . x)
  (cond
   ((null? x)
    (if (all-form? formula)
	(all-form-to-final-kernel (all-form-to-kernel formula))
	formula))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (all-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res formula (all-form-to-kernel res)))
	  ((or (= n i) (not (all-form? rho)))
	   (if (= n i)
	       res
	       (myerror "all-form-to-final-kernel:"
			n "vars expected in"
			formula))))))
   (else (myerror "all-form-to-final-kernel" "non-negative integer expected"
		  (car x)))))

(define (all-form-to-vars-and-final-kernel formula)
  (if (all-form? formula)
      (let* ((rec-result (all-form-to-vars-and-final-kernel
			  (all-form-to-kernel formula)))
	     (vars (car rec-result))
	     (final-kernel (cadr rec-result)))
	(list (cons (all-form-to-var formula) vars) final-kernel))
      (list '() formula)))

(define (mk-ex x . rest)
  (if (null? rest)
      x
      (make-ex x (apply mk-ex rest))))

; ex-form-to-vars computes the first (car x) vars of a formula.

(define (ex-form-to-vars formula . x)
  (cond
   ((null? x)
    (if (ex-form? formula)
	(cons (ex-form-to-var formula)
	      (ex-form-to-vars (ex-form-to-kernel formula)))
	'()))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (ex-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res '() (cons (ex-form-to-var rho) res)))
	  ((or (= n i) (not (ex-form? rho)))
	   (if (= n i)
	       (reverse res)
	       (myerror "ex-form-to-vars:" n "vars expected in"
			formula))))))
   (else (myerror "ex-form-to-vars" "non-negative integer expected"
		  (car x)))))

; ex-form-to-final-kernel computes the final kernel (kernel
; after removing the first (car x) vars) of a formula. 

(define (ex-form-to-final-kernel formula . x)
  (cond
   ((null? x)
    (if (ex-form? formula)
	(ex-form-to-final-kernel (ex-form-to-kernel formula))
	formula))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (ex-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res formula (ex-form-to-kernel res)))
	  ((or (= n i) (not (ex-form? rho)))
	   (if (= n i)
	       res
	       (myerror "ex-form-to-final-kernel:"
			n "vars expected in"
			formula))))))
   (else (myerror "ex-form-to-final-kernel" "non-negative integer expected"
		  (car x)))))

(define (ex-form-to-vars-and-final-kernel formula)
  (if (ex-form? formula)
      (let* ((rec-result (ex-form-to-vars-and-final-kernel
			  (ex-form-to-kernel formula)))
	     (vars (car rec-result))
	     (final-kernel (cadr rec-result)))
	(list (cons (ex-form-to-var formula) vars) final-kernel))
      (list '() formula)))

; (mk-allnc var1 ... formula) results from formula by first generalizing
; var1, then var2 etc.

(define (mk-allnc x . rest)
  (if (null? rest)
      x
      (make-allnc x (apply mk-allnc rest))))

; allnc-form-to-vars computes the first (car x) vars of a formula.

(define (allnc-form-to-vars formula . x)
  (cond
   ((null? x)
    (if (allnc-form? formula)
	(cons (allnc-form-to-var formula)
	      (allnc-form-to-vars (allnc-form-to-kernel formula)))
	'()))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (allnc-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res '() (cons (allnc-form-to-var rho) res)))
	  ((or (= n i) (not (allnc-form? rho)))
	   (if (= n i)
	       (reverse res)
	       (myerror "allnc-form-to-vars:" n "vars expected in"
			formula))))))
   (else (myerror "allnc-form-to-vars" "non-negative integer expected"
		  (car x)))))

; allnc-form-to-final-kernel computes the final kernel (kernel
; after removing the first (car x) vars) of a formula. 

(define (allnc-form-to-final-kernel formula . x)
  (cond
   ((null? x)
    (if (allnc-form? formula)
	(allnc-form-to-final-kernel (allnc-form-to-kernel formula))
	formula))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (allnc-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res formula (allnc-form-to-kernel res)))
	  ((or (= n i) (not (allnc-form? rho)))
	   (if (= n i)
	       res
	       (myerror "allnc-form-to-final-kernel:"
			n "vars expected in"
			formula))))))
   (else (myerror "allnc-form-to-final-kernel" "non-negative integer expected"
		  (car x)))))

(define (allnc-form-to-vars-and-final-kernel formula)
  (if (allnc-form? formula)
      (let* ((rec-result (allnc-form-to-vars-and-final-kernel
			  (allnc-form-to-kernel formula)))
	     (vars (car rec-result))
	     (final-kernel (cadr rec-result)))
	(list (cons (allnc-form-to-var formula) vars) final-kernel))
      (list '() formula)))

; (mk-exnc var1 ... formula) results from formula by first quantifying
; var1, then var2 etc.

(define (mk-exnc x . rest)
  (if (null? rest)
      x
      (make-exnc x (apply mk-exnc rest))))

; exnc-form-to-vars computes the first (car x) vars of a formula.

(define (exnc-form-to-vars formula . x)
  (cond
   ((null? x)
    (if (exnc-form? formula)
	(cons (exnc-form-to-var formula)
	      (exnc-form-to-vars (exnc-form-to-kernel formula)))
	'()))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (exnc-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res '() (cons (exnc-form-to-var rho) res)))
	  ((or (= n i) (not (exnc-form? rho)))
	   (if (= n i)
	       (reverse res)
	       (myerror "exnc-form-to-vars:" n "vars expected in"
			formula))))))
   (else (myerror "exnc-form-to-vars" "non-negative integer expected"
		  (car x)))))

; exnc-form-to-final-kernel computes the final kernel (kernel
; after removing the first (car x) vars) of a formula. 

(define (exnc-form-to-final-kernel formula . x)
  (cond
   ((null? x)
    (if (exnc-form? formula)
	(exnc-form-to-final-kernel (exnc-form-to-kernel formula))
	formula))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (exnc-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res formula (exnc-form-to-kernel res)))
	  ((or (= n i) (not (exnc-form? rho)))
	   (if (= n i)
	       res
	       (myerror "exnc-form-to-final-kernel:"
			n "vars expected in"
			formula))))))
   (else (myerror "exnc-form-to-final-kernel" "non-negative integer expected"
		  (car x)))))

(define (exnc-form-to-vars-and-final-kernel formula)
  (if (exnc-form? formula)
      (let* ((rec-result (exnc-form-to-vars-and-final-kernel
			  (exnc-form-to-kernel formula)))
	     (vars (car rec-result))
	     (final-kernel (cadr rec-result)))
	(list (cons (exnc-form-to-var formula) vars) final-kernel))
      (list '() formula)))

(define (mk-exca x . rest)
  (if (null? rest)
      x
      (let ((kernel-and-rev-vars (reverse rest)))
	(make-exca (cons x (reverse (cdr kernel-and-rev-vars)))
		   (car kernel-and-rev-vars)))))

(define (mk-excl x . rest)
  (if (null? rest)
      x
      (let ((kernel-and-rev-vars (reverse rest)))
	(make-excl (cons x (reverse (cdr kernel-and-rev-vars)))
		   (car kernel-and-rev-vars)))))

; Occasionally it is useful to have constructors for quantified
; formulas accepting lists of variables.

(define (make-quant-formula quant vars kernel)
  (if (memv quant '(all ex allnc exnc))
      (list quant (car vars) kernel)
      ;else exca oder excl, expecting a list of variables
      (list quant vars kernel)))

(define (mk-quant x . rest) ;x list consisting of quant and vars
  (if (null? rest)
      x
      (if (and (list? x) (= 2 (length x)))
	  (let ((quant (car x))
		(vars (cadr x)))
	    (make-quant-formula quant vars (apply mk-quant rest)))
	  (myerror "mk-quant" "list of quant and vars expected" x))))

(define (exc-form? x)
  (and (list? x) (= 3 (length x)) (memq (car x) '(exca excl))
       (list? (cadr x)) (pair? (cadr x)) (apply and-op (map var? (cadr x)))))

(define exc-form-to-quant car)
(define exc-form-to-vars cadr)
(define exc-form-to-kernel caddr)

; By means of inductively defined predicate constants, we can add
; disjunctionm existential quantification ExI and ExID and conjunction
; AndL AndR AndLR AndNC with the infix notation andl andr andlr andnc
; (so "and" is for the boolean and-op).

(define (make-or formula1 formula2)
  (make-predicate-formula
   (make-idpredconst
    "OrID" '() (list (make-cterm formula1) (make-cterm formula2)))))

(define (or-form-to-left formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (car cterms))))

(define (or-form-to-right formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (cadr cterms))))

(define (or-form? x)
  (and (predicate-form? x)
       (let ((pred (predicate-form-to-predicate x)))
	 (and (idpredconst-form? pred)
	      (equal? "OrID" (idpredconst-to-name pred))))))

(define (make-exid var kernel)
  (make-predicate-formula
   (make-idpredconst
    "ExID" (list (var-to-type var)) (list (make-cterm var kernel)))))
  
(define (exid-form-to-var formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (car (cterm-to-vars (car cterms)))))

(define (exid-form-to-kernel formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (car cterms))))

(define (exid-form? x)
  (and (predicate-form? x)
       (let ((pred (predicate-form-to-predicate x)))
	 (and (idpredconst-form? pred)
	      (equal? "ExID" (idpredconst-to-name pred))))))

(define (mk-exid x . rest)
  (if (null? rest)
      x
      (make-exid x (apply mk-exid rest))))

(define (exid-form-to-vars-and-final-kernel formula)
  (if (exid-form? formula)
      (let* ((rec-result (exid-form-to-vars-and-final-kernel
			  (exid-form-to-kernel formula)))
	     (vars (car rec-result))
	     (final-kernel (cadr rec-result)))
	(list (cons (exid-form-to-var formula) vars) final-kernel))
      (list '() formula)))

; exid-form-to-vars computes the first (car x) vars of a formula.

(define (exid-form-to-vars formula . x)
  (cond
   ((null? x)
    (if (exid-form? formula)
	(cons (exid-form-to-var formula)
	      (exid-form-to-vars (exid-form-to-kernel formula)))
	'()))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (exid-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res '() (cons (exid-form-to-var rho) res)))
	  ((or (= n i) (not (exid-form? rho)))
	   (if (= n i)
	       (reverse res)
	       (myerror "exid-form-to-vars:" n "vars expected in"
			formula))))))
   (else (myerror "exid-form-to-vars" "non-negative integer expected"
		  (car x)))))

; exid-form-to-final-kernel computes the final kernel (kernel
; after removing the first (car x) vars) of a formula. 

(define (exid-form-to-final-kernel formula . x)
  (cond
   ((null? x)
    (if (exid-form? formula)
	(exid-form-to-final-kernel (exid-form-to-kernel formula))
	formula))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (exid-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res formula (exid-form-to-kernel res)))
	  ((or (= n i) (not (exid-form? rho)))
	   (if (= n i)
	       res
	       (myerror "exid-form-to-final-kernel:"
			n "vars expected in"
			formula))))))
   (else (myerror "exid-form-to-final-kernel" "non-negative integer expected"
		  (car x)))))

(define (make-exi var kernel)
  (make-predicate-formula
   (make-idpredconst
    "ExI" (list (var-to-type var)) (list (make-cterm var kernel)))))
  
(define (exi-form-to-var formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (car (cterm-to-vars (car cterms)))))

(define (exi-form-to-kernel formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (car cterms))))

(define (exi-form? x)
  (and (predicate-form? x)
       (let ((pred (predicate-form-to-predicate x)))
	 (and (idpredconst-form? pred)
	      (equal? "ExI" (idpredconst-to-name pred))))))

(define (mk-exi x . rest)
  (if (null? rest)
      x
      (make-exi x (apply mk-exi rest))))

(define (exi-form-to-vars-and-final-kernel formula)
  (if (exi-form? formula)
      (let* ((rec-result (exi-form-to-vars-and-final-kernel
			  (exi-form-to-kernel formula)))
	     (vars (car rec-result))
	     (final-kernel (cadr rec-result)))
	(list (cons (exi-form-to-var formula) vars) final-kernel))
      (list '() formula)))

; exi-form-to-vars computes the first (car x) vars of a formula.

(define (exi-form-to-vars formula . x)
  (cond
   ((null? x)
    (if (exi-form? formula)
	(cons (exi-form-to-var formula)
	      (exi-form-to-vars (exi-form-to-kernel formula)))
	'()))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (exi-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res '() (cons (exi-form-to-var rho) res)))
	  ((or (= n i) (not (exi-form? rho)))
	   (if (= n i)
	       (reverse res)
	       (myerror "exi-form-to-vars:" n "vars expected in"
			formula))))))
   (else (myerror "exi-form-to-vars" "non-negative integer expected"
		  (car x)))))

; exi-form-to-final-kernel computes the final kernel (kernel
; after removing the first (car x) vars) of a formula. 

(define (exi-form-to-final-kernel formula . x)
  (cond
   ((null? x)
    (if (exi-form? formula)
	(exi-form-to-final-kernel (exi-form-to-kernel formula))
	formula))
   ((and (integer? (car x)) (not (negative? (car x))))
    (let ((n (car x)))
      (do ((rho formula (exi-form-to-kernel rho))
	   (i 0 (+ 1 i))
	   (res formula (exi-form-to-kernel res)))
	  ((or (= n i) (not (exi-form? rho)))
	   (if (= n i)
	       res
	       (myerror "exi-form-to-final-kernel:"
			n "vars expected in"
			formula))))))
   (else (myerror "exi-form-to-final-kernel" "non-negative integer expected"
		  (car x)))))

(define (make-andl formula1 formula2)
  (make-predicate-formula
   (make-idpredconst
    "AndL" '() (list (make-cterm formula1) (make-cterm formula2)))))

(define (andl-form-to-left formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (car cterms))))

(define (andl-form-to-right formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (cadr cterms))))

(define (andl-form? x)
  (and (predicate-form? x)
       (let ((pred (predicate-form-to-predicate x)))
	 (and (idpredconst-form? pred)
	      (equal? "AndL" (idpredconst-to-name pred))))))

(define (make-andr formula1 formula2)
  (make-predicate-formula
   (make-idpredconst
    "AndR" '() (list (make-cterm formula1) (make-cterm formula2)))))

(define (andr-form-to-left formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (car cterms))))

(define (andr-form-to-right formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (cadr cterms))))

(define (andr-form? x)
  (and (predicate-form? x)
       (let ((pred (predicate-form-to-predicate x)))
	 (and (idpredconst-form? pred)
	      (equal? "AndR" (idpredconst-to-name pred))))))

(define (make-andlr formula1 formula2)
  (make-predicate-formula
   (make-idpredconst
    "AndLR" '() (list (make-cterm formula1) (make-cterm formula2)))))

(define (andlr-form-to-left formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (car cterms))))

(define (andlr-form-to-right formula)
  (let* ((idpredconst (predicate-form-to-predicate formula))
	 (cterms (idpredconst-to-cterms idpredconst)))
    (cterm-to-formula (cadr cterms))))

(define (andlr-form? x)
  (and (predicate-form? x)
       (let ((pred (predicate-form-to-predicate x)))
	 (and (idpredconst-form? pred)
	      (equal? "AndLR" (idpredconst-to-name pred))))))

(define (make-eqid term1 term2)
  (let* ((type1 (term-to-type term1))
         (type2 (term-to-type term2))
	 (equal-idpredconst
	  (if (equal? type1 type2)
	      (make-idpredconst "EqID" (list type1) '())
	      (myerror "make-eqid" "equal types expected"
		       type1
		       type2))))
    (make-predicate-formula equal-idpredconst term1 term2)))

(define (quant-form? x)
  (and (list? x) (= 3 (length x))
       (memq (car x) '(all ex allnc exnc exca excl))))

(define quant-form-to-quant car)

(define (quant-form-to-vars x)
  (let ((quant (quant-form-to-quant x)))
    (if (memq quant '(all ex allnc exnc))
	(list (cadr x))
	(cadr x))))

(define quant-form-to-kernel caddr)

(define (quant-prime-form? formula)
  (or (prime-form? formula)
      (and (quant-form? formula)
	   (quant-prime-form? (quant-form-to-kernel formula)))))

(define (quant-free? formula)
  (case (tag formula)
    ((atom predicate) #t)
    ((imp) (and (quant-free? (imp-form-to-premise formula))
		(quant-free? (imp-form-to-conclusion formula))))
    ((and) (and (quant-free? (and-form-to-left formula))
		(quant-free? (and-form-to-right formula))))
    ((tensor) (and (quant-free? (tensor-form-to-left formula))
		   (quant-free? (tensor-form-to-right formula))))
    ((all ex allnc exnc exca excl) #f)
    (else (myerror "quant-free?" "formula expected" formula))))

(define (formula-to-head formula)
  (case (tag formula)
    ((atom predicate) formula)
    ((imp) (formula-to-head (imp-form-to-conclusion formula)))
    ((all) (formula-to-head (all-form-to-kernel formula)))
    ((allnc) (formula-to-head (allnc-form-to-kernel formula)))
    (else (myerror
	   "formula-to-head" "prime, imp, all or allnc formula expected"
	   formula))))

(define (unfold-formula formula)
  (case (tag formula)
    ((atom predicate) formula)
    ((imp) (mk-imp (unfold-formula (imp-form-to-premise formula))
		   (unfold-formula (imp-form-to-conclusion formula))))
    ((and) (mk-and (unfold-formula (and-form-to-left formula))
		   (unfold-formula (and-form-to-right formula))))
    ((tensor) (mk-tensor (unfold-formula (tensor-form-to-left formula))
			 (unfold-formula (tensor-form-to-right formula))))
    ((all) (mk-all (all-form-to-var formula)
		   (unfold-formula (all-form-to-kernel formula))))
    ((ex) (mk-ex (ex-form-to-var formula)
		 (unfold-formula (ex-form-to-kernel formula))))
    ((allnc) (mk-allnc (allnc-form-to-var formula)
		       (unfold-formula (allnc-form-to-kernel formula))))
    ((exnc) (mk-exnc (exnc-form-to-var formula)
		     (unfold-formula (exnc-form-to-kernel formula))))
    ((exca) (mk-neg
	     (apply
	      mk-all (append (exca-form-to-vars formula)
			     (list (apply mk-imp
					  (append (map unfold-formula
						       (tensor-form-to-parts
							(exca-form-to-kernel
							 formula)))
						  (list falsity))))))))
    ((excl) (mk-neg-log
	     (apply
	      mk-all (append (excl-form-to-vars formula)
			     (list (apply mk-imp
					  (append (map unfold-formula
						       (tensor-form-to-parts
							(excl-form-to-kernel
							 formula)))
						  (list falsity-log))))))))
    (else (myerror "unfold-formula" "formula expected" formula))))

(define (fold-formula formula)
  (cond ((prime-form? formula) formula)
	((foldable-exca-form? formula)
	 (let* ((prem (imp-form-to-premise formula))
		(vars-and-final-kernel
		 (all-form-to-vars-and-final-kernel prem))
		(vars (car vars-and-final-kernel))
		(kernel (cadr vars-and-final-kernel)))
	   (make-exca vars
		      (apply mk-tensor
			     (map fold-formula
				  (imp-form-to-premises kernel))))))
	((foldable-excl-form? formula)
	 (let* ((prem (imp-form-to-premise formula))
		(vars-and-final-kernel
		 (all-form-to-vars-and-final-kernel prem))
		(vars (car vars-and-final-kernel))
		(kernel (cadr vars-and-final-kernel)))
	   (make-excl vars
		      (apply mk-tensor
			     (map fold-formula
				  (imp-form-to-premises kernel))))))
	((imp-form? formula)
	 (mk-imp (fold-formula (imp-form-to-premise formula))
		 (fold-formula (imp-form-to-conclusion formula))))
	((and-form? formula)
	 (mk-and (fold-formula (and-form-to-left formula))
		 (fold-formula (and-form-to-right formula))))
	((tensor-form? formula)
	 (mk-tensor (fold-formula (tensor-form-to-left formula))
		    (fold-formula (tensor-form-to-right formula))))
	((quant-form? formula)
	 (make-quant-formula (quant-form-to-quant formula)
			     (quant-form-to-vars formula)
			     (fold-formula (quant-form-to-kernel formula))))
	(else (myerror "fold-formula" "formula expected" formula))))

(define (foldable-exca-form? formula)
  (and
   (imp-form? formula) (equal? falsity (imp-form-to-conclusion formula))
   (let ((prem (imp-form-to-premise formula)))
     (and (all-form? prem)
	  (let ((kernel (cadr (all-form-to-vars-and-final-kernel prem))))
	    (and (imp-form? kernel)
		 (equal? falsity (imp-form-to-final-conclusion kernel))))))))

(define (foldable-excl-form? formula)
  (and
   (imp-form? formula)
   (equal? falsity-log (imp-form-to-conclusion formula))
   (let ((prem (imp-form-to-premise formula)))
     (and (all-form? prem)
	  (let ((kernel (cadr (all-form-to-vars-and-final-kernel prem))))
	    (and (imp-form? kernel)
		 (equal? falsity-log
			 (imp-form-to-final-conclusion kernel))))))))

; The tensor ! is used with classical existential quantifiers only.
; Hence it should not appear in unfolded formulas.

(define (formula-with-illegal-tensor? formula)
  (case (tag formula)
    ((atom predicate) #f)
    ((imp)
     (or (formula-with-illegal-tensor? (imp-form-to-premise formula))
	 (formula-with-illegal-tensor? (imp-form-to-conclusion formula))))
    ((and)
     (or (formula-with-illegal-tensor? (and-form-to-left formula))
    	 (formula-with-illegal-tensor? (and-form-to-right formula))))
    ((tensor) #t)
    ((all ex allnc exnc)
     (formula-with-illegal-tensor? (quant-form-to-kernel formula)))
    (else (myerror "formula-with-illegal-tensor?"
		   "unfolded formula expected" formula))))

; Moreover we need

(define (formula-to-free formula)
  (case (tag formula)
    ((atom) (term-to-free (atom-form-to-kernel formula)))
    ((predicate)
     (let ((pred (predicate-form-to-predicate formula)))
       (cond
	((or (pvar-form? pred) (predconst-form? pred))
	 (apply union (map term-to-free (predicate-form-to-args formula))))
	((idpredconst-form? pred)
	 (let ((cterms (idpredconst-to-cterms pred)))
	   (apply union
		  (append (map term-to-free (predicate-form-to-args formula))
			  (map cterm-to-free cterms)))))
	(else (myerror "formula-to-pvars" "predicate expected" pred)))))
    ((imp) (union (formula-to-free (imp-form-to-premise formula))
		  (formula-to-free (imp-form-to-conclusion formula))))
    ((and) (union (formula-to-free (and-form-to-left formula))
		  (formula-to-free (and-form-to-right formula))))
    ((tensor) (union (formula-to-free (tensor-form-to-left formula))
		     (formula-to-free (tensor-form-to-right formula))))
    ((all) (remove (all-form-to-var formula)
		   (formula-to-free (all-form-to-kernel formula))))
    ((ex) (remove (ex-form-to-var formula)
		  (formula-to-free (ex-form-to-kernel formula))))
    ((allnc) (remove (allnc-form-to-var formula)
		     (formula-to-free (allnc-form-to-kernel formula))))
    ((exnc) (remove (exnc-form-to-var formula)
		    (formula-to-free (exnc-form-to-kernel formula))))
    ((exca) (set-minus (formula-to-free (exca-form-to-kernel formula))
		       (exca-form-to-vars formula)))
    ((excl) (set-minus (formula-to-free (exc-form-to-kernel formula))
		       (excl-form-to-vars formula)))
    (else (myerror "formula-to-free" "formula expected" formula))))

(define (formula-to-tvars formula)
  (case (tag formula)
    ((atom) (term-to-tvars (atom-form-to-kernel formula)))
    ((predicate)
     (apply union
	    (cons (predicate-to-tvars (predicate-form-to-predicate formula))
		  (map term-to-tvars (predicate-form-to-args formula)))))
    ((imp) (union (formula-to-tvars (imp-form-to-premise formula))
		  (formula-to-tvars (imp-form-to-conclusion formula))))
    ((and) (union (formula-to-tvars (and-form-to-left formula))
		  (formula-to-tvars (and-form-to-right formula))))
    ((tensor) (union (formula-to-tvars (tensor-form-to-left formula))
		     (formula-to-tvars (tensor-form-to-right formula))))
    ((all ex allnc exnc exca excl)
     (let* ((vars (quant-form-to-vars formula))
	    (kernel (quant-form-to-kernel formula)))
       (apply union (append (map (lambda (x) (type-to-free (var-to-type x)))
				 vars)
			    (list (formula-to-tvars kernel))))))
    (else (myerror "formula-to-tvars" "formula expected" formula))))

(define (formula-to-pvars formula)
  (case (tag formula)
    ((atom) '())
    ((predicate)
     (let ((pred (predicate-form-to-predicate formula)))
       (cond
	((pvar-form? pred) (list pred))
	((predconst-form? pred) '())
	((idpredconst-form? pred)
	 (let* ((cterms (idpredconst-to-cterms pred))
		(formulas (map cterm-to-formula cterms)))
	   (apply union (map formula-to-pvars formulas))))
	(else (myerror "formula-to-pvars" "predicate expected" pred)))))
    ((imp) (union (formula-to-pvars (imp-form-to-premise formula))
		  (formula-to-pvars (imp-form-to-conclusion formula))))
    ((and) (union (formula-to-pvars (and-form-to-left formula))
		  (formula-to-pvars (and-form-to-right formula))))
    ((tensor) (union (formula-to-pvars (tensor-form-to-left formula))
		     (formula-to-pvars (tensor-form-to-right formula))))
    ((all ex allnc exnc exca excl)
     (formula-to-pvars (quant-form-to-kernel formula)))
    (else (myerror "formula-to-pvars" "formula expected" formula))))

(define (ex-free-formula? formula)
  (case (tag formula)
    ((atom predicate) #t)
    ((imp) (and (ex-free-formula? (imp-form-to-premise formula))
		(ex-free-formula? (imp-form-to-conclusion formula))))
    ((and) (and (ex-free-formula? (and-form-to-left formula))
		(ex-free-formula? (and-form-to-right formula))))
    ((tensor) (and (ex-free-formula? (tensor-form-to-left formula))
		   (ex-free-formula? (tensor-form-to-right formula))))
    ((all ex allnc exnc exca excl)
     (and (not (eq? 'ex (quant-form-to-quant formula)))
	  (ex-free-formula? (quant-form-to-kernel formula))))
    (else (myerror "ex-free-formula?" "formula expected" formula))))

; nbe-formula-to-type needs a procedure associating type variables to
; predicate variables, which remembers the assignment done so far.
; Therefore it refers to the global variable PVARS-TO-TVARS.  This
; machinery will be used to assign recursion constants to induction
; constants.  There we need to associate type variables with predicate
; variables, in such a way that we can later refer to this assignment.

(define (nbe-formula-to-type formula)
  (case (tag formula)
    ((atom) (make-tconst "atomic"))
    ((predicate)
     (let ((pred (predicate-form-to-predicate formula)))
       (cond
	((pvar-form? pred) (PVAR-TO-TVAR pred))
	((predconst-form? pred) (make-tconst "prop"))
	((idpredconst-form? pred)
	 (let* ((name (idpredconst-to-name pred))
		(types (idpredconst-to-types pred))
		(param-cterms (idpredconst-to-cterms pred))
		(param-cterm-types
		 (map nbe-formula-to-type
		      (map cterm-to-formula param-cterms)))
		(nbe-alg-name (idpredconst-name-to-nbe-alg-name name)))
	   (apply make-alg (cons nbe-alg-name
				 (append types param-cterm-types))))))))
    ((imp)
     (make-arrow (nbe-formula-to-type (imp-form-to-premise formula))
		 (nbe-formula-to-type (imp-form-to-conclusion formula))))
    ((and)
     (make-star (nbe-formula-to-type (and-form-to-left formula))
		(nbe-formula-to-type (and-form-to-right formula))))
    ((tensor)
     (make-alg "ytensor" 
	       (nbe-formula-to-type (tensor-form-to-left formula))
	       (nbe-formula-to-type (tensor-form-to-right formula))))
    ((all)
     (let* ((var (all-form-to-var formula))
	    (kernel (all-form-to-kernel formula))
	    (type1 (var-to-type var))
	    (type2 (nbe-formula-to-type kernel)))
       (make-arrow type1 type2)))
    ((ex exnc) (make-tconst "existential"))
    ((allnc)
     (let* ((var (allnc-form-to-var formula))
	    (kernel (allnc-form-to-kernel formula))
	    (type1 (var-to-type var))
	    (type2 (nbe-formula-to-type kernel)))
       (make-arrow type1 type2)))
    ((exca excl) (nbe-formula-to-type (unfold-formula formula)))
    (else (myerror "nbe-formula-to-type" "formula expected" formula))))

(define (formula-to-prime-subformulas formula)
  (case (tag formula)
    ((atom predicate) (list formula))
    ((imp)
     (union (formula-to-prime-subformulas (imp-form-to-premise formula))
	    (formula-to-prime-subformulas (imp-form-to-conclusion formula))))
    ((and)
     (union (formula-to-prime-subformulas (and-form-to-left formula))
	    (formula-to-prime-subformulas (and-form-to-right formula))))
    ((tensor)
     (union (formula-to-prime-subformulas (tensor-form-to-left formula))
	    (formula-to-prime-subformulas (tensor-form-to-right formula))))
    ((all ex allnc exnc exca excl)
     (formula-to-prime-subformulas (quant-form-to-kernel formula)))
    (else (myerror "formula-to-prime-subformulas" "formula expected"
		   formula))))

(define (formula-to-positive-existential-subformulas formula)
  (case (tag formula)
    ((atom predicate) '())
    ((imp)
     (union (formula-to-negative-existential-subformulas
	     (imp-form-to-premise formula))
	    (formula-to-positive-existential-subformulas
	     (imp-form-to-conclusion formula))))
    ((and)
     (union (formula-to-positive-existential-subformulas
	     (and-form-to-left formula))
	    (formula-to-positive-existential-subformulas
	     (and-form-to-right formula))))
    ((tensor)
     (union (formula-to-positive-existential-subformulas
	     (tensor-form-to-left formula))
	    (formula-to-positive-existential-subformulas
	     (tensor-form-to-right formula))))
    ((all allnc exca excl)
     (formula-to-positive-existential-subformulas
      (quant-form-to-kernel formula)))
    ((ex exnc)
     (cons formula
	   (formula-to-positive-existential-subformulas
	    (quant-form-to-kernel formula))))
    (else (myerror
	   "formula-to-positive-existential-subformulas" "formula expected"
	   formula))))

(define (formula-to-negative-existential-subformulas formula)
  (case (tag formula)
    ((atom predicate) '())
    ((imp)
     (union (formula-to-positive-existential-subformulas
	     (imp-form-to-premise formula))
	    (formula-to-negative-existential-subformulas
	     (imp-form-to-conclusion formula))))
    ((and)
     (union (formula-to-negative-existential-subformulas
	     (and-form-to-left formula))
	    (formula-to-negative-existential-subformulas
	     (and-form-to-right formula))))
    ((tensor)
     (union (formula-to-negative-existential-subformulas
	     (tensor-form-to-left formula))
	    (formula-to-negative-existential-subformulas
	     (tensor-form-to-right formula))))
    ((all allnc exca excl)
     (formula-to-negative-existential-subformulas
      (quant-form-to-kernel formula)))
    ((ex exnc)
     (formula-to-negative-existential-subformulas
      (quant-form-to-kernel formula)))
    (else (myerror
	   "formula-to-negative-existential-subformulas" "formula expected"
	   formula))))

(define (formula-to-beta-nf formula)
  (case (tag formula)
    ((atom)
     (make-atomic-formula (term-to-beta-nf (atom-form-to-kernel formula))))
    ((predicate)
     (let ((normal-args
	    (map term-to-beta-nf (predicate-form-to-args formula))))
       (apply make-predicate-formula
	      (cons (predicate-form-to-predicate formula) normal-args))))
    ((imp)
     (make-imp (formula-to-beta-nf (imp-form-to-premise formula))
	       (formula-to-beta-nf (imp-form-to-conclusion formula))))
    ((and)
     (make-and (formula-to-beta-nf (and-form-to-left formula))
	       (formula-to-beta-nf (and-form-to-right formula))))
    ((tensor)
     (make-tensor (formula-to-beta-nf (tensor-form-to-left formula))
		  (formula-to-beta-nf (tensor-form-to-right formula))))
    ((all ex allnc exnc exca excl)
     (let ((quant (quant-form-to-quant formula))
	   (vars (quant-form-to-vars formula))
	   (kernel (quant-form-to-kernel formula)))
       (make-quant-formula quant vars (formula-to-beta-nf kernel))))
    (else (myerror "formula-to-beta-nf" "formula expected" formula))))

; 2007-01-20 Moved here from atr.scm
(define (qf-to-term formula)
  (case (tag formula)
    ((atom) (atom-form-to-kernel formula))
    ((predicate)
     (if (formula=? falsity-log formula)
	 (make-term-in-const-form false-const)
	 (myerror "qf-to-term" "unexpected predicate"
		  formula)))
    ((imp)
     (mk-term-in-app-form
      (make-term-in-const-form imp-const)
      (qf-to-term (imp-form-to-premise formula))
      (qf-to-term (imp-form-to-conclusion formula))))
    ((and)
     (mk-term-in-app-form
      (make-term-in-const-form and-const)
      (qf-to-term (and-form-to-left formula))
      (qf-to-term (and-form-to-right formula))))
    (else (myerror "qf-to-term" "quantifier free formula expected"
		   formula))))

(define (alpha-equal-formulas-to-renaming formula1 formula2)
  (cond
   ((and (prime-form? formula1) (prime-form? formula2)) '())
   ((and (imp-form? formula1) (imp-form? formula2))
    (append (alpha-equal-formulas-to-renaming
	     (imp-form-to-premise formula1)
	     (imp-form-to-premise formula2))
	    (alpha-equal-formulas-to-renaming
	     (imp-form-to-conclusion formula1)
	     (imp-form-to-conclusion formula2))))
   ((and (and-form? formula1) (and-form? formula2))
    (append (alpha-equal-formulas-to-renaming
	     (and-form-to-left formula1)
	     (and-form-to-left formula2))
	    (alpha-equal-formulas-to-renaming
	     (and-form-to-right formula1)
	     (and-form-to-right formula2))))
   ((and (tensor-form? formula1) (tensor-form? formula2))
    (append (alpha-equal-formulas-to-renaming
	     (tensor-form-to-left formula1)
	     (tensor-form-to-left formula2))
	    (alpha-equal-formulas-to-renaming
	     (tensor-form-to-right formula1)
	     (tensor-form-to-right formula2))))
   ((and
     (quant-form? formula1) (quant-form? formula2)
     (equal? (quant-form-to-quant formula1) (quant-form-to-quant formula2)))
    (let ((vars1 (quant-form-to-vars formula1))
	  (vars2 (quant-form-to-vars formula2)))
      (if (not (= (length vars1) (length vars2)))
	  (myerror "alpha-equal-formulas-to-renaming"
		   "quantified variables of the same length expected"
		   formula1 formula2))
      (append (list-transform-positive
		  (map (lambda (x y) (list x y)) vars1 vars2)
		(lambda (p)
		  (not (equal? (car p) (cadr p)))))
	      (alpha-equal-formulas-to-renaming
	       (quant-form-to-kernel formula1)
	       (quant-form-to-kernel formula2)))))
   (else (myerror "alpha-equal-formulas-to-renaming"
		  "alpha equal formula expected"
		  formula1 formula2))))

; Comprehension terms have the form (cterm vars formula), where
; formula may contain further free variables.

(define (make-cterm x . rest)
  (if (null? rest)
      (list 'cterm (list) x)
      (let ((prev (apply make-cterm rest)))
	(list 'cterm (cons x (cterm-to-vars prev))
	      (cterm-to-formula prev)))))

(define cterm-to-vars cadr)
(define cterm-to-formula caddr)

(define (cterm-form? x)
  (and (pair? x) (eq? 'cterm (car x))))

(define (cterm? x)
  (and (cterm-form? x)
       (list? x)
       (= 3 (length x))
       (let ((vars (cadr x))
	     (formula (caddr x)))
	 (and (apply and-op (map var? vars))
	      (formula? formula)))))

(define (classical-cterm=? cterm1 cterm2)
  (or (equal? cterm1 cterm2)
      (classical-formula=?
       (apply mk-all (append (cterm-to-vars cterm1)
			     (list (cterm-to-formula cterm1))))
       (apply mk-all (append (cterm-to-vars cterm2)
			     (list (cterm-to-formula cterm2)))))))

(define (cterm=? cterm1 cterm2)
  (or (equal? cterm1 cterm2)
      (formula=?
       (apply mk-all (append (cterm-to-vars cterm1)
			     (list (cterm-to-formula cterm1))))
       (apply mk-all (append (cterm-to-vars cterm2)
			     (list (cterm-to-formula cterm2)))))))

(define (cterm-to-free cterm)
  (set-minus (formula-to-free (cterm-to-formula cterm))
	     (cterm-to-vars cterm)))

(define (fold-cterm cterm)
  (list 'cterm
	(cterm-to-vars cterm)
	(fold-formula (cterm-to-formula cterm))))

(define (unfold-cterm cterm)
  (list 'cterm
	(cterm-to-vars cterm)
	(unfold-formula (cterm-to-formula cterm))))


; 7-2. Normalization
; ==================

(define (normalize-formula formula)
  (case (tag formula)
    ((atom)
     (make-atomic-formula (nt (atom-form-to-kernel formula))))
    ((predicate)
     (let ((normal-args (map nt (predicate-form-to-args formula))))
       (apply make-predicate-formula
	      (cons (predicate-form-to-predicate formula) normal-args))))
    ((imp)
     (make-imp (normalize-formula (imp-form-to-premise formula))
	       (normalize-formula (imp-form-to-conclusion formula))))
    ((and)
     (make-and (normalize-formula (and-form-to-left formula))
	       (normalize-formula (and-form-to-right formula))))
    ((tensor)
     (make-tensor (normalize-formula (tensor-form-to-left formula))
		  (normalize-formula (tensor-form-to-right formula))))
    ((all ex allnc exnc exca excl)
     (let ((quant (quant-form-to-quant formula))
	   (vars (quant-form-to-vars formula))
	   (kernel (quant-form-to-kernel formula)))
       (make-quant-formula quant vars (normalize-formula kernel))))
    (else (myerror "normalize-formula" "formula expected" formula))))

(define nf normalize-formula)


; 7-3. Alpha-equality
; ===================

(define (classical-formula=? formula1 formula2)
  (or (equal? formula1 formula2)
      (let ((for1 (unfold-formula formula1))
	    (for2 (unfold-formula formula2)))
	(or (equal? for1 for2)
	    (let ((nf1 (normalize-formula for1))
		  (nf2 (normalize-formula for2)))
	      (or (equal? nf1 nf2)
		  (formula=-aux? nf1 nf2 '() '())))))))

(define (formula=? formula1 formula2)
  (formula=-aux? formula1 formula2 '() '()))

(define (formula=-aux? formula1 formula2 alist alistrev)
  (or (and (atom-form? formula1) (atom-form? formula2)
           (term=-aux? (atom-form-to-kernel formula1)
		       (atom-form-to-kernel formula2)
		       alist alistrev))
      (and (predicate-form? formula1) (predicate-form? formula2)
           (let ((pred1 (predicate-form-to-predicate formula1))
                 (pred2 (predicate-form-to-predicate formula2))
                 (args1 (predicate-form-to-args formula1))
                 (args2 (predicate-form-to-args formula2)))
             (and (predicate-equal? pred1 pred2)
                  (terms=-aux? args1 args2 alist alistrev))))
      (and (imp-form? formula1) (imp-form? formula2)
           (let ((prem1 (imp-form-to-premise formula1))
                 (concl1 (imp-form-to-conclusion formula1))
                 (prem2 (imp-form-to-premise formula2))
                 (concl2 (imp-form-to-conclusion formula2)))
             (and (formula=-aux? prem1 prem2 alist alistrev)
                  (formula=-aux? concl1 concl2 alist alistrev))))
      (and (and-form? formula1) (and-form? formula2)
           (let ((left1 (and-form-to-left formula1))
                 (right1 (and-form-to-right formula1))
                 (left2 (and-form-to-left formula2))
                 (right2 (and-form-to-right formula2)))
             (and (formula=-aux? left1 left2 alist alistrev)
                  (formula=-aux? right1 right2 alist alistrev))))
      (and (tensor-form? formula1) (tensor-form? formula2)
           (let ((left1 (tensor-form-to-left formula1))
                 (right1 (tensor-form-to-right formula1))
                 (left2 (tensor-form-to-left formula2))
                 (right2 (tensor-form-to-right formula2)))
             (and (formula=-aux? left1 left2 alist alistrev)
                  (formula=-aux? right1 right2 alist alistrev))))
      (and (all-form? formula1) (all-form? formula2)
           (let ((var1 (all-form-to-var formula1))
                 (var2 (all-form-to-var formula2))
                 (kernel1 (all-form-to-kernel formula1))
                 (kernel2 (all-form-to-kernel formula2)))
             (and (equal? (var-to-type var1) (var-to-type var2))
                  (equal? (var-to-t-deg var1) (var-to-t-deg var2))
                  (formula=-aux?
                   kernel1 kernel2
                   (cons (list var1 var2) alist)
                   (cons (list var2 var1) alistrev)))))
      (and (ex-form? formula1) (ex-form? formula2)
           (let ((var1 (ex-form-to-var formula1))
                 (var2 (ex-form-to-var formula2))
                 (kernel1 (ex-form-to-kernel formula1))
                 (kernel2 (ex-form-to-kernel formula2)))
             (and (equal? (var-to-type var1) (var-to-type var2))
                  (equal? (var-to-t-deg var1) (var-to-t-deg var2))
                  (formula=-aux?
                   kernel1 kernel2
                   (cons (list var1 var2) alist)
                   (cons (list var2 var1) alistrev)))))
      (and (allnc-form? formula1) (allnc-form? formula2)
           (let ((var1 (allnc-form-to-var formula1))
                 (var2 (allnc-form-to-var formula2))
                 (kernel1 (allnc-form-to-kernel formula1))
                 (kernel2 (allnc-form-to-kernel formula2)))
             (and (equal? (var-to-type var1) (var-to-type var2))
                  (equal? (var-to-t-deg var1) (var-to-t-deg var2))
                  (formula=-aux?
                   kernel1 kernel2
                   (cons (list var1 var2) alist)
                   (cons (list var2 var1) alistrev)))))
      (and (exnc-form? formula1) (exnc-form? formula2)
           (let ((var1 (exnc-form-to-var formula1))
                 (var2 (exnc-form-to-var formula2))
                 (kernel1 (exnc-form-to-kernel formula1))
                 (kernel2 (exnc-form-to-kernel formula2)))
             (and (equal? (var-to-type var1) (var-to-type var2))
                  (equal? (var-to-t-deg var1) (var-to-t-deg var2))
                  (formula=-aux?
                   kernel1 kernel2
                   (cons (list var1 var2) alist)
                   (cons (list var2 var1) alistrev)))))))

  
; 7-4. Display
; ============

; For a readable display of formulas we use

(define (predicate-to-token-tree pred)
  (cond
   ((pvar-form? pred)
    (make-token-tree 'pvar (pvar-to-string pred)))
   ((predconst-form? pred)
    (make-token-tree 'predconst (predconst-to-string pred)))
   ((idpredconst-form? pred)
    (make-token-tree 'idpredconst (idpredconst-to-string pred))) ;unfold?
   (else (myerror "predicate-to-token-tree" "predicate expected" pred))))  

(define (formula-to-token-tree formula)
  (case (tag formula)
    ((atom)
     (let ((kernel (atom-form-to-kernel formula)))
       (cond
	((and (term-in-const-form? kernel)
	      (string=? "True"
			(const-to-name (term-in-const-form-to-const kernel))))
	 (make-token-tree 'atom "T"))
	((and (term-in-const-form? kernel)
	      (string=? "False"
			(const-to-name (term-in-const-form-to-const kernel))))
	 (make-token-tree 'atom "F"))
	(else (make-token-tree 'atom "" (term-to-token-tree kernel))))))
    ((predicate)
     (let* ((pred (predicate-form-to-predicate formula))
	    (args (predicate-form-to-args formula))
	    (pred-string
	     (cond ((pvar-form? pred) (pvar-to-string pred))
		   ((predconst-form? pred) (predconst-to-string pred))
		   ((idpredconst-form? pred) (idpredconst-to-string pred))
		   (else (myerror "formula-to-token-tree" "predicate expected"
				  pred))))
	    (name
	     (cond ((pvar-form? pred) (pvar-to-name pred))
		   ((predconst-form? pred) (predconst-to-name pred))
		   ((idpredconst-form? pred) (idpredconst-to-name pred))
		   (else (myerror "formula-to-token-tree" "predicate expected"
				  pred))))
	    (infix? (or (and (predconst-form? pred)
			     (let ((info (assoc name PREDCONST-DISPLAY)))
			       (and info (eq? 'pred-infix (cadr info)))))
			(and (idpredconst-form? pred)
			     (let ((info (assoc name IDPREDCONST-DISPLAY)))
			       (and info (eq? 'pred-infix (cadr info))))))))
       (if infix?
	   (apply make-token-tree (append (list 'pred-infix pred-string)
					  (map term-to-token-tree args)))
	   (do ((l args (cdr l))
		(res (predicate-to-token-tree pred)
		     (let ((arg (car l)))
		       (make-token-tree
			'predapp ""
			res
			(term-to-token-tree arg)))))
	       ((null? l) res)))))
    ((imp)
     (make-token-tree
      'imp-op " -> "
      (formula-to-token-tree (imp-form-to-premise formula))
      (formula-to-token-tree (imp-form-to-conclusion formula))))
    ((and)
     (make-token-tree
      'and-op " & "
      (formula-to-token-tree (and-form-to-left formula))
      (formula-to-token-tree (and-form-to-right formula))))
    ((tensor)
     (make-token-tree
      'tensor-op " ! "
      (formula-to-token-tree (tensor-form-to-left formula))
      (formula-to-token-tree (tensor-form-to-right formula))))
    ((all)
     (make-token-tree
      'all-op (var-to-string (all-form-to-var formula))
      (formula-to-token-tree (all-form-to-kernel formula))))
    ((ex)
     (make-token-tree
      'ex-op (var-to-string (ex-form-to-var formula))
      (formula-to-token-tree (ex-form-to-kernel formula))))
    ((allnc)
     (make-token-tree
      'allnc-op (var-to-string (allnc-form-to-var formula))
      (formula-to-token-tree (allnc-form-to-kernel formula))))
    ((exnc)
     (make-token-tree
      'exnc-op (var-to-string (exnc-form-to-var formula))
      (formula-to-token-tree (exnc-form-to-kernel formula))))
    ((exca)
     (make-token-tree
      'exca-op (map var-to-string (exca-form-to-vars formula))
      (formula-to-token-tree (exca-form-to-kernel formula))))
    ((excl)
     (make-token-tree
      'excl-op (map var-to-string (excl-form-to-vars formula))
      (formula-to-token-tree (excl-form-to-kernel formula))))
    (else
     (myerror "formula-to-token-tree" "unexpected tag" (tag formula)))))


; 7-6. Check
; ==========

; check-formula is a test function for formulas.  If the argument is
; not a formula, an error is returned.

(define (check-formula x)
  (if (not (pair? x)) (myerror "check-formula" "formula expected"))
  (cond
   ((atom-form? x)
    (let ((kernel (atom-form-to-kernel x)))
      (check-term kernel)
      (if (not (string=? "boole" (type-to-string (term-to-type kernel))))
	  (myerror "check-formula"
		   "atom should have an argument of type boole"
		   (term-to-type kernel)))
      #t))
   ((predicate-form? x)
    (let ((pred (predicate-form-to-predicate x))
	  (args (predicate-form-to-args x)))
      (map check-term args)
      (let ((arity (predicate-to-arity pred))
	    (types (map term-to-type args)))
	(if (not (equal? (arity-to-types arity) types))
	    (myerror "check-formula" "equal types expected"
		     (arity-to-types arity) types))))
    #t)
   ((imp-form? x)
    (let ((prem (imp-form-to-premise x))
	  (concl (imp-form-to-conclusion x)))
      (check-formula prem)
      (check-formula concl)))
   ((and-form? x)
    (let ((left (and-form-to-left x))
	  (right (and-form-to-right x)))
      (check-formula left)
      (check-formula right)))
   ((tensor-form? x)
    (let ((left (tensor-form-to-left x))
	  (right (tensor-form-to-right x)))
      (check-formula left)
      (check-formula right)))
   ((all-form? x)
    (let ((var (all-form-to-var x))
	  (kernel (all-form-to-kernel x)))
      (if (not (var? var))
	  (myerror "check-formula" "variable expected" var))
      (check-formula kernel)))
   ((ex-form? x)
    (let ((var (ex-form-to-var x))
	  (kernel (ex-form-to-kernel x)))
      (if (not (var? var))
	  (myerror "check-formula" "variable expected" var))
      (check-formula kernel)))
   ((allnc-form? x)
    (let ((var (allnc-form-to-var x))
	  (kernel (allnc-form-to-kernel x)))
      (if (not (var? var))
	  (myerror "check-formula" "variable expected" var))
      (check-formula kernel)))
   ((exnc-form? x)
    (let ((var (exnc-form-to-var x))
	  (kernel (exnc-form-to-kernel x)))
      (if (not (var? var))
	  (myerror "check-formula" "variable expected" var))
      (check-formula kernel)))
   ((exca-form? x)
    (let ((vars (exca-form-to-vars x))
	  (kernel (exca-form-to-kernel x)))
      (do ((l vars (cdr l)))
	  ((null? l))
	(if (not (var? (car l)))
	    (myerror "check-formula" "variable expected" (car l))))
      (check-formula kernel)))
   ((excl-form? x)
    (let ((vars (excl-form-to-vars x))
	  (kernel (excl-form-to-kernel x)))
      (do ((l vars (cdr l)))
	  ((null? l))
	(if (not (var? (car l)))
	    (myerror "check-formula" "variable expected" (car l))))
      (check-formula kernel)))
   (else (myerror "check-formula" "formula expected" x))))

; formula? is a complete test for formula.  Returns true or false.

(define (formula? x)
  (if (not (pair? x)) #f
      (cond
       ((atom-form? x)
	(let ((kernel (atom-form-to-kernel x)))
	  (and (term? kernel)
	       (string=? "boole" (type-to-string (term-to-type kernel))))))
       ((predicate-form? x)
	(let ((pred (predicate-form-to-predicate x))
	      (args (predicate-form-to-args x)))
	  (and (apply and-op (map term? args))
	       (let ((arity (predicate-to-arity pred))
		     (types (map term-to-type args)))
		 (equal? (arity-to-types arity) types)))))
       ((imp-form? x)
	(let ((prem (imp-form-to-premise x))
	      (concl (imp-form-to-conclusion x)))
	  (and (formula? prem) (formula? concl))))
       ((and-form? x)
	(let ((left (and-form-to-left x))
	      (right (and-form-to-right x)))
	  (and (formula? left) (formula? right))))
       ((tensor-form? x)
	(let ((left (tensor-form-to-left x))
	      (right (tensor-form-to-right x)))
	  (and (formula? left) (formula? right))))
       ((all-form? x)
	(let ((var (all-form-to-var x))
	      (kernel (all-form-to-kernel x)))
	  (and (var? var) (formula? kernel))))
       ((ex-form? x)
	(let ((var (ex-form-to-var x))
	      (kernel (ex-form-to-kernel x)))
	  (and (var? var) (formula? kernel))))
       ((allnc-form? x)
	(let ((var (allnc-form-to-var x))
	      (kernel (allnc-form-to-kernel x)))
	  (and (var? var) (formula? kernel))))
       ((exnc-form? x)
	(let ((var (exnc-form-to-var x))
	      (kernel (exnc-form-to-kernel x)))
	  (and (var? var) (formula? kernel))))
       ((exca-form? x)
	(let ((vars (exca-form-to-vars x))
	      (kernel (exca-form-to-kernel x)))
	  (and (apply and-op (map var? vars))
	       (formula? kernel))))
       ((excl-form? x)
	(let ((vars (excl-form-to-vars x))
	      (kernel (excl-form-to-kernel x)))
	  (and (apply and-op (map var? vars))
	       (formula? kernel))))
       (else #f))))

(define cf check-formula)

; Test function: (cterm-check-to-string x node).  If x is a
; comprehension term, the result is a string (which should be a
; readable representation of the comprehension term).  If x is not a
; comprehension term, an error is raised.

(define (cterm-check-to-string x node)
  (let* ((vars (cterm-to-vars x))
	 (formula (cterm-to-formula x))
	 (string-of-formula
	  (formula-check-to-string formula (append node (list 1))))
	 (result (string-append "(cterm " (vars-to-string vars) " "
				(formula-to-string formula) ")")))
    (do ((l vars (cdr l))
	 (res '() (if
		   (member (car l) (remove (car l) vars))
		   (myerror "cterm-check-to-string" "repeated vars in cterm"
			    vars)
		   (if (variable? (car l))
		       (cons (car l) res)
		       (myerror "cterm-check-to-string" "variable expected"
				   (car l))))))
	((null? l)))
    (display-comment (make-string (length node) #\.))
    (display result) (newline)
    result))


; 7-7. Substitution
; =================

; We define simultaneous substitution for type, object and predicate
; variables in a formula, via tsubst, subst and psubst.  It is assumed
; that subst only affects those vars whose type is not changed by
; tsubst, and that psubst only affects those pvars whose arity is not
; changed by tsubst.

; In the quantifier case of the recursive definition, the abstracted
; variable may need to be renamed.  However, its type can be affected
; by tsubst.  Then the renaming cannot be made part of subst, because
; then the condition above would be violated.  Therefore we carry
; along a procedure rename renaming variables, which remembers the
; renaming of variables done so far.

; We will also need formula substitution to compute the formula of an
; assumption constant.  However, there (type and) predicate variables
; are (implicitely) considered to be bound.  Therefore, we also have
; to carry along a procedure prename renaming pvars, which remembers
; the renaming of predicate variables done so far.

(define (make-prename tsubst)
  ;returns a procedure renaming predicate variables, 
  ;which remembers the renaming of predicate variables done so far.
  (let ((assoc-list '()))
    (lambda (pvar)
      (let* ((arity (pvar-to-arity pvar))
	     (types (arity-to-types arity))
	     (new-types (map (lambda (x) (type-substitute x tsubst)) types))
	     (new-arity (apply make-arity new-types)))
	(if (apply and-op (map equal? types new-types))
	    pvar
	    (let ((info (assoc pvar assoc-list)))
	      (if info
		  (cadr info)
		  (let ((new-pvar (arity-to-new-pvar new-arity pvar)))
		    (set! assoc-list (cons (list pvar new-pvar) assoc-list))
		    new-pvar))))))))

(define (formula-substitute formula topsubst)
  (let* ((tsubst-and-subst-and-psubst
	  (do ((l topsubst (cdr l))
	       (tsubst '() (if (tvar? (caar l))
			       (cons (car l) tsubst)
			       tsubst))
	       (subst '() (if (var? (caar l))
			       (cons (car l) subst)
			       subst))
	       (psubst '() (if (pvar-form? (caar l))
			       (cons (car l) psubst)
			       psubst)))
	      ((null? l)
	       (list (reverse tsubst) (reverse subst) (reverse psubst)))))
	 (tsubst (car tsubst-and-subst-and-psubst))
	 (subst (cadr tsubst-and-subst-and-psubst))
	 (psubst (caddr tsubst-and-subst-and-psubst))
	 (rename (make-rename tsubst))
	 (prename (make-prename tsubst)))
    (formula-substitute-aux formula tsubst subst psubst rename prename)))

(define (pvar-cterm-equal? pvar cterm)
  (let ((formula (cterm-to-formula cterm)))
    (and (predicate-form? formula)
	 (equal? pvar (predicate-form-to-predicate formula))
	 (equal? (map make-term-in-var-form (cterm-to-vars cterm))
		 (predicate-form-to-args formula)))))

(define (formula-subst formula arg val)
  (let ((equality?
	 (cond
	  ((and (tvar? arg) (type? val)) equal?)
	  ((and (var-form? arg) (term-form? val)) var-term-equal?)
	  ((and (pvar? arg) (cterm-form? val)) pvar-cterm-equal?)
	  (else
	   (myerror "formula-subst" "unexpected arg" arg "and val" val)))))
    (formula-substitute formula (make-subst-wrt equality? arg val))))

; In formula-substitute-aux we always first rename, when a predicate
; variable is encountered.

(define (formula-substitute-aux formula tsubst subst psubst rename prename)
  (case (tag formula)
    ((atom)
     (make-atomic-formula
      (term-substitute-aux (atom-form-to-kernel formula)
			   tsubst subst rename)))
    ((predicate)
     (let* ((pred (predicate-form-to-predicate formula))
	    (args (predicate-form-to-args formula))
	    (new-args (map (lambda (x)
			     (term-substitute-aux x tsubst subst rename))
			   args)))
       (cond
	((pvar-form? pred)
	 (let* ((pvar (prename pred))
		(info (assoc pvar psubst)))
	   (if info
	       (let* ((cterm (cadr info))
		      (vars (cterm-to-vars cterm))
		      (fla (cterm-to-formula cterm))
		      (new-subst (make-substitution vars new-args)))
		 (if (and (not (pvar-with-positive-content? pvar))
; 			  (not (equal? pvar (predicate-form-to-predicate
; 					     falsity-log))) ;for atr
			  (not (nulltype? (formula-to-et-type fla))))
		     (myerror "formula-substitute-aux"
			      "formula without positive content expected"
			      fla)
		     (formula-substitute fla new-subst)))
	       (apply make-predicate-formula (cons pvar new-args)))))
	((predconst-form? pred)
	 (let* ((tsubst0 (predconst-to-tsubst pred))
		(composed-tsubst (compose-t-substitutions tsubst0 tsubst))
		(arity (predconst-to-uninst-arity pred))
		(new-tsubst (restrict-substitution-wrt
			     composed-tsubst
			     (lambda (x)
			       (member x (apply union
						(map type-to-free
						     (arity-to-types
						      arity)))))))
		(predconst (make-predconst arity
					   new-tsubst
					   (predconst-to-index pred)
					   (predconst-to-name pred))))
	   (apply make-predicate-formula (cons predconst new-args))))
	((idpredconst-form? pred)
	 (let* ((name (idpredconst-to-name pred))
		(types (idpredconst-to-types pred))
		(cterms (idpredconst-to-cterms pred))
		(subst-types
		 (map (lambda (x) (type-substitute x tsubst)) types))
		(subst-cterms
		 (map (lambda (x) (cterm-substitute-aux
				   x tsubst subst psubst rename prename))
		      cterms))
		(idpredconst (make-idpredconst name subst-types subst-cterms)))
	   (apply make-predicate-formula (cons idpredconst new-args))))
	(else (myerror "formula-substitute-aux" "predicate expected" pred)))))
    ((imp)
     (make-imp
      (formula-substitute-aux
       (imp-form-to-premise formula) tsubst subst psubst rename prename)
      (formula-substitute-aux
       (imp-form-to-conclusion formula) tsubst subst psubst rename prename)))
    ((and)
     (make-and
      (formula-substitute-aux
       (and-form-to-left formula) tsubst subst psubst rename prename)
      (formula-substitute-aux
       (and-form-to-right formula) tsubst subst psubst rename prename)))
    ((tensor)
     (make-tensor
      (formula-substitute-aux
       (tensor-form-to-left formula) tsubst subst psubst rename prename)
      (formula-substitute-aux
       (tensor-form-to-right formula) tsubst subst psubst rename prename)))
    ((all ex allnc exnc exca excl)
     (let* ((quant (quant-form-to-quant formula))
	    (vars (map rename (quant-form-to-vars formula)))
	    (kernel (quant-form-to-kernel formula))
	    (substvars (map car subst))
	    (active-substvars
	     (intersection substvars (map rename (formula-to-free formula))))
	    (active-subst (list-transform-positive subst
			    (lambda (x) (member (car x) active-substvars))))
	    (active-terms (map cadr active-subst))
	    (psubstvars (map car psubst))
	    (active-psubstvars
	     (intersection psubstvars
			   (map prename (formula-to-pvars formula))))
	    (active-psubst (list-transform-positive psubst
			    (lambda (x) (member (car x) active-psubstvars))))
	    (active-cterms (map cadr active-psubst))
	    (free (apply union (append (map term-to-free active-terms)
				       (map cterm-to-free active-cterms))))
	    (new-vars (map (lambda (x) (if (member x free)
					   (var-to-new-var x)
					   x))
			   vars))
	    (new-subst (append (make-substitution-wrt var-term-equal?
				vars (map make-term-in-var-form new-vars))
			       active-subst)))
       (make-quant-formula
	quant new-vars (formula-substitute-aux
			kernel tsubst new-subst psubst rename prename))))
    (else (myerror "formula-substitute-aux" "formula expected" formula))))
			      
(define (formula-substitute-and-beta0-nf formula subst) ;simult. subst
  (if
   (null? subst)
   formula
   (case (tag formula)
     ((atom)
      (make-atomic-formula
       (term-substitute-and-beta0-nf (atom-form-to-kernel formula) subst)))
     ((predicate)
      (apply make-predicate-formula
	     (cons (predicate-form-to-predicate formula)
		   (map (lambda (arg)
			  (term-substitute-and-beta0-nf arg subst))
			(predicate-form-to-args formula)))))
     ((imp)
      (make-imp
       (formula-substitute-and-beta0-nf
	(imp-form-to-premise formula) subst)
       (formula-substitute-and-beta0-nf
	(imp-form-to-conclusion formula) subst)))
     ((and)
      (make-and (formula-substitute-and-beta0-nf
		 (and-form-to-left formula) subst)
                (formula-substitute-and-beta0-nf
		 (and-form-to-right formula) subst)))
     ((tensor)
      (make-tensor
       (formula-substitute-and-beta0-nf
	(tensor-form-to-left formula) subst)
       (formula-substitute-and-beta0-nf
	(tensor-form-to-right formula) subst)))
     ((all ex allnc exnc exca excl)
      (let* ((quant (quant-form-to-quant formula))
	     (vars (quant-form-to-vars formula))
	     (kernel (quant-form-to-kernel formula))
	     (substvars (map car subst))
	     (active-substvars
	      (intersection substvars (formula-to-free formula)))
	     (active-subst
	      (do ((l subst (cdr l))
		   (res '() (if (member (caar l) active-substvars)
				(cons (car l) res)
				res)))
		  ((null? l) (reverse res))))
	     (active-terms (map cadr active-subst))
	     (critical-vars (apply union (map term-to-free active-terms)))
	     (new-vars-and-renaming
	      (do ((l vars (cdr l))
		   (res
		    '(() ())
		    (if (member (car l) critical-vars)
			(let ((new-var (var-to-new-var (car l))))
			  (list (cons new-var (car res))
				(cons (list (car l) (make-term-in-var-form
						     new-var))
				      (cadr res))))
			(list (cons (car l) (car res)) (cadr res)))))
		  ((null? l) (list (reverse (car res))
				   (reverse (cadr res))))))
	     (new-vars (car new-vars-and-renaming))
	     (renaming (cadr new-vars-and-renaming)))
	(make-quant-formula
	 quant new-vars
	 (formula-substitute kernel (append renaming active-subst)))))
     (else (myerror "formula-substitute-and-beta0-nf" "formula expected"
		    formula)))))

(define (formula-subst-and-beta0-nf formula var term)
  (formula-substitute-and-beta0-nf
   formula (make-subst-wrt var-term-equal? var term)))

; (formula-gen-substitute formula gen-subst) substitutes simultaneously
; the left hand sides of the alist gen-subst at all occurrences in
; formula with no free variables captured by the corresponding right
; hand sides.  gen-subst is an alist associating terms to terms.
; Renaming takes place if and only if a free variable would become
; bound.

(define (formula-gen-substitute formula gen-subst)
  (car (formula-gen-substitute-and-newfreeoccs formula gen-subst)))

(define (formula-gen-substitute-and-newfreeoccs formula gen-subst)
  (if
   (null? gen-subst)
   (list formula '())
   (case (tag formula)
     ((atom)
      (let* ((pair (term-gen-substitute-and-newfreeoccs
		    (atom-form-to-kernel formula) gen-subst))
	     (new-kernel (car pair))
	     (new-free-occs (cadr pair)))
	(list (make-atomic-formula new-kernel) new-free-occs)))
     ((predicate)
      (let* ((pred (predicate-form-to-predicate formula))
	     (args (predicate-form-to-args formula))
	     (new-pairs (map (lambda (t) (term-gen-substitute-and-newfreeoccs
					  t gen-subst))
			     args))
	     (new-terms (map car new-pairs))
	     (new-free-occs (apply union (map cadr new-pairs))))
	(list (apply make-predicate-formula (cons pred new-terms))
	      new-free-occs)))
     ((imp)
      (let* ((pair1 (formula-gen-substitute-and-newfreeoccs
		     (imp-form-to-premise formula) gen-subst))
	     (pair2 (formula-gen-substitute-and-newfreeoccs
		     (imp-form-to-conclusion formula) gen-subst))
	     (new-premise (car pair1))
	     (new-conclusion (car pair2))
	     (new-free-occs (union (cadr pair1) (cadr pair2))))
	(list (make-imp new-premise new-conclusion) new-free-occs)))
     ((and)
      (let* ((pair1 (formula-gen-substitute-and-newfreeoccs
		     (and-form-to-left formula) gen-subst))
	     (pair2 (formula-gen-substitute-and-newfreeoccs
		     (and-form-to-right formula) gen-subst))
	     (new-left (car pair1))
	     (new-right (car pair2))
	     (new-free-occs (union (cadr pair1) (cadr pair2))))
	(list (make-and new-left new-right) new-free-occs)))
     ((tensor)
      (let* ((pair1 (formula-gen-substitute-and-newfreeoccs
		     (tensor-form-to-left formula) gen-subst))
	     (pair2 (formula-gen-substitute-and-newfreeoccs
		     (tensor-form-to-right formula) gen-subst))
	     (new-left (car pair1))
	     (new-right (car pair2))
	     (new-free-occs (union (cadr pair1) (cadr pair2))))
	(list (make-tensor new-left new-right) new-free-occs)))
     ((all ex allnc exnc exca excl)
      (let* ((quant (quant-form-to-quant formula))
	     (vars (quant-form-to-vars formula))
	     (kernel (quant-form-to-kernel formula))
	     ; substitute only those lhss without var from vars
	     (new-subst (do ((s gen-subst (cdr s))
			     (res '() (if (pair? (intersection
						  vars
						  (term-to-free (caar s))))
					  res
					  (cons (car s) res))))
			    ((null? s) (reverse res))))
	     (pair (formula-gen-substitute-and-newfreeoccs
		    kernel new-subst))
	     (new-kernel (car pair))
	     (new-free-occs (cadr pair))
	     (new-vars-and-renaming
	      (do ((l vars (cdr l))
		   (res
		    '(() ())
		    (if (member (car l) new-free-occs)
			(let ((new-var (var-to-new-var (car l))))
			  (list (cons new-var (car res))
				(cons (list (car l) (make-term-in-var-form
						     new-var))
				      (cadr res))))
			(list (cons (car l) (car res)) (cadr res)))))
		  ((null? l) (list (reverse (car res))
				   (reverse (cadr res))))))
	     (new-vars (car new-vars-and-renaming))
	     (renaming (cadr new-vars-and-renaming)))
	(if (null? renaming)
	    (list (make-quant-formula quant vars new-kernel) new-free-occs)
	    (list (make-quant-formula
		   quant new-vars (formula-substitute new-kernel renaming))
		  (set-minus new-free-occs vars)))))
     (else (myerror "formula-gen-substitute-and-newfreeoccs" "formula expected"
		    formula)))))

(define (formula-gen-subst formula term1 term2)
  (formula-gen-substitute formula (make-subst-wrt term=? term1 term2)))

; Substitution for cterms is done essentially as for formulas.

(define (cterm-substitute cterm topsubst)
  (let* ((tsubst-and-subst-and-psubst
	  (do ((l topsubst (cdr l))
	       (tsubst '() (if (tvar? (caar l))
			       (cons (car l) tsubst)
			       tsubst))
	       (subst '() (if (var? (caar l))
			       (cons (car l) subst)
			       subst))
	       (psubst '() (if (pvar-form? (caar l))
			       (cons (car l) psubst)
			       psubst)))
	      ((null? l)
	       (list (reverse tsubst) (reverse subst) (reverse psubst)))))
	 (tsubst (car tsubst-and-subst-and-psubst))
	 (subst (cadr tsubst-and-subst-and-psubst))
	 (psubst (caddr tsubst-and-subst-and-psubst))
	 (rename (make-rename tsubst))
	 (prename (make-prename tsubst)))
    (cterm-substitute-aux cterm tsubst subst psubst rename prename)))

(define (cterm-subst cterm arg val)
  (let ((equality?
	 (cond
	  ((and (tvar? arg) (type? val)) equal?)
	  ((and (var-form? arg) (term-form? val)) var-term-equal?)
	  ((and (pvar? arg) (cterm-form? val)) pvar-cterm-equal?)
	  (else (myerror "cterm-subst" "unexpected arg" arg "and val" val)))))
    (cterm-substitute cterm (make-subst-wrt equality? arg val))))

(define (cterm-substitute-aux cterm tsubst subst psubst rename prename)
  (let* ((vars (cterm-to-vars cterm))
	 (l (length vars))
	 (for (cterm-to-formula cterm))
	 (aux-formula (apply mk-all (append vars (list for))))
	 (subst-aux-formula (formula-substitute-aux
			     aux-formula tsubst subst psubst rename prename))
	 (aux-vars-and-kernel
	  (all-form-to-vars-and-final-kernel subst-aux-formula))
	 (aux-vars (car aux-vars-and-kernel))
	 (kernel (cadr aux-vars-and-kernel))
	 (new-vars (list-head aux-vars l))
	 (rest-vars (list-tail aux-vars l))
	 (subst-for (apply mk-all (append rest-vars (list kernel)))))
    (apply make-cterm (append new-vars (list subst-for)))))
			      
; Display functions for predicate substitutions:

(define (display-p-substitution psubst)
  (display-comment "Predicate substitution:") (newline)
  (for-each (lambda (x)
	      (let* ((pvar (car x))
		     (cterm (cadr x)))
		(if (pvar-form? pvar)
		    (display-comment (pvar-to-string pvar))
		    (myerror "display-p-substitution:"
			     "predicate variable expected" pvar))
		(display tab)
		(display "->")
		(display tab)
		(if (cterm-form? cterm)
		    (display (cterm-to-string cterm))
		    (myerror "display-p-substitution" "cterm expected" cterm))
		(newline)))
	    psubst))

(define (p-substitution-to-string psubst)
  (do ((l (reverse psubst) (cdr l))
       (res ""
	    (let* ((x (car l))
		   (pvar
		    (if (and (list? x) (= 2 (length x)))
			(car x)
			(myerror
			 "p-substitution-to-string" "subst pair expected" x)))
		   (cterm (cadr x)))
	      (string-append
	       (if (pvar? pvar)
		   (pvar-to-string pvar)
		   (myerror "p-substitution-to-string" "pvar expected" pvar))
	       " -> "
	       (if (cterm-form? cterm)
		   (cterm-to-string cterm)
		   (myerror "p-substitution-to-string" "cterm expected" cterm))
	       (if (string=? "" res) "" ", ")
	       res))))
      ((null? l) res)))

(define (display-substitutions top-subst)
  (if (not top-subst) #f)
  (if (not (and (list? top-subst)
		(map and-op (map (lambda (item)
				   (and (list? item)
					(= 2 (length item))))
				 top-subst))))
      (myerror "substitution expected" top-subst))
  (let ((tsubst (list-transform-positive top-subst
		  (lambda (x) (tvar-form? (car x)))))
	(subst (list-transform-positive top-subst
		 (lambda (x) (var-form? (car x)))))
	(psubst (list-transform-positive top-subst
		  (lambda (x) (pvar-form? (car x))))))
    (if (pair? tsubst) (display-t-substitution tsubst))
    (if (pair? subst) (display-substitution subst))
    (if (pair? psubst) (display-p-substitution psubst))))

