/*======================================================================*\
|*		Editor mined						*|
|*		auxiliary functions					*|
\*======================================================================*/

#include "mined.h"
#include "io.h"
#include "textfile.h"	/* for panicwrite (), unlock_file () */

#include <errno.h>	/* ENOENT */
#include <signal.h>
#include <sys/stat.h>

#ifndef pc
#ifndef vms
#include <pwd.h>	/* for getpwuid () */
#endif
#endif

#define use_debuglog
#ifdef __TURBOC__
#undef use_debuglog
#endif
#ifdef vms
#undef use_debuglog
#endif

#ifdef use_debuglog
#include <time.h>	/* for localtime (), strftime () */
#include <sys/time.h>	/* for gettimeofday () */
#endif


/*======================================================================*\
|*			Debug stuff					*|
\*======================================================================*/

/**
   Write debug log entry.
 */
void
debuglog (tag, l1, l2)
  char * tag;
  char * l1;
  char * l2;
{
#ifdef use_debuglog
  if (debug_mined) {
    char logfn [maxFILENAMElen];
    FILE * log;
    build_string (logfn, "%s/.minedlog", gethomedir ());
    log = fopen (logfn, "a");
    if (log) {
	char timbuf [99];
	struct timeval now;
	gettimeofday (& now, 0);
	strftime (timbuf, sizeof (timbuf), "%m-%d %H:%M:%S", localtime (& now.tv_sec));
	fprintf (log, "[%d@%s.%03d] %s: <%s> <%s>\n",
				getpid (), 
				timbuf, now.tv_usec / 1000,
				tag,
				unnull (l1), unnull (l2)
		);
	fclose (log);
    }
  }
#endif
}


/*======================================================================*\
|*			Auxiliary routines				*|
\*======================================================================*/

/*
 * Unnull () changes a NULL string pointer into an empty string pointer.
 */
char *
unnull (s)
  char * s;
{
  if (s == NIL_PTR) {
	return "";
  } else {
	return s;
  }
}

/*
 * Output an (unsigned) long in a 10 digit field without leading zeros.
 * It returns a pointer to the first digit in the buffer.
 */
char *
num_out (number, radix)
  long number;
  long radix;
{
  static char num_buf [11];		/* Buffer to build number */
  register long digit;			/* Next digit of number */
  register long pow;			/* Highest radix power of long */
  FLAG digit_seen = False;
  int i;
  int maxdigs;
  char * bufp = num_buf;

  if (radix == 16) {
	pow = 268435456L;
	maxdigs = 8;
  } else {
	pow = 1000000000L;
	maxdigs = 10;
	if (number < 0) {
		* bufp ++ = '-';
		number = - number;
	}
  }

  for (i = 0; i < maxdigs; i ++) {
	digit = number / pow;		/* Get next digit */
	if (digit == 0L && digit_seen == False && i != 9) {
	} else {
		if (digit > 9) {
			* bufp ++ = 'A' + (char) (digit - 10);
		} else {
			* bufp ++ = '0' + (char) digit;
		}
		number -= digit * pow;	/* Erase digit */
		digit_seen = True;
	}
	pow /= radix;			/* Get next digit */
  }
  * bufp = '\0';
  return num_buf;
}

char *
dec_out (number)
  long number;
{
  return num_out (number, 10);
}

/*
 * scan_int () converts a string into a natural number
 * returns the character pointer behind the last digit
 */
char *
scan_int (str, nump)
  char * str;
  int * nump;
{
  register char * chpoi = str;
  int negative = False;

  while (* chpoi == ' ') {
	chpoi ++;
  }
  if (* chpoi == '-') {
	negative = True;
	chpoi ++;
  }
  if (* chpoi >= '0' && * chpoi <= '9') {
	* nump = 0;
  } else {
	return str;
  }
  while (* chpoi >= '0' && * chpoi <= '9' && quit == False) {
	* nump *= 10;
	* nump += * chpoi - '0';
	chpoi ++;
  }
  if (negative) {
	* nump = - * nump;
  }
  return chpoi;
}

/*
 * copy_string () copies the string 'from' into the string 'to' which 
   must be long enough
 */
void
copy_string (to, from)
  register char * to;
  register char * from;
{
  while ((* to ++ = * from ++) != '\0')
	{}
}

/*
 * length_of () returns the number of bytes in the string 'string'
 * excluding the '\0'.
 */
int
length_of (string)
  register char * string;
{
  register int count = 0;

  if (string != NIL_PTR) {
	while (* string ++ != '\0') {
		count ++;
	}
  }
  return count;
}

char *
dupstr (s)
  char * s;
{
	char * s1 = alloc (strlen (s) + 1);
	copy_string (s1, s);
	return s1;
}

char *
nextutf8 (s)
  char * s;
{
  int follow = UTF8_len (* s) - 1;
  s ++;
  while (follow > 0 && (* s & 0xC0) == 0x80) {
	s ++;
	follow --;
  }
  return s;
}

int
stringcompare_ (s1, s2)
  char * s1;
  char * s2;
{
  if (* s1) {
	if (* s2) {
		unsigned long c1 = case_convert (utf8value (s1), -1);
		unsigned long c2 = case_convert (utf8value (s2), -1);
		if (c1 < c2) {
			return -1;
		} else if (c1 > c2) {
			return 1;
		} else {
			return stringcompare_ (nextutf8 (s1), nextutf8 (s2));
		}
	} else {
		return 1;
	}
  } else {
	if (* s2) {
		return -1;
	} else {
		return 0;
	}
  }
}

int
stringprefix_ (s1, s2)
  char * s1;
  char * s2;
{
  if (* s1) {
	if (* s2) {
		unsigned long c1 = case_convert (utf8value (s1), -1);
		unsigned long c2 = case_convert (utf8value (s2), -1);
		if (c1 == c2) {
			return stringprefix_ (nextutf8 (s1), nextutf8 (s2));
		} else {
			return 0;
		}
	} else {
		return 0;
	}
  } else {
	return 1;
  }
}


/*======================================================================*\
|*			Environment stuff				*|
\*======================================================================*/

/**
   Return the value of an environment variable, or NULL.
 */
char *
envvar (name)
  char * name;
{
  return getenv (name);
}

/**
   Return the string value of an environment variable.
   Do not return NULL.
 */
char *
envstr (name)
  char * name;
{
  char * val = envvar (name);
  if (val) {
	return val;
  } else {
	return "";
  }
}


/**
   Get a user name
 */
char *
getusername ()
{
  char * username;

#ifdef unix
  username = envvar ("USER");
  if (! username) {
	username = dec_out (geteuid ());
  }
#endif
#ifdef vms
  username = envvar ("USER");
  if (! username) {
	username = dec_out (geteuid ());
  }
#endif
#ifdef msdos
  if (is_Windows ()) {
	username = envvar ("USERNAME");
	if (! username) {
		username = "000";
	}
  } else {
	username = "000";
  }
#endif

  return username;
}


/**
   Return current working directory
 */
char *
get_cwd (dirbuf)
  char * dirbuf;
{
  char * res;
#ifdef unix
# ifdef sysV
  res = getcwd (dirbuf, maxFILENAMElen);
# else
  char bsdbuf [4096];
  res = getwd (bsdbuf);
  if (res) {
	dirbuf [0] = '\0';
	strncat (dirbuf, bsdbuf, maxFILENAMElen - 1);
  }
# endif
#else
  res = getcwd (dirbuf, maxFILENAMElen);
#endif

  if (res) {
	return res;
  } else {
	return "";
  }
}

static char * _homedir = NIL_PTR;

void
sethomedir (hd)
  char * hd;
{
  if (hd && * hd) {
	_homedir = hd;
#ifdef __CYGWIN__
	if (streq (hd, "%HOME%")) {
		char * regkey = "/proc/registry/HKEY_CURRENT_USER/Software/Microsoft/Windows/CurrentVersion/Explorer/Shell Folders/Personal";
		int regfd = open (regkey, O_RDONLY, 0);
		_homedir = envvar ("USERPROFILE");
		if (regfd >= 0) {
			static char buf [maxFILENAMElen];
			int rd = read (regfd, buf, maxFILENAMElen - 1);
			if (rd > 0) {
				buf [rd] = '\0';
				_homedir = buf;
			}
			(void) close (regfd);
		}
	}
#endif
  }
}

char *
gethomedir ()
{
  if (! _homedir) {
    _homedir = envvar ("HOME");
    if (! _homedir) {
#ifdef pc
#ifdef msdos
	if (is_Windows ()) {
		_homedir = envvar ("USERPROFILE");
	}
#else
#ifdef __CYGWIN__
	sethomedir ("%HOME%");
#else
	_homedir = envvar ("USERPROFILE");
#endif
#endif
	if (! _homedir) {
		_homedir = "/";
	}
#else
#ifdef vms
	_homedir = "SYS$LOGIN";
#else
	struct passwd * pwd = getpwuid (geteuid ());
	if (pwd) {
		_homedir = pwd->pw_dir;
	} else {
		_homedir = "/";
	}
#endif
#endif
    }
  }
  return _homedir;
}


/*======================================================================*\
|*			Process stuff					*|
\*======================================================================*/

/**
   System call with screen mode adjustment.
   Optional message before call.
   Optional delay after call.
 */
int
systemcall (msg, cmd, delay)
  char * msg;
  char * cmd;
  int delay;
{
  int res;

  raw_mode (False);
  if (msg != NIL_PTR) {
	unidisp_on ();
	putstring ("mined: ");
	unidisp_off ();
	reverse_on ();
	putstring (msg);
	reverse_off ();
	putstring ("\r\n");
	flush ();
  }

  res = system (cmd);

  if (delay) {
	sleep (delay);
  }
  raw_mode (True);
  return res;
}

#ifndef msdos

/**
   Program call with screen mode adjustment.
   Optional message before call.
   Optional delay after call.
 */
int
progcallpp (msg, delay, binprepath, dir, prog, p1, p2, p3, p4)
  char * msg;
  int delay;
  char * binprepath [];
  char * dir;
  char * prog;
  char * p1;
  char * p2;
  char * p3;
  char * p4;
{
  int pid;
  int res;
  int status;

  if (delay >= 0) {
	raw_mode (False);
  }
  if (msg != NIL_PTR) {
	unidisp_on ();
	putstring ("mined: ");
	unidisp_off ();
	reverse_on ();
	putstring (msg);
	reverse_off ();
	putstring ("\r\n");
	flush ();
  }

#ifdef FORK
  pid = fork ();
#else
  pid = vfork ();
#endif

  switch (pid) {
	case 0:		/* child process */
	    {
		int pathi = 0;
		if (dir) {
			chdir (dir);
		}
		while (binprepath && binprepath [pathi]) {
			char pathbuf [maxFILENAMElen];
			int cmpi = 0;
			while (cmpi < pathi) {
				if (streq (binprepath [cmpi], binprepath [pathi])) {
					break;
				}
				cmpi ++;
			}
			if (cmpi == pathi) {
				build_string (pathbuf, "%s/bin/%s", binprepath [pathi], prog);
				execl (pathbuf, prog, p1, p2, p3, p4, NIL_PTR);
			}
			pathi ++;
		}
		execlp (prog, prog, p1, p2, p3, p4, NIL_PTR);
		_exit (127);	/* Exit with 127 */
	    }
	    /* NOTREACHED */
	case -1:	/* fork error */
		res = -1;
	default:	/* parent after successful fork */
		do {
			res = wait (& status);
		} while (res != -1 && res != pid);
		/* ignore geterrno (); */

		if (res == -1) {
			res = -2;
		} else {
			res = status >> 8;
		}
  }

  if (delay > 0) {
	sleep (delay);
  }
  if (delay >= 0) {
	raw_mode (True);
  }
  return res;
}

#endif


/*======================================================================*\
|*			System stuff					*|
\*======================================================================*/

#ifdef msdos

int
is_Windows ()
{
#if defined(__TURBOC__) || defined(__EMX__)
  return 0;
#else
#ifdef unreliable_windir_test
  /* for obscure reasons, windir is not seen in djgpp programs */
  return envvar ("windir") || /* mangled by cygwin */ envvar ("WINDIR");
#else
  /* try some heuristics */
  return strisprefix ("Windows", envstr ("OS")) ||
	 (envvar ("USERPROFILE") && envvar ("APPDATA"));
#endif
#endif
}

#endif


/*======================================================================*\
|*			File operations					*|
\*======================================================================*/

/**
   Return base file name of path name
 */
char *
getbasename (s)
  char * s;
{
  char * b;
#ifdef pc
  b = strrchr (s, '\\');
  if (b) {
	s = b + 1;
  }
#endif
  b = strrchr (s, '/');
  if (b == NIL_PTR) {
	return s;
  } else {
	b ++;
	return b;
  }
}

/**
   Check whether path name is absolute
 */
int
is_absolute_path (fn)
  char * fn;
{
#ifdef pc
  if (fn [0] && fn [1] == ':') {
	fn += 2;	/* skip drive */
  }
#endif
  return
#ifdef pc
	fn [0] == '\\' ||
#endif
	fn [0] == '/';
}


/*
 * Delete file.
 */
int
delete_file (file)
  char * file;
{
#ifdef unix
  return unlink (file);
#endif
#ifdef msdos
  return unlink (file);
#endif
#ifdef vms
  return delete (file);
#endif
}

/**
   Copy file.
   Return NOT_VALID if source file does not exist.
   Return False if it fails otherwise.
 */
FLAG
copyfile (from_file, to_file)
  char * from_file;
  char * to_file;
{
  int from_fd;
  int to_fd;
  int cnt;	/* count check for read/write */
  int ret = True;
  struct stat fstat_buf;
  PROT copyprot;

  if ((from_fd = open (from_file, O_RDONLY | O_BINARY, 0)) < 0) {
	if (geterrno () == ENOENT) {
		return NOT_VALID;
	} else {
		return False;
	}
  }
  if (fstat (from_fd, & fstat_buf) == 0) {
	copyprot = fstat_buf.st_mode;
  } else {
	copyprot = bufprot;
  }

  if ((to_fd = open (to_file, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, copyprot)) < 0) {
	(void) close (from_fd);
	return False;
  }
#ifndef msdos
#ifdef vms
  (void) chmod (to_file, copyprot);
#else
  (void) fchmod (to_fd, copyprot);
#endif
#endif

  while ((cnt = read (from_fd, text_buffer, sizeof (text_buffer))) > 0) {
	if (write (to_fd, text_buffer, (unsigned int) cnt) != cnt) {
		ret = False;
		break;
	}
  }
  if (cnt < 0) {
	ret = False;
  }

  (void) close (from_fd);
  if (close (to_fd) < 0) {
	ret = False;
  }
  return ret;
}


/*======================================================================*\
|*			Memory allocation				*|
\*======================================================================*/

#ifdef msdos
# ifdef __TURBOC__
#include <alloc.h>
#define allocate farmalloc
#define freemem farfree
# else
extern void * malloc ();
extern void free ();
#define allocate malloc
#define freemem free
# endif
#else
extern void * malloc ();
extern void free ();
#define allocate malloc
#define freemem free
#endif

#define dont_debug_out_of_memory
#ifdef debug_out_of_memory
static int allocount = 0;
#endif

void *
alloc (bytes)
  int bytes;
{
#ifdef debug_out_of_memory
  allocount ++;
  if (allocount == 500) {
	return NIL_PTR;
  }
#endif
  return allocate ((unsigned) bytes);
/*
  char * p;

  if ((p = allocate ((unsigned) bytes)) == NIL_PTR) {
	panic ("Out of memory", NIL_PTR);
  }
  return p;
*/
}

void
free_space (p)
  void * p;
{
  freemem (p);
}

/*
 * free header list
 */

#define pointersize	sizeof (void *)
#define blocksizeof(typ) ((sizeof (typ) + pointersize - 1) / pointersize * pointersize)

LINE * free_header_list = NIL_LINE;

static
void
alloc_headerblock (n)
  int n;
{
  LINE * new_header;
  LINE * new_list;
  int i = 0;

  new_list = alloc (n * blocksizeof (LINE));
  if (new_list == NIL_LINE) {
	free_header_list = NIL_LINE;
  } else {
	while (i < n) {
		new_header = (LINE *) ((long) new_list + i * blocksizeof (LINE));
		new_header->next = free_header_list;
		free_header_list = new_header;
		i ++;
	}
  }
}

LINE *
alloc_header ()
{
/*  return alloc (sizeof (LINE)); */
  LINE * new_header;

  if (free_header_list == NIL_LINE) {
	alloc_headerblock (64);
	if (free_header_list == NIL_LINE) {
		alloc_headerblock (16);
		if (free_header_list == NIL_LINE) {
			alloc_headerblock (4);
			if (free_header_list == NIL_LINE) {
				alloc_headerblock (1);
				if (free_header_list == NIL_LINE) {
					return NIL_LINE;
				}
			}
		}
	}
  }
  new_header = free_header_list;
  free_header_list = free_header_list->next;
  return new_header;
}

void
free_header (hp)
  LINE * hp;
{
/*  freemem (hp); */
  hp->next = free_header_list;
  free_header_list = hp;
}


/*======================================================================*\
|*			Interrupt condition handling			*|
\*======================================================================*/

#define dont_debug_intr

static int panic_level = 0;	/* To adjust error handling to situation */

/*
 * Panic () is called with a mined error msg and an optional system error msg.
 * It is called when something unrecoverable has happened.
 * It writes the message to the terminal, resets the tty and exits.
 * Ask the user if he wants to save his file.
 */
static
void
panic_msg (msg)
  char * msg;
{
  if (isscreenmode) {
	/* don't use status_msg here, msg may contain filename characters */
	status_line (msg, NIL_PTR);
	sleep (2);
  } else {
	(void) printf ("%s\n", msg);
  }
}

static
void
panicking (message, err, signum)
  register char * message;
  register char * err;
  register int signum;
{
  panic_level ++;
#ifdef debug_intr
  printf ("panicking (%s), panic_level ++: %d\n", message, panic_level);
#endif

  if (panic_level <= 2) {
	if (loading == False && modified) {
		if (panicwrite () == ERRORS) {
			sleep (2);
			build_string (text_buffer, "Error writing panic file %s", panic_file);
		} else {
			build_string (text_buffer, "Panic file %s written", panic_file);
		}
		ring_bell ();
		panic_msg (text_buffer);
	}
	if (signum != 0) {
		build_string (text_buffer, message, signum);
	} else if (err == NIL_PTR) {
		build_string (text_buffer, "%s", message);
	} else {
		build_string (text_buffer, "%s (%s)", message, err);
	}
	panic_msg (text_buffer);

	/* "normal" panic handling: */
	if (loading == False) {
		QUED ();	/* Try to save the file and quit */
		/* QUED returned: something wrong */
		if (tty_closed == False) {
			sleep (2);
			panic_msg ("Aborted writing file in panic mode - trying to continue");
			panic_level --;
#ifdef debug_intr
			printf ("panic_level --: %d\n", panic_level);
#endif
		}
		return;
	}
  }

  if (panic_level <= 3) {
	if (isscreenmode) {
		set_cursor (0, YMAX);
		putchar ('\n');
#ifdef unix
		clear_window_title ();
#endif
		raw_mode (False);
	}
	/* Remove file lock (if any) */
	unlock_file ();
	delete_yank_files ();
  }
#ifdef debug_intr
  printf ("exiting\n");
#endif
  exit (1) /* abort () sends IOT which would again be caught */;
}

void
panic (message, err)
  register char * message;
  register char * err;
{
  panicking (message, err, 0);
}

void
panicio (message, err)
  register char * message;
  register char * err;
{
/* Should panic_level already be increased here ? */
  panic (message, err);
}


void
handle_interrupt (signum)
  int signum;
{
#ifdef debug_intr
  printf ("handle_interrupt %d @ panic level %d\n", signum, panic_level);
#endif
  if (signum == SIGTERM && panic_level == 0) {
	panic_level ++;
#ifdef debug_intr
	printf ("handle_interrupt exiting, panic_level ++: %d\n", panic_level);
#endif
	EXMINED ();
	panic_level --;
#ifdef debug_intr
	printf ("handle_interrupt exiting, panic_level --: %d\n", panic_level);
#endif
#ifdef SIGHUP
  } else if (signum == SIGHUP && panic_level == 0) {
	panic_level ++;
#ifdef debug_intr
	printf ("handle_interrupt quitting, panic_level ++: %d\n", panic_level);
#endif
	QUED ();
	hup = True;
	panic_level --;
#ifdef debug_intr
	printf ("handle_interrupt quitting, panic_level --: %d\n", panic_level);
#endif
#endif
  } else {
	panicking ("External signal %d caught - terminating", NIL_PTR, signum);
  }
}


/*======================================================================*\
|*				End					*|
\*======================================================================*/
