/*
 * This file is part of the MicroPython project, http://micropython.org/
 *
 * The MIT License (MIT)
 *
 * Copyright (c) 2013, 2014 Damien P. George
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef MICROPY_INCLUDED_PY_RUNTIME0_H
#define MICROPY_INCLUDED_PY_RUNTIME0_H

// The first four must fit in 8 bits, see emitbc.c
// The remaining must fit in 16 bits, see scope.h
#define MP_SCOPE_FLAG_VARARGS      (0x01)
#define MP_SCOPE_FLAG_VARKEYWORDS  (0x02)
#define MP_SCOPE_FLAG_GENERATOR    (0x04)
#define MP_SCOPE_FLAG_DEFKWARGS    (0x08)
#define MP_SCOPE_FLAG_REFGLOBALS   (0x10) // used only if native emitter enabled
#define MP_SCOPE_FLAG_HASCONSTS    (0x20) // used only if native emitter enabled
#define MP_SCOPE_FLAG_VIPERRET_POS    (6) // 3 bits used for viper return type

// types for native (viper) function signature
#define MP_NATIVE_TYPE_OBJ  (0x00)
#define MP_NATIVE_TYPE_BOOL (0x01)
#define MP_NATIVE_TYPE_INT  (0x02)
#define MP_NATIVE_TYPE_UINT (0x03)
#define MP_NATIVE_TYPE_PTR  (0x04)
#define MP_NATIVE_TYPE_PTR8 (0x05)
#define MP_NATIVE_TYPE_PTR16 (0x06)
#define MP_NATIVE_TYPE_PTR32 (0x07)

typedef enum {
    // These ops may appear in the bytecode. Changing this group
    // in any way requires changing the bytecode version.
    MP_UNARY_OP_POSITIVE,
    MP_UNARY_OP_NEGATIVE,
    MP_UNARY_OP_INVERT,
    MP_UNARY_OP_NOT,

    // Following ops cannot appear in the bytecode
    MP_UNARY_OP_NUM_BYTECODE,

    MP_UNARY_OP_BOOL = MP_UNARY_OP_NUM_BYTECODE, // __bool__
    MP_UNARY_OP_LEN, // __len__
    MP_UNARY_OP_HASH, // __hash__; must return a small int
    MP_UNARY_OP_ABS, // __abs__
    MP_UNARY_OP_INT, // __int__
    MP_UNARY_OP_SIZEOF, // for sys.getsizeof()

    MP_UNARY_OP_NUM_RUNTIME,
} mp_unary_op_t;

// Note: the first 9+12+12 of these are used in bytecode and changing
// them requires changing the bytecode version.
typedef enum {
    // 9 relational operations, should return a bool
    MP_BINARY_OP_LESS,
    MP_BINARY_OP_MORE,
    MP_BINARY_OP_EQUAL,
    MP_BINARY_OP_LESS_EQUAL,
    MP_BINARY_OP_MORE_EQUAL,
    MP_BINARY_OP_NOT_EQUAL,
    MP_BINARY_OP_IN,
    MP_BINARY_OP_IS,
    MP_BINARY_OP_EXCEPTION_MATCH,

    // 12 inplace arithmetic operations
    MP_BINARY_OP_INPLACE_OR,
    MP_BINARY_OP_INPLACE_XOR,
    MP_BINARY_OP_INPLACE_AND,
    MP_BINARY_OP_INPLACE_LSHIFT,
    MP_BINARY_OP_INPLACE_RSHIFT,
    MP_BINARY_OP_INPLACE_ADD,
    MP_BINARY_OP_INPLACE_SUBTRACT,
    MP_BINARY_OP_INPLACE_MULTIPLY,
    MP_BINARY_OP_INPLACE_FLOOR_DIVIDE,
    MP_BINARY_OP_INPLACE_TRUE_DIVIDE,
    MP_BINARY_OP_INPLACE_MODULO,
    MP_BINARY_OP_INPLACE_POWER,

    // 12 normal arithmetic operations
    MP_BINARY_OP_OR,
    MP_BINARY_OP_XOR,
    MP_BINARY_OP_AND,
    MP_BINARY_OP_LSHIFT,
    MP_BINARY_OP_RSHIFT,
    MP_BINARY_OP_ADD,
    MP_BINARY_OP_SUBTRACT,
    MP_BINARY_OP_MULTIPLY,
    MP_BINARY_OP_FLOOR_DIVIDE,
    MP_BINARY_OP_TRUE_DIVIDE,
    MP_BINARY_OP_MODULO,
    MP_BINARY_OP_POWER,

    // Operations below this line don't appear in bytecode, they
    // just identify special methods.
    MP_BINARY_OP_NUM_BYTECODE,

    // MP_BINARY_OP_REVERSE_* must follow immediately after MP_BINARY_OP_*
#if MICROPY_PY_REVERSE_SPECIAL_METHODS
    MP_BINARY_OP_REVERSE_OR = MP_BINARY_OP_NUM_BYTECODE,
    MP_BINARY_OP_REVERSE_XOR,
    MP_BINARY_OP_REVERSE_AND,
    MP_BINARY_OP_REVERSE_LSHIFT,
    MP_BINARY_OP_REVERSE_RSHIFT,
    MP_BINARY_OP_REVERSE_ADD,
    MP_BINARY_OP_REVERSE_SUBTRACT,
    MP_BINARY_OP_REVERSE_MULTIPLY,
    MP_BINARY_OP_REVERSE_FLOOR_DIVIDE,
    MP_BINARY_OP_REVERSE_TRUE_DIVIDE,
    MP_BINARY_OP_REVERSE_MODULO,
    MP_BINARY_OP_REVERSE_POWER,
#endif

    // This is not emitted by the compiler but is supported by the runtime
    MP_BINARY_OP_DIVMOD
        #if !MICROPY_PY_REVERSE_SPECIAL_METHODS
        = MP_BINARY_OP_NUM_BYTECODE
        #endif
    ,

    // The runtime will convert MP_BINARY_OP_IN to this operator with swapped args.
    // A type should implement this containment operator instead of MP_BINARY_OP_IN.
    MP_BINARY_OP_CONTAINS,

    MP_BINARY_OP_NUM_RUNTIME,

    // These 2 are not supported by the runtime and must be synthesised by the emitter
    MP_BINARY_OP_NOT_IN,
    MP_BINARY_OP_IS_NOT,
} mp_binary_op_t;

typedef enum {
    MP_F_CONST_NONE_OBJ = 0,
    MP_F_CONST_FALSE_OBJ,
    MP_F_CONST_TRUE_OBJ,
    MP_F_CONVERT_OBJ_TO_NATIVE,
    MP_F_CONVERT_NATIVE_TO_OBJ,
    MP_F_NATIVE_SWAP_GLOBALS,
    MP_F_LOAD_NAME,
    MP_F_LOAD_GLOBAL,
    MP_F_LOAD_BUILD_CLASS,
    MP_F_LOAD_ATTR,
    MP_F_LOAD_METHOD,
    MP_F_LOAD_SUPER_METHOD,
    MP_F_STORE_NAME,
    MP_F_STORE_GLOBAL,
    MP_F_STORE_ATTR,
    MP_F_OBJ_SUBSCR,
    MP_F_OBJ_IS_TRUE,
    MP_F_UNARY_OP,
    MP_F_BINARY_OP,
    MP_F_BUILD_TUPLE,
    MP_F_BUILD_LIST,
    MP_F_LIST_APPEND,
    MP_F_BUILD_MAP,
    MP_F_STORE_MAP,
#if MICROPY_PY_BUILTINS_SET
    MP_F_STORE_SET,
    MP_F_BUILD_SET,
#endif
    MP_F_MAKE_FUNCTION_FROM_RAW_CODE,
    MP_F_NATIVE_CALL_FUNCTION_N_KW,
    MP_F_CALL_METHOD_N_KW,
    MP_F_CALL_METHOD_N_KW_VAR,
    MP_F_NATIVE_GETITER,
    MP_F_NATIVE_ITERNEXT,
    MP_F_NLR_PUSH,
    MP_F_NLR_POP,
    MP_F_NATIVE_RAISE,
    MP_F_IMPORT_NAME,
    MP_F_IMPORT_FROM,
    MP_F_IMPORT_ALL,
#if MICROPY_PY_BUILTINS_SLICE
    MP_F_NEW_SLICE,
#endif
    MP_F_UNPACK_SEQUENCE,
    MP_F_UNPACK_EX,
    MP_F_DELETE_NAME,
    MP_F_DELETE_GLOBAL,
    MP_F_NEW_CELL,
    MP_F_MAKE_CLOSURE_FROM_RAW_CODE,
    MP_F_ARG_CHECK_NUM_SIG,
    MP_F_SETUP_CODE_STATE,
    MP_F_SMALL_INT_FLOOR_DIVIDE,
    MP_F_SMALL_INT_MODULO,
    MP_F_NATIVE_YIELD_FROM,
    MP_F_NUMBER_OF,
} mp_fun_kind_t;

extern const void *const mp_fun_table[MP_F_NUMBER_OF];

#endif // MICROPY_INCLUDED_PY_RUNTIME0_H
