#
# partitionedfs.py: partitioned files system class, extends fs.py
#
# Copyright 2007-2008, Red Hat  Inc.
# Copyright 2008, Daniel P. Berrange
# Copyright 2008,  David P. Huff
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import os
import os.path
import glob
import shutil
import subprocess
import logging

from mic.imgcreate.errors import *
from mic.imgcreate.fs import *

class PartitionedMount(Mount):
    def __init__(self, disks, mountdir, skipformat = False):
        Mount.__init__(self, mountdir)
        self.disks = {}
        for name in disks.keys():
            self.disks[name] = { 'disk': disks[name],  # Disk object
                                 'mapped': False, # True if kpartx mapping exists
                                 'numpart': 0, # Number of allocate partitions
                                 'partitions': [], # indexes to self.partitions
                                 # Partitions with part num higher than 3 will 
                                 # be put inside extended partition.
                                 'extended': 0, # Size of extended partition
                                 # Sector 0 is used by the MBR and can't be used
                                 # as the start, so setting offset to 1.
                                 'offset': 1 } # Offset of next partition (in sectors)

        self.partitions = []
        self.mapped = False
        self.mountOrder = []
        self.unmountOrder = []
        self.parted=find_binary_path("parted")
        self.kpartx=find_binary_path("kpartx")
        self.mkswap=find_binary_path("mkswap")
        self.skipformat = skipformat
        # Size of a sector used in calculations
        self.sector_size = 512

    def add_partition(self, size, disk, mountpoint, fstype = None, boot = False):
        # Converting M to s for parted
        size = size * 1024 * 1024 / self.sector_size
        self.partitions.append({'size': size, # In sectors
                                'mountpoint': mountpoint, # Mount relative to chroot
                                'fstype': fstype, # Filesystem type
                                'disk': disk, # physical disk name holding partition
                                'device': None, # kpartx device node for partition
                                'mount': None, # Mount object
                                'num': None, # Partition number
                                'boot': boot}) # Bootable flag

    def __create_part_to_image(self,device, parttype, fstype, start, size):
        # Start is included to the size so we need to substract one from the end.
        end = start+size-1
        logging.debug("Added '%s' part at %d of size %d" % (parttype,start,end))
        part_cmd = [self.parted, "-s", device, "unit", "s", "mkpart", parttype]
        if fstype:
            part_cmd.extend([fstype])
        part_cmd.extend(["%d" % start, "%d" % end])
        logging.debug(part_cmd)
        p1 = subprocess.Popen(part_cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        (out,err) = p1.communicate()
        logging.debug(out)
        return p1

    def __format_disks(self):
        logging.debug("Assigning partitions to disks")
        
        mbr_sector_skipped = False
        
        for n in range(len(self.partitions)):
            p = self.partitions[n]

            if not self.disks.has_key(p['disk']):
                raise MountError("No disk %s for partition %s" % (p['disk'], p['mountpoint']))
            
            if not mbr_sector_skipped:
                # This hack is used to remove one sector from the first partition,
                # that is the used to the MBR.
                p['size'] -= 1
                mbr_sector_skipped = True

            d = self.disks[p['disk']]
            d['numpart'] += 1
            if d['numpart'] > 3:
                # Increase allocation of extended partition to hold this partition
                d['extended'] += p['size']
                p['type'] = 'logical'
                p['num'] = d['numpart'] + 1
            else:
                p['type'] = 'primary'
                p['num'] = d['numpart']

            p['start'] = d['offset']
            d['offset'] += p['size']
            d['partitions'].append(n)
            logging.debug("Assigned %s to %s%d at %d at size %d" % (p['mountpoint'], p['disk'], p['num'], p['start'], p['size']))

        if self.skipformat:
            logging.debug("Skipping disk format, because skipformat flag is set.")
            return
            
        for dev in self.disks.keys():
            d = self.disks[dev]
            logging.debug("Initializing partition table for %s" % (d['disk'].device))
            p1 = subprocess.Popen([self.parted, "-s", d['disk'].device, "mklabel", "msdos"],
                                 stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            (out,err) = p1.communicate()            
            logging.debug(out)
            
            if p1.returncode != 0:
                # NOTE: We don't throw exception when return code is not 0, because
                # parted always fails to reload part table with loop devices.
                # This prevents us from distinguishing real errors based on return code.
                logging.debug("WARNING: parted returned '%s' instead of 0 when creating partition-table for disk '%s'." % (p1.returncode,d['disk'].device))

        logging.debug("Creating partitions")

        for p in self.partitions:
            d = self.disks[p['disk']]
            if p['num'] == 5:
                self.__create_part_to_image(d['disk'].device,"extended",None,p['start'],d['extended'])
                
            if p['fstype'] == "swap":
                parted_fs_type = "linux-swap"
            elif p['fstype'] == "vfat":
                parted_fs_type = "fat32"
            elif p['fstype'] == "msdos":
                parted_fs_type = "fat16"
            else:
                # Type for ext2/ext3/ext4/btrfs
                parted_fs_type = "ext2"

            # Boot ROM of OMAP boards require vfat boot partition to have an 
            # even number of sectors.
            if p['mountpoint'] == "/boot" and p['fstype'] in ["vfat","msdos"] and p['size'] % 2:
                logging.debug("Substracting one sector from '%s' partition to get even number of sectors for the partition." % (p['mountpoint']))
                p['size'] -= 1
                
            p1 = self.__create_part_to_image(d['disk'].device,p['type'], 
                                             parted_fs_type, p['start'], 
                                             p['size'])

            if p1.returncode != 0:
                # NOTE: We don't throw exception when return code is not 0, because
                # parted always fails to reload part table with loop devices.
                # This prevents us from distinguishing real errors based on return code.
                logging.debug("WARNING: parted returned '%s' instead of 0 when creating partition '%s' for disk '%s'." % (p1.returncode,p['mountpoint'],d['disk'].device))

            if p['boot']:
                logging.debug("Setting boot flag for partition '%s' on disk '%s'." % (p['num'],d['disk'].device))
                boot_cmd = [self.parted, "-s", d['disk'].device, "set", "%d" % p['num'], "boot", "on"]
                logging.debug(boot_cmd)
                p1 = subprocess.Popen(boot_cmd,
                                      stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
                (out,err) = p1.communicate()            
                logging.debug(out)

                if p1.returncode != 0:
                    # NOTE: We don't throw exception when return code is not 0, because
                    # parted always fails to reload part table with loop devices.
                    # This prevents us from distinguishing real errors based on return code.
                    logging.debug("WARNING: parted returned '%s' instead of 0 when adding boot flag for partition '%s' disk '%s'." % (p1.returncode,p['num'],d['disk'].device))

    def __map_partitions(self):
        """Load it if dm_snapshot isn't loaded"""
        load_module("dm_snapshot")

        dev_null = os.open("/dev/null", os.O_WRONLY)
        for dev in self.disks.keys():
            d = self.disks[dev]
            if d['mapped']:
                continue

            logging.debug("Running kpartx on %s" % d['disk'].device )
            kpartx = subprocess.Popen([self.kpartx, "-l", "-v", d['disk'].device],
                                      stdout=subprocess.PIPE, stderr=dev_null)

            kpartxOutput = kpartx.communicate()[0].strip().split("\n")

            if kpartx.returncode:
                os.close(dev_null)
                raise MountError("Failed to query partition mapping for '%s'" %
                                 d['disk'].device)

            # Strip trailing blank and mask verbose output
            i = 0
            while i < len(kpartxOutput) and kpartxOutput[i][0:4] != "loop":
               i = i + 1
            kpartxOutput = kpartxOutput[i:]

            # Quick sanity check that the number of partitions matches
            # our expectation. If it doesn't, someone broke the code
            # further up
            if len(kpartxOutput) != d['numpart']:
                os.close(dev_null)
                raise MountError("Unexpected number of partitions from kpartx: %d != %d" %
                                 (len(kpartxOutput), d['numpart']))

            for i in range(len(kpartxOutput)):
                line = kpartxOutput[i]
                newdev = line.split()[0]
                mapperdev = "/dev/mapper/" + newdev
                loopdev = d['disk'].device + newdev[-1]

                logging.debug("Dev %s: %s -> %s" % (newdev, loopdev, mapperdev))
                pnum = d['partitions'][i]
                self.partitions[pnum]['device'] = loopdev

                # grub's install wants partitions to be named
                # to match their parent device + partition num
                # kpartx doesn't work like this, so we add compat
                # symlinks to point to /dev/mapper
                if os.path.lexists(loopdev):
                    os.unlink(loopdev)
                os.symlink(mapperdev, loopdev)

            logging.debug("Adding partx mapping for %s" % d['disk'].device)
            p1 = subprocess.Popen([self.kpartx, "-v", "-a", d['disk'].device],
                                  stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            
            (out,err) = p1.communicate()
            logging.debug(out)
            
            if p1.returncode != 0:
                # Make sure that the device maps are also removed on error case.
                # The d['mapped'] isn't set to True if the kpartx fails so
                # failed mapping will not be cleaned on cleanup either.
                subprocess.call([self.kpartx, "-d", d['disk'].device],
                                stdout=dev_null, stderr=dev_null)
                os.close(dev_null)
                raise MountError("Failed to map partitions for '%s'" %
                                 d['disk'].device)
            d['mapped'] = True
        os.close(dev_null)


    def __unmap_partitions(self):
        dev_null = os.open("/dev/null", os.O_WRONLY)
        for dev in self.disks.keys():
            d = self.disks[dev]
            if not d['mapped']:
                continue

            logging.debug("Removing compat symlinks")
            for pnum in d['partitions']:
                if self.partitions[pnum]['device'] != None:
                    os.unlink(self.partitions[pnum]['device'])
                    self.partitions[pnum]['device'] = None

            logging.debug("Unmapping %s" % d['disk'].device)
            rc = subprocess.call([self.kpartx, "-d", d['disk'].device],
                                 stdout=dev_null, stderr=dev_null)
            if rc != 0:
                os.close(dev_null)
                raise MountError("Failed to unmap partitions for '%s'" %
                                 d['disk'].device)

            d['mapped'] = False
            os.close(dev_null)


    def __calculate_mountorder(self):
        logging.debug("Calculating mount order")
        for p in self.partitions:
            self.mountOrder.append(p['mountpoint'])
            self.unmountOrder.append(p['mountpoint'])

        self.mountOrder.sort()
        self.unmountOrder.sort()
        self.unmountOrder.reverse()

    def cleanup(self):
        Mount.cleanup(self)
        self.__unmap_partitions()
        for dev in self.disks.keys():
            d = self.disks[dev]
            try:
                d['disk'].cleanup()
            except:
                pass

    def unmount(self):
        for mp in self.unmountOrder:
            if mp == 'swap':
                continue
            p = None
            for p1 in self.partitions:
                if p1['mountpoint'] == mp:
                    p = p1
                    break

            if p['mount'] != None:
                try:
                    p['mount'].cleanup()
                except:
                    pass
                p['mount'] = None

    def mount(self):
        for dev in self.disks.keys():
            d = self.disks[dev]
            d['disk'].create()

        self.__format_disks()
        self.__map_partitions()
        self.__calculate_mountorder()

        for mp in self.mountOrder:
            p = None
            for p1 in self.partitions:
                if p1['mountpoint'] == mp:
                    p = p1
                    break

            if mp == 'swap':
                subprocess.call([self.mkswap, p['device']])
                continue

            rmmountdir = False
            if p['mountpoint'] == "/":
                rmmountdir = True
            if p['fstype'] == "vfat" or p['fstype'] == "msdos":
                myDiskMount = VfatDiskMount
            elif p['fstype'] in ("ext2", "ext3", "ext4"):
                myDiskMount = ExtDiskMount
            elif p['fstype'] == "btrfs":
                myDiskMount = BtrfsDiskMount
            else:
                raise MountError("Fail to support file system " + p['fstype'])

            pdisk = myDiskMount(RawDisk(p['size'] * self.sector_size, p['device']),
                                 self.mountdir + p['mountpoint'],
                                 p['fstype'],
                                 4096,
                                 p['mountpoint'],
                                 rmmountdir,
                                 self.skipformat)

            pdisk.mount()
            p['mount'] = pdisk

    def resparse(self, size = None):
        # Can't re-sparse a disk image - too hard
        pass
