/***************************** LICENSE START ***********************************

 Copyright 2018 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "Metview.h"
#include "MvPath.hpp"
#include "MvDate.h"
#include "MvMiscelaneous.h"
#include "Tokenizer.h"
#include "Path.h"

#include <assert.h>
#include <iostream>
#include <stdexcept>
#include <algorithm>

#define ECCHARTS_CHK(str) ({if(!(str)) {setError(13); return;} })


class MvMacroCallerService : public MvService
{
public:
    MvMacroCallerService(char *a);

    static void progressCb(svcid* id, request* r, void* obj);
    static void replyCb(svcid* id, request* r, void* obj);

protected:
    void registerCallbacks();

    virtual void reply(const MvRequest&)=0;
    virtual void progress(const MvRequest&)=0;
};


MvMacroCallerService::MvMacroCallerService(char *a) : MvService(a)
{
}

void MvMacroCallerService::registerCallbacks()
{
    add_progress_callback(MvApplication::getService(),0,
            progressCb, (void*)this);

    add_reply_callback(MvApplication::getService(),0,
            replyCb, (void*)this);
}

void MvMacroCallerService::progressCb(svcid* id, request* r, void* obj)
{
    std::string s(get_value(r,"PROGRESS",0));
    std::cout << "PROG callback: " << s << std::endl;
    MvRequest in(r);
    if(MvMacroCallerService* b=static_cast<MvMacroCallerService*>(obj))
        b->progress(in);
}

void MvMacroCallerService::replyCb(svcid* id, request* r, void* obj)
{
    std::cout << "REPLY: " << std::endl;
    print_all_requests(r);
    MvRequest in(r);
    //in.print();
    if(MvMacroCallerService* b=static_cast<MvMacroCallerService*>(obj))
        b->reply(in);
}

class Eccharts : public MvMacroCallerService
{
public:
    Eccharts(char *req);
    void serve(MvRequest&,MvRequest&);

protected:
    void reply(const MvRequest&);
    void progress(const MvRequest&);
    void generateData(const MvRequest& in,MvRequest& out,const std::string& macroPath,const std::string& reqPath);

    std::string id_;
    std::string paramPrefix_;
    std::string outReqType_;
    bool hasReply_;
    MvRequest replyReq_;
};

Eccharts::Eccharts(char* req) :
    MvMacroCallerService(req),
    id_(req),
    hasReply_(false)
{
}

void Eccharts::progress(const MvRequest& in)
{
    if(const char* c=in("PROGRESS"))
    {
        marslog(LOG_INFO,"%s",c);
    }   
}

void Eccharts::reply(const MvRequest& in)
{
    hasReply_=true;
    replyReq_=in;
}

void Eccharts::serve(MvRequest& inReq, MvRequest& out)
{
    std::cout << "--------------Eccharts::serve()--------------" << std::endl;
    inReq.print();

    std::string macroPath;
    std::string reqPath;
    std::string prepPy;
	char *mvbin=getenv("METVIEW_BIN");
	if (mvbin == 0)  
	{	
        marslog(LOG_EROR,"No METVIEW_BIN env variable is defined. Cannot locate script mv_eccharts.py!");
		setError(13);
		return;
	}
	else
	{
        prepPy=std::string(mvbin) +"/mv_eccharts.py";
	}

    //Get the verb. We accept two verbs:
    // -ECCHARTS: it is sent when we call the action on a non-cached icon
    // -GRIB: it sent when the result (basically a GRIB) is cached and we call the
    //        "export" action on the icon! In this case to correctly perform the action we need
    //        the original ECCHARTS request and we store int the "_ECHHARTS" parameter of the
    //        cached GRIB request!!
    std::string verb;
    if(const char* chverb=inReq.getVerb())
        verb=std::string(chverb);
    else
    {
        marslog(LOG_EROR,"Verb is not defined!");
        setError(13);
    }

    //We will work  with a local copy oif the incoming request!
    MvRequest in = inReq;

    //Determine the user action
    const char* actionC = (const char*)in("_ACTION");
    std::string action=(actionC)?std::string(actionC):"execute";

    //When the incoming request is GRIB we need to recover the original
    //request from the _ECCHARTS parameter
    if(verb  == "GRIB")
    {
        if(action != "export")
            return;

        in = inReq("_ECCHARTS");
    }

    //Get params
    std::vector<std::string> param;

    //Mode
    std::string mode="data";
    if(action == "export")
    {
        mode = "macro";
    }
    assert(mode == "data" || mode == "macro");
    param.push_back(mode);

    //Layer definition json file
    std::string layersFile=metview::ecchartsDirFile("layers.json");
    param.push_back(layersFile);

    //mars definition json file
    std::string marsFile=metview::ecchartsDirFile("layers_mars.json");
    param.push_back(marsFile);

    //layer name
    std::string layer;
    ECCHARTS_CHK(in.getValue("LAYER",layer));

    if(mode == "data")
    {
        //Macro file to be generated
        char* macroPathCh = marstmp();
        macroPath=std::string(macroPathCh);
        param.push_back(macroPath);

        //Request file to be generated
        char* reqPathCh = marstmp();
        reqPath=std::string(reqPathCh);in.print();
        param.push_back(reqPath);
    }
    else if(mode == "macro")
    {                
        in("MACRO_OUTPUT_PATH") = std::string("eccharts_" + layer + ".mv").c_str();
        if(!in.getPath("MACRO_OUTPUT_PATH",macroPath,false))
        {
            setError(13);
            return;
        }

        Path p(macroPath);
        macroPath = p.directory().add(p.uniqueNameInDir()).str();
        param.push_back(macroPath);
        param.push_back("_dummy_path_");
    }

    //add layer name to params
    param.push_back(layer);

    //layer style - for contouring
    std::string style;
    ECCHARTS_CHK(in.getValue("STYLE",style,true));
    param.push_back(style);

    //date
    std::string dateStr;
    ECCHARTS_CHK(in.getValue("DATE",dateStr));
    param.push_back("date:" + dateStr);

    //time
    std::string timeStr;
    ECCHARTS_CHK(in.getValue("TIME",timeStr));
    param.push_back("time:" + timeStr);

    //grid
    std::vector<std::string> grid;
    ECCHARTS_CHK(in.getValue("GRID",grid));
    param.push_back("grid:" + metview::toMacroList(grid));

    //step
    std::vector<std::string> step;
    ECCHARTS_CHK(in.getValue("STEP",step));
    param.push_back("step:" + metview::toMacroList(step));

    //expver
    std::string expver;
    ECCHARTS_CHK(in.getValue("EXPVER",expver));
    param.push_back("expver:" + expver);

    //Run python script to generate the macro and requests
    std::string cmd = "python3 -u " + prepPy;
    for(size_t i=0; i < param.size(); i++)
    {
        cmd += " \"" + param[i] + "\"";
    }

    char* pyLogCh=marstmp();
    std::string pyLog(pyLogCh);
    cmd += " > \"" + pyLog + "\" 2>&1";

    marslog(LOG_INFO,"Execute command: %s",cmd.c_str());
    int ret=system(cmd.c_str());

    metview::writeFileToLogErr(pyLog);

    //If the script failed read log file and
    //write it into LOG_EROR
    if(ret == -1 || WEXITSTATUS(ret) != 0)
    {                   
        marslog(LOG_EROR,"Command to generate macro failed with exit code: %d", WEXITSTATUS(ret));
        marslog(LOG_EROR,"Command: %s\nError message:",cmd.c_str());
        metview::writeFileToLogErr(pyLog);

        setError(13);
        return;
    }

    if(mode == "data")
    {
        generateData(inReq,out,macroPath,reqPath);
    }
}

void Eccharts::generateData(const MvRequest& in,MvRequest& out,const std::string& macroPath,const std::string& reqPath)
{
    //=========================================
    // Read requests needed to built the plot
    //=========================================

    //Read the requests
    MvRequest vReq;
    vReq.read(reqPath.c_str());

    //=========================================
    // Retrieve and process data
    //=========================================

    //Build request to be sent to Macro
    MvRequest req("MACRO");

    std::string processName = MakeProcessName("Eccharts");
    //MvRequest macroReq("MACRO");
    req("PATH") = macroPath.c_str();
    req("_CLASS") = "MACRO";
    req("_ACTION") = "execute";
    req("_REPLY") = processName.c_str();
    req("_EXTENDMESSAGE") ="1";

    std::vector<std::string> param;

    //Resulting gib file
    char* dataPath = marstmp();
    param.push_back(std::string(dataPath));

    //Define argument list for the macro!
    for(std::vector<std::string>::iterator it=param.begin(); it != param.end(); it++)
    {
        req.addValue("_ARGUMENTS",(*it).c_str());
    }

    //Run macro
    marslog(LOG_INFO,"Execute macro: %s",macroPath.c_str());

    svc* mysvc=MvApplication::getService();
    registerCallbacks();
    MvApplication::callService("macro",req,this);

    for(;;)
    {
        if(hasReply_)
            break;

        svc_connect(mysvc);
        if(process_service(mysvc))
            break; /* Timeout */
    }

    if(const char *v=replyReq_.getVerb())
    {
        if(strcmp(v,"ERROR") == 0)
        {
            marslog(LOG_EROR,"Failed to run Eccharts data preparation! Macro failed!");
            if(const char* cc=replyReq_("MESSAGE"))
            {
                marslog(LOG_EROR,"Message: %s",cc);
            }
            setError(13);
        }
        else if(strcmp(v,"NUMBER") == 0)
        {
            if(const char *numCh=replyReq_("VALUE"))
            {
                if(strcmp(numCh,"0") != 0)
                {
                    marslog(LOG_EROR,"Failed to run Eccharts data preparation!");
                    marslog(LOG_EROR,"Macro exited with code:: %s",numCh);
                }
            }
        }
    }
    else
    {
        marslog(LOG_EROR,"Failed to run Eccharts data preparation!");
        setError(13);
    }

    out=MvRequest("GRIB");
    out("TEMPORARY")=1;
    out("PATH")=dataPath;

    out=out + vReq;

    //Add the original request as a hidden parameter. We need it to export
    //to macro under certain circumstances.
    out("_ECCHARTS") = in;

    out.print();
}

int main( int argc, char** argv )
{
    MvApplication theApp( argc, argv,"Eccharts");

    Eccharts eccharts("ECCHARTS");
    Eccharts ecchartsExport("GRIB");

    theApp.run();
}
