/***************************** LICENSE START ***********************************

 Copyright 2014 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "Metview.h"

#include "MetviewTask.h"
#include "IconObject.h"
#include "Log.h"
//#include "Editor.h"
#include "Request.h"
#include "Dependancy.h"
#include "IconFactory.h"
#include "IconClass.h"

//=================================================================

MetviewTask::MetviewTask(const string& service,const Action& action,IconObject* o):
	service_(service),
	action_(action),
	object_(o),
	error_(false),
	waiting_(0)	
{
}

MetviewTask::~MetviewTask()
{
	cout << "MetviewTask::~MetviewTask " << *object_ << endl;
}

void MetviewTask::start()
{
	if(!object_->iconClass().skipDepandancies(action_.name()))	
	{
  		const set<DependancyH>& dep = object_->dependancies();

		Action action("prepare","*");

		for(set<DependancyH>::const_iterator j = dep.begin(); j != dep.end(); ++j)
		{
			Task* t = (*j)->action(action);
			if(t) {
				tasks_[t] = *j;
				waiting_++;
				t->add(this);
			}

		}
	}
	
	check();

}

void MetviewTask::check()
{
	if(waiting_) 
		return;

	if(error_)
	{
		Task::failure();
		return;
	}

	MvRequest r   = object_->fullRequest();
	const char* null = 0;

	if(null == r("_NAME"))    r("_NAME")    = object_->fullName().c_str();
	if(null == r("_CLASS"))   r("_CLASS")   = object_->className().c_str();
	if(null == r("_ACTION"))  r("_ACTION")  = action_.name().c_str();
	if(null == r("_SERVICE")) r("_SERVICE") = service_.c_str();

	// Add extra information from the dropped target object
	if ( context_ )
		r("_CONTEXT") = context_;

	cout << " ---> send " << *this << endl;
	r.print();

	callService(service_,r);

	cout << " <--- send " << *this << endl;
}

//Sends a special progress message back to Desktop to indicate that the
//GUI app started up. Desktop then turns the icon green. Otherwise the icon would stay
//orange and all other actions would be blocked on it until the GUI finishes!!
//Ideally we should send a reply back to Desktop but this CAN ONLY BE SENT when
//the forked process (the GUI) has finished!!!

void MetviewTask::progress(const Request& r)
{    
    if(const char* v=r.getVerb())
    {
        //When a GUI_STARTED request is received in progress() it means
        //the interactive (GUI) app we requested for has been started up and we can
        //regard this task as finished and can tun the icon green so that it could
        //receive another action. This typically happens when we start up an examiner.
        if(strcmp(v,"GUI_STARTED") == 0)
        {
            Task::success(r);
        }
    }
}

void MetviewTask::reply(const Request& r,int err)
{
	if(err) 
		Task::failure();
	else {
		cout << " ---> reply " << *this << endl;
		r.print();
        cout << "action = " << action_.name() << endl;
		cout << " <--- reply " << *this << endl;

		if(r) {
			IconObjectH o = IconFactory::createTemporary(object_,r);
			TaskH       t = o->action(action_);
		 }
		
		Task::success(r);
	}
}

void MetviewTask::message(const string& s)
{
	Log::info(object_) << s << endl;
}

void MetviewTask::success(Task* t,const Request& r)
{
	cout << "MetviewTask::success " << *t << endl;
	tasks_[t]->success(r);
	waiting_--;
	check();
}

void MetviewTask::failure(Task* t)
{
	cout << "MetviewTask::failure " << *t << endl;
	error_ = true;
	tasks_[t]->failure();
	waiting_--;
	check();
}

void MetviewTask::print(ostream& s) const
{
	s << "MetviewTask[" 
	  << service_ 
	  << "," 
      << action_.name() 
	  << ","  
	  << action_.mode() 
	  << ","
      << *object_ 
	  << "]";
}
