/***************************** LICENSE START ***********************************

 Copyright 2016 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQFileDialog.h"

#include <QtGlobal>
#include <QDebug>
#include <QDir>
#include <QFileSystemModel>
#include <QUrl>

#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif

#include "MvQPixmapCache.h"

//===========================
// MvQFileIconProvider
//===========================

MvQFileIconProvider::MvQFileIconProvider()
{
    if (char* iconHome = getenv("METVIEW_DIR_SHARE")) {
        path_ = QString(iconHome);
        path_.append("/icons/");
    }

    pixCache_ = new MvQPixmapCache(QPixmap());

    char* mv = getenv("METVIEW_USER_DIRECTORY");
    if (mv)
        mvHome_ = QString(mv);
}

QIcon MvQFileIconProvider::icon(const QFileInfo& info) const
{
#ifdef METVIEW
    if (info.isDir() && !path_.isEmpty()) {
        QString pixName = "FOLDER";

        if (!mvHome_.isEmpty() && info.filePath() == mvHome_) {
            pixName = "HOME";
        }

        QPixmap pix = pixCache_->pixmap(pixName);
        if (pix.isNull()) {
            pix = QPixmap(path_ + pixName + ".svg");
            if (!pix.isNull())
                pixCache_->add(pixName, pix);
        }

        return QIcon(pix);
    }
#endif
    return QFileIconProvider::icon(info);
}

//===========================
// MvQFileDialogFilterModel
//===========================

MvQFileDialogFilterModel::MvQFileDialogFilterModel(QObject* parent) :
    QSortFilterProxyModel(parent)
{
}

bool MvQFileDialogFilterModel::filterAcceptsRow(int sourceRow, const QModelIndex& sourceParent) const
{
    QModelIndex index = sourceModel()->index(sourceRow, 0, sourceParent);
    auto* fsModel     = qobject_cast<QFileSystemModel*>(sourceModel());

    if (!fsModel)
        return false;

    QString fileName = fsModel->fileName(index);

    if (fsModel->isDir(index)) {
        if (fileName.endsWith("#") == false)
            return true;
        else
            return false;
    }

    if (fileName.startsWith(".") == false &&
        fileName.startsWith("..") == false) {
        return true;
    }

    return false;
}

bool MvQFileDialogFilterModel::lessThan(const QModelIndex& left, const QModelIndex& right) const
{
    auto* fsModel = qobject_cast<QFileSystemModel*>(sourceModel());

    if (!fsModel)
        return false;

    if (fsModel->isDir(right)) {
        if (!fsModel->isDir(left))
            return false;
    }
    else {
        if (fsModel->isDir(left))
            return true;
    }

    return QSortFilterProxyModel::lessThan(left, right);
}

//===========================
// MvQFileDialog
//===========================

MvQFileDialog::MvQFileDialog(QString startDir, QString title, QWidget* parent) :
    QFileDialog(parent)
{
    setWindowTitle(title);

    setOptions(QFileDialog::DontUseNativeDialog);

    //url sidebar
    QList<QUrl> urlLst;

    urlLst << QUrl::fromLocalFile(QDir::rootPath());

    Q_FOREACH (QString s, QStandardPaths::standardLocations(QStandardPaths::HomeLocation)) {
        urlLst << QUrl::fromLocalFile(s);
    }

#ifdef METVIEW
    if (char* mvDir = getenv("METVIEW_USER_DIRECTORY")) {
        urlLst << QUrl::fromLocalFile(QString(mvDir));
    }
#else
    if (char* scratchDir = getenv("SCRATCH")) {
        urlLst << QUrl::fromLocalFile(QString(scratchDir));
    }
#endif

    setSidebarUrls(urlLst);

    //Filter model
    auto* proxy = new MvQFileDialogFilterModel(this);
    setProxyModel(proxy);

    //Dir
    setDirectory(startDir);

#ifdef METVIEW
    //Icons
    auto* iconProvider = new MvQFileIconProvider;
    setIconProvider(iconProvider);
#endif
}
