/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <stdlib.h>
#include <time.h>
#include <MvRequest.h>

#include "macro.h"
#include "script.h"

#include "MvFlextra.h"

//===========================================================================

class FlextraCountFunction : public Function {
public:
	FlextraCountFunction(char *n) : Function(n,1,tnumber)
    {info = "Causes Macro to always wait(1)/not wait(0) for functions to complete before continuing.";}
	virtual int ValidArguments(int arity,Value *arg);
	virtual Value Execute(int,Value*);
};

int FlextraCountFunction::ValidArguments(int arity,Value *arg)
{
        // arguments can be: (FLEXTRA_FILE definition)

	if(arity !=1 ) 
		return false;
     
   	if(arg[0].GetType() != trequest) 
      		return false;
	
	request *r;
	arg[0].GetValue(r);
	
	if(r && strcmp(r->name,"FLEXTRA_FILE")==0)
			return true;
	
	return false;
}

Value FlextraCountFunction::Execute(int /*arity*/,Value *arg)
{
	request *r;
	arg[0].GetValue(r);
	const char* path=get_value(r,"PATH",0);
	
	if(path)
	{
	  	string s(path);
	  	MvFlextra flx(s);
		return Value(flx.blockNum());
	}	
  
  	int zero=0;
	return Value(zero);
}

//===========================================================================

class FlextraGroupGetFunction : public Function {
    
 public:
	FlextraGroupGetFunction(const char *n) : Function(n,3,tnumber)
	{info = "Returns a list of available meta data keys for the given table";}
	virtual int ValidArguments(int arity,Value *arg);
	virtual Value Execute(int arity,Value *arg);
};

int FlextraGroupGetFunction::ValidArguments(int arity,Value *arg)
{
        // arguments can be: (FLEXTRA_FILE definition, number)const map
	
	if(arity !=2) 
		return false;
     
   	if(arg[0].GetType() != trequest)   //first must be a request
      		return false;
		
	if(arg[1].GetType() != tstring && arg[1].GetType() != tlist) 
    	{
        	return false;
    	}
   	
	request *r;
	arg[0].GetValue(r);
	
	if(r && strcmp(r->name,"FLEXTRA_FILE")==0)
			return true;
	
	return false;
}

Value FlextraGroupGetFunction::Execute(int /*arity*/,Value *arg)
{
	request *r;
	arg[0].GetValue(r);
	const char* path=get_value(r,"PATH",0);
	
	if(!path)
		return Value();	
	  	 	  
	string s(path);
	MvFlextra flx(s);
	
	if(flx.blockNum() ==0)
		return Value();
	
	int id=0;	
	if(id <0 || id >= flx.blockNum())
		return Value();	
	
	const map<string,string> &metaData = flx.blocks().at(id)->metaData();
	
	map<string,string>::const_iterator it;
	
	if(arg[1].GetType() == tstring)
	{
	  	const char *key;
        	arg[1].GetValue(key);
		if((it=metaData.find(key)) != metaData.end())
			return Value((*it).second.c_str());
	}
	else if(arg[1].GetType() == tlist) 
	{
		CList *keys;
        	arg[1].GetValue(keys);
        	CList *values = new CList (keys->Count());

        	for(int i=0;i< keys->Count();i++)
        	{
            		const char *key;
            		(*keys)[i].GetValue(key);
			if((it=metaData.find(key)) != metaData.end())
				(*values)[i]=Value((*it).second.c_str());
        	}  
        	     	
        	return values;
	}
	
  	return Value();
}

//===========================================================================

class FlextraTrGetFunction : public Function {
    
 public:
	FlextraTrGetFunction(const char *n) : Function(n,3,tnumber)
	{info = "Returns a list of available meta data keys for the given table";}
	virtual int ValidArguments(int arity,Value *arg);
	virtual Value Execute(int arity,Value *arg);
};

int FlextraTrGetFunction::ValidArguments(int arity,Value *arg)
{
        // arguments can be: (FLEXTRA_FILE definition, number)
	
	if(arity !=3) 
		return false;
     
   	if(arg[0].GetType() != trequest)   //first must be a request
      		return false;
		
	if(arg[1].GetType() != tnumber)   //second must be a number
	     return false;
	
	if(arg[2].GetType() != tstring && arg[2].GetType() != tlist) 
    	{
        	return false;
    	}
   	
	request *r;
	arg[0].GetValue(r);
	
	if(r && strcmp(r->name,"FLEXTRA_FILE")==0)
			return true;
	
	return false;
}

Value FlextraTrGetFunction::Execute(int /*arity*/,Value *arg)
{
	request *r;
	arg[0].GetValue(r);
	const char* path=get_value(r,"PATH",0);
	
	if(!path)
		return Value();	
	  	 	  
	string s(path);
	MvFlextra flx(s);
	
	if(flx.blockNum() ==0)
		return Value();
	
	int trId;
	arg[1].GetValue(trId);
	trId--;			   
	
	if(trId < 0 || trId >= flx.blocks().at(0)->itemNum())
		return Value();

	MvFlextraItem *item=flx.blocks().at(0)->items().at(trId);
	
	const map<string,string> &metaData = item->metaData();
	
	map<string,string>::const_iterator it;
		
	//A single key is specified
	if(arg[2].GetType() == tstring)
	{
	  	const char *key;
        	arg[2].GetValue(key);
		if((it=metaData.find(key)) != metaData.end())
			return Value((*it).second.c_str());
		else
		{
		  	vector<string> vec;
			MvFlextraItem::DataType type;
			item->pointData(key,vec,type);
			if(vec.size() > 0)
			{
        			if(type == MvFlextraItem::IntType || 
				   type == MvFlextraItem::FloatType )
				{
					CVector *values = new CVector(vec.size());	  
			  		for(unsigned int i=0;i< vec.size() ;i++)
        				{
						istringstream iss(vec.at(i));
 						double d;
						iss >> d; 
					  	values->setIndexedValue(i, d);
					}
					
					return values;
				}	
				else if(type == MvFlextraItem::DateType)		
				{
			  		CList *values = new CList (vec.size());
					for(unsigned int i=0;i< vec.size() ;i++)
        				{
						Date d(vec.at(i).c_str());
						(*values)[i]=Value(d);
					}						
					return values;								
				}	
			}
		}
	}	
		
	else if(arg[2].GetType() == tlist) 
	{
		CList *keys;
        	arg[2].GetValue(keys);
        	CList *values = new CList (keys->Count());

        	for(int i=0;i< keys->Count();i++)
        	{
            		const char *key;
            		(*keys)[i].GetValue(key);
			if((it=metaData.find(key)) != metaData.end())
				(*values)[i]=Value((*it).second.c_str());
			
			else
			{
			  	vector<string> vec;
				MvFlextraItem::DataType type;
				item->pointData(key,vec,type);
				
				if(type == MvFlextraItem::IntType || 
				   type == MvFlextraItem::FloatType )
				{
					CVector *data = new CVector(vec.size());	  
			  		for(unsigned int j=0;j< vec.size() ;j++)
        				{
						istringstream iss(vec.at(j));
 						double d;
						iss >> d; 
					  	data->setIndexedValue(j, d);
					}
					
					(*values)[i]=data;
				}	
				else if(type == MvFlextraItem::DateType)		
				{
			  		CList *data = new CList (vec.size());
					for(unsigned int j=0;j< vec.size() ;j++)
        				{
						Date d(vec.at(j).c_str());
						(*data)[j]=Value(d);
					}						
					(*values)[i]=data;								
				}				  
			}
        	}  
        	     	
        	return values;
	}
	
  	return Value();
}

//=============================================================================

class FlextraElemFunction : public Function {
public:
	FlextraElemFunction(char *n) : Function(n,2,tlist,tnumber) {};
	virtual Value Execute(int arity,Value *arg);
	virtual int ValidArguments(int arity,Value *arg);
};

int FlextraElemFunction::ValidArguments(int arity,Value *arg)
{
	if(arity != 2) 
	  	return false;

	if(arg[0].GetType() != trequest) 
	  	return false;

	if(arg[1].GetType() != tnumber) 
	  	return false;
	
	request *r;
	arg[0].GetValue(r);
	
	if(r && strcmp(r->name,"FLEXTRA_FILE")==0)
			return true;
	
	return false;
}

Value FlextraElemFunction::Execute(int arity,Value *arg)
{
	request *r;
	arg[0].GetValue(r);
	const char* path=get_value(r,"PATH",0);
	
	if(!path)
		return Value();	
	  	 	  
	string s(path);
	MvFlextra flx(s);
	
	if(flx.blockNum() ==0)
		return Value();
	
 	int id;
	if(arity == 1)
		id=0;	
	else if(arity == 2)
	{
		arg[1].GetValue(id);
		id--;			  
	} 
	
	if(id <0 || id >= flx.blockNum())
		return Value();
	
	if(id ==0 && flx.blockNum() == 1)
	{
	 	return Value(r);
	}	
	else
	{  
		string outfile(marstmp());	
		flx.write(outfile,id);
	
		request *outr = empty_request("FLEXTRA_FILE");
		set_value(outr,"PATH",outfile.c_str());
		
		return Value(outr);
	}
	
	return Value();
}


//=============================================================================

static void install(Context *c)
{	
	c->AddFunction(new FlextraCountFunction("count"));
	c->AddFunction(new FlextraGroupGetFunction("flextra_group_get"));
	c->AddFunction(new FlextraTrGetFunction("flextra_tr_get"));
	c->AddFunction(new FlextraElemFunction("[]"));
}

static Linkage Link(install);

