#include "mars.h"
#include "mars_client_config.h"

#ifdef HAVE_NETCDF
#include <netcdf.h>


static err read_offset(const char *path, FILE *f, int length, unsigned long *v) {
    unsigned char buf[8];
    int i;

    *v = 0;

    if (fread(buf, 1, length, f) != length) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        return -2;
    }

    for (i = length - 1; i >= 0; i--) {
        *v <<= 8;
        *v |= buf[i];
    }

    return 0;
}

err check_hdf5_superblock(const char *path) {


    /* http://www.hdfgroup.org/HDF5/doc/H5.format.html#Superblock */
    FILE *f = fopen(path, "r");
    int e, nc, format, number_of_groups, number_of_types;

    struct stat st;

    unsigned char buf[8];
    unsigned char  version_of_superblock,  size_of_offsets, size_of_lengths, consistency_flags;

    unsigned long base_address, superblock_extension_address, end_off_file_address;


    marslog(LOG_INFO, "Check if %s is supported NetCDF format", path);

    if (!f) {
        marslog(LOG_EROR | LOG_PERR, "check_hdf5_superblock: cannot open %s", path);
        return -2;
    }

    if (fread(buf, 1, 8, f) != 8) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;
    }

    /* http://www.hdfgroup.org/HDF5/doc/H5.format.html#FileMetaData */

    if ( !(buf[0] == 137 && buf[1] == 'H' && buf[2] == 'D' && buf[3] == 'F' && buf[4] == '\r' && buf[5] == '\n' && buf[6] == 26 && buf[7] == '\n'))  {
        marslog(LOG_WARN, "check_hdf5_superblock: wrong magic %s", path);
        fclose(f);
    }


    if (fread(&version_of_superblock, 1, 1, f) != 1) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;
    }

    /*if (version_of_superblock != 2) {
        marslog(LOG_WARN, "check_hdf5_superblock: only version 2 of hdf5 superblock suported %s (version in file is %ld)", path, (long) version_of_superblock);
        fclose(f);
        return -2;
    }
*/

    if (fread(&size_of_offsets, 1, 1, f) != 1) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;
    }

    if (fread(&size_of_lengths, 1, 1, f) != 1) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;
    }

    if (fread(&consistency_flags, 1, 1, f) != 1) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;
    }

    if (read_offset(path, f, size_of_offsets, &base_address)) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;

    }

    if (read_offset(path, f, size_of_offsets, &superblock_extension_address)) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;

    }

    if (read_offset(path, f, size_of_offsets, &end_off_file_address)) {
        marslog(LOG_WARN, "check_hdf5_superblock: Cannot read enough bytes from %s", path);
        fclose(f);
        return -2;

    }

    fclose(f);

    if(stat(path, &st) <0) {
        marslog(LOG_EROR|LOG_PERR, "Cannot stat %s", path);
        return -2;
    }

    if(st.st_size != end_off_file_address) {
        marslog(LOG_EROR, "%p: file is %ld bytes long, HDF5 header says it is %ld bytes", path, st.st_size, end_off_file_address);
        return -2;
    }

    if ( (e = nc_open(path, NC_NOWRITE, &nc)) != NC_NOERR) {
        marslog(LOG_EROR, "nc_open(%s): %s", path, nc_strerror(e));
        return -2;
    }

    if ( (e = nc_inq_format(nc, &format)) != NC_NOERR) {
        marslog(LOG_EROR, "nc_inq_format(%s): %s", path, nc_strerror(e));
        return -2;
    }

    switch (format) {

    case NC_FORMAT_CLASSIC:
        marslog(LOG_DBUG, "%s is NC_FORMAT_CLASSIC", path);
        break;

    case NC_FORMAT_64BIT:
        marslog(LOG_DBUG, "%s is NC_FORMAT_64BIT", path);
        break;

    case NC_FORMAT_NETCDF4:
        marslog(LOG_DBUG, "%s is NC_FORMAT_NETCDF4", path);
        break;

    case NC_FORMAT_NETCDF4_CLASSIC:
        marslog(LOG_DBUG, "%s is NC_FORMAT_NETCDF4_CLASSIC", path);
        break;

    default:
        marslog(LOG_EROR, "Unknow netcdf format: %s (%d)", path, format);
        return -2;
    }
    /* Check that NetCDF is 'classic' */

    if ( (e = nc_inq_grps(nc, &number_of_groups, NULL)) != NC_NOERR) {
        marslog(LOG_EROR, "nc_inq_grps(%s): %s", path, nc_strerror(e));
        return -2;
    }

    if (number_of_groups > 1) {
        marslog(LOG_EROR, "NetCDF is not 'classic' (%s): Number of groups is %d", path, number_of_groups);
        return -2;
    }

    if ( (e = nc_inq_typeids(nc, &number_of_types, NULL)) != NC_NOERR) {
        marslog(LOG_EROR, "nc_inq_typeids(%s): %s", path, nc_strerror(e));
        return -2;
    }

    if (number_of_types > 0) {
        marslog(LOG_EROR, "NetCDF is not 'classic' (%s): Number of types is %d", path, number_of_types);
        return -2;
    }

     if ( (e = nc_close(nc)) != NC_NOERR) {
        marslog(LOG_EROR, "nc_close(%s): %s", path, nc_strerror(e));
        return -2;
    }

    marslog(LOG_INFO, "%s is supported NetCDF format", path);

    return 0;

}

#endif
