// Copyright (c) 2010, Jens Peter Secher <jpsecher@gmail.com>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

// This implementation was inspired by John Goerzen's hg-importdsc.

typedef Dsc =
{
	var dir : String;
	var changeLog : DebianChangeLog;
	var diff : String;
	var balls : Array<String>;
};

class MercurialImportDsc
{
	public static function main()
	{
		var dscFileName = parseCommandLine();
		var dsc = parseDsc( dscFileName );
		Mercurial.exitIfUncommittedChanges();
		// Check that the new version is not already imported.
		if( Mercurial.hasTag( Util.fullTag( dsc.changeLog ) ) )
		{
			Util.die
			(
				"Version " + dsc.changeLog.version +
				" already exists in repository."
			);
		}
		// Abort if new version is already incorporated.
		if( ! Util.tagOk( Mercurial.getTags(), dsc.changeLog ) )
		{
			Util.die
			(
				"Version " + dsc.changeLog.version +
				" is not newer than existing Debian versions."
			);
		}
		try
		{
			// Is there a Debian diff? 
			if( dsc.diff.length > 0 ) processUpstream( dsc );
			// Import tarballs.
			importDsc( dscFileName );
			// Make sure debian/rules is executable.
			Process.runButExitOnError( "chmod", ["+x","debian/rules"] );
			// Add and remove files from dsc, but do a lot to detect renames.
			info( 1, "Starting hg addremove -s " + Constants.similarity() );
			var addremove = new Process
			(
				"hg", [ "addremove", "-s", Constants.similarity() ]
			);
			infos( 2, addremove.stdout() );
			if( addremove.process.exitCode() != 0 )
			{
				Util.writeError( "addremove failed:" );
				Util.writeErrors( addremove.stderr() );
				throw "Addremove dsc failed.";
			}
			// Commit the new dsc contents.
			info( 2, "Starting hg commit" );
			var msg = Constants.importDsc() + " imported "
				+ dsc.changeLog.version;
			var commit = new Process( "hg", [ "commit", "-m", msg ] );
			if( commit.process.exitCode() != 0 )
			{
				Util.writeError( "commit failed:" );
				Util.writeErrors( commit.stderr() );
				throw "Commit dsc failed.";
			}
			infos( 2, commit.stdout() );
		}		
		catch( e : Dynamic )
		{
			Util.writeError( Std.string( e ) );
			info( 0, "Reverting default branch to state before import." );
			Mercurial.revert();
			neko.Sys.exit( 1 );
		}
		// Put a new tag in default branch.
		info( 1, "Tagging with " + Util.fullTag( dsc.changeLog ) );
		var tag = Util.fullTag( dsc.changeLog );
		var msg = Constants.importDsc() + " added tag " + tag;
		var tagger = Mercurial.tag( tag, msg );
		infos( 2, tagger );
	}

	//
	// Setup global variables and abort if something is wrong with the command
	// line.  Returns the dsc file name.
	//
	static function parseCommandLine() : String
	{
		var usage = "Usage: " + Constants.importDsc() + " [-V|--version]" +
			" [-v|--verbose] dscFile";
		var options = new GetPot( neko.Sys.args() );
		Util.maybeReportVersion( options, usage );
		// Collect verbosity options.
		verbosity = 0;
		while( options.got( ["--verbose","-v"] ) ) ++verbosity;
		// Reject other options.
		Util.exitOnExtraneousOptions( options, usage );
		// Get the dsc file name.
		var dscFileName = options.unprocessed();
		if( dscFileName == null ) Util.die( usage );
		Util.exitOnExtraneousArguments( options, usage );
		// Check that the dsc file exists.
		if( ! FileUtil.fileExists( dscFileName ) )
		{
			Util.die( "File " + dscFileName + " does not exist." );
		}
		return dscFileName;
	}

	static function processUpstream( dsc : Dsc )
	{
		info( 1, "Processing upstream tarball(s)." );
		var args : Array<String> = [ "--no-merge" ];
		for( ball in dsc.balls ) args.push( dsc.dir + "/" + ball );
		for( i in 0...verbosity-1 ) args.push( "-v" );
		var importOrig = new Process( Constants.importOrig(), args );
		for( line in importOrig.stdout() )
		{
			// Filter out "to pull in new version" from importorig.
			var pullInUpstream = ~/to pull in new version/;
			if( ! pullInUpstream.match( line ) ) info( 1, line );
		}
		if( importOrig.process.exitCode() != 0 )
		{
			Util.writeError
			(
				Constants.importOrig() + " " + args.join(" ") + " failed:"
			);
			Util.dies( importOrig.stderr() );
		}		
	}

	static function parseDsc( dscFileName : String ) : Dsc
	{
		// Extract the source and version info.
		var dsc = neko.io.File.getContent( dscFileName );
		// Get the source from dsc contents.		
		var sourceRegExp = ~/^Source:[ \t]+(.+)$/m;
		if( ! sourceRegExp.match( dsc ) )
		{
			Util.die( "File " + dscFileName + " does not contain source info." );
		}
		var source = sourceRegExp.matched( 1 );
		// Get the version from dsc contents.
		var versionRegExp = ~/^Version:[ \t]+(.+)$/m;
		if( ! versionRegExp.match( dsc ) )
		{
			Util.die( "File " + dscFileName + " does not contain version info." );
		}
		var version = versionRegExp.matched( 1 );
		// Get the dpkg format.
		var formatRegExp = ~/^Format: (.*)$/m;
		if( ! formatRegExp.match( dsc ) )
		{
			Util.die ( "File " + dscFileName + " does not contain format info." );
		}
		var format = formatRegExp.matched( 1 );
		info( 1, "Source format " + format );
		// Get the tarball file names.
		var balls = new Array<String>();
		var diff = "";
		if( format == "1.0" )
		{
			var origRegExp = ~/ ([^ ]+\.tar\.gz)$/m;
			if( origRegExp.match( dsc ) )
			{
				var ball = origRegExp.matched( 1 );
				info( 1, "Tarball " + ball );
				balls.push( ball );
				var diffRegExp = ~/ ([^ ]+\.diff\.gz)$/m;
				if( diffRegExp.match( dsc ) )
				{
					diff = diffRegExp.matched( 1 );
					info( 1, "Diff " + diff );
				}
			}
		}
		else if( format == "3.0 (native)" )
		{
			var origRegExp = ~/ ([^ ]+\.tar\.(gz|bz2|lzma))$/m;
			if( origRegExp.match( dsc ) )
			{
				var ball = origRegExp.matched( 1 );
				balls.push( ball );
				info( 1, "Tarball " + ball );
			}
		}
		else if( format == "3.0 (quilt)" )
		{
			// Find the debian diff.
			var diffRegExp = ~/ ([^ ]+\.debian\.tar\.(gz|bz2|lzma))$/m;
			if( ! diffRegExp.match( dsc ) )
			{
				Util.die( "No debian diff in dsc file." );
			}
			diff = diffRegExp.matched( 1 );
			info( 1, "Debian tarball " + diff );
			// Find the main tarball.
			var mainRegExp = ~/ ([^ ]+\.orig\.tar\.(gz|bz2|lzma))$/m;
			if( mainRegExp.match( dsc ) )
			{
				var main = mainRegExp.matched( 1 );
				info( 1, "Main tarball " + main );
				balls.push( main );
				// Collect additional tarballs.
				var searchable = dsc;
				while( searchable.length > 0 )
				{
					var componentRegExp =
					~/ ([^ ]+\.orig-[A_Za-z0-9][-A_Za-z0-9]*\.tar\.(gz|bz2|lzma))$/m;
					if( componentRegExp.match( searchable ) )
					{
						var ball = componentRegExp.matched( 1 );
						if( ! Lambda.has( balls, ball ) )
						{
							info( 1, "Additional tarball " + ball );
							balls.push( ball );
							searchable = componentRegExp.matchedRight();
						}
						else
						{
							// Stop the search.
							searchable = "";
						}
					}
					else
					{
						// Stop the search.
						searchable = "";
					}
				}
			}
		}
		if( balls.length == 0 )
		{
			// Search for 
			Util.die( "No source ball in dsc file." );
		}
		return
		{
			dir: neko.io.Path.directory( dscFileName ),
			changeLog: new DebianChangeLog( source, version ),
			diff: diff,
			balls: balls
		};
	}

	static function importDsc( dscFileName : String )
	{
		// Delete everything except the Mercurial and Quilt files.
		for( entry in neko.FileSystem.readDirectory( "." ) )
		{			
			if( ! Constants.precious().match( entry ) ) FileUtil.unlink( entry );
		}
		// Make sure the dsc file has an absolute path.
		dscFileName = neko.FileSystem.fullPath( dscFileName );
		// Use dpkg-source to unpack everything in a temporary directory.  The
		// path cannot contain ".." because dpkg-source will complain about it.
		info( 1, "Importing dsc." );
		var tmpDir =
			neko.FileSystem.fullPath( neko.Sys.getCwd() + ".." ) +
			"/,,importdsc-" + neko.Sys.time();
		var dpkgArgs = [ "--no-copy", "-x", dscFileName, tmpDir ];
		info( 2, "Starting dpkg-source " + dpkgArgs.join(" ") + " in .." );
		infos( 3, Process.runButExitOnError( "dpkg-source", dpkgArgs, ".." ) );
		// Delete empty directories because they are not tracked by Mercurial.
		FileUtil.prune( tmpDir );
		// Move the unpacked source back into the root, unless it is a Mercurial
		// file.
		var entries = neko.FileSystem.readDirectory( tmpDir );
		for( entry in entries )
		{
			if( ! Constants.precious().match( entry ) )
			{
				info( 3, "Copying " + tmpDir + "/" + entry );
				FileUtil.copy( tmpDir + "/" + entry, "." );
			}
		}
		info( 2, "Deleting " + tmpDir );
		FileUtil.unlink( tmpDir );
	}

	//
	// Higher values means more output.
	//
	static var verbosity : Int;

	//
	// Write a line of info to stdout if the verbosity level is high enough.
	//
	static function info( level : Int, line : String )
	{
		if( verbosity >= level ) Util.writeInfo( line );
	}	

	//
	// Write lines of info to stdout if the verbosity level is high enough.
	//
	static function infos( level : Int, lines : Iterator<String> )
	{
		for( line in lines )
		{
			if( verbosity >= level ) Util.writeInfo( line );
		}
	}
}

