 /*
   MemAid, a memorisation tool.
   Copyright (C) 2003 David Calinski
   Portions of the code: Copyright (c) 1996 Karsten Kutza

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307  USA
*/

#include <math.h>
#include <stdarg.h>
#include "ann.h"

#ifdef MEMAID_DEBUG
FILE *F_debug;
#endif
static FILE *F_err;
char ma_Path[MAX_PATH_LEN];

void ma_error (const char *s, ...) {
	va_list argp;
	va_start (argp, s);
	vfprintf (F_err, s, argp);
	va_end (argp);
	fflush (F_err);
	return;
}

void ma_debug (const char *s, ...) {
#ifdef MEMAID_DEBUG
	va_list argp;
	va_start (argp, s);
	fprintf (F_debug, "DEBUG: ");
	vfprintf (F_debug, s, argp);
	va_end (argp);
	fflush (F_debug);
#endif
	return;
}

INLINE unsigned int ma_rand (unsigned long up_to) {
	assert (up_to > 0);
#ifdef __BSD_VISIBLE
	return (int) (random() % up_to);
#else
	return (rand() / (RAND_MAX / up_to + 1L));
#endif
}

#define NUM_LAYERS   3  /* Number of layers */
#define IN_N         4  /* number of neurons in input  layer  */
#define HID_N       20  /* number of neurons in middle layer  */
#define OUT_N        1  /* number of neurons in output layer  */
#define IN_OUT_N (IN_N+OUT_N)

typedef struct {              /* A LAYER OF A NET:                   */
	u_short  Units;       /*  number of units in this layer      */
	float*   Output;      /*  output of ith unit                 */
	float*   Error;       /*  error term of ith unit             */
	float**  Weight;      /*  connection weights to ith unit     */
	float**  WeightSave;  /*  saved weights for stopped training */
} LAYER;
static struct {               /* A NET:                    */
	LAYER**  Layer;       /*  layers of this net       */
	LAYER*   OutputLayer; /*  output layer             */
	float    Eta;         /*  learning rate            */
	float    Error;       /*  total net error          */
} Net;

static float **Data;
u_long NData;          /* how many cases for ANN to learn from */
u_short NData_general; /* how many cases for ANN not learn from user */

#define N_DATA_JUMP 512 /* keep it always _higher_ than N_HARD_CODED_CASES below!
			 * AND always power of 2 */

#define N_HARD_CODED_CASES 366


/* We have hard-coded initial data for ANN here,
 * not used if data.bin file exist
 *
 * format:
 *  	1) NN input 1: last_interval_planned_by_ann
 *	2) NN input 2: real_last_interval
 *	3) NN input 3: number_of_repetitions
 *	4) NN input 4: grade
 *	5) NN output: next_interval
 */
static const float Hard_coded_data [N_HARD_CODED_CASES][IN_OUT_N] = {
{ 0.038273,0.044194,0.171875,1.000000,0.049411 },
{ 0.255792,0.255792,0.148438,0.800000,0.270633 },
{ 0.038273,0.038273,0.148438,0.800000,0.044194 },
{ 0.125000,0.125000,0.140625,1.000000,0.134411 },
{ 0.054127,0.054127,0.140625,0.800000,0.058463 },
{ 0.031250,0.031250,0.140625,0.600000,0.031250 },
{ 0.143205,0.143205,0.140625,0.200000,0.022097 },
{ 0.141490,0.141490,0.132812,0.800000,0.146575 },
{ 0.062500,0.062500,0.132812,0.600000,0.044194 },
{ 0.116927,0.116927,0.125000,0.800000,0.121031 },
{ 0.044194,0.044194,0.125000,0.400000,0.022097 },
{ 0.022097,0.000000,0.117188,0.800000,0.022097 },
{ 0.196403,0.246063,0.117188,0.200000,0.031250 },
{ 0.163876,0.163876,0.109375,0.800000,0.165359 },
{ 0.066291,0.066291,0.109375,0.800000,0.079672 },
{ 0.112673,0.114820,0.109375,0.200000,0.022097 },
{ 0.250975,0.250975,0.101562,0.800000,0.296464 },
{ 0.178152,0.178152,0.101562,0.800000,0.197642 },
{ 0.076547,0.076547,0.101562,0.800000,0.079672 },
{ 0.058463,0.058463,0.101562,0.800000,0.073288 },
{ 0.038273,0.022097,0.101562,0.800000,0.038273 },
{ 0.022097,0.022097,0.101562,0.800000,0.038273 },
{ 0.082680,0.082680,0.078125,1.000000,0.110485 },
{ 0.350780,0.350780,0.078125,0.800000,0.512062 },
{ 0.096319,0.096319,0.078125,0.800000,0.128847 },
{ 0.069877,0.069877,0.078125,0.800000,0.082680 },
{ 0.062500,0.062500,0.078125,0.800000,0.073288 },
{ 0.049411,0.049411,0.078125,0.800000,0.066291 },
{ 0.038273,0.000000,0.078125,0.800000,0.038273 },
{ 0.460871,0.460871,0.078125,0.600000,0.397133 },
{ 0.141490,0.141490,0.078125,0.600000,0.139754 },
{ 0.101262,0.103645,0.078125,0.600000,0.093750 },
{ 0.062500,0.062500,0.078125,0.600000,0.062500 },
{ 0.022097,0.031250,0.078125,0.600000,0.022097 },
{ 0.022097,0.022097,0.078125,0.400000,0.022097 },
{ 0.031250,0.038273,0.078125,0.000000,0.022097 },
{ 0.243068,0.243068,0.070312,1.000000,0.308569 },
{ 0.066291,0.066291,0.070312,1.000000,0.121031 },
{ 0.260521,0.260521,0.070312,0.800000,0.366439 },
{ 0.121031,0.101262,0.070312,0.800000,0.132583 },
{ 0.082680,0.082680,0.070312,0.800000,0.091109 },
{ 0.066291,0.096319,0.070312,0.800000,0.116927 },
{ 0.058463,0.000000,0.070312,0.800000,0.058463 },
{ 0.049411,0.000000,0.070312,0.800000,0.049411 },
{ 0.044194,0.044194,0.070312,0.800000,0.054127 },
{ 0.038273,0.038273,0.070312,0.800000,0.058463 },
{ 0.031250,0.000000,0.070312,0.800000,0.031250 },
{ 0.022097,0.031250,0.070312,0.800000,0.062500 },
{ 0.460871,0.511107,0.070312,0.600000,0.421006 },
{ 0.098821,0.098821,0.070312,0.600000,0.098821 },
{ 0.082680,0.085582,0.070312,0.600000,0.082680 },
{ 0.073288,0.073288,0.070312,0.600000,0.058463 },
{ 0.066291,0.066291,0.070312,0.600000,0.062500 },
{ 0.044194,0.044194,0.070312,0.600000,0.031250 },
{ 0.031250,0.038273,0.070312,0.600000,0.031250 },
{ 0.022097,0.000000,0.070312,0.600000,0.022097 },
{ 0.069877,0.069877,0.070312,0.400000,0.031250 },
{ 0.062500,0.062500,0.070312,0.400000,0.031250 },
{ 0.031250,0.031250,0.070312,0.400000,0.022097 },
{ 0.038273,0.044194,0.070312,0.000000,0.000000 },
{ 0.031250,0.031250,0.070312,0.000000,0.000000 },
{ 0.022097,0.031250,0.070312,0.000000,0.000000 },
{ 0.198874,0.198874,0.062500,1.000000,0.276876 },
{ 0.069877,0.069877,0.062500,1.000000,0.125000 },
{ 0.282981,0.282981,0.062500,0.800000,0.409840 },
{ 0.149870,0.149870,0.062500,0.800000,0.200098 },
{ 0.103645,0.103645,0.062500,0.800000,0.143205 },
{ 0.062500,0.000000,0.062500,0.800000,0.062500 },
{ 0.058463,0.058463,0.062500,0.800000,0.096319 },
{ 0.054127,0.000000,0.062500,0.800000,0.054127 },
{ 0.038273,0.000000,0.062500,0.800000,0.038273 },
{ 0.031250,0.038273,0.062500,0.800000,0.054127 },
{ 0.022097,0.022097,0.062500,0.800000,0.044194 },
{ 0.069877,0.069877,0.062500,0.600000,0.073288 },
{ 0.069877,0.098821,0.062500,0.600000,0.091109 },
{ 0.058463,0.058463,0.062500,0.600000,0.062500 },
{ 0.038273,0.038273,0.062500,0.600000,0.038273 },
{ 0.031250,0.038273,0.062500,0.600000,0.044194 },
{ 0.022097,0.000000,0.062500,0.600000,0.022097 },
{ 0.088388,0.088388,0.062500,0.400000,0.038273 },
{ 0.066291,0.073288,0.062500,0.400000,0.031250 },
{ 0.038273,0.038273,0.062500,0.400000,0.022097 },
{ 0.022097,0.022097,0.062500,0.200000,0.022097 },
{ 0.038273,0.038273,0.062500,0.000000,0.000000 },
{ 0.227503,0.227503,0.054688,1.000000,0.374348 },
{ 0.066291,0.066291,0.054688,1.000000,0.108253 },
{ 0.337297,0.337297,0.054688,0.800000,0.547098 },
{ 0.091109,0.091109,0.054688,0.800000,0.121031 },
{ 0.079672,0.079672,0.054688,0.800000,0.134411 },
{ 0.073288,0.073288,0.054688,0.800000,0.105974 },
{ 0.066291,0.000000,0.054688,0.800000,0.066291 },
{ 0.062500,0.000000,0.054688,0.800000,0.062500 },
{ 0.058463,0.058463,0.054688,0.800000,0.091109 },
{ 0.038273,0.038273,0.054688,0.800000,0.058463 },
{ 0.031250,0.031250,0.054688,0.800000,0.054127 },
{ 0.022097,0.022097,0.054688,0.800000,0.038273 },
{ 0.171163,0.171163,0.054688,0.600000,0.182217 },
{ 0.105974,0.105974,0.054688,0.600000,0.096319 },
{ 0.096319,0.096319,0.054688,0.600000,0.091109 },
{ 0.069877,0.069877,0.054688,0.600000,0.069877 },
{ 0.058463,0.058463,0.054688,0.600000,0.058463 },
{ 0.038273,0.038273,0.054688,0.600000,0.031250 },
{ 0.031250,0.038273,0.054688,0.600000,0.038273 },
{ 0.022097,0.000000,0.054688,0.600000,0.022097 },
{ 0.220971,0.220971,0.054688,0.400000,0.076547 },
{ 0.073288,0.073288,0.054688,0.400000,0.031250 },
{ 0.058463,0.058463,0.054688,0.400000,0.022097 },
{ 0.038273,0.038273,0.054688,0.000000,0.000000 },
{ 0.443045,0.443045,0.046875,1.000000,0.671330 },
{ 0.302980,0.302980,0.046875,1.000000,0.450694 },
{ 0.085582,0.085582,0.046875,1.000000,0.156250 },
{ 0.069877,0.069877,0.046875,1.000000,0.132583 },
{ 0.038273,0.000000,0.046875,1.000000,0.044194 },
{ 0.443045,0.443045,0.046875,0.800000,0.641196 },
{ 0.098821,0.098821,0.046875,0.800000,0.154680 },
{ 0.085582,0.085582,0.046875,0.800000,0.141490 },
{ 0.076547,0.076547,0.046875,0.800000,0.123031 },
{ 0.069877,0.062500,0.046875,0.800000,0.105974 },
{ 0.069877,0.069877,0.046875,0.800000,0.108253 },
{ 0.066291,0.066291,0.046875,0.800000,0.110485 },
{ 0.062500,0.000000,0.046875,0.800000,0.062500 },
{ 0.058463,0.062500,0.046875,0.800000,0.085582 },
{ 0.038273,0.038273,0.046875,0.800000,0.066291 },
{ 0.031250,0.038273,0.046875,0.800000,0.062500 },
{ 0.031250,0.031250,0.046875,0.800000,0.058463 },
{ 0.022097,0.000000,0.046875,0.800000,0.022097 },
{ 0.022097,0.022097,0.046875,0.800000,0.049411 },
{ 0.460871,0.460871,0.046875,0.600000,0.470310 },
{ 0.229640,0.229640,0.046875,0.600000,0.242061 },
{ 0.160869,0.160869,0.046875,0.600000,0.173993 },
{ 0.110485,0.110485,0.046875,0.600000,0.114820 },
{ 0.079672,0.082680,0.046875,0.600000,0.082680 },
{ 0.066291,0.066291,0.046875,0.600000,0.069877 },
{ 0.038273,0.044194,0.046875,0.600000,0.054127 },
{ 0.022097,0.000000,0.046875,0.600000,0.022097 },
{ 0.054127,0.054127,0.046875,0.400000,0.031250 },
{ 0.031250,0.038273,0.046875,0.400000,0.022097 },
{ 0.022097,0.031250,0.046875,0.400000,0.022097 },
{ 0.208464,0.209631,0.046875,0.200000,0.022097 },
{ 0.121031,0.171163,0.046875,0.200000,0.031250 },
{ 0.098821,0.098821,0.046875,0.200000,0.038273 },
{ 0.054127,0.054127,0.046875,0.200000,0.022097 },
{ 0.038273,0.038273,0.046875,0.000000,0.022097 },
{ 0.022097,0.022097,0.046875,0.000000,0.000000 },
{ 0.392806,0.444695,0.039062,1.000000,0.625000 },
{ 0.302980,0.331456,0.039062,1.000000,0.487139 },
{ 0.302980,0.302980,0.039062,1.000000,0.466662 },
{ 0.215376,0.215376,0.039062,1.000000,0.319454 },
{ 0.088388,0.031250,0.039062,1.000000,0.098821 },
{ 0.076547,0.076547,0.039062,1.000000,0.139754 },
{ 0.062500,0.062500,0.039062,1.000000,0.121031 },
{ 0.049411,0.069877,0.039062,1.000000,0.130728 },
{ 0.044194,0.044194,0.039062,1.000000,0.091109 },
{ 0.022097,0.022097,0.039062,1.000000,0.073288 },
{ 0.445792,0.445792,0.039062,0.800000,0.625781 },
{ 0.201314,0.201314,0.039062,0.800000,0.297286 },
{ 0.116927,0.116927,0.039062,0.800000,0.208464 },
{ 0.098821,0.098821,0.039062,0.800000,0.169731 },
{ 0.088388,0.000000,0.039062,0.800000,0.088388 },
{ 0.079672,0.079672,0.039062,0.800000,0.126938 },
{ 0.073288,0.073288,0.039062,0.800000,0.112673 },
{ 0.069877,0.000000,0.039062,0.800000,0.069877 },
{ 0.069877,0.069877,0.039062,0.800000,0.123031 },
{ 0.066291,0.000000,0.039062,0.800000,0.066291 },
{ 0.066291,0.066291,0.039062,0.800000,0.118996 },
{ 0.062500,0.000000,0.039062,0.800000,0.062500 },
{ 0.058463,0.000000,0.039062,0.800000,0.058463 },
{ 0.054127,0.000000,0.039062,0.800000,0.054127 },
{ 0.049411,0.049411,0.039062,0.800000,0.096319 },
{ 0.044194,0.000000,0.039062,0.800000,0.049411 },
{ 0.044194,0.044194,0.039062,0.800000,0.082680 },
{ 0.031250,0.031250,0.039062,0.800000,0.062500 },
{ 0.022097,0.000000,0.039062,0.800000,0.022097 },
{ 0.022097,0.022097,0.039062,0.800000,0.049411 },
{ 0.244070,0.244070,0.039062,0.600000,0.294812 },
{ 0.210793,0.210793,0.039062,0.600000,0.231756 },
{ 0.121031,0.121031,0.039062,0.600000,0.134411 },
{ 0.096319,0.096319,0.039062,0.600000,0.101262 },
{ 0.062500,0.066291,0.039062,0.600000,0.069877 },
{ 0.058463,0.058463,0.039062,0.600000,0.066291 },
{ 0.044194,0.044194,0.039062,0.600000,0.054127 },
{ 0.038273,0.044194,0.039062,0.600000,0.054127 },
{ 0.031250,0.038273,0.039062,0.600000,0.038273 },
{ 0.217631,0.217631,0.039062,0.400000,0.088388 },
{ 0.126938,0.128847,0.039062,0.400000,0.062500 },
{ 0.085582,0.085582,0.039062,0.400000,0.044194 },
{ 0.082680,0.082680,0.039062,0.400000,0.044194 },
{ 0.066291,0.066291,0.039062,0.400000,0.031250 },
{ 0.054127,0.054127,0.039062,0.400000,0.022097 },
{ 0.022097,0.022097,0.039062,0.400000,0.022097 },
{ 0.173993,0.173993,0.039062,0.200000,0.022097 },
{ 0.121031,0.121031,0.039062,0.200000,0.031250 },
{ 0.058463,0.058463,0.039062,0.200000,0.031250 },
{ 0.044194,0.044194,0.039062,0.200000,0.022097 },
{ 0.022097,0.022097,0.039062,0.200000,0.022097 },
{ 0.069877,0.069877,0.039062,0.000000,0.022097 },
{ 0.054127,0.054127,0.039062,0.000000,0.022097 },
{ 0.031250,0.038273,0.039062,0.000000,0.000000 },
{ 0.022097,0.031250,0.039062,0.000000,0.000000 },
{ 0.022097,0.022097,0.039062,0.000000,0.000000 },
{ 0.354243,0.354243,0.031250,1.000000,0.608373 },
{ 0.222073,0.222073,0.031250,1.000000,0.366439 },
{ 0.076547,0.076547,0.031250,1.000000,0.144900 },
{ 0.069877,0.000000,0.031250,1.000000,0.073288 },
{ 0.049411,0.096319,0.031250,1.000000,0.153093 },
{ 0.132583,0.132583,0.031250,0.800000,0.211948 },
{ 0.130728,0.098821,0.031250,0.800000,0.178152 },
{ 0.114820,0.114820,0.031250,0.800000,0.200098 },
{ 0.088388,0.088388,0.031250,0.800000,0.151490 },
{ 0.085582,0.000000,0.031250,0.800000,0.085582 },
{ 0.079672,0.079672,0.031250,0.800000,0.148232 },
{ 0.076547,0.000000,0.031250,0.800000,0.076547 },
{ 0.076547,0.076547,0.031250,0.800000,0.126938 },
{ 0.069877,0.000000,0.031250,0.800000,0.069877 },
{ 0.069877,0.069877,0.031250,0.800000,0.123031 },
{ 0.066291,0.066291,0.031250,0.800000,0.125000 },
{ 0.062500,0.000000,0.031250,0.800000,0.076547 },
{ 0.062500,0.062500,0.031250,0.800000,0.112673 },
{ 0.049411,0.049411,0.031250,0.800000,0.088388 },
{ 0.038273,0.000000,0.031250,0.800000,0.038273 },
{ 0.022097,0.022097,0.031250,0.800000,0.058463 },
{ 0.352862,0.352862,0.031250,0.600000,0.376948 },
{ 0.183552,0.183552,0.031250,0.600000,0.200098 },
{ 0.143205,0.143205,0.031250,0.600000,0.144900 },
{ 0.110485,0.103645,0.031250,0.600000,0.098821 },
{ 0.103645,0.103645,0.031250,0.600000,0.103645 },
{ 0.088388,0.088388,0.031250,0.600000,0.098821 },
{ 0.069877,0.069877,0.031250,0.600000,0.073288 },
{ 0.066291,0.062500,0.031250,0.600000,0.079672 },
{ 0.066291,0.066291,0.031250,0.600000,0.076547 },
{ 0.058463,0.058463,0.031250,0.600000,0.066291 },
{ 0.054127,0.054127,0.031250,0.600000,0.062500 },
{ 0.054127,0.022097,0.031250,0.600000,0.044194 },
{ 0.038273,0.038273,0.031250,0.600000,0.049411 },
{ 0.031250,0.038273,0.031250,0.600000,0.044194 },
{ 0.022097,0.000000,0.031250,0.600000,0.022097 },
{ 0.022097,0.022097,0.031250,0.600000,0.031250 },
{ 0.230700,0.230700,0.031250,0.400000,0.123031 },
{ 0.103645,0.103645,0.031250,0.400000,0.073288 },
{ 0.096319,0.096319,0.031250,0.400000,0.058463 },
{ 0.085582,0.085582,0.031250,0.400000,0.058463 },
{ 0.076547,0.076547,0.031250,0.400000,0.038273 },
{ 0.066291,0.066291,0.031250,0.400000,0.038273 },
{ 0.058463,0.076547,0.031250,0.400000,0.031250 },
{ 0.054127,0.058463,0.031250,0.400000,0.031250 },
{ 0.044194,0.044194,0.031250,0.400000,0.022097 },
{ 0.038273,0.049411,0.031250,0.400000,0.022097 },
{ 0.022097,0.022097,0.031250,0.400000,0.000000 },
{ 0.069877,0.073288,0.031250,0.200000,0.022097 },
{ 0.066291,0.069877,0.031250,0.200000,0.022097 },
{ 0.054127,0.054127,0.031250,0.200000,0.022097 },
{ 0.022097,0.022097,0.031250,0.200000,0.022097 },
{ 0.054127,0.054127,0.031250,0.000000,0.000000 },
{ 0.038273,0.038273,0.031250,0.000000,0.000000 },
{ 0.022097,0.031250,0.031250,0.000000,0.000000 },
{ 0.022097,0.022097,0.031250,0.000000,0.000000 },
{ 0.268823,0.268823,0.023438,1.000000,0.482608 },
{ 0.136216,0.136216,0.023438,1.000000,0.250975 },
{ 0.125000,0.125000,0.023438,1.000000,0.236965 },
{ 0.058463,0.058463,0.023438,1.000000,0.112673 },
{ 0.054127,0.054127,0.023438,1.000000,0.118996 },
{ 0.049411,0.049411,0.023438,1.000000,0.108253 },
{ 0.044194,0.044194,0.023438,1.000000,0.088388 },
{ 0.022097,0.022097,0.023438,1.000000,0.079672 },
{ 0.255792,0.255792,0.023438,0.800000,0.499022 },
{ 0.156250,0.156250,0.023438,0.800000,0.270633 },
{ 0.128847,0.128847,0.023438,0.800000,0.220971 },
{ 0.096319,0.096319,0.023438,0.800000,0.183552 },
{ 0.093750,0.093750,0.023438,0.800000,0.165359 },
{ 0.088388,0.000000,0.023438,0.800000,0.085582 },
{ 0.079672,0.079672,0.023438,0.800000,0.151490 },
{ 0.069877,0.000000,0.023438,0.800000,0.069877 },
{ 0.066291,0.000000,0.023438,0.800000,0.073288 },
{ 0.058463,0.000000,0.023438,0.800000,0.062500 },
{ 0.058463,0.058463,0.023438,0.800000,0.101262 },
{ 0.054127,0.000000,0.023438,0.800000,0.054127 },
{ 0.049411,0.049411,0.023438,0.800000,0.096319 },
{ 0.044194,0.000000,0.023438,0.800000,0.049411 },
{ 0.044194,0.049411,0.023438,0.800000,0.093750 },
{ 0.038273,0.044194,0.023438,0.800000,0.073288 },
{ 0.038273,0.000000,0.023438,0.800000,0.044194 },
{ 0.031250,0.000000,0.023438,0.800000,0.049411 },
{ 0.031250,0.038273,0.023438,0.800000,0.058463 },
{ 0.022097,0.000000,0.023438,0.800000,0.022097 },
{ 0.022097,0.022097,0.023438,0.800000,0.062500 },
{ 0.176777,0.176777,0.023438,0.600000,0.213097 },
{ 0.143205,0.143205,0.023438,0.600000,0.173993 },
{ 0.091109,0.091109,0.023438,0.600000,0.105974 },
{ 0.066291,0.066291,0.023438,0.600000,0.076547 },
{ 0.054127,0.054127,0.023438,0.600000,0.058463 },
{ 0.044194,0.049411,0.023438,0.600000,0.054127 },
{ 0.044194,0.069877,0.023438,0.600000,0.066291 },
{ 0.022097,0.031250,0.023438,0.600000,0.038273 },
{ 0.022097,0.000000,0.023438,0.600000,0.022097 },
{ 0.022097,0.022097,0.023438,0.600000,0.031250 },
{ 0.134411,0.134411,0.023438,0.400000,0.069877 },
{ 0.103645,0.103645,0.023438,0.400000,0.058463 },
{ 0.069877,0.069877,0.023438,0.400000,0.038273 },
{ 0.066291,0.066291,0.023438,0.400000,0.022097 },
{ 0.058463,0.062500,0.023438,0.400000,0.022097 },
{ 0.054127,0.054127,0.023438,0.400000,0.031250 },
{ 0.038273,0.096319,0.023438,0.400000,0.038273 },
{ 0.022097,0.022097,0.023438,0.400000,0.000000 },
{ 0.204920,0.204920,0.023438,0.200000,0.049411 },
{ 0.118996,0.118996,0.023438,0.200000,0.054127 },
{ 0.069877,0.096319,0.023438,0.200000,0.022097 },
{ 0.049411,0.049411,0.023438,0.200000,0.022097 },
{ 0.022097,0.022097,0.023438,0.200000,0.022097 },
{ 0.088388,0.088388,0.023438,0.000000,0.022097 },
{ 0.062500,0.062500,0.023438,0.000000,0.000000 },
{ 0.022097,0.022097,0.023438,0.000000,0.000000 },
{ 0.333659,0.333659,0.015625,1.000000,0.608373 },
{ 0.121031,0.121031,0.015625,1.000000,0.250975 },
{ 0.076547,0.076547,0.015625,1.000000,0.151490 },
{ 0.069877,0.069877,0.015625,1.000000,0.148232 },
{ 0.058463,0.058463,0.015625,1.000000,0.132583 },
{ 0.049411,0.049411,0.015625,1.000000,0.116927 },
{ 0.038273,0.038273,0.015625,1.000000,0.103645 },
{ 0.022097,0.000000,0.015625,1.000000,0.031250 },
{ 0.022097,0.022097,0.015625,1.000000,0.066291 },
{ 0.222073,0.222073,0.015625,0.800000,0.380173 },
{ 0.128847,0.128847,0.015625,0.800000,0.243068 },
{ 0.088388,0.088388,0.015625,0.800000,0.163876 },
{ 0.076547,0.062500,0.015625,0.800000,0.123031 },
{ 0.069877,0.069877,0.015625,0.800000,0.136216 },
{ 0.066291,0.000000,0.015625,0.800000,0.069877 },
{ 0.062500,0.038273,0.015625,0.800000,0.085582 },
{ 0.058463,0.000000,0.015625,0.800000,0.062500 },
{ 0.054127,0.054127,0.015625,0.800000,0.103645 },
{ 0.044194,0.044194,0.015625,0.800000,0.082680 },
{ 0.038273,0.038273,0.015625,0.800000,0.073288 },
{ 0.031250,0.000000,0.015625,0.800000,0.031250 },
{ 0.022097,0.031250,0.015625,0.800000,0.058463 },
{ 0.222073,0.222073,0.015625,0.600000,0.272431 },
{ 0.112673,0.165359,0.015625,0.600000,0.188798 },
{ 0.112673,0.112673,0.015625,0.600000,0.144900 },
{ 0.088388,0.088388,0.015625,0.600000,0.110485 },
{ 0.038273,0.044194,0.015625,0.600000,0.049411 },
{ 0.031250,0.000000,0.015625,0.600000,0.038273 },
{ 0.022097,0.022097,0.015625,0.600000,0.038273 },
{ 0.382733,0.382733,0.015625,0.400000,0.162380 },
{ 0.082680,0.085582,0.015625,0.400000,0.044194 },
{ 0.044194,0.049411,0.015625,0.400000,0.031250 },
{ 0.022097,0.022097,0.015625,0.400000,0.022097 },
{ 0.085582,0.085582,0.015625,0.200000,0.022097 },
{ 0.069877,0.069877,0.015625,0.200000,0.031250 },
{ 0.062500,0.066291,0.015625,0.200000,0.000000 },
{ 0.022097,0.022097,0.007812,1.000000,0.091109 },
{ 0.091109,0.091109,0.007812,0.800000,0.173993 },
{ 0.066291,0.066291,0.007812,0.800000,0.118996 },
{ 0.054127,0.058463,0.007812,0.800000,0.116927 },
{ 0.038273,0.000000,0.007812,0.800000,0.054127 },
{ 0.186193,0.186193,0.007812,0.600000,0.235932 },
{ 0.118996,0.105974,0.007812,0.600000,0.134411 },
{ 0.076547,0.076547,0.007812,0.600000,0.091109 },
{ 0.069877,0.069877,0.007812,0.600000,0.082680 },
{ 0.066291,0.066291,0.007812,0.600000,0.082680 },
{ 0.058463,0.038273,0.000000,0.800000,0.098821 },
{ 0.038273,0.022097,0.000000,0.800000,0.069877 },
{ 0.022097,0.038273,0.000000,0.800000,0.085582 },
{ 0.066291,0.069877,0.000000,0.600000,0.091109 },
{ 0.054127,0.054127,0.000000,0.600000,0.062500 },
{ 0.044194,0.044194,0.000000,0.600000,0.058463 },
{ 0.022097,0.022097,0.000000,0.600000,0.044194 },
{ 0.076547,0.079672,0.000000,0.400000,0.038273 },
{ 0.022097,0.022097,0.000000,0.400000,0.022097 },
}; /* 366 cases */

static const float Hard_coded_weights [(IN_N * HID_N) + (HID_N * OUT_N)] = {
3.235386,0.956375,-2.263822,-0.938662,-3.259808,1.973934,
0.681164,-2.761034,0.974361,-5.049428,-2.369908,5.078909,
-1.543787,1.230283,-3.997151,0.775778,0.279331,-0.812299,
-7.308127,0.529813,0.013233,1.538310,-0.065645,1.711841,
-0.040194,0.014488,-0.499831,-0.589988,-1.619819,-0.348417,
1.556765,1.071569,-1.030902,-0.479584,0.984601,-1.459256,
-8.520793,-12.118382,-0.722938,-0.263354,0.408103,0.053183,
-1.046983,-0.554245,-0.422486,-0.253554,-0.399313,-0.152260,
-1.808737,0.685275,3.574663,3.017537,0.798322,-0.238852,
-0.683935,-0.568840,1.497294,-0.247287,-1.599820,-3.199625,
0.531417,0.477442,4.081839,0.313259,-0.840464,-0.182970,
1.807147,3.113346,0.089809,0.860311,1.797278,0.265959,
-0.379720,-0.703635,-0.772423,-0.749950,-4.279593,-3.167622,
2.876281,2.207856,0.867398,-2.071645,1.436387,-0.371106,
0.326914,0.383471,-0.319191,-1.649726,-1.253920,-3.778691,
0.234376,-0.177537,2.548882,0.876411,-5.838309,
-4.265553,2.273356,-0.802679,-0.930389,-3.361013,
};

static int	Units[NUM_LAYERS] = {IN_N, HID_N, OUT_N};
float		TestError;
static void	save_weights (void);


/* ANN uses floats (from 0.0 to 1.0),
 * but we do like integers
 * (grades, intervals, numbers of repetitions are integers)
 */

/*  grades: 0 to MA_MAX_GRADE (5)
 *  interval 0 to MA_MAX_N_OF_RPTS (2048)
 *  number of repetitions: 0 to MA_MAX_INTERVAL (128)
 */

INLINE float f_interval (u_short in) { /* "f" stands for "float" */
	if (in > MA_MAX_INTERVAL) {
		ma_error ("f_interval(in): in > MA_MAX_INTERVAL, "
				"truncated to MA_MAX_INTERVAL. (in=%d, MA_MAX_INTERVAL=%d)\n",
				in, MA_MAX_INTERVAL);
		in = MA_MAX_INTERVAL;
	}
	return sqrt ( (float)in / (float)MA_MAX_INTERVAL );
}

u_short us_interval (const float in) { /* "us" stands for "unsigned short" */
	assert (in >= 0.0f  &&  in <= 1.0f);

	return (u_short) (in*in * (float)MA_MAX_INTERVAL + 0.5f);
}

INLINE float f_n_of_rpt (const u_short in) {
	if (in > MA_MAX_N_OF_RPTS) {
		ma_error ("Warning! Input data in f_n_of_rpt() > MA_MAX_N_OF_RPTS (%d).\n"\
				"Truncated to MA_MAX_N_OF_RPTS\n", MA_MAX_N_OF_RPTS);
		return 1;
	}
	return (((float)in) / (float)MA_MAX_N_OF_RPTS);
}

u_short us_n_of_rpt (const float in) {
	assert (in >= 0.0f && in <= 1.0f);

	return (u_short)(in * (float)MA_MAX_N_OF_RPTS + 0.5f);
}

INLINE float f_grade (const u_short in) {
	static float lut[6] = { 0.0, 0.2, 0.4, 0.6, 0.8, 1.0 };

	if (in > 5) {
		ma_error ("ERROR. grade > 5 in f_grade(), truncated to 5\n");
		return lut[5];
	}
	return lut[in];	
}

u_short us_grade (const float in) {
	assert (in >= 0.0f  && in <= 1.0f);

	return (u_short)(in * 5.0f + 0.5f);
}

char *pathed (const char *filename) {
	/* 4 buffers to make possible up to 5 calls, e.g.:
	 * something (pathed("blah"), pathed("blah2")...)
	 * and keep own data for each call
	 */
	static char buf[4][MAX_PATH_LEN + MAX_FILENAME_LEN];
	static int i = 0;

	i = ++i > 3 ? 0 : i;
	strcpy (buf[i], ma_Path);
	assert (strlen (filename) < MAX_FILENAME_LEN);
	strcat (buf[i], filename);
	return buf[i];
}

static int load_weights (void) {
	register int l, i;
	int counter;
	FILE *f_weights = fopen (pathed ("weights.bin"), "rb");

	if (!f_weights) 
		goto use_hard_coded_weights;

	for (l = 1; l < NUM_LAYERS; l++)
		for (i = 0; i < Net.Layer[l]->Units; i++)
			if (fread (Net.Layer[l]->Weight[i],
						sizeof(float), Net.Layer[l-1]->Units, f_weights) 
					!= Net.Layer[l-1]->Units) {
				ma_error ("load_weights(): "
						"Fatal error while reading weights.bin\n");
				goto use_hard_coded_weights;
			}
	fclose (f_weights);
	save_weights();
	return 0;

use_hard_coded_weights:

	for (l = 1, counter = 0; l < NUM_LAYERS; l++)
		for (i = 0; i < Net.Layer[l]->Units; i++) {
			memcpy (Net.Layer[l]->Weight[i], &Hard_coded_weights[counter],
					Net.Layer[l-1]->Units * sizeof(float));
			counter += Net.Layer[l-1]->Units;
		}

	save_weights();
	return 0;
}

static void free_mem_after_ann (void) {
	register u_short l, i;

	for (l = 0; l < NUM_LAYERS; l++) {
		if (l > 0) {
			for (i = 0; i < Units[l]; i++)
				free (Net.Layer[l]->Weight[i]);
			free (Net.Layer[l]->Weight);
		}
		free (Net.Layer[l]->Output);
		free (Net.Layer[l]);
	}
	free (Net.Layer);
}

static void generate_ann (void) {
	register int l,i;

	Data = (float**) malloc (N_DATA_JUMP * sizeof (float*));

	Net.Layer = (LAYER**) malloc (NUM_LAYERS * sizeof (LAYER*));

	for (l = 0; l < NUM_LAYERS; l++) {
		Net.Layer[l] = (LAYER*) malloc (sizeof (LAYER));
		Net.Layer[l]->Units      = Units[l];
		Net.Layer[l]->Output     = (float*)  calloc (Units[l] << 1, sizeof (float));
		Net.Layer[l]->Error      = Net.Layer[l]->Output + Units[l];
		if (l > 0) {
			Net.Layer[l]->Weight     = (float**) malloc ((Units[l]<<1) * sizeof (float*));
			Net.Layer[l]->WeightSave = Net.Layer[l]->Weight + Units[l];
			for (i = 0; i < Units[l]; i++) {
				Net.Layer[l]->Weight[i] = (float*) malloc ((Units[l-1]<<1) * sizeof (float));
				Net.Layer[l]->WeightSave[i] = Net.Layer[l]->Weight[i] + Units[l-1];
			}
		}
	}
	Net.OutputLayer = Net.Layer[NUM_LAYERS - 1];
	Net.Eta         = 0.1;
}

static void save_weights (void) {
	register int i, l;

	for (l = NUM_LAYERS - 1; l > 0; l--) {
		for (i = 0; i < Net.Layer[l]->Units; i++)
			memcpy (Net.Layer[l]->WeightSave[i], Net.Layer[l]->Weight[i],
					sizeof(float) * (Net.Layer[l-1]->Units));
	}
}

static void restore_weights (void) {
	register int i, l;

	for (l = NUM_LAYERS - 1; l > 0; l--) {
		for (i = 0; i < Net.Layer[l]->Units; i++)
			memcpy (Net.Layer[l]->Weight[i], Net.Layer[l]->WeightSave[i],
					sizeof(float) * (Net.Layer[l-1]->Units));
	}
}

INLINE static void propagate_net (void) {
	register int l, i, j;
	register float Sum;

	for (l = 0; l < NUM_LAYERS - 1; l++)
		for (i = 0; i < Net.Layer[l+1]->Units; i++) {
			Sum = 0;
			for (j = 0; j < Net.Layer[l]->Units; j++)
				Sum += Net.Layer[l+1]->Weight[i][j] * Net.Layer[l]->Output[j];
#if defined(_ANSI_SOURCE) && defined(_POSIX_SOURCE)
			Net.Layer[l+1]->Output[i] = (1.0f / (1.0f + expf (-Sum)));
#else
			Net.Layer[l+1]->Output[i] = (1.0f / (1.0f + exp  (-Sum)));
#endif
		}
}

INLINE static void compute_output_error (float Target) {
	register float Out, Err;

	Net.Error = 0;
	Err = Target - (Out = Net.OutputLayer->Output[0]);
	Net.OutputLayer->Error[0] = Out * (1.0f - Out) * Err;
	Net.Error += 0.5f * Err * Err;
}

INLINE static void backpropagate_net (void) {
	register short i, j, l;
	register float Out, Err;

	for (l = NUM_LAYERS - 1; l > 0; l--)
		for (i = 0; i < Net.Layer[l-1]->Units; i++) {
			Out = Net.Layer[l-1]->Output[i];
			Err = 0;
			for (j = 0; j < Net.Layer[l]->Units; j++)
				Err += Net.Layer[l]->Weight[j][i] * Net.Layer[l]->Error[j];
			/*
			 * an idea: if we would use one table for weights and errors,
			 * we could potentially use more efficently cache *here*.
			 * table[] = {weight_for_a_neuron, error, weight_for_a_neuron, error, .. }
			 * We could also use one pointer here to go through the table.
			 */

			Net.Layer[l-1]->Error[i] = Out * (1.0f - Out) * Err;
		}
}

INLINE static void simulate_net (float *input, float *output, char learn) {
	memcpy (Net.Layer[0]->Output, input,  sizeof (float) * Net.Layer[0]->Units);
	propagate_net ();
	if (learn == 0)
		memcpy (output, Net.OutputLayer->Output, sizeof (float) * Net.OutputLayer->Units);
	else {
		register short l, i, j;
		compute_output_error (input[IN_N]);
		backpropagate_net();
		for (l = 1; l < NUM_LAYERS; l++) /* adjusting weights */
			for (i = 0; i < Net.Layer[l]->Units; i++)
				for (j = 0; j < Net.Layer[l-1]->Units; j++)
					Net.Layer[l]->Weight[i][j] +=
						Net.Layer[l]->Error[i] * Net.Layer[l-1]->Output[j] * Net.Eta;
	}
	return;
}

static void train_net (int epochs) {
	epochs *= NData;
	do
		simulate_net (Data[ ma_rand(NData) ], NULL, 1);
	while  (--epochs > 0);
}

float test_net (void) {
	register u_long i;

	/* TestError should be computed always on the same number of cases.
	 * Otherwise:
	 * 1) TestError would grow as number of elements grows,
	 *    and we wouldn't be able to see
	 *    if TestError grows because ANN is not trained for new elements, or
	 *    _just_ number of cases got higher.
	 * 2) test_net() would be getting slower as number of cases gets higher
	 *
	 * TestError computed on not all cases (a constant number of cases is taken randomly)
	 * tends to be *a bit* random, but it's the price we have to pay.
	 * (We actually do some stuff to have a stable value of TestError:
	 *  we compute TestError few times and then take mean of results).
	 *
	 * NData_general (number of cases shipped with MemAid - hardcoded, or in data.bin)
	 * seems to be apriopriate to use as the test number.
	 */

	TestError = 0.0f;
	i = NData_general << 2;
	
	do {
		simulate_net (Data[ ma_rand (NData) ], NULL, 1);
		TestError += Net.Error;
	} while (--i);
		
	TestError /= 4.0f;

	if (TestError > 0.4f)
		ma_error ("*WARNING*: TestError = %f (enormously high!)\n", TestError);
	return TestError;
}

INLINE static char dup (const int first, const int last) {
	register int i;
	
	for (i = first; i < last; i++)
		if (memcmp (Data[i], Data[last], IN_N * sizeof(float)) == 0) {
			Data[i][IN_N] = (Data[last][IN_N] + Data[i][IN_N]) / 2.0;
			return 1;
		}
	return 0;
}

static void add_data_from_file (FILE *f_in) {
	do {
		if (dup(NData_general, NData-1)) {
			NData--;
			continue;
		}
		if ((NData & (N_DATA_JUMP-1)) == 0) /* N_DATA_JUMP is power of 2 */
			Data = (float**) realloc (Data, (NData + N_DATA_JUMP + 1) * sizeof(float*));
		Data[NData] = (float*) malloc (IN_OUT_N * sizeof(float));
	} while (fread (Data[ NData++ ], sizeof(float), IN_OUT_N, f_in) == IN_OUT_N);

	free (Data[ --NData ]);
}

static void load_data (void) {
	FILE  *f_in;
	
	NData = 0;
	NData_general = 0;

	if (!(f_in = fopen (pathed ("ann/data.bin"), "rb")))
		f_in = fopen (pathed ("data.bin"), "rb");
	
	if (f_in) {
		add_data_from_file (f_in);
		NData_general = NData;
		fclose (f_in);
	}
	else {
		for (NData = 0; NData < N_HARD_CODED_CASES; NData++) {
			Data[NData] = (float*) malloc (sizeof(float) * IN_OUT_N);
			memcpy (Data[NData], Hard_coded_data[NData], sizeof(float) * IN_OUT_N);
		}
		NData_general = N_HARD_CODED_CASES;
	}
	if (!(f_in = fopen (pathed ("ann/user_data.bin"), "rb")))
		f_in = fopen (pathed ("user_data.bin"), "rb");
		
	if (f_in) {
		add_data_from_file (f_in);
		fclose (f_in);
	}
	return;
}

int nn_init (void) {
#ifdef __BSD_VISIBLE
	srandom ((unsigned int)time (NULL) + (unsigned int)clock ());
#endif
	/*  we will use common rand(), anyways,
	 *  when there is less need for so good random number
	 */
	srand  ((unsigned int)time (NULL) + (unsigned int)clock ());
	F_err = stderr;
	generate_ann ();
	load_data();
	return (load_weights());
}

void ma_train_ann (int n) {
	float MinTestError = 1000.0f;

	do {
		train_net (8);
		test_net();
		if (TestError < MinTestError) {
			MinTestError = TestError;
			save_weights();
		}
		else
			if (TestError > 1.2f * MinTestError)
				restore_weights ();
		if (TestError > 0.1)
			Net.Eta = 0.2f;
		else
			Net.Eta = TestError * 2.0f;
	} while (--n > 0);
	
	restore_weights();
	return;
}

void ma_train_ann_for_x_secs (int secs) {
	long init_time = time (NULL);

	do
		ma_train_ann (16);
	while (time(NULL) - init_time < secs);
	return;
}

void nn_save_weights_to_file (void) {
	register int l, i;
	FILE *f_weights;
	
	if (!(f_weights = fopen (pathed ("ann/weights.bin"), "wb")))
		f_weights = fopen (pathed ("weights.bin"), "wb");

	if (!f_weights)
		ma_error ("Cannot save ANN weights!\n");
	else {
		for (l = 1; l < NUM_LAYERS; l++) {
			for (i = 0; i < Net.Layer[l]->Units; i++) {
				fwrite (Net.Layer[l]->WeightSave[i], sizeof(float),
						Net.Layer[l-1]->Units, f_weights);
			}
		}
		fclose (f_weights);
	}
	return;
}

int nn_deinit (void) {
	nn_save_weights_to_file();
	free_mem_after_ann(); 
	return 0;
}

u_short ma_new_interval (u_short interval, u_short real_interval, u_short rpt_counter, u_short grade) {
	float in[IN_OUT_N], out;
	double d;
	u_short result, final_result;
	int i, j;

	in[0] = f_interval (interval);
	in[1] = f_interval (real_interval);
	in[2] = f_n_of_rpt (rpt_counter);
	in[3] = f_grade    (grade);

	simulate_net (in, &out, 0);
	result = us_interval (out);
	if (result == 0)
		result++; /* we never return 0, it would complicate things (moving element) */

	/* small randomness, -/+15% */

	i = ma_rand(31) - 15; /* random number from -15 to 15 */
	j = result * (100+i);
	d = ((double)j / 100.0);
	final_result = (u_short) (d + 0.5); /* round to closest integer */

	return (final_result);
}

static int add_data (u_short l_ivl, u_short real_l_ivl, u_short rp, u_short l_gr, u_short better_ivl) {
	FILE *out;

	if ((NData & (N_DATA_JUMP-1)) == 0)
		Data = (float**) realloc (Data, (NData + N_DATA_JUMP + 1) * sizeof(float*));

	Data[NData] = (float*) malloc (IN_OUT_N * sizeof(float));

	Data[NData][0] = f_interval (l_ivl);
	Data[NData][1] = f_interval (real_l_ivl);
	Data[NData][2] = f_n_of_rpt (rp);
	Data[NData][3] = f_grade    (l_gr);
	Data[NData][4] = f_interval (better_ivl);

	NData++;
	
	if (!(out = fopen (pathed ("ann/user_data.bin"), "ab")))
		out = fopen (pathed ("user_data.bin"), "ab");

	if (!out) {
		ma_error ("ERROR: cannot open %s nor %s files in add_data()\n",
				pathed ("ann/user_data.bin"), pathed ("user_data.bin"));
		return 1;
	}

	fwrite (Data[NData-1], sizeof(float), IN_OUT_N, out);
	fclose (out);
	return 0;
}

int feedback_to_ann (u_short l_ivl, u_short real_l_ivl, u_short rp, u_short l_gr, u_short ivl_by_ann,
		u_short real_act_ivl, u_short act_grade) {
	float factor;
	u_short n_ivl; /* new interval as feedback */
	
	if (real_act_ivl == 0) /* don't add data from Final drill - it would add a lot of mess */
		return 1;

	n_ivl = real_act_ivl;
	
	switch (act_grade) {
		case 0:
			if (real_act_ivl > ivl_by_ann)
				n_ivl = (real_act_ivl + ivl_by_ann) >> 1;
			factor = 0.4;
			break;
		case 1:
			if (real_act_ivl > ivl_by_ann)
				n_ivl = (real_act_ivl + ivl_by_ann) >> 1;	
			factor = 0.55;
			break;
		case 2:
			if (real_act_ivl > ivl_by_ann)
				n_ivl = (real_act_ivl + ivl_by_ann) >> 1;
			factor = 0.7;
			break;
		case 3:
			if (real_act_ivl > ivl_by_ann)
				n_ivl = (real_act_ivl + ivl_by_ann) >> 1;
			factor = 0.85;
			break;
		case 4: /* grade 4 is perfect - interval is ok */
			factor = 1;
			break;
		case 5:
			if (real_act_ivl < ivl_by_ann) 
				n_ivl = (real_act_ivl + ivl_by_ann) >> 1;
			factor = 1.2;
			break;
		default:
			ma_error ("feedback_to_ann(): ERROR, default case\n");
			return 1;
	}
	n_ivl = (u_short) ((float)n_ivl * factor + 0.5f);

	ma_debug ("real_last_interval = %hu, new_ivl_as_feedback = %hu\n", real_act_ivl, n_ivl);
	
	/*
	 * add_data() add case to current cases in memory and add to user_data.bin file
	 * This would be bad because:
	 * - queue with cases would grow very fast (ANN would be slower and slower)
	 * - frequent cases would too dimish role of these less frequent
	 *
	 * But this is not that bad because when loading user_data.bin we look for doubling cases
	 * (that has the same inputs values),
	 * and we eliminate these doubling cases (taking as opput mean of their outputs (n_ivl)).
	 */
	add_data (l_ivl, real_l_ivl, rp, l_gr, n_ivl);
	
	return 0;
}

