import All
import Control.Monad.State
import Control.Monad.Error
import System.Directory
import System.Environment
import System.Console.GetOpt
import System.Exit
import ImperativeState
import Data.Maybe
import System.FilePath.Posix
data Flag = Verbose |Vector | Version | Templates String | Resolution String | Paper String
              |Copy String | Input String | Output String | LibDir String | MediaWiki | HTML | InternalTemplates | MegaFont
  deriving (Show,Eq)

versionOption::String
versionOption="version"
resolutionOption::String
resolutionOption="resolution"
output::String
output="output"
templates::String
templates="templates"
url::String
url="url"
mediawiki::String
mediawiki="mediawiki"
html::String
html="html"    
paperOption::String
paperOption="paper"
internal::String
internal="internal"
vectorOption::String
vectorOption="vector"
copyOption::String
copyOption="copy"
fontOption::String
fontOption="font"

options :: [OptDescr Flag]
options = [
   Option ['V','?','v'] [versionOption,"help"] (NoArg Version) "show version number"
   , Option ['o']     [output]  (ReqArg Output "FILE")  "output FILE (REQUIRED)"
   , Option ['t']     [templates]  (ReqArg Templates "FILE")  "user template map FILE"
   , Option ['r']     [resolutionOption]  (ReqArg Resolution "INTEGER")  "maximum image resolution in dpi INTEGER"
   , Option ['u']     [url]          (ReqArg  Input "URL")  "input URL (REQUIRED)"
   , Option ['p']      [paperOption]          (ReqArg  Paper "PAPER")  "paper size, on of A4,A5,B5,letter,legal,executive"
   , Option ['m'] [mediawiki] (NoArg MediaWiki)  "use MediaWiki to expand templates"
   , Option ['h'] [html] (NoArg Main.HTML)  "use MediaWiki generated html as input (default)"
   , Option ['g'] [vectorOption] (NoArg Main.Vector)  "keep vector graphics in vector form"

   , Option ['i'] [internal] (NoArg Main.InternalTemplates)  "use internal template definitions"   
   , Option ['c'] [copyOption] (ReqArg Main.Copy "DIRECTORY")  "copy LaTeX tree to DIRECTORY"
   , Option ['f'] [fontOption] (NoArg MegaFont)  "cover full unicode range using megafont ttf files"   
   ]
    
    
compilerOpts :: [String] -> IO ([Flag], [String])
compilerOpts argv = 
       case getOpt Permute options argv of
          (o,n,[]  ) -> return (o,n)
          (_,_,errs) -> ioError (userError (concat errs ++ usageInfo header options))

header :: String
header = "Usage: mediawiki2latex [OPTION...]"


versionHeader :: String
versionHeader = "mediawiki2latex version 6.9\n"++( usageInfo header options)

printVersion:: Eq a => ([Flag], [a]) -> IO ()
printVersion o = if (Version `elem` (fst o)) || o == ([],[]) then putStrLn versionHeader>>exitSuccess else return ()

exactlyOne :: (a -> Maybe b) -> String -> [a] -> Either MyError b
exactlyOne predicate s o = case filter isJust (map predicate o) of
                    ((Just x):[])->Right x
                    _ -> Left (NotExcatlyOneError s)

atMostOne :: (a1 -> Maybe a) -> String -> [a1] -> Either MyError (Maybe a)
atMostOne predicate s o = case filter isJust (map predicate o) of
                    (x:[])->Right x
                    ([])->Right Nothing
                    _ -> Left (NotAtMostOneError s)


resolutionPredicate :: Flag -> Maybe String
resolutionPredicate (Resolution x) = Just x
resolutionPredicate _ = Nothing

copyPredicate :: Flag -> Maybe String
copyPredicate (Copy x) = Just x
copyPredicate _ = Nothing

outputPredicate :: Flag -> Maybe String
outputPredicate (Output x) = Just x
outputPredicate _ = Nothing


inputPredicate :: Flag -> Maybe String
inputPredicate (Input x) = Just x
inputPredicate _ = Nothing

templatesPredicate :: Flag -> Maybe String
templatesPredicate (Templates x) = Just x
templatesPredicate _ = Nothing

paperPredicate :: Flag -> Maybe String
paperPredicate (Paper x) = Just x
paperPredicate _ = Nothing

defaultResolution :: Integer
defaultResolution= 300

defaultPaper :: String
defaultPaper= "A4"

maybeToInt :: Num a => Maybe t -> a
maybeToInt (Just _) = 1
maybeToInt _ = 0

boolToInt :: Num a => Bool -> a
boolToInt True = 1
boolToInt _ = 0

checkOpts :: FilePath-> [Flag] -> Either MyError FullConfig
checkOpts cwd o = 
           do resolutionOpt <- atMostOne resolutionPredicate resolutionOption o
              resolutionVal<-case resolutionOpt of 
                           (Just x)-> case reads x of
                                        [(z, _)] -> Right z
                                        _->  Left (NotIntegerError resolutionOption)
                           _-> Right defaultResolution
              outputVal <- exactlyOne outputPredicate output o
              inputVal <- exactlyOne inputPredicate url o
              templatesVal <- atMostOne templatesPredicate templates o
              copyVal <- atMostOne copyPredicate copyOption o
              paperOpt <- atMostOne paperPredicate paperOption o
              paperVal<- case paperOpt of 
                           Just x -> if x `elem` ["A4","A5","B5","letter","legal","executive"] then Right x else Left PaperError 
                           _-> Right defaultPaper 
              let mediaWikiVal = (MediaWiki `elem` o)
              let htmlVal = (Main.HTML `elem` o)
              let temVal = (Main.InternalTemplates `elem` o)
              let vectorVal = (Main.Vector `elem` o)
              let fontVal = (Main.MegaFont `elem` o)
              let mysum=(boolToInt temVal)+(boolToInt mediaWikiVal)+(boolToInt htmlVal)+(maybeToInt templatesVal)
              if mysum>1 then Left ToManyOptionsError else Right ()
              runModeVal <- if mysum==(1::Integer) 
                              then case templatesVal of 
                                Just xx -> Right (UserTemplateFile xx)
                                _ -> if mediaWikiVal 
                                  then Right ExpandedTemplates
                                  else if htmlVal then Right ImperativeState.HTML else Right (StandardTemplates) 
                              else Right ImperativeState.HTML  
              return FullConfig{resolution=resolutionVal, outputFilename=outputVal,inputUrl=inputVal,runMode=runModeVal,paper=paperVal,vector=vectorVal,copy=copyVal >>=(return.(  cwd </>)),font=fontVal,mainPath=cwd}

main :: IO ()
main= do a<-getArgs
         o<-compilerOpts a
         printVersion o
         cwd<-getCurrentDirectory
         case (checkOpts cwd (fst o)) of
           Right x -> do (xx,_)<-(runStateT (runErrorT (All.all x)) imperativeStateZero)
                         case xx of 
                           Left n->print n
                           _->return ()
           Left y -> print y
         return ()


