/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/filesystemwalker.h"

// System Libraries
#include <sys/syscall.h>
#include <unistd.h>

// GStreamer
#include <gst/pbutils/pbutils.h>

// Boost C++
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/filesystem.hpp>
#include <boost/locale/format.hpp>
#include <boost/regex.hpp>
#include <boost/variant.hpp>

// Standard Library
#include <limits>
#include <list>
#include <map>
#include <string>
#include <vector>
#include <memory>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/mediautils.h"
#include "mediascanner/metadataresolver.h"
#include "mediascanner/property.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/taskfacades.h"
#include "mediascanner/utilities.h"
#include "mediascanner/writablemediaindex.h"

namespace mediascanner {

// Boost C++
using boost::locale::format;
using boost::posix_time::from_time_t;
using boost::posix_time::microseconds;
using boost::posix_time::ptime;

// Context specific logging domains
static const logging::Domain kError("error/fs-walker", logging::error());
static const logging::Domain kInfo("info/fs-walker", logging::info());
static const logging::Domain kWarning("warning/fs-walker", logging::warning());
static const logging::Domain kDebug("debug/fs-walker", logging::debug());
static const logging::Domain kTrace("trace/fs-walker", logging::trace());

class FileSystemWalker::Private {
public:
    typedef TaskFacade::ErrorFunction ErrorFunction;

    typedef std::map<std::string, Wrapper<GFileMonitor> > FileMonitorMap;

    enum ScanDetails {
        ScanAttributes = (1 << 0),
        ScanStreams = (1 << 1),
        ScanMetadata = (1 << 2),
        ScanAll = (ScanAttributes | ScanStreams | ScanMetadata)
    };

    struct FileTaskInfo {
        FileTaskInfo(unsigned task_id, ScanDetails details)
            : task_id(task_id)
            , details(details) {
        }

        unsigned task_id;
        ScanDetails details;
    };

    typedef std::map<std::string, FileTaskInfo> FileTaskInfoMap;

    Private(size_t group_id,
            const MediaRoot &media_root,
            MetadataResolverPtr resolver,
            TaskManagerPtr file_task_manager,
            TaskManagerPtr index_task_manager,
            TaskFacadePtr index_task_facade,
            const ErrorFunction &report_fatal_error);

    FileSystemPath media_index_path() const {
        return index_task_facade_->media_index_path();
    }

    MediaRoot media_root() const {
        return media_root_;
    }

    size_t group_id() const {
        return group_id_;
    }

    bool is_cancelled() const {
        return g_cancellable_is_cancelled(cancellable_.get());
    }

    void set_file_monitor_enabled(bool enable) {
        if (not (file_monitor_enabled_ = enable))
            file_monitors_.clear();
    }

    bool file_monitor_enabled() const {
        return file_monitor_enabled_;
    }

    void ReportError(const format &error_message) {
        ReportErrorLiteral(error_message.str());
    }

    ErrorFunction bind_report_error_literal() {
        return std::bind(&Private::ReportErrorLiteral, this, std::placeholders::_1);
    }

    void ReportErrorLiteral(const std::string &error_message);
    void cancel();

    typedef void (Private::*FileTaskFunction)(Wrapper<GFile> file,
                                              ScanDetails details);

    typedef std::function<void(WritableMediaIndex *index)> IndexTaskFunction;

    void RunIndexTask(const IndexTaskFunction &task);
    void push_index_task(const IndexTaskFunction &task);

    void StoreMediaInfo(const std::wstring &url,
                        const MediaInfo &metadata);

    void store_media_info_async(const std::wstring &url,
                                const MediaInfo &metadata);

    void push_file_task(FileTaskFunction task,
                        Wrapper<GFile> file,
                        ScanDetails details);

    void cancel_file_task(const std::string &path);

    void take_file_task(const std::string &path);

    void CancelDiscoverer();

    void ScanDirectory(Wrapper<GFile> directory, ScanDetails details);
    void ScanFile(Wrapper<GFile> file, ScanDetails details);
    bool start(ScanDetails details);

    bool MarkRunning();
    void NotifyFinished();
    bool Join();

    const std::string& error_message() const {
        return error_message_;
    }

private:
    static void on_file_changed(Private *d, GFile *file, GFile *other,
                                GFileMonitorEvent event);

    void on_file_deleted(GFile *const file);
    void on_file_moved(GFile *file, GFile *other);

    MetadataResolver::StoreFunction store_function() {
        return std::bind(&Private::store_media_info_async, this, std::placeholders::_1, std::placeholders::_2);
    }

    // related objects
    const ErrorFunction report_fatal_error_;
    const Wrapper<GCancellable> cancellable_;
    const MetadataResolverPtr metadata_resolver_;
    const TaskManagerPtr file_task_manager_;
    const TaskManagerPtr index_task_manager_;
    const TaskFacadePtr index_task_facade_;
    FileMonitorMap file_monitors_;

    // attributes
    const size_t group_id_;
    MediaRoot media_root_;
    bool file_monitor_enabled_;
    std::string error_message_;

    // state attributes
    volatile bool walking_;
    FileTaskInfoMap file_tasks_;
    GMutex mutex_;
    GCond cond_;
};

static std::string get_url(GFile *file) {
    return file ? take_string(g_file_get_uri(file))
                : std::string();
}

static std::string get_path(GFile *file) {
    return file ? take_string(g_file_get_path(file))
                : std::string();
}

FileSystemWalker::Private::Private(size_t group_id,
                                   const MediaRoot &media_root,
                                   MetadataResolverPtr resolver,
                                   TaskManagerPtr file_task_manager,
                                   TaskManagerPtr index_task_manager,
                                   TaskFacadePtr index_task_facade,
                                   const ErrorFunction &report_fatal_error)
    : report_fatal_error_(report_fatal_error)
    , cancellable_(take(g_cancellable_new()))
    , metadata_resolver_(resolver)
    , file_task_manager_(file_task_manager)
    , index_task_manager_(index_task_manager)
    , index_task_facade_(index_task_facade)
    , group_id_(group_id)
    , media_root_(media_root)
    , file_monitor_enabled_(true)
    , walking_(false) {
    g_mutex_init(&mutex_);
    g_cond_init(&cond_);

    // Check if an error occured during initialization
    if (media_root_.is_valid()) {
        push_index_task(std::bind(&WritableMediaIndex::AddMediaRoot,
                                    std::placeholders::_1, media_root_));
    } else {
        ReportErrorLiteral(media_root_.error_message());
    }
}

FileSystemWalker::FileSystemWalker(const MediaRoot &media_root,
                                   MetadataResolverPtr resolver,
                                   TaskManagerPtr file_task_manager,
                                   TaskManagerPtr index_task_manager,
                                   TaskFacadePtr index_task_facade)
    : d(new Private(reinterpret_cast<size_t>(this),
                    media_root, resolver, file_task_manager,
                    index_task_manager, index_task_facade,
                    d->bind_report_error_literal())) {
}

FileSystemWalker::~FileSystemWalker() {
    cancel();
    Join();
    delete d;
}

// Attributes //////////////////////////////////////////////////////////////////

bool FileSystemWalker::is_cancelled() const {
    return d->is_cancelled();
}

void FileSystemWalker::set_file_monitor_enabled(bool enable) {
    d->set_file_monitor_enabled(enable);
}

bool FileSystemWalker::file_monitor_enabled() const {
    return d->file_monitor_enabled();
}

MediaRoot FileSystemWalker::media_root() const {
    return d->media_root();
}

std::string FileSystemWalker::error_message() const {
    return d->error_message();
}

size_t FileSystemWalker::task_group() const {
    return d->group_id();
}

// Actions /////////////////////////////////////////////////////////////////////

static std::string PrintStreams(GList *streams) {
    std::ostringstream text;

    text << ", streams=[";

    for (GList *l = streams; l; l = l->next) {
        if (l->prev)
            text << ", ";

        const Wrapper<GstDiscovererStreamInfo> si =
                wrap(GST_DISCOVERER_STREAM_INFO(l->data));

        text << gst_discoverer_stream_info_get_stream_type_nick(si.get())
             << ", "
             << to_string(wrap(gst_discoverer_stream_info_get_caps(si.get())));

        if (GstDiscovererContainerInfo *const ci =
                si.get<GstDiscovererContainerInfo>())
            text << PrintStreams(gst_discoverer_container_info_get_streams(ci));

        if (GstDiscovererAudioInfo *const ai =
                si.get<GstDiscovererAudioInfo>()) {
            text << ", channels="
                 << gst_discoverer_audio_info_get_channels(ai)
                 << ", sample-rate="
                 << gst_discoverer_audio_info_get_sample_rate(ai)
                 << ", depth="
                 << gst_discoverer_audio_info_get_depth(ai)
                 << ", bitrate="
                 << gst_discoverer_audio_info_get_bitrate(ai)
                 << ", max-bitrate="
                 << gst_discoverer_audio_info_get_max_bitrate(ai);
        }

        if (GstDiscovererVideoInfo *const vi =
                si.get<GstDiscovererVideoInfo>()) {
            text << ", width="
                 << gst_discoverer_video_info_get_width(vi)
                 << ", height="
                 << gst_discoverer_video_info_get_height(vi)
                 << ", depth="
                 << gst_discoverer_video_info_get_depth(vi)
                 << ", framerate="
                 << gst_discoverer_video_info_get_framerate_num(vi) << '/'
                 << gst_discoverer_video_info_get_framerate_denom(vi)
                 << ", par="
                 << gst_discoverer_video_info_get_par_num(vi) << '/'
                 << gst_discoverer_video_info_get_par_denom(vi)
                 << ", is-interlaced="
                 << gst_discoverer_video_info_is_interlaced(vi)
                 << ", bitrate="
                 << gst_discoverer_video_info_get_bitrate(vi)
                 << ", max_bitrate="
                 << gst_discoverer_video_info_get_max_bitrate(vi)
                 << ", is-image="
                 << gst_discoverer_video_info_is_image(vi);
        }
    }

    text << "]";

    return text.str();
}

static bool merge_properties_visitor(const Property &property,
                                     GstDiscovererInfo *media,
                                     GstDiscovererStreamInfo *stream,
                                     Property::ValueMap *properties) {
    property.MergeStreamInfo(media, stream, properties);
    return false;
}

static void CollectMediaProperties(GstDiscovererInfo *const media,
                                   MediaInfo *metadata) {
    Property::ValueMap media_properties;

    Property::VisitAll(std::bind(merge_properties_visitor,
                                   std::placeholders::_1, media, nullptr, &media_properties));
    metadata->add_related(media_properties);

    const GList *const stream_list = gst_discoverer_info_get_stream_list(media);

    for (const GList *l = stream_list; l; l = l->next) {
        GstDiscovererStreamInfo *const stream =
                GST_DISCOVERER_STREAM_INFO(l->data);

        Property::ValueMap stream_properties;
        Property::VisitAll(std::bind(merge_properties_visitor,
                                       std::placeholders::_1, media, stream, &stream_properties));
        metadata->add_related(stream_properties);
    }
}

static ptime GetTimeStamp(GFileInfo *file, const char *attr_sec,
                          const char *attr_usec) {
    BOOST_ASSERT(g_file_info_has_attribute(file, attr_sec));
    ptime t = from_time_t(g_file_info_get_attribute_uint64(file, attr_sec));

    if (g_file_info_has_attribute(file, attr_usec))
        t += microseconds(g_file_info_get_attribute_uint32(file, attr_usec));

    return t;
}

void FileSystemWalker::Private::ReportErrorLiteral
                                        (const std::string &error_message) {
    error_message_ = error_message;
    cancel();
}

void FileSystemWalker::Private::cancel() {
    if (metadata_resolver_)
        metadata_resolver_->cancel();

    g_cancellable_cancel(cancellable_.get());
    BOOST_ASSERT(is_cancelled());
}

void FileSystemWalker::cancel() {
    d->cancel();
}

void FileSystemWalker::Private::RunIndexTask(const IndexTaskFunction &task) {
    index_task_manager_->RunTask
            (index_task_facade_->bind(task, report_fatal_error_));
}

void FileSystemWalker::Private::push_index_task(const IndexTaskFunction &task) {
    index_task_manager_->AppendTask
            (index_task_facade_->bind(task, report_fatal_error_));
}

bool FileSystemWalker::start() {
    return d->start(Private::ScanAll);
}

void FileSystemWalker::Private::push_file_task(FileTaskFunction task,
                                               Wrapper<GFile> file,
                                               ScanDetails details) {
    g_mutex_lock(&mutex_);

    const std::string file_path = get_path(file.get());
    const FileTaskInfoMap::iterator it = file_tasks_.find(file_path);
    unsigned task_id = 0;

    if (it != file_tasks_.end()) {
        if (it->second.details > details)
            // FIXME(M5): actually we should have a smart lock class
            goto unlock;

        kTrace("Updating scan request for \"{1}\".") % file_path;
        file_task_manager_->CancelTaskByTaskId(it->second.task_id);
        file_tasks_.erase(it);
    } else {
        kTrace("Adding new scan request for \"{1}\".") % file_path;
    }

    // FIXME(M5): avoid "crosses initialization of..." error
    task_id = file_task_manager_->AppendGroupedTask
            (group_id_, std::bind(task, this, file, details),
             group_id_);

    file_tasks_.insert(std::make_pair(file_path,
                                      FileTaskInfo(task_id, details)));

unlock:
    g_mutex_unlock(&mutex_);
}

void FileSystemWalker::Private::cancel_file_task(const std::string &path) {
    g_mutex_lock(&mutex_);

    const FileTaskInfoMap::iterator it = file_tasks_.find(path);

    if (it != file_tasks_.end()) {
        file_task_manager_->CancelTaskByTaskId(it->second.task_id);
        file_tasks_.erase(it);
    }

    g_mutex_unlock(&mutex_);
}


void FileSystemWalker::Private::take_file_task(const std::string &path) {
    g_mutex_lock(&mutex_);

    const FileTaskInfoMap::iterator it = file_tasks_.find(path);

    if (it != file_tasks_.end())
        file_tasks_.erase(it);

    g_mutex_unlock(&mutex_);
}

bool FileSystemWalker::Private::start(ScanDetails details) {
    if (not MarkRunning())
        return false;

    const std::string root_path = media_root().path();
    const FileSystemPath index_path = media_index_path();
    kInfo("Scanning \"{1}\" using media index at \"{2}\"")
            % root_path % index_path;

    error_message_.clear();

    push_file_task(&Private::ScanDirectory, media_root().file(), details);

    file_task_manager_->AppendGroupedTask
            (group_id_, std::bind(&Private::NotifyFinished, this),
             std::numeric_limits<unsigned>::max());

    return true;
}

bool FileSystemWalker::Private::MarkRunning() {
    g_mutex_lock(&mutex_);

    if (is_cancelled())
        return false;

    walking_ = true;
    g_mutex_unlock(&mutex_);
    return true;
}

void FileSystemWalker::Private::NotifyFinished() {
    const std::string root_path = media_root_.path();
    kInfo("Finished scanning directories below \"{1}\".") % root_path;

    // Wakeup joiners
    g_mutex_lock(&mutex_);
    walking_ = false;
    g_cond_broadcast(&cond_);
    g_mutex_unlock(&mutex_);
}


bool FileSystemWalker::Private::Join() {
    bool was_walking = false;
    g_mutex_lock(&mutex_);

    while (walking_) {
        g_cond_wait(&cond_, &mutex_);
        was_walking = true;
    }

    g_mutex_unlock(&mutex_);
    return was_walking;
}

bool FileSystemWalker::Join() {
    return d->Join();
}

void FileSystemWalker::Private::ScanDirectory(Wrapper<GFile> directory,
                                              ScanDetails details) {
    const std::string path = get_path(directory.get());
    take_file_task(path);

    kInfo("Scanning directory \"{1}\"") % path;

    Wrapper<GError> error;

    // Install file monitor if needed.
    if (file_monitor_enabled_ && not file_monitors_.count(path)) {
        const Wrapper<GFileMonitor> monitor =
                take(g_file_monitor_directory(directory.get(),
                                              G_FILE_MONITOR_SEND_MOVED,
                                              cancellable_.get(),
                                              error.out_param()));

        if (not monitor) {
            const std::string error_message = to_string(error);
            kWarning("Cannot create file monitor for \"{1}\": {2}")
                    % path % error_message;
            error.reset();
        } else {
            g_signal_connect_swapped(monitor.get(), "changed",
                                     G_CALLBACK(&Private::on_file_changed),
                                     this);

            file_monitors_.insert(std::make_pair(path, monitor));
        }
    }

    const Wrapper<GFileEnumerator> children =
            take(g_file_enumerate_children(directory.get(),
                                           G_FILE_ATTRIBUTE_STANDARD_NAME,
                                           G_FILE_QUERY_INFO_NONE,
                                           cancellable_.get(),
                                           error.out_param()));

    if (not children) {
        const std::string error_message = to_string(error);
        kWarning("Cannot list content of \"{1}\": {2}")
                % path % error_message;
        return;
    }

    while (const Wrapper<GFileInfo> child =
           take(g_file_enumerator_next_file(children.get(),
                                            cancellable_.get(),
                                            error.out_param()))) {
        if (is_cancelled())
            return;

        if (error) {
            const std::string error_message = to_string(error);
            kWarning("Cannot list content of \"{1}\": {2}")
                    % path % error_message;
            return;
        }

        // Skip hidden and backup files
        if (g_file_info_get_is_hidden(child.get())
                || g_file_info_get_is_backup(child.get()))
            continue;

        const char *const name = g_file_info_get_name(child.get());

        push_file_task(&Private::ScanFile,
                       take(g_file_get_child(directory.get(), name)),
                       details);
    }
}

static void RemoveMediaInfo(WritableMediaIndex *media_index,
                            const std::wstring &url) {
    media_index->Delete(url);
}

void FileSystemWalker::Private::on_file_moved(GFile *file, GFile *other) {
    const std::string file_path = get_path(file);
    const std::string other_path = get_path(other);

    // Skip symlinks to avoid getting into infinite loops for things
    // like each process' "root" link in procfs.
    if (g_file_test(file_path.c_str(), G_FILE_TEST_IS_SYMLINK))
        return;

    kTrace("File moved from \"{1}\" to \"{2}\".")
            % file_path % other_path;

    ScanDetails details = ScanAttributes;

    g_mutex_lock(&mutex_);

    const FileTaskInfoMap::iterator it = file_tasks_.find(file_path);

    if (it != file_tasks_.end()) {
        file_task_manager_->CancelTaskByTaskId(it->second.task_id);
        details = (details | it->second.details);
    }

    g_mutex_unlock(&mutex_);

    FileTaskFunction task = &Private::ScanFile;

    if (g_file_test(file_path.c_str(), G_FILE_TEST_IS_DIR))
        task = &Private::ScanDirectory;

    push_file_task(task, wrap(other), details);
}

void FileSystemWalker::Private::on_file_deleted(GFile *const file) {
    const std::string path = get_path(file);
    kTrace("File \"{1}\" deleted.") % path;

    cancel_file_task(path);
    push_index_task(std::bind(&RemoveMediaInfo, std::placeholders::_1,
                                ToUnicode(get_url(file))));
}

void FileSystemWalker::Private::on_file_changed(Private *d,
                                                GFile *file, GFile *other,
                                                GFileMonitorEvent event) {
    switch (event) {
    case G_FILE_MONITOR_EVENT_CREATED:
    case G_FILE_MONITOR_EVENT_CHANGED:
        d->push_file_task(&Private::ScanFile, wrap(file), ScanAll);
        break;

    case G_FILE_MONITOR_EVENT_ATTRIBUTE_CHANGED:
        d->push_file_task(&Private::ScanFile, wrap(file), ScanAttributes);
        break;

    case G_FILE_MONITOR_EVENT_MOVED:
        d->on_file_moved(file, other);
        break;

    case G_FILE_MONITOR_EVENT_DELETED:
        d->on_file_deleted(file);
        break;

    case G_FILE_MONITOR_EVENT_PRE_UNMOUNT:
    case G_FILE_MONITOR_EVENT_UNMOUNTED:
        // Ignoring this events, they are handled by the volume monitor.
        break;

    case G_FILE_MONITOR_EVENT_CHANGES_DONE_HINT:
        // TODO(M5): Figure out if we can do something with "changes-done".
        break;
    }
}

static void compare_etags(WritableMediaIndex *media_index,
                          const std::wstring &uri, const std::wstring &etag,
                          bool *etags_are_equal) {
    const std::wstring stored_etag = media_index->Lookup(uri, schema::kETag);
    *etags_are_equal = (stored_etag == etag);
}

static void InsertMediaInfo(WritableMediaIndex *media_index,
                            const std::wstring &uri,
                            const MediaInfo &metadata) {
    media_index->Insert(uri, metadata);
}

void FileSystemWalker::Private::StoreMediaInfo(const std::wstring &url,
                                               const MediaInfo &metadata) {
    RunIndexTask(std::bind(&InsertMediaInfo, std::placeholders::_1, url, metadata));
}

void FileSystemWalker::Private::store_media_info_async
                        (const std::wstring &url, const MediaInfo &metadata) {
    push_index_task(std::bind(&InsertMediaInfo, std::placeholders::_1, url, metadata));
}

template<typename T>
static std::string enum_nick(T value) {
    static GEnumClass *const enum_class = static_cast<GEnumClass *>
            (g_type_class_ref(internal::GetGType<T>()));

    if (enum_class) {
        GEnumValue *const enum_value = g_enum_get_value(enum_class, value);

        if (enum_value)
            return enum_value->value_nick;
    }

    std::ostringstream oss;
    oss << g_type_name(internal::GetGType<T>()) << "(" << value << ")";
    return oss.str();
}

void FileSystemWalker::Private::ScanFile(Wrapper<GFile> file,
                                         ScanDetails details) {
    Wrapper<GError> error;

    std::unique_ptr<GstDiscoverer, void(*)(GstDiscoverer*)> discoverer(gst_discoverer_new(GST_SECOND, error.out_param()),
            [](GstDiscoverer *p) { g_object_unref(p);});
    if (!discoverer) {
        const std::string error_message = to_string(error);
        ReportError(format("Cannot create discoverer: {1}") % error_message);
        return;
    }
    const std::string path = get_path(file.get());
    take_file_task(path);

    kDebug("  Scanning \"{1}\"") % path;

    const Wrapper<GFileInfo> file_info =
            take(g_file_query_info(file.get(), "standard::*,etag::*,time::*",
                                   G_FILE_QUERY_INFO_NONE, cancellable_.get(),
                                   error.out_param()));

    if (not file_info) {
        const std::string error_message = to_string(error);
        kWarning("Failed to query information about file \"{1}\": {2}")
                % path % error_message;
        return;
    }

    const GFileType file_type = g_file_info_get_file_type(file_info.get());

    if (file_type == G_FILE_TYPE_DIRECTORY) {
        if (g_file_info_get_is_symlink(file_info.get())) {
            // FIXME(future): Maybe only skip if the symlink creates a cycle
            // which could be detected by tracking G_FILE_ATTRIBUTE_ID_FILE.
            // But actually not sure if we really want to have duplicates
            // in the index, and such.
            kDebug("  Skipping symbolic link at \"{1}\"") % path;
        } else {
            // Queue the directory for scanning it later.
            push_file_task(&Private::ScanDirectory, file, details);
        }

        return;
    }

    // Skip anything that isn't a regular file.
    if (file_type != G_FILE_TYPE_REGULAR) {
        const std::string file_type_name = enum_nick(file_type);
        kWarning("Unexpected file type \"{1}\" for \"{2}\".")
                 % file_type_name % path;
        return;
    }

    // Resolve the file's URI.
    const std::string url = get_url(file.get());
    const std::wstring w_url = ToUnicode(url);

    // Skip if ETag indicates the file has not changed.
    const std::wstring etag = ToUnicode(g_file_info_get_etag(file_info.get()));
    bool etags_are_equal = false;

    RunIndexTask(std::bind(&compare_etags, std::placeholders::_1, w_url, etag,
                             &etags_are_equal));

    if (etags_are_equal) {
        kDebug("  Not changed, ignoring \"{1}\"") % path;
        return;
    }

    // Skip files according to their MIME type.
    const MimeType mime_type(g_file_info_get_content_type(file_info.get()));

    if (not mime_type.is_audio()
            && not mime_type.is_image()
            && not mime_type.is_video()) {
        kDebug("  Unknown MIME type, ignoring \"{1}\"") % path;
        return;
    }

    // FIXME(M5): Obey ScanStreams flag
    // FIXME(M5): Discover in background
    const Wrapper<GstDiscovererInfo> media =
            take(gst_discoverer_discover_uri(discoverer.get(), url.c_str(),
                                             error.out_param()));

    if (error) {
        const std::string error_message = to_string(error);
        kError("Content discovery failed for \"{1}\": {2}")
                % path % error_message;
        return;
    }

    if (media) {
        GList *const streams = gst_discoverer_info_get_stream_list(media.get());

        // Skip any file that doesn't provide GStreamer compatible streams.
        if (streams == nullptr) {
            kDebug("  No media streams found in \"{1}\"") % path;
            return;
        }

        // File must be an image if duration is zero. Check that.
        if (gst_discoverer_info_get_duration(media) == 0) {
            // No streams at all? That's suspecious, but well.
            if (not GST_IS_DISCOVERER_VIDEO_INFO(streams->data))
                return;

            GstDiscovererVideoInfo *const video_info =
                    static_cast<GstDiscovererVideoInfo *>(streams->data);

            // Duration is zero, but it's also no image? Let's skip!
            if (not gst_discoverer_video_info_is_image(video_info))
                return;
        }
    }

    if (kDebug.enabled()) {
        std::ostringstream text;

        text << "  Found " << url
             << ": type=" << FromUnicode(mime_type.str())
             << ", etag=" << g_file_info_get_etag(file_info.get());

        if (media) {
            text << ", gstinfo={result="
                 << gst_discoverer_info_get_result(media);

            GList *streams = gst_discoverer_info_get_stream_list(media.get());
            text << PrintStreams(streams);

            text << "}";
        }

        kDebug(text.str());
    }

    // FIXME(M3): https://live.gnome.org/MediaArtStorageSpec
    // else if (tag_name == GST_TAG_IMAGE)
    // else if (tag_name == GST_TAG_PREVIEW_IMAGE)
    // -> attach as data: URL

    // Update media index with findings
    Property::ValueMap file_properties;

    file_properties.insert(schema::kMimeType.bind_value(mime_type.str()));
    file_properties.insert(schema::kETag.bind_value(etag));

    const uint64_t file_size = g_file_info_get_size(file_info.get());
    file_properties.insert(schema::kFileSize.bind_value(file_size));

    const ptime last_modified = GetTimeStamp
            (file_info.get(), G_FILE_ATTRIBUTE_TIME_MODIFIED,
             G_FILE_ATTRIBUTE_TIME_MODIFIED_USEC);
    file_properties.insert(schema::kLastModified.bind_value(last_modified));

    const ptime last_accessed = GetTimeStamp
            (file_info.get(), G_FILE_ATTRIBUTE_TIME_ACCESS,
             G_FILE_ATTRIBUTE_TIME_ACCESS_USEC);
    file_properties.insert(schema::kLastAccessed.bind_value(last_accessed));

    MediaInfo metadata;
    metadata.add_related(file_properties);

    if (media)
        CollectMediaProperties(media.get(), &metadata);

    // Generate title property if not available in media properties.
    if (metadata.first(schema::kTitle).empty()) {
        static const boost::wregex::flag_type rx_flags =
                boost::regex_constants::extended |
                boost::regex_constants::optimize |
                boost::regex_constants::icase;
        static const boost::wregex
                rx_extensions(L"(\\.\\w+)+$", rx_flags);
        static const boost::wregex
                rx_format_info(L"[-_ ]\\d+([ip][-_a-z0-9]*|x\\d+)$", rx_flags);
        static const boost::wregex
                rx_separators(L"(\\S)[-_]+(\\S)", rx_flags);

        // Strip extensions from filename.
        std::string title = g_file_info_get_edit_name(file_info.get());
        std::wstring w_title = ToUnicode(title);

        w_title = boost::regex_replace(w_title, rx_extensions, std::wstring());
        w_title = boost::regex_replace(w_title, rx_format_info, std::wstring());
        w_title = boost::regex_replace(w_title, rx_separators, "$1 $2");

        metadata.add_single(schema::kTitle.bind_value(w_title));
    }

    StoreMediaInfo(w_url, metadata);

    if ((details & ScanMetadata) && metadata_resolver_)
        metadata_resolver_->push(url, metadata, store_function());
}

} // namespace mediascanner

// TODO(M5): Maybe rescan when new plugins get added?
