/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */

// C Library
#include <stdlib.h>

// Google Tests
#include <gtest/gtest.h>

// Boost C++
#include <boost/algorithm/string/predicate.hpp>

// Standard Library
#include <sstream>
#include <string>

// Media Scanner
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"

// Test Suite
#include "testlib/loggingsink.h"
#include "testlib/testutils.h"

namespace mediascanner {

static std::string to_string(unsigned n) {
    std::ostringstream oss;
    oss << n;
    return oss.str();
}

TEST(LoggingTest, DefaultDomains) {
    LoggingSink sink;

    // Verify initial state of the domains.
    ASSERT_TRUE(logging::error()->enabled());
    ASSERT_TRUE(logging::warning()->enabled());
    ASSERT_TRUE(logging::info()->enabled());
    ASSERT_FALSE(logging::debug()->enabled());
    ASSERT_FALSE(logging::trace()->enabled());

    // Verify initial behavior of the domains.
    (*logging::debug())(to_string(__LINE__));
    EXPECT_TRUE(sink.take().empty());

    (*logging::info())(to_string(__LINE__));
    EXPECT_FALSE(sink.take().empty());
    EXPECT_TRUE(sink.get().empty());

    (*logging::warning())(to_string(__LINE__));
    EXPECT_FALSE(sink.take().empty());
    EXPECT_TRUE(sink.get().empty());

    (*logging::error())(to_string(__LINE__));
    EXPECT_FALSE(sink.take().empty());
    EXPECT_TRUE(sink.get().empty());

    // Verify that the debug domain can be enabled.
    logging::debug()->set_enabled(true);
    ASSERT_TRUE(logging::debug()->enabled());
    ASSERT_TRUE(logging::info()->enabled());

    (*logging::debug())(to_string(__LINE__));
    EXPECT_FALSE(sink.take().empty());
    EXPECT_TRUE(sink.get().empty());

    // Verify that the info domain can be disabled.
    // Note that the debug domain remains active, as it was enabled explicitly.
    logging::info()->set_enabled(false);
    ASSERT_TRUE(logging::debug()->enabled());
    ASSERT_FALSE(logging::info()->enabled());

    (*logging::debug())(to_string(__LINE__));
    EXPECT_FALSE(sink.take().empty());
    EXPECT_TRUE(sink.get().empty());

    (*logging::info())(to_string(__LINE__));
    EXPECT_TRUE(sink.take().empty());

    // Verify that the debug domain inherts the info domain's state after reset.
    logging::debug()->reset();
    ASSERT_FALSE(logging::debug()->enabled());
    ASSERT_FALSE(logging::info()->enabled());

    (*logging::debug())(to_string(__LINE__));
    EXPECT_TRUE(sink.take().empty());

    (*logging::info())(to_string(__LINE__));
    EXPECT_TRUE(sink.take().empty());
}

TEST(LoggingTest, CustomDomain) {
    LoggingSink default_sink;
    std::ostringstream other_stream;

    const logging::MessageSinkPtr other_sink
            (new logging::DefaultMessageSink(&other_stream));

    logging::Domain other_domain("other", logging::info());
    other_domain.set_message_sink(other_sink);

    ASSERT_TRUE(logging::info()->enabled());
    EXPECT_TRUE(default_sink.get().empty());
    EXPECT_TRUE(other_stream.str().empty());

    (*logging::info())(to_string(__LINE__));

    EXPECT_FALSE(default_sink.take().empty());
    EXPECT_TRUE(default_sink.get().empty());
    EXPECT_TRUE(other_stream.str().empty());

    other_domain(to_string(__LINE__));

    EXPECT_TRUE(default_sink.get().empty());
    EXPECT_FALSE(other_stream.str().empty());
}

TEST(LoggingTest, Strings) {
    LoggingSink sink;

    ASSERT_TRUE(logging::info()->enabled());

    (*logging::info())("Format {1}") % 123;
    EXPECT_TRUE(boost::algorithm::ends_with(sink.take(), "Format 123\n"));
    EXPECT_TRUE(sink.get().empty());

    (*logging::info())(L"Wide {1}") % 123;
    EXPECT_TRUE(boost::algorithm::ends_with(sink.take(), "Wide 123\n"));
    EXPECT_TRUE(sink.get().empty());
}

struct EnvTestParam {
    EnvTestParam(const std::string &environment_value,
                 const bool first_domain_enabled,
                 const bool second_domain_enabled,
                 const bool third_domain_enabled,
                 const bool nested_domain_enabled)
        : environment_value(environment_value)
        , first_domain_enabled(first_domain_enabled)
        , second_domain_enabled(second_domain_enabled)
        , third_domain_enabled(third_domain_enabled)
        , nested_domain_enabled(nested_domain_enabled) {
    }

    const std::string environment_value;
    const bool first_domain_enabled;
    const bool second_domain_enabled;
    const bool third_domain_enabled;
    const bool nested_domain_enabled;
};

static void PrintTo(const EnvTestParam &p, std::ostream *os) {
    *os << '"' << p.environment_value << '"';
}

class EnvTest : public testing::TestWithParam<EnvTestParam> {
};

INSTANTIATE_TEST_CASE_P(, EnvTest, testing::Values
        (EnvTestParam("", true, true, false, false),
         EnvTestParam("::", true, true, false, false),
         EnvTestParam("third", true, true, true, true),
         EnvTestParam("-second", true, false, false, false),
         EnvTestParam("third:-second", true, false, true, true),
         EnvTestParam("-second:third", true, false, true, true),
         EnvTestParam("-second:second", true, true, false, false),
         EnvTestParam("+second:-second", true, false, false, false),
         EnvTestParam("third/child", true, true, false, true)));

TEST_P(EnvTest, Test) {
    const EnvTestParam &p = GetParam();

    ::setenv("MEDIASCANNER_DEBUG", p.environment_value.c_str(), true);
    ASSERT_EQ(p.environment_value, safe_string(::getenv("MEDIASCANNER_DEBUG")));

    logging::Domain first("first");
    logging::Domain second("second", &first);
    logging::Domain third("third", logging::Domain::Disabled, &second);
    logging::Domain nested("third/child", &third);

    EXPECT_EQ(p.first_domain_enabled, first.enabled());
    EXPECT_EQ(p.second_domain_enabled, second.enabled());
    EXPECT_EQ(p.third_domain_enabled, third.enabled());
    EXPECT_EQ(p.nested_domain_enabled, nested.enabled());
}

} // namespace mediascanner

int main(int argc, char *argv[]) {
    ::setenv("MEDIASCANNER_DEBUG", 0, true);
    mediascanner::InitTests(&argc, argv);
    return RUN_ALL_TESTS();
}
