/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/filesystemscanner.h"

// C++ Standard Library
#include <algorithm>
#include <limits>
#include <memory>
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/filesystemwalker.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/logging.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/metadataresolver.h"
#include "mediascanner/taskfacades.h"
#include "mediascanner/taskmanager.h"
#include "mediascanner/writablemediaindex.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Context specific logging domains
static const logging::Domain kError("error/fs-scanner", logging::error());
static const logging::Domain kWarning("warning/fs-scanner", logging::warning());
static const logging::Domain kDebug("debug/fs-scanner", logging::debug());
static const logging::Domain kInfo("info/fs-scanner", logging::info());

typedef std::shared_ptr<FileSystemWalker> FileSystemWalkerPtr;

class FileSystemScanner::Private : public MediaRootManager::Listener {
    friend class FileSystemScanner;

    typedef std::vector<FileSystemWalkerPtr> MediaRootVector;
    typedef MediaRootVector::iterator MediaRootIterator;
    typedef std::vector<FileSystemScanner::Listener *> ListenerVector;

    Private(MetadataResolverPtr resolver,
            TaskManagerPtr index_task_manager,
            TaskFacadePtr index_task_facade)
        : metadata_resolver_(resolver)
        , file_task_manager_(new TaskManager("file system scanner"))
        , index_task_manager_(index_task_manager)
        , index_task_facade_(index_task_facade)
        , file_monitor_enabled_(true) {
        index_task_facade_->root_manager()->add_listener(this);
        g_mutex_init(&mutex_);
    }

    ~Private() {
        index_task_facade_->root_manager()->remove_listener(this);

        // Ensure no more tasks are running in background.
        CancelWalkers();
        file_task_manager_->Shutdown();

        // Verify we really shutdown all walkers.
        for (const auto &walker: walkers_) {
            BOOST_ASSERT_MSG(walker.use_count() == 1,
                             walker->media_root().path().c_str());
        }

        walkers_.clear();

        // Ensure we really shutdown the task manager.
        BOOST_ASSERT(file_task_manager_.use_count() == 1);
        file_task_manager_.reset();
    }

    MediaRootManagerPtr root_manager() const {
        return index_task_facade_->root_manager();
    }

    bool is_idle() const {
        if (not pending_roots_.empty())
            return false;
        if (metadata_resolver_ && not metadata_resolver_->is_idle())
            return false;

        return true;
    }

    bool add_media_root(const MediaRoot &root);
    bool remove_media_root(const std::string &path);
    bool remove_media_root(MediaRootIterator it);

    void OnMediaRootAdded(const MediaRoot &root);
    void OnMediaRootRemoved(const MediaRoot &root);

    void OnMediaRootFinished(FileSystemWalkerPtr walker);
    void NotifyScanningFinished(std::string error_message);

    void CancelWalkers() {
        for (const auto &walker: walkers_) {
            file_task_manager_->CancelByGroupId(walker->task_group());
        }
    }

    MetadataResolverPtr metadata_resolver_;
    TaskManagerPtr file_task_manager_;
    TaskManagerPtr index_task_manager_;
    TaskFacadePtr index_task_facade_;

    bool file_monitor_enabled_;

    typedef std::vector<unsigned> PendingRootVector;
    PendingRootVector pending_roots_;

    MediaRootVector walkers_;
    ListenerVector listeners_;

    std::vector<std::string> directories_;
    std::vector<std::string> metadata_sources_;

    GMutex mutex_;
};

FileSystemScanner::FileSystemScanner(MetadataResolverPtr resolver,
                                     TaskManagerPtr media_task_manager,
                                     TaskFacadePtr media_task_facade)
    : d(new Private(resolver, media_task_manager, media_task_facade)) {
}

FileSystemScanner::~FileSystemScanner() {
    delete d;
}

// Attributes //////////////////////////////////////////////////////////////////

bool FileSystemScanner::Private::add_media_root(const MediaRoot &root) {
    for (const auto &walker: walkers_) {
        if (walker->media_root().path() == root.path())
            return false;
    }

    FileSystemWalkerPtr walker(new FileSystemWalker(root, metadata_resolver_,
                                                    file_task_manager_,
                                                    index_task_manager_,
                                                    index_task_facade_));
    // Don't add failed media roots.
    if (walker->is_cancelled())
        return false;

    const std::string relative_path = root.relative_path();
    const std::string base_path = root.base_path();
    kInfo("Starting to monitor \"/{1}\" mounted at \"{2}\"")
            % relative_path % base_path;
    walkers_.push_back(walker);

    return true;
}

bool FileSystemScanner::Private::remove_media_root(const std::string &path) {
    for (MediaRootIterator it = walkers_.begin(); it != walkers_.end(); ++it) {
        if ((*it)->media_root().path() == path)
            return remove_media_root(it);
    }

    return false;
}

bool FileSystemScanner::Private::remove_media_root(MediaRootIterator it) {
    // Move into scoped pointer to delete at return.
    const FileSystemWalkerPtr walker = *it;

    const std::string relative_path = walker->media_root().relative_path();
    const std::string base_path = walker->media_root().base_path();
    kInfo("Stopping to monitor \"/{1}\" mounted at \"{2}\"")
            % relative_path
            % base_path;

    // First remove from list and cancel later to minimize possible races.
    walkers_.erase(it);

    file_task_manager_->CancelByGroupId(walker->task_group());

    return true;
}

void FileSystemScanner::set_directories(const std::vector<std::string> &paths) {
    g_mutex_lock(&d->mutex_);

    d->CancelWalkers();
    d->walkers_.clear();

    for (const auto &root_path: paths) {
        d->add_media_root(d->root_manager()->make_root(root_path));
    }

    g_mutex_unlock(&d->mutex_);
}

bool FileSystemScanner::add_directory(const std::string &path) {
    g_mutex_lock(&d->mutex_);
    const bool result = d->add_media_root(d->root_manager()->make_root(path));
    g_mutex_unlock(&d->mutex_);
    return result;
}

bool FileSystemScanner::remove_directory(const std::string &path) {
    g_mutex_lock(&d->mutex_);
    const bool result = d->remove_media_root(path);
    g_mutex_unlock(&d->mutex_);
    return result;
}

std::vector<std::string> FileSystemScanner::directories() const {
    std::vector<std::string> paths;
    g_mutex_lock(&d->mutex_);

    for (const auto &walker: d->walkers_)
        paths.push_back(walker->media_root().path());

    g_mutex_unlock(&d->mutex_);
    return paths;
}

void FileSystemScanner::add_listener(Listener *listener) {
    d->listeners_.push_back(listener);
}

void FileSystemScanner::remove_listener(Listener *listener) {
    Private::ListenerVector::iterator it = d->listeners_.begin();

    while (it != d->listeners_.end()) {
        if (*it == listener) {
            it = d->listeners_.erase(it);
            continue;
        }

        ++it;
    }
}

void FileSystemScanner::set_file_monitor_enabled(bool enable) {
    if (enable != d->file_monitor_enabled_) {
        for (const auto &walker: d->walkers_) {
            walker->set_file_monitor_enabled(enable);
        }
    }

    // FIXME(M5): Trigger rescan?
}

bool FileSystemScanner::file_monitor_enabled() const {
    return d->file_monitor_enabled_;
}

bool FileSystemScanner::is_idle() const {
    g_mutex_lock(&d->mutex_);
    const bool result = d->is_idle();
    g_mutex_unlock(&d->mutex_);
    return result;
}

// Actions /////////////////////////////////////////////////////////////////////

bool FileSystemScanner::start_scanning() {
    bool started = false;
    g_mutex_lock(&d->mutex_);

    if (not d->is_idle()) {
        kWarning("File system scanner was started already.");
    } else {
        for (const auto walker: d->walkers_) {
            g_mutex_unlock(&d->mutex_);
            const bool walker_started = walker->start();
            g_mutex_lock(&d->mutex_);

            if (walker_started) {
                d->file_task_manager_->AppendGroupedTask
                        (walker->task_group(),
                         std::bind(&Private::OnMediaRootFinished, d, walker));

                d->pending_roots_.push_back(walker->task_group());
            }
        }

        started = true;
    }

    g_mutex_unlock(&d->mutex_);
    return started;
}

void FileSystemScanner::Private::OnMediaRootAdded(const MediaRoot &root) {
    g_mutex_lock(&mutex_);
    add_media_root(root);
    g_mutex_unlock(&mutex_);
}

void FileSystemScanner::Private::OnMediaRootRemoved(const MediaRoot &root) {
    g_mutex_lock(&mutex_);
    remove_media_root(root.path());
    g_mutex_unlock(&mutex_);
}

void FileSystemScanner::Private::OnMediaRootFinished
                                                (FileSystemWalkerPtr walker) {
    const std::string root_path = walker->media_root().path();
    kDebug("Scanning \"{1}\" just finished.") % root_path;
    g_mutex_lock(&mutex_);

    const std::string error_message = walker->error_message();

    if (not error_message.empty()) {
        // FIXME(future): Maybe should escalate this even further,
        // e.g. to the media scanner service?
        kWarning("Scanning failed for \"{1}\": {2}.")
                % root_path % error_message;
    }

    // Stop tracking this walker as pending.
    const PendingRootVector::iterator it =
            std::find(pending_roots_.begin(), pending_roots_.end(),
                      walker->task_group());

    if (it != pending_roots_.end()) {
        pending_roots_.erase(it);
    } else {
        kWarning("Cannot find scanner task for \"{1}\".")
                % root_path;
    }

    // Wait for remaining metadata resolvers if this was the last walker.
    if (pending_roots_.empty()) {
        kDebug("Last media root scanned, waiting for pending metadata.");

        if (metadata_resolver_)
            metadata_resolver_->WaitForFinished();

        BOOST_ASSERT(is_idle());

        Idle::AddOnce(std::bind(&Private::NotifyScanningFinished,
                                  this, std::string()));
    }

    g_mutex_unlock(&mutex_);
}

void FileSystemScanner::Private::NotifyScanningFinished
                                                (std::string error_message) {
    std::for_each(listeners_.begin(), listeners_.end(),
                  std::bind(&FileSystemScanner::Listener::OnScanningFinished,
                              std::placeholders::_1, error_message));
}

} // namespace mediascanner
