/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/taskmanager.h"

// Standard Library
#include <list>
#include <string>

// Media Scanner
#include "mediascanner/glibutils.h"
#include "mediascanner/logging.h"

namespace mediascanner {

// Context specific logging domains
static const logging::Domain kDebug("debug/tasks", logging::debug());
static const logging::Domain kTrace("trace/tasks", logging::trace());

// Last id assigned to a task
static unsigned last_task_id = 0;

/**
  * @brief Description of a queued task.
  */
class TaskManager::TaskInfo {
public:
    /**
     * @brief Constructs a new task description.
     * @param priority a number used to sort tasks
     * @param group a number used identify groups of tasks
     * @param task the function to execute
     */
    TaskInfo(unsigned priority,
             unsigned group_id,
             const TaskFunction &task)
        : task_(task)
        , priority_(priority)
        , group_id_(group_id)
        , task_id_(++last_task_id) {
    }

    /**
     * @brief This number is used to sort tasks.
     */
    unsigned priority() const {
        return priority_;
    }

    /**
     * @brief This number is used to identify groups of tasks.
     */
    size_t group_id() const {
        return group_id_;
    }

    /**
     * @brief This number is used to identify tasks.
     */
    size_t task_id() const {
        return task_id_;
    }

    /**
     * @brief Runs the function associated with this task.
     */
    void RunTask() const {
        task_();
    }

    bool less_by_priority(const TaskManager::TaskInfo &other) const {
        return priority_ < other.priority_;
    }

private:
    TaskFunction task_;
    unsigned priority_;
    unsigned group_id_;
    unsigned task_id_;
};

class TaskManager::Private {
    friend class TaskManager;

    typedef std::list<TaskInfo> TaskQueue;

    explicit Private(const std::string &name);
    ~Private();

    TaskQueue::iterator find_predecessor(const TaskInfo &task) {
        if (task.priority() == 0)
            return tasks_.begin();

        return std::find_if
                (tasks_.begin(), tasks_.end(),
                 std::bind(&TaskInfo::less_by_priority, task, std::placeholders::_1));
    }


    TaskQueue::iterator find_successor(const TaskInfo &task) {
        if (task.priority() == 0)
            return tasks_.end();

        return std::find_if
                (tasks_.begin(), tasks_.end(),
                 [&](const TaskInfo &t) {return ! t.less_by_priority(task); });
    }

    static void *BackgroundThread(void *data);
    void Shutdown();

    GThread *const foreground_thread_;
    GThread *background_thread_;

    std::string name_;
    TaskQueue tasks_;
    GMutex mutex_;
    GCond cond_;

    volatile bool destructing_;
};

TaskManager::Private::Private(const std::string &name)
    : foreground_thread_(g_thread_self())
    , background_thread_(nullptr)
    , name_(name)
    , destructing_(false) {
    // ensure the GThread primitives are in usable state
    g_mutex_init(&mutex_);
    g_cond_init(&cond_);

    // acquire mutex to prepare waiting for the background thread
    g_mutex_lock(&mutex_);

    // create the background thread
    background_thread_ = g_thread_new(name_.c_str(),
                                      &Private::BackgroundThread,
                                      this);

    // wait for background thread getting ready
    g_cond_wait(&cond_, &mutex_);
    g_mutex_unlock(&mutex_);

    kDebug("Task manager for {1} ready, "
           "with foreground thread {2} and background thread: {3}")
            % name_ % foreground_thread_ % background_thread_;
}

TaskManager::Private::~Private() {
    kDebug("Destroying the task manager for {1}.") % name_;
    Shutdown();
}

void TaskManager::Private::Shutdown()  {
    if (background_thread_) {
        // Signal the background thread we are tearing down.
        g_mutex_lock(&mutex_);

        destructing_ = true;
        tasks_.clear();

        g_cond_signal(&cond_);
        g_mutex_unlock(&mutex_);

        // Wait for the background thread to follow our advice.
        kDebug("Waiting for the background thread of {1} to finish.") % name_;
        g_thread_join(background_thread_);

        // Destroy the thread object.
        g_thread_unref(background_thread_);
        background_thread_ = nullptr;
    }
}

TaskManager::TaskManager(const std::string &name)
    : d(new Private(name)) {
}

TaskManager::~TaskManager() {
    delete d;
}

unsigned TaskManager::PrependGroupedTask(unsigned group_id,
                                         const TaskFunction &task,
                                         unsigned priority) {
    g_mutex_lock(&d->mutex_);

    const TaskInfo task_info(priority, group_id, task);
    d->tasks_.insert(d->find_predecessor(task_info), task_info);
    const unsigned task_id = task_info.task_id();

    const size_t num_tasks = d->tasks_.size();
    kTrace("Prepending task {1}, now queuing {2} tasks for {3}.")
            % priority % num_tasks % d->name_;

    g_cond_signal(&d->cond_);
    g_mutex_unlock(&d->mutex_);

    return task_id;
}

unsigned TaskManager::AppendGroupedTask(unsigned group_id,
                                        const TaskFunction &task,
                                        unsigned priority) {
    g_mutex_lock(&d->mutex_);

    const TaskInfo task_info(priority, group_id, task);
    d->tasks_.insert(d->find_successor(task_info), task_info);
    const unsigned task_id = task_info.task_id();

    const size_t num_tasks = d->tasks_.size();
    kTrace("Appending task {1}, now queuing {2} tasks for {3}")
            % priority % num_tasks % d->name_;

    g_cond_signal(&d->cond_);
    g_mutex_unlock(&d->mutex_);

    return task_id;
}

static void RunAndNotifyOnExit(const TaskManager::TaskFunction &task,
                               GCond *cond, GMutex *mutex) {
    kTrace("Starting sync task...");

    task();

    kTrace("Synchronius task finished.");

    g_mutex_lock(mutex);
    g_cond_signal(cond);
    g_mutex_unlock(mutex);

    kTrace("Synchronius task signaled.");
}

void TaskManager::RunGroupedTask(unsigned group_id,
                                 const TaskFunction &task,
                                 unsigned priority) {
    BOOST_ASSERT_MSG(g_thread_self() != d->background_thread_,
                     "Function must not be called from task thread");

    GCond cond;
    g_cond_init(&cond);

    GMutex mutex;
    g_mutex_init(&mutex);
    g_mutex_lock(&mutex);

    AppendGroupedTask(group_id,
                      std::bind(&RunAndNotifyOnExit, task, &cond, &mutex),
                      priority);

    kTrace("Waiting for queued task to finish...");

    g_cond_wait(&cond, &mutex);
    g_mutex_unlock(&mutex);
}

unsigned TaskManager::CancelByGroupId(unsigned group_id) {
    g_mutex_lock(&d->mutex_);

    Private::TaskQueue::iterator it = d->tasks_.begin();
    unsigned task_count = 0;

    while (it != d->tasks_.end()) {
        if (group_id == it->group_id()) {
            const size_t task_id = it->task_id();
            kDebug("Cancelling task {1} for group {2} of {3}")
                % task_id % group_id % d->name_;

            it = d->tasks_.erase(it);
            ++task_count;
            continue;
        }

        ++it;
    }

    const size_t num_tasks = d->tasks_.size();
    kTrace("Number of queued tasks: {1}") % num_tasks;
    g_mutex_unlock(&d->mutex_);

    return task_count;
}

bool TaskManager::CancelTaskByTaskId(unsigned task_id) {
    g_mutex_lock(&d->mutex_);

    Private::TaskQueue::iterator it = d->tasks_.begin();
    bool task_cancelled = false;

    while (it != d->tasks_.end()) {
        if (task_id == it->task_id()) {
            kDebug("Cancelling task {1} of {2}") % task_id % d->name_;
            it = d->tasks_.erase(it);
            task_cancelled = true;
            break;
        }

        ++it;
    }

    const size_t num_tasks = d->tasks_.size();
    kTrace("Number of queued tasks: {1}") % num_tasks;
    g_mutex_unlock(&d->mutex_);

    return task_cancelled;
}

// Main routine of the task manager's background thread.
void *TaskManager::Private::BackgroundThread(void *data) {
    Private *const d = static_cast<Private *>(data);

    // Signal that the backround thread got ready for work
    g_mutex_lock(&d->mutex_);
    g_cond_signal(&d->cond_);

    for (;;) {
        // Abort on destructor invokation.
        if (d->destructing_)
            break;

        // Wait for the next event to happen.
        g_cond_wait(&d->cond_, &d->mutex_);

        // Process pending tasks.
        while (not d->tasks_.empty()) {
            const TaskInfo task = d->tasks_.front();
            d->tasks_.pop_front();

            const unsigned task_priority = task.priority();
            const size_t num_tasks = d->tasks_.size();
            kTrace("Running task {1}, now queuing {2} tasks for {3}.")
                    % task_priority % num_tasks % d->name_;

            // Run the next task.
            g_mutex_unlock(&d->mutex_);
            task.RunTask();
            g_mutex_lock(&d->mutex_);
        }
    }

    g_mutex_unlock(&d->mutex_);

    return 0;
}

void TaskManager::Shutdown() {
    d->Shutdown();
}

} // namespace mediascanner
