/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/propertyschema.h"

// GStreamer
#include <gst/pbutils/gstdiscoverer.h>
#include <gst/tag/tag.h>

// Boost C++
#include <boost/algorithm/string/predicate.hpp>

// C++ Standard Library
#include <sstream>
#include <string>

// Media Scanner
#include "mediascanner/glibutils.h"
#include "mediascanner/propertyprivate.h"
#include "mediascanner/utilities.h"

// TODO(M3): L10N: properly pull gettext()/boost::translate
#define N_(MsgId) (MsgId)

namespace mediascanner {
namespace schema {

// Generic Properties //////////////////////////////////////////////////////////

ExternalUrl::ExternalUrl()
    : StringProperty(L"external-url",
                     GRL_METADATA_KEY_EXTERNAL_URL,
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_LOCATION)) {
}

HomepageUrl::HomepageUrl()
    : StringProperty(L"homepage-url",
                     GRL_METADATA_KEY_SITE,
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_HOMEPAGE)) {
}

Author::Author()
    : TextProperty(L"author",
                   GRL_METADATA_KEY_AUTHOR,
                   Property::Category::Generic,
                   Property::MergeAppend,
                   merge_nothing) {
}

License::License()
    : StringProperty(L"license",
                     GRL_METADATA_KEY_LICENSE,
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_LICENSE)) {
}

LicenseUri::LicenseUri()
    : StringProperty(L"license-uri",
                     define("lucene-license-uri",
                            N_("License Uri (Lucene)"),
                            N_("...")),
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_LICENSE_URI)) {
}

Copyright::Copyright()
    : StringProperty(L"copyright",
                     define("lucene-copyright",
                            N_("Copyright (Lucene)"),
                            N_("Copyright notice of this media")),
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_COPYRIGHT)) {
}

CopyrightUri::CopyrightUri()
    : StringProperty(L"copyright-uri",
                     define("lucene-copyright-uri",
                            N_("Copyright Uri (Lucene)"),
                            N_("...")),
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_COPYRIGHT_URI)) {
}

Description::Description()
    : TextProperty(L"description",
                   GRL_METADATA_KEY_DESCRIPTION,
                   Property::Category::Generic,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_DESCRIPTION)) {
}

Rating::Rating()
    : NumericProperty(L"rating",
                      GRL_METADATA_KEY_RATING,
                      Property::Category::Generic,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_USER_RATING)) {
}

Title::Title()
    : TextProperty(L"title",
                   GRL_METADATA_KEY_TITLE,
                   Property::Category::Generic,
                   Property::MergeReplace,
                   bind_tag(GST_TAG_TITLE)) {
}

OriginalTitle::OriginalTitle()
    : TextProperty(L"original-title",
                   GRL_METADATA_KEY_ORIGINAL_TITLE,
                   Property::Category::Generic,
                   Property::MergeReplace,
                   merge_nothing) {
}

Certification::Certification()
    : StringProperty(L"certification",
                     GRL_METADATA_KEY_CERTIFICATE,
                     Property::Category::Generic,
                     Property::MergeAppend,
                     merge_nothing) {
}

Region::Region()
    : StringProperty(L"region",
                     GRL_METADATA_KEY_REGION,
                     Property::Category::Generic,
                     Property::MergeAppend,
                     merge_nothing) {
}

Comment::Comment()
    : TextProperty(L"comment",
                   define("lucene-comment",
                          N_("Comment (Lucene)"),
                          N_("User comment about this media")),
                   Property::Category::Generic,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_COMMENT)) {
}

// FIXME(M3): Do we need a special tag merger here?
Keyword::Keyword()
    : StringProperty(L"keyword",
                     GRL_METADATA_KEY_KEYWORD,
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_KEYWORDS)) {
}

Language::Language()
    : StringProperty(L"language",
                     define("lucene-language",
                            N_("Language (Lucene)"),
                            N_("Languages this media is adopted to")),
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_LANGUAGE_CODE)) {
}

Version::Version()
    : StringProperty(L"version",
                     define("lucene-version",
                            N_("Version (Lucene)"),
                            N_("...")),
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_VERSION)) {
}

Organization::Organization()
    : StringProperty(L"organization",
                     define("lucene-organization",
                            N_("Organization (Lucene)"),
                            N_("...")),
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_ORGANIZATION)) {
}

Contact::Contact()
    : StringProperty(L"contact",
                     define("lucene-contact",
                            N_("Contact (Lucene)"),
                            N_("...")),
                     Property::Category::Generic,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CONTACT)) {
}

Favourite::Favourite()
    : NumericProperty(L"favourite",
                      GRL_METADATA_KEY_FAVOURITE,
                      Property::Category::Generic,
                      Property::MergeReplace,
                      merge_nothing) {
}

// File Properties /////////////////////////////////////////////////////////////

Url::Url()
    : StringProperty(L"url",
                     GRL_METADATA_KEY_URL,
                     Property::Category::File,
                     Property::MergeAppend,
                     merge_nothing) {
}

ETag::ETag()
    : StringProperty(L"etag",
                     define("lucene-etag",
                            N_("ETag (Lucene)"),
                            N_("A sophisticated last change indicator, "
                               "similiar to the ETag header of HTTP 1.1")),
                     Property::Category::File,
                     Property::MergeAppend,
                     merge_nothing) {
}

MimeType::MimeType()
    : StringProperty(L"mime-type",
                     GRL_METADATA_KEY_MIME,
                     Property::Category::File,
                     Property::MergeAppend,
                     merge_nothing) {
}

LastModified::LastModified()
    : DateTimeProperty(L"modification-date",
                       GRL_METADATA_KEY_MODIFICATION_DATE,
                       Property::Category::File,
                       Property::MergeAppend,
                       merge_nothing) {
}

FileSize::FileSize()
    : NumericProperty(L"file-size",
                      define("lucene-file-size",
                             N_("File Size (Lucene"),
                             N_("Size of the media file in bytes")),
                      Property::Category::File,
                      Property::MergeAppend,
                      merge_nothing) {
}

LastAccessed::LastAccessed()
    : DateTimeProperty(L"last-accessed",
                       define("lucene-last-accessed",
                              N_("Last access time (Lucene)"),
                              N_("Time when the media file was accessed "
                                 "the last time")),
                       Property::Category::File,
                       Property::MergeAppend,
                       merge_nothing) {
}

// Media Properties ////////////////////////////////////////////////////////////

static int get_duration(const GstDiscovererInfo *media) {
    return GST_TIME_AS_SECONDS(gst_discoverer_info_get_duration(media));
}

Duration::Duration()
    : NumericProperty(L"duration",
                      GRL_METADATA_KEY_DURATION,
                      Property::Category::Media,
                      Property::MergeAppend,
                      bind_any(bind_attr(get_duration),
                               bind_tag(GST_TAG_DURATION))) {
}

static bool get_seekable(const GstDiscovererInfo *media) {
    return gst_discoverer_info_get_seekable(media);
}

Seekable::Seekable()
    : NumericProperty(L"seekable",
                      define("lucene-seekable",
                             N_("Seekable (Lucene)"),
                             N_("The media is seekable"), true),
                      Property::Category::Media,
                      Property::MergeAppend,
                      bind_attr(get_seekable)) {
}

LastPlayed::LastPlayed()
    : DateTimeProperty(L"last-played",
                       GRL_METADATA_KEY_LAST_PLAYED,
                       Property::Category::Media,
                       Property::MergeReplace,
                       merge_nothing) {
}

PlayCount::PlayCount()
    : NumericProperty(L"play-count",
                      GRL_METADATA_KEY_PLAY_COUNT,
                      Property::Category::Media,
                      Property::MergeReplace,
                      merge_nothing) {
}

LastPosition::LastPosition()
    : NumericProperty(L"last-position",
                      GRL_METADATA_KEY_LAST_POSITION,
                      Property::Category::Media,
                      Property::MergeReplace,
                      merge_nothing) {
}

Producer::Producer()
    : TextProperty(L"producer",
                   GRL_METADATA_KEY_PRODUCER,
                   Property::Category::Media,
                   Property::MergeAppend,
                   merge_nothing) {
}

Performer::Performer()
    : TextProperty(L"performer",
                   GRL_METADATA_KEY_PERFORMER,
                   Property::Category::Media,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_PERFORMER)) {
}

// TODO(jamesh): GST_TAG_IMAGE is actually a GstSegment containing the
// image data rather than a file URI.
Cover::Cover()
    : StringProperty(L"cover",
                     GRL_METADATA_KEY_THUMBNAIL,
                     Property::Category::Media,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_IMAGE)) {
}

Poster::Poster()
    : StringProperty(L"poster",
                     define("tmdb-poster",
                            N_("Poster (Lucene, TMDB)"),
                            N_("URLs pointing to posters for this media")),
                     Property::Category::Media,
                     Property::MergeAppend,
                     merge_nothing) {
}

PublicationDate::PublicationDate()
    : DateTimeProperty(L"publication-date",
                       GRL_METADATA_KEY_PUBLICATION_DATE,
                       Property::Category::Media,
                       Property::MergeAppend,
                       merge_nothing) {
}

Studio::Studio()
    : StringProperty(L"studio",
                     GRL_METADATA_KEY_STUDIO,
                     Property::Category::Media,
                     Property::MergeAppend,
                     merge_nothing) {
}

Genre::Genre()
    : StringProperty(L"genre",
                     GRL_METADATA_KEY_GENRE,
                     Property::Category::Media,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_GENRE)) {
}

EncodedBy::EncodedBy()
    : StringProperty(L"encoded-by",
                     define("lucene-encoded-by",
                            N_("Encoded By (Lucene)"),
                            N_("...")),
                     Property::Category::Media,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_ENCODED_BY)) {
}

ContainerFormat::ContainerFormat()
    : StringProperty(L"container-format",
                     define("lucene-container-format",
                            N_("Container Format (Lucene)"),
                            N_("...")),
                     Property::Category::Media,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CONTAINER_FORMAT)) {
}

Encoder::Encoder()
    : StringProperty(L"encoder",
                     define("lucene-encoder",
                            N_("Encoder (Lucene)"),
                            N_("...")),
                     Property::Category::Media,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_ENCODER)) {
}

EncoderVersion::EncoderVersion()
    : NumericProperty(L"encoder-version",
                      define("lucene-encoder-version",
                             N_("Encoder Version (Lucene)"),
                             N_("...")),
                      Property::Category::Media,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_ENCODER_VERSION)) {
}

// Music Properties ////////////////////////////////////////////////////////////

static int get_audio_bitrate(const GstDiscovererAudioInfo *info) {
    return gst_discoverer_audio_info_get_bitrate(info);
}

AudioBitRate::AudioBitRate()
    : NumericProperty(L"audio-bitrate",
                      GRL_METADATA_KEY_BITRATE,
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_attr(get_audio_bitrate)) {
}

static int get_max_audio_bitrate(const GstDiscovererAudioInfo *info) {
    return gst_discoverer_audio_info_get_max_bitrate(info);
}

MaximumAudioBitRate::MaximumAudioBitRate()
    : NumericProperty(L"maximum-audio-bitrate",
                      define("lucene-maximum-audio-bitrate",
                             N_("Maximum Audio Bitrate (Lucene)"),
                             N_("The maximum bitrate of this media's "
                                "audio stream")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_attr(get_max_audio_bitrate)) {
}

AudioCodec::AudioCodec()
    : StringProperty(L"audio-codec",
                     define("lucene-audio-codec",
                            N_("Audio Codec (Lucene)"),
                            N_("The codec used for this media's audio stream")),
                     Property::Category::Music,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_AUDIO_CODEC)) {
}

int get_sample_rate(const GstDiscovererAudioInfo *info) {
    return gst_discoverer_audio_info_get_sample_rate(info);
}

SampleRate::SampleRate()
    : NumericProperty(L"sample-rate",
                      define("lucene-sample-rate",
                             N_("Sample Rate (Lucene"),
                             N_("The sample rate of this media's "
                                "audio stream")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_attr(get_sample_rate)) {
}

int get_channels(const GstDiscovererAudioInfo *info) {
    return gst_discoverer_audio_info_get_channels(info);
}

ChannelCount::ChannelCount()
    : NumericProperty(L"channel-count",
                      define("lucene-channel-count",
                             N_("Channel Count (Lucene"),
                             N_("The number of this media's "
                                "audio stream channels")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_attr(get_channels)) {
}

AlbumArtist::AlbumArtist()
    : TextProperty(L"album-artist",
                   define("lucene-artist",
                          N_("Artist (Lucene)"),
                          N_("The artists of the album "
                             "this song was released with")),
                   Property::Category::Music,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_ALBUM_ARTIST)) {
}

Composer::Composer()
    : TextProperty(L"composer",
                   define("lucene-composer",
                          N_("Composer (Lucene)"),
                          N_("The composers of this song")),
                   Property::Category::Music,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_COMPOSER)) {
}

TrackCount::TrackCount()
    : NumericProperty(L"track-count",
                      define("lucene-track-count",
                             N_("Track Count (Lucene"),
                             N_("The number of individual tracks "
                                "on this song's album")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_TRACK_COUNT)) {
}

DiscNumber::DiscNumber()
    : NumericProperty(L"disc-number",
                      define("lucene-disc-number",
                             N_("Disc Number (Lucene"),
                             N_("The disc number within the album "
                                "of this song")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_ALBUM_VOLUME_NUMBER)) {
}

Artist::Artist()
    : TextProperty(L"artist",
                   GRL_METADATA_KEY_ARTIST,
                   Property::Category::Music,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_ARTIST)) {
}

Album::Album()
    : TextProperty(L"album",
                   GRL_METADATA_KEY_ALBUM,
                   Property::Category::Music,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_ALBUM)) {
}

Lyrics::Lyrics()
    : TextProperty(L"lyrics",
                   GRL_METADATA_KEY_LYRICS,
                   Property::Category::Music,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_LYRICS)) {
}

TrackNumber::TrackNumber()
    : NumericProperty(L"track-number",
                      GRL_METADATA_KEY_TRACK_NUMBER,
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_TRACK_NUMBER)) {
}

AlbumVolumeNumber::AlbumVolumeNumber()
    : NumericProperty(L"album-volume-number",
                      define("lucene-album-volume-number",
                             N_("Album Volume Number (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_ALBUM_VOLUME_NUMBER)) {
}

AlbumVolumeCount::AlbumVolumeCount()
    : NumericProperty(L"album-volume-count",
                      define("lucene-album-volume-count",
                             N_("Album Volume Count (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_ALBUM_VOLUME_COUNT)) {
}

ISRC::ISRC()
    : StringProperty(L"isrc",
                     define("lucene-isrc",
                            N_("ISRC (Lucene)"),
                            N_("International Standard Recording Code")),
                     Property::Category::Music,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_ISRC)) {
}

TrackGain::TrackGain()
    : NumericProperty(L"track-gain",
                      define("lucene-track-gain",
                             N_("Track Gain (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_TRACK_GAIN)) {
}

TrackPeak::TrackPeak()
    : NumericProperty(L"track-peak",
                      define("lucene-track-peak",
                             N_("Track Peak (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_TRACK_PEAK)) {
}

AlbumGain::AlbumGain()
    : NumericProperty(L"album-gain",
                      define("lucene-album-gain",
                             N_("Album Gain (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_ALBUM_GAIN)) {
}

AlbumPeak::AlbumPeak()
    : NumericProperty(L"album-peak",
                      define("lucene-album-peak",
                             N_("Album Peak (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_ALBUM_PEAK)) {
}

ReferenceLevel::ReferenceLevel()
    : NumericProperty(L"reference-level",
                      define("lucene-reference-level",
                             N_("Reference Level (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_REFERENCE_LEVEL)) {
}

BeatsPerMinute::BeatsPerMinute()
    : NumericProperty(L"beats-per-minute",
                      define("lucene-beats-per-minute",
                             N_("Beats Per Minute (Lucene)"),
                             N_("...")),
                      Property::Category::Music,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_BEATS_PER_MINUTE)) {
}

// Image Properties ////////////////////////////////////////////////////////////

static int get_width(const GstDiscovererVideoInfo *info) {
    return gst_discoverer_video_info_get_width(info);
}

Width::Width()
    : NumericProperty(L"width",
                      GRL_METADATA_KEY_WIDTH,
                      Property::Category::Image,
                      Property::MergeAppend,
                      bind_attr(get_width)) {
}

static int get_height(const GstDiscovererVideoInfo *info) {
    return gst_discoverer_video_info_get_height(info);
}

Height::Height()
    : NumericProperty(L"height",
                      GRL_METADATA_KEY_HEIGHT,
                      Property::Category::Image,
                      Property::MergeAppend,
                      bind_attr(get_height)) {
}

class ImageOrientation::Private : public StringProperty::Private {
public:
    typedef StringProperty::Private inherited;

    Private(const String &field_name,
            const Property::MetadataKey &metadata_key,
            Property::Category category,
            Property::MergeStrategy merge_strategy,
            const Property::StreamInfoFunction &stream_info)
        : inherited(field_name, metadata_key, category,
                    merge_strategy, stream_info) {
    }

    bool TransformGriloValue(const GValue *input, Value *output) const {
        if (G_VALUE_TYPE(input) != G_TYPE_INT) {
            // TODO(M5): Maybe also permit GStreamer orientationt tags?
            GValue safe_value = G_VALUE_INIT;
            g_value_init(&safe_value, G_TYPE_INT);

            // FIXME(M3): Print warning, move to property.cpp?
            if (not g_value_transform(input, &safe_value))
                return false;

            return TransformGriloValue(&safe_value, output);
        }

        std::wostringstream oss;
        oss << "rotate-" << (g_value_get_int(input) % 360);
        *output = oss.str();
        return true;
    }

    Wrapper<GValue> MakeGriloValue(const Value &value) const {
        std::wstring text = boost::get<std::wstring>(value);

        const std::wstring kFlip = L"flip-";
        if (boost::algorithm::starts_with(text, kFlip)) {
            // FIXME(M3): Add related image-flipped metadata key
            text = text.substr(kFlip.length());
        }

        const std::wstring kRotate = L"rotate-";
        if (boost::algorithm::starts_with(text, kRotate)) {
            text = text.substr(kRotate.length());
            int rotation;

            if (not (std::wistringstream(text) >> rotation))
                return Wrapper<GValue>();

            GValue result = G_VALUE_INIT;
            g_value_init(&result, G_TYPE_INT);
            g_value_set_int(&result, rotation % 360);
            return wrap(&result);
        }

        return Wrapper<GValue>();
    }
};

ImageOrientation::ImageOrientation()
    : StringProperty(new Private(L"orientation",
                                 GRL_METADATA_KEY_ORIENTATION,
                                 Property::Category::Image,
                                 Property::MergeAppend,
                                 bind_tag(GST_TAG_IMAGE_ORIENTATION))) {
}

HorizontalResolution::HorizontalResolution()
    : NumericProperty(L"horizontal-resolution",
                      define("lucene-horizontal-resolution",
                             N_("Horizontal Image Resolution (Lucene)"),
                             N_("...")),
                      Property::Category::Image,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_IMAGE_HORIZONTAL_PPI)) {
}

VerticalResolution::VerticalResolution()
    : NumericProperty(L"vertical-resolution",
                      define("lucene-vertical-resolution",
                             N_("Vertical Image Resolution (Lucene)"),
                             N_("...")),
                      Property::Category::Image,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_IMAGE_VERTICAL_PPI)) {
}

// Photo Properties ////////////////////////////////////////////////////////////

DateTaken::DateTaken()
    : DateTimeProperty(L"date-taken",
                       GRL_METADATA_KEY_CREATION_DATE,
                       Property::Category::Photo,
                       Property::MergeAppend,
                       bind_any(bind_tag(GST_TAG_DATE_TIME),
                                bind_tag(GST_TAG_DATE))) {
}

DeviceModel::DeviceModel()
    : TextProperty(L"camera-model",
                   GRL_METADATA_KEY_CAMERA_MODEL,
                   Property::Category::Photo,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_DEVICE_MODEL)) {
}

IsoSpeed::IsoSpeed()
    : NumericProperty(L"iso-speed",
                      GRL_METADATA_KEY_ISO_SPEED,
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_CAPTURING_ISO_SPEED)) {
}

ExposureTime::ExposureTime()
    : NumericProperty(L"exposure-time",
                      GRL_METADATA_KEY_EXPOSURE_TIME,
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_CAPTURING_SHUTTER_SPEED)) {
}

FlashUsed::FlashUsed()
    : NumericProperty(L"flash-used",
                      GRL_METADATA_KEY_FLASH_USED,
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_CAPTURING_FLASH_FIRED)) { }

DeviceManufacturer::DeviceManufacturer()
    : TextProperty(L"camera-manufacturer",
                   define("lucene-camera-manufacturer",
                          N_("Camera Manufacturer (Lucene)"),
                          N_("The manufacturer of the camera used "
                             "for capturing this photo")),
                   Property::Category::Photo,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_DEVICE_MANUFACTURER)) {
}

FocalRatio::FocalRatio()
    : NumericProperty(L"focal-ratio",
                      define("lucene-focal-ratio",
                             N_("Focal Ratio (Lucene"),
                             N_("Focal length used when capturing this photo")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_CAPTURING_FOCAL_RATIO)) {
}

FocalLength::FocalLength()
    : NumericProperty(L"focal-length",
                      define("lucene-focal-length",
                             N_("Focal Length (Lucene"),
                             N_("Denominator of the focal length used "
                                "when capturing this photo")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_CAPTURING_FOCAL_LENGTH)) {
}

MeteringMode::MeteringMode()
    : StringProperty(L"metering-mode",
                     define("lucene-metering-mode",
                            N_("Metering mode (Lucene)"),
                            N_("The metering mode used when capturing "
                               "this photo")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_METERING_MODE)) {
}

WhiteBalance::WhiteBalance()
    : StringProperty(L"white-balance",
                     define("lucene-white-balance",
                            N_("White Balance (Lucene)"),
                            N_("The white balance parameters "
                               "applied to this photo")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_WHITE_BALANCE)) {
}

GeoLocationName::GeoLocationName()
    : StringProperty(L"geo-location-name",
                     define("lucene-geo-location-name",
                            N_("Geo Location Name (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_GEO_LOCATION_NAME)) {
}

GeoLocationLatitude::GeoLocationLatitude()
    : NumericProperty(L"geo-location-latitude",
                      define("lucene-geo-location-latitude",
                             N_("Geo Location Latitude (Lucene)"),
                             N_("...")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_GEO_LOCATION_LATITUDE)) {
}

GeoLocationLongitude::GeoLocationLongitude()
    : NumericProperty(L"geo-location-longitude",
                      define("lucene-geo-location-longitude",
                             N_("Geo Location Longitude (Lucene)"),
                             N_("...")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_GEO_LOCATION_LONGITUDE)) {
}

GeoLocationElevation::GeoLocationElevation()
    : NumericProperty(L"geo-location-elevation",
                      define("lucene-geo-location-elevation",
                             N_("Geo Location Elevation (Lucene)"),
                             N_("...")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_GEO_LOCATION_ELEVATION)) {
}

GeoLocationCountry::GeoLocationCountry()
    : StringProperty(L"geo-location-country",
                     define("lucene-geo-location-country",
                            N_("Geo Location Country (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_GEO_LOCATION_COUNTRY)) {
}

GeoLocationCity::GeoLocationCity()
    : StringProperty(L"geo-location-city",
                     define("lucene-geo-location-city",
                            N_("Geo Location City (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_GEO_LOCATION_CITY)) {
}

GeoLocationSublocation::GeoLocationSublocation()
    : StringProperty(L"geo-location-sublocation",
                     define("lucene-geo-location-sublocation",
                            N_("Geo Location Sublocation (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_GEO_LOCATION_SUBLOCATION)) {
}

GeoLocationHorizontalError::GeoLocationHorizontalError()
    : NumericProperty(L"geo-location-horizontal-error",
                      define("lucene-geo-location-horizontal-error",
                             N_("Geo Location Horizontal Error (Lucene)"),
                             N_("...")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_GEO_LOCATION_HORIZONTAL_ERROR)) {
}

GeoLocationMovementSpeed::GeoLocationMovementSpeed()
    : NumericProperty(L"geo-location-movementSpeed",
                      define("lucene-geo-location-movement-speed",
                             N_("Geo Location Movement Speed (Lucene)"),
                             N_("...")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_GEO_LOCATION_MOVEMENT_SPEED)) {
}

GeoLocationMovementDirection::GeoLocationMovementDirection()
    : NumericProperty(L"geo-location-movementDirection",
                      define("lucene-geo-location-movement-direction",
                             N_("Geo Location Movement Direction (Lucene)"),
                             N_("..."),
                             0.0, 360.0),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_GEO_LOCATION_MOVEMENT_DIRECTION)) {
}

GeoLocationCaptureDirection::GeoLocationCaptureDirection()
    : NumericProperty(L"geo-location-captureDirection",
                      define("lucene-geo-location-capture-direction",
                             N_("Geo Location Capture Direction (Lucene)"),
                             N_("..."),
                             0.0, 360.0),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_GEO_LOCATION_CAPTURE_DIRECTION)) {
}

Grouping::Grouping()
    : StringProperty(L"grouping",
                     define("lucene-grouping",
                            N_("Grouping (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_GROUPING)) {
}

ApplicationName::ApplicationName()
    : StringProperty(L"application-name",
                     define("lucene-application-name",
                            N_("Application Name (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_APPLICATION_NAME)) {
}

CapturingDigitalZoomRatio::CapturingDigitalZoomRatio()
    : NumericProperty(L"capturing-digital-zoom-ratio",
                      define("lucene-capturing-digital-zoom-ratio",
                             N_("Capturing Digital Zoom Ratio (Lucene)"),
                             N_("...")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_CAPTURING_DIGITAL_ZOOM_RATIO)) {
}

CapturingExposureProgram::CapturingExposureProgram()
    : StringProperty(L"capturing-exposure-program",
                     define("lucene-capturing-exposure-program",
                            N_("Capturing Exposure Program (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_EXPOSURE_PROGRAM)) {
}

CapturingExposureMode::CapturingExposureMode()
    : StringProperty(L"capturing-exposure-mode",
                     define("lucene-capturing-exposure-mode",
                            N_("Capturing Exposure Mode (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_EXPOSURE_MODE)) {
}

CapturingExposureCompensation::CapturingExposureCompensation()
    : NumericProperty(L"capturing-exposure-compensation",
                      define("lucene-capturing-exposure-compensation",
                             N_("Capturing Exposure Compensation (Lucene)"),
                             N_("...")),
                      Property::Category::Photo,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_CAPTURING_EXPOSURE_COMPENSATION)) {
}

CapturingSceneCaptureType::CapturingSceneCaptureType()
    : StringProperty(L"capturing-scene-captureType",
                     define("lucene-capturing-scene-capture-type",
                            N_("Capturing Scene Capture Type (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_SCENE_CAPTURE_TYPE)) {
}

CapturingGainAdjustment::CapturingGainAdjustment()
    : StringProperty(L"capturing-gain-adjustment",
                     define("lucene-capturing-gain-adjustment",
                            N_("Capturing Gain Adjustment (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_GAIN_ADJUSTMENT)) {
}

CapturingContrast::CapturingContrast()
    : StringProperty(L"capturing-contrast",
                     define("lucene-capturing-contrast",
                            N_("Capturing Contrast (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_CONTRAST)) {
}

CapturingSaturation::CapturingSaturation()
    : StringProperty(L"capturing-saturation",
                     define("lucene-capturing-saturation",
                            N_("Capturing Saturation (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_SATURATION)) {
}

CapturingSharpness::CapturingSharpness()
    : StringProperty(L"capturing-sharpness",
                     define("lucene-capturing-sharpness",
                            N_("Capturing Sharpness (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_SHARPNESS)) {
}

CapturingFlashMode::CapturingFlashMode()
    : StringProperty(L"capturing-flash-mode",
                     define("lucene-capturing-flash-mode",
                            N_("Capturing Flash Mode (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_FLASH_MODE)) {
}

CapturingSource::CapturingSource()
    : StringProperty(L"capturing-source",
                     define("lucene-capturing-source",
                            N_("Capturing Source (Lucene)"),
                            N_("...")),
                     Property::Category::Photo,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_CAPTURING_SOURCE)) {
}

// Movie Properties ////////////////////////////////////////////////////////////

static Fraction get_framerate(const GstDiscovererVideoInfo *info) {
    return Fraction(gst_discoverer_video_info_get_framerate_num(info),
                    gst_discoverer_video_info_get_framerate_denom(info));
}

FrameRate::FrameRate()
    : NumericProperty(L"frame-rate",
                      GRL_METADATA_KEY_FRAMERATE,
                      Property::Category::Movie,
                      Property::MergeAppend,
                      bind_attr(get_framerate)) {
}

ShowName::ShowName()
    : TextProperty(L"show",
                   GRL_METADATA_KEY_SHOW,
                   Property::Category::Movie,
                   Property::MergeAppend,
                   bind_tag(GST_TAG_SHOW_NAME)) {
}

Episode::Episode()
    : NumericProperty(L"episode",
                      GRL_METADATA_KEY_EPISODE,
                      Property::Category::Movie,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_SHOW_EPISODE_NUMBER)) {
}

Season::Season()
    : NumericProperty(L"season",
                      GRL_METADATA_KEY_SEASON,
                      Property::Category::Movie,
                      Property::MergeAppend,
                      bind_tag(GST_TAG_SHOW_SEASON_NUMBER)) {
}

Backdrop::Backdrop()
    : StringProperty(L"backdrop",
                     define("tmdb-backdrop",
                            N_("Backdrop (Lucene, TMDB)"),
                            N_("Background information on this movie")),
                     Property::Category::Movie,
                     Property::MergeAppend,
                     merge_nothing) {
}

VideoCodec::VideoCodec()
    : StringProperty(L"video-codec",
                     define("lucene-video-code",
                            N_("Video Codec (Lucene)"),
                            N_("The codec used for this media's video stream")),
                     Property::Category::Movie,
                     Property::MergeAppend,
                     bind_tag(GST_TAG_VIDEO_CODEC)) {
}

int get_video_bitrate(const GstDiscovererVideoInfo *info) {
    return gst_discoverer_video_info_get_bitrate(info);
}

VideoBitRate::VideoBitRate()
    : NumericProperty(L"video-bitrate",
                      define("lucene-video-bitrate",
                             N_("Video Bitrate (Lucene"),
                             N_("The bitrate of this media's video stream")),
                      Property::Category::Movie,
                      Property::MergeAppend,
                      bind_attr(get_video_bitrate)) {
}

int get_max_video_bitrate(const GstDiscovererVideoInfo *info) {
    return gst_discoverer_video_info_get_max_bitrate(info);
}

MaximumVideoBitRate::MaximumVideoBitRate()
    : NumericProperty(L"maximum-video-bitrate",
                      define("lucene-maximum-video-bitrate",
                             N_("Maximum Video Bitrate (Lucene)"),
                             N_("The maximum bitrate of this media's "
                                "video stream")),
                      Property::Category::Movie,
                      Property::MergeAppend,
                      bind_attr(get_max_video_bitrate)) {
}

Director::Director()
    : TextProperty(L"director",
                   GRL_METADATA_KEY_DIRECTOR,
                   Property::Category::Movie,
                   Property::MergeAppend,
                   merge_nothing) {
}

ImdbId::ImdbId()
    : StringProperty(L"imdb-id",
                     define("tmdb-imdb-id",
                            N_("IMDB ID (Lucene, TMDB)"),
                            N_("Internet Movie Database ID of this movie")),
                     Property::Category::Movie,
                     Property::MergeReplace,
                     merge_nothing) {
}

TmdbId::TmdbId()
    : StringProperty(L"tmdb-id",
                     define("tmdb-id",
                            N_("TMDB ID (Lucene)"),
                            N_("The Movie Database ID of this movie")),
                     Property::Category::Movie,
                     Property::MergeReplace,
                     merge_nothing) {
}

// Property Definitions ////////////////////////////////////////////////////////

const AlbumArtist kAlbumArtist;
const AlbumGain kAlbumGain;
const Album kAlbum;
const AlbumPeak kAlbumPeak;
const AlbumVolumeCount kAlbumVolumeCount;
const AlbumVolumeNumber kAlbumVolumeNumber;
const ApplicationName kApplicationName;
const Artist kArtist;
const AudioBitRate kAudioBitRate;
const AudioCodec kAudioCodec;
const Author kAuthor;
const Backdrop kBackdrop;
const BeatsPerMinute kBeatsPerMinute;
const CapturingContrast kCapturingContrast;
const CapturingDigitalZoomRatio kCapturingDigitalZoomRatio;
const CapturingExposureCompensation kCapturingExposureCompensation;
const CapturingExposureMode kCapturingExposureMode;
const CapturingExposureProgram kCapturingExposureProgram;
const CapturingFlashMode kCapturingFlashMode;
const CapturingGainAdjustment kCapturingGainAdjustment;
const CapturingSaturation kCapturingSaturation;
const CapturingSceneCaptureType kCapturingSceneCaptureType;
const CapturingSharpness kCapturingSharpness;
const CapturingSource kCapturingSource;
const Certification kCertification;
const ChannelCount kChannelCount;
const Comment kComment;
const Composer kComposer;
const Contact kContact;
const ContainerFormat kContainerFormat;
const Copyright kCopyright;
const CopyrightUri kCopyrightUri;
const Cover kCover;
const DateTaken kDateTaken;
const Description kDescription;
const DeviceManufacturer kDeviceManufacturer;
const DeviceModel kDeviceModel;
const Director kDirector;
const DiscNumber kDiscNumber;
const Duration kDuration;
const EncodedBy kEncodedBy;
const Encoder kEncoder;
const EncoderVersion kEncoderVersion;
const Episode kEpisode;
const ETag kETag;
const ExposureTime kExposureTime;
const ExternalUrl kExternalUrl;
const Favourite kFavourite;
const FileSize kFileSize;
const FlashUsed kFlashUsed;
const FocalLength kFocalLength;
const FocalRatio kFocalRatio;
const FrameRate kFrameRate;
const Genre kGenre;
const GeoLocationCaptureDirection kGeoLocationCaptureDirection;
const GeoLocationCity kGeoLocationCity;
const GeoLocationCountry kGeoLocationCountry;
const GeoLocationElevation kGeoLocationElevation;
const GeoLocationHorizontalError kGeoLocationHorizontalError;
const GeoLocationLatitude kGeoLocationLatitude;
const GeoLocationLongitude kGeoLocationLongitude;
const GeoLocationMovementDirection kGeoLocationMovementDirection;
const GeoLocationMovementSpeed kGeoLocationMovementSpeed;
const GeoLocationName kGeoLocationName;
const GeoLocationSublocation kGeoLocationSublocation;
const Grouping kGrouping;
const Height kHeight;
const HomepageUrl kHomepageUrl;
const HorizontalResolution kHorizontalResolution;
const ImageOrientation kImageOrientation;
const ImdbId kImdbId;
const IsoSpeed kIsoSpeed;
const ISRC kISRC;
const Keyword kKeyword;
const Language kLanguage;
const LastAccessed kLastAccessed;
const LastModified kLastModified;
const LastPlayed kLastPlayed;
const LastPosition kLastPosition;
const License kLicense;
const LicenseUri kLicenseUri;
const Lyrics kLyrics;
const MaximumAudioBitRate kMaximumAudioBitRate;
const MaximumVideoBitRate kMaximumVideoBitRate;
const MeteringMode kMeteringMode;
const MimeType kMimeType;
const Organization kOrganization;
const OriginalTitle kOriginalTitle;
const Performer kPerformer;
const PlayCount kPlayCount;
const Poster kPoster;
const Producer kProducer;
const PublicationDate kPublicationDate;
const Rating kRating;
const ReferenceLevel kReferenceLevel;
const Region kRegion;
const SampleRate kSampleRate;
const Season kSeason;
const Seekable kSeekable;
const ShowName kShowName;
const Studio kStudio;
const Title kTitle;
const TmdbId kTmdbId;
const TrackCount kTrackCount;
const TrackGain kTrackGain;
const TrackNumber kTrackNumber;
const TrackPeak kTrackPeak;
const Url kUrl;
const Version kVersion;
const VerticalResolution kVerticalResolution;
const VideoBitRate kVideoBitRate;
const VideoCodec kVideoCodec;
const WhiteBalance kWhiteBalance;
const Width kWidth;

} // namespace schema
} // namespace mediascanner
