/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "grlmediascanner/mediasource.h"

// Boost C++
#include <boost/filesystem.hpp>
#include <boost/variant.hpp>

// C++ Standard Library
#include <algorithm>
#include <set>
#include <string>
#include <vector>
#include <functional>

// MediaScanner Plugin for Grilo
#include "grlmediascanner/enums.h"

// Media Scanner Library
#include "mediascanner/dbusservice.h"
#include "mediascanner/filter.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/mediaindex.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/mediautils.h"
#include "mediascanner/property.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/taskfacades.h"
#include "mediascanner/taskmanager.h"
#include "mediascanner/utilities.h"

// TODO(M3): L10N: properly pull gettext()/boost::translate
#define _(MsgId) (MsgId)

enum {
    PROP_NONE,
    PROP_INDEX_PATH,
    PROP_SEARCH_METHOD,
    N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES] = { NULL, };

struct _GrlMediaScannerSourcePrivate {
    _GrlMediaScannerSourcePrivate()
        : root_manager_(new mediascanner::MediaRootManager)
        , task_facade_(root_manager_)
        , task_manager_("media_scanner media source")
        , change_id_(0) {
        root_manager_->initialize();
    }

    mediascanner::MediaRootManagerPtr root_manager_;
    mediascanner::MediaIndexFacade<mediascanner::MediaIndex> task_facade_;
    mediascanner::TaskManager task_manager_;

    mediascanner::Wrapper<GrlCaps> simple_caps_;
    mediascanner::Wrapper<GrlCaps> filter_caps_;

    GrlMediaScannerSearchMethod search_method_;

    mediascanner::dbus::MediaScannerProxy service_proxy_;
    unsigned change_id_;
};

namespace mediascanner {

// Boost C++
using boost::locale::format;
using boost::gregorian::date;
using boost::posix_time::time_duration;

// Standard Library
using std::string;
using std::vector;
using std::wstring;

// Context specific logging domains
static const logging::Domain kWarning("warning/plugin", logging::warning());
static const logging::Domain kDebug("debug/plugin", logging::debug());
static const logging::Domain kTrace("trace/plugin", logging::trace());

G_DEFINE_TYPE(GrlMediaScannerSource, grl_media_scanner_source, GRL_TYPE_SOURCE)

static const GrlTypeFilter kSupportedTypeFilters =
        GRL_TYPE_FILTER_AUDIO | GRL_TYPE_FILTER_VIDEO | GRL_TYPE_FILTER_IMAGE;

static bool StoreMedia(dbus::MediaScannerProxy *service,
                       const Wrapper<GrlMedia> media,
                       const GList *const keys_to_store,
                       GList **failed_keys, string *error_message) {
    MediaInfo metadata;

    if (not metadata.fill_from_media(media.get(), keys_to_store,
                                     failed_keys, error_message))
        return false;

    // Connected to D-Bus if needed.
    Wrapper<GError> error;

    if (not service->handle()
            && not service->ConnectAndWait(Wrapper<GCancellable>(),
                                           error.out_param())) {
        const string message = to_string(error);
        *error_message =
                (format("Mediascanner service not available: {1}")
                 % message).str();
        return false;
    }

    const std::set<string> service_failed_keys =
            service->StoreMediaInfo(metadata, error.out_param());

    for (const string &field_name: service_failed_keys) {
        const Property p = Property::FromFieldName(ToUnicode(field_name));

        if (p) {
            *failed_keys = g_list_prepend
                (*failed_keys, GRLKEYID_TO_POINTER(p.metadata_key().id()));
        }
    }

    if (error) {
        const string message = to_string(error);
        *error_message =
                (format("Store operation failed: {1}")
                 % message).str();
        return false;
    }

    return true;
}

// Filter Creation Routines ////////////////////////////////////////////////////

static bool make_type_filters(GrlOperationOptions *options,
                              BooleanFilter *filter) {
    const GrlTypeFilter types =
            grl_operation_options_get_type_filter(options);

    if (types == GRL_TYPE_FILTER_NONE)
        return true;

    // FIXME(M3): Error reporting on bad meta data keys.
    if (types & ~kSupportedTypeFilters)
        return false;

    BooleanFilter type_filter;

    if (types & GRL_TYPE_FILTER_AUDIO) {
        type_filter.add_clause(PrefixFilter(schema::kMimeType,
                                            MimeType::kAudioPrefix.str()),
                               BooleanFilter::SHOULD);
        type_filter.add_clause(ValueFilter(schema::kMimeType,
                                           MimeType::kApplicationOgg.str()),
                               BooleanFilter::SHOULD);
    }

    if (types & GRL_TYPE_FILTER_IMAGE) {
        type_filter.add_clause(PrefixFilter(schema::kMimeType,
                                            MimeType::kImagePrefix.str()),
                               BooleanFilter::SHOULD);
    }

    if (types & GRL_TYPE_FILTER_VIDEO) {
        type_filter.add_clause(PrefixFilter(schema::kMimeType,
                                            MimeType::kVideoPrefix.str()),
                               BooleanFilter::SHOULD);
    }

    filter->add_clause(type_filter, BooleanFilter::MUST);

    return true;
}

static bool make_key_value_filters(GrlOperationOptions *options,
                                   BooleanFilter *filter) {
    const Wrapper<GList> key_list =
            take(grl_operation_options_get_key_filter_list(options));

    for (GList *l = key_list.get(); l; l = l->next) {
        const GrlKeyID key = GRLPOINTER_TO_KEYID(l->data);
        const Property property = Property::FromMetadataKey(key);

        // FIXME(M3): Error reporting on bad meta data keys.
        if (not property)
            return false;

        Property::Value value;

        // FIXME(M3): Error reporting on bad meta data keys.
        if (not property.TransformGriloValue
                (grl_operation_options_get_key_filter(options, key), &value))
            return false;

        filter->add_clause(ValueFilter(property, value), BooleanFilter::MUST);
    }

    return true;
}

static bool make_key_range_filters(GrlOperationOptions *options,
                                   BooleanFilter *filter) {
    const Wrapper<GList> key_list =
            take(grl_operation_options_get_key_range_filter_list(options));

    for (GList *l = key_list.get(); l; l = l->next) {
        const GrlKeyID key = GRLPOINTER_TO_KEYID(l->data);
        const Property property = Property::FromMetadataKey(key);

        // FIXME(M3): Error reporting on bad meta data keys.
        if (not property)
            return false;

        GValue *lower_value_ptr;
        GValue *upper_value_ptr;

        grl_operation_options_get_key_range_filter
                (options, key, &lower_value_ptr, &upper_value_ptr);

        Property::Value lower_value, upper_value;

        // FIXME(M3): Error reporting on bad meta data keys.
        if (not property.TransformGriloValue(lower_value_ptr, &lower_value))
            return false;
        if (not property.TransformGriloValue(upper_value_ptr, &upper_value))
            return false;

        filter->add_clause(RangeFilter(property, lower_value, upper_value),
                           BooleanFilter::MUST);
    }

    return true;
}

static bool make_filter(GrlOperationOptions *options, BooleanFilter *filter) {
    return make_type_filters(options, filter)
            && make_key_value_filters(options, filter)
            && make_key_range_filters(options, filter);
}

// Task Handling ///////////////////////////////////////////////////////////////

typedef MediaIndexFacade<MediaIndex>::TaskFunction TaskFunction;
typedef std::function<void(Wrapper<GError> error)> ErrorFunction;

static Wrapper<GError> make_error(int code, const string &message) {
    Wrapper<GError> error;

    g_set_error_literal(error.out_param(),
                        GRL_CORE_ERROR, code, message.c_str());

    return error;
}

static void report_wrapped_error(const ErrorFunction &report_error,
                                 const std::string &error_message) {
    report_error(make_error(GRL_CORE_ERROR_LOAD_PLUGIN_FAILED, error_message));
}

static void push_task(GrlMediaScannerSource *source,
                      unsigned opid, const TaskFunction &task,
                      const ErrorFunction &report_error) {
    const MediaIndexFacade<MediaIndex>::ErrorFunction &on_error =
            std::bind(report_wrapped_error, report_error, std::placeholders::_1);
    const TaskManager::TaskFunction &decorated_task =
            source->priv->task_facade_.bind(task, on_error);

    source->priv->task_manager_.AppendTask(decorated_task, opid);
}

// Browse Operation ////////////////////////////////////////////////////////////

static void browse_report_error(const GrlSourceBrowseSpec *bs,
                                Wrapper<GError> error) {
    Idle::AddOnce(std::bind(bs->callback, bs->source, bs->operation_id,
                       nullptr, 0, bs->user_data, error));
}

static void browse_report_error_message(const GrlSourceBrowseSpec *bs,
                                        const char *message) {
    browse_report_error(bs, make_error(GRL_CORE_ERROR_BROWSE_FAILED, message));
}

static void browse_visit_item(GrlSourceBrowseSpec *bs,
                              const MediaInfo &metadata,
                              int32_t remaining_items) {
    Idle::AddOnce(std::bind(bs->callback, bs->source, bs->operation_id,
                       metadata.make_media(nullptr), remaining_items,
                       bs->user_data, nullptr));
}

static void browse_run(GrlSourceBrowseSpec *bs, MediaIndex *media_index) {
    BooleanFilter filter;

    // TODO(M3): L10N
    if (not make_filter(bs->options, &filter)) {
        browse_report_error_message(bs, "Cannot create filter from options");
        return;
    }

    if (media_index->Query(std::bind(&browse_visit_item, bs, std::placeholders::_1, std::placeholders::_2), filter,
                               grl_operation_options_get_count(bs->options),
                               grl_operation_options_get_skip(bs->options))) {
        Idle::AddOnce(std::bind(bs->callback, bs->source, bs->operation_id,
                           nullptr, 0, bs->user_data, nullptr));
    } else {
        browse_report_error_message(bs, media_index->error_message().c_str());
    }
}

static void browse(GrlSource *source, GrlSourceBrowseSpec *bs) {
    kTrace("{1}: opid={2}") % __func__ % bs->operation_id;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run browse operation
    push_task(media_scanner_source, bs->operation_id,
              std::bind(&browse_run, bs, std::placeholders::_1),
              std::bind(&browse_report_error, bs, std::placeholders::_1));
}

// Resolve Operation ///////////////////////////////////////////////////////////

static void resolve_report_error(const GrlSourceResolveSpec *rs,
                                 Wrapper<GError> error) {
    Idle::AddOnce(std::bind(rs->callback, rs->source, rs->operation_id,
                       rs->media, rs->user_data, error));
}

static void resolve_run(GrlSourceResolveSpec *rs, MediaIndex *media_index) {
    const char *const url = grl_media_get_id(rs->media);
    MediaInfo metadata;

    if (url)
        metadata = media_index->Lookup(ToUnicode(url));

    if (metadata.empty()) {
        const string message =
                (format("Media not found for <{1}>.") % url).str();
        resolve_report_error
                (rs, make_error(GRL_CORE_ERROR_MEDIA_NOT_FOUND, message));
    } else {
        metadata.copy_to_media(rs->keys, rs->media);

        Idle::AddOnce(std::bind(rs->callback, rs->source, rs->operation_id,
                           rs->media, rs->user_data, nullptr));
    }
}

static void resolve(GrlSource *source, GrlSourceResolveSpec *rs) {
    kTrace("{1}: opid={2}") % __func__ % rs->operation_id;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run browse operation
    push_task(media_scanner_source, rs->operation_id,
              std::bind(&resolve_run, rs, std::placeholders::_1),
              std::bind(&resolve_report_error, rs, std::placeholders::_1));
}

// TestMediaFromUri Operation //////////////////////////////////////////////////

static void test_media_from_uri_report_error(const char *uri,
                                             Wrapper<GError> error,
                                             GMutex *mutex, GCond *waiter) {
    const string error_string = to_string(error);
    kWarning("Testing URI failed for {1}: {2}") % uri % error_string;

    g_mutex_lock(mutex);
    g_cond_broadcast(waiter);
    g_mutex_unlock(mutex);
}

static void test_media_from_uri_run(const char *uri, MediaIndex *media_index,
                                    GMutex *mutex, GCond *waiter,
                                    bool *result) {
    g_mutex_lock(mutex);
    *result = uri && media_index->Exists(ToUnicode(uri));
    g_cond_broadcast(waiter);
    g_mutex_unlock(mutex);
}

static gboolean test_media_from_uri(GrlSource *source, const char *uri) {
    kTrace("{1}: uri=<{2}>") % __func__ % uri;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_val_if_fail(media_scanner_source != NULL, false);

    // Run operation
    static GMutex mutex;
    static GCond waiter;
    bool result = false;

    g_mutex_lock(&mutex);

    push_task(media_scanner_source, TaskManager::kInstantly,
              std::bind(&test_media_from_uri_run, uri,
                   std::placeholders::_1,  &mutex, &waiter, &result),
              std::bind(&test_media_from_uri_report_error, uri,
                   std::placeholders::_1, &mutex, &waiter));

    g_cond_wait(&waiter, &mutex);
    g_mutex_unlock(&mutex);

    return result;
}

// MediaFromUri Operation //////////////////////////////////////////////////////

static void media_from_uri_report_error(const GrlSourceMediaFromUriSpec *mfus,
                                        Wrapper<GError> error) {
    Idle::AddOnce(std::bind(mfus->callback, mfus->source, mfus->operation_id,
                       nullptr, mfus->user_data, error));
}

static void media_from_uri_run(GrlSourceMediaFromUriSpec *mfus,
                               MediaIndex *media_index) {
    const wstring url = ToUnicode(mfus->uri);
    const MediaInfo metadata = media_index->Lookup(url);

    if (metadata.empty()) {
        const string message =
                (format("Media not found for <{1}>.") % mfus->uri).str();
        media_from_uri_report_error
                (mfus, make_error(GRL_CORE_ERROR_MEDIA_NOT_FOUND, message));
    } else {
        Idle::AddOnce(std::bind(mfus->callback, mfus->source, mfus->operation_id,
                           metadata.make_media(mfus->keys), mfus->user_data,
                           nullptr));
    }
}

static void media_from_uri(GrlSource *source, GrlSourceMediaFromUriSpec *mfus) {
    kTrace("{1}: opid={2}") % __func__ % mfus->operation_id;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run browse operation
    push_task(media_scanner_source, mfus->operation_id,
              std::bind(&media_from_uri_run, mfus, std::placeholders::_1),
              std::bind(&media_from_uri_report_error, mfus, std::placeholders::_1));
}

// Query Operation /////////////////////////////////////////////////////////////

static void query_report_error(const GrlSourceQuerySpec *qs,
                               Wrapper<GError> error) {
    Idle::AddOnce(std::bind(qs->callback, qs->source, qs->operation_id,
                       nullptr, 0, qs->user_data, error));
}

static void query_report_error_message(const GrlSourceQuerySpec *qs,
                                       const char *message) {
    query_report_error(qs, make_error(GRL_CORE_ERROR_QUERY_FAILED, message));
}

static void query_visit_item(GrlSourceQuerySpec *qs, const MediaInfo &metadata,
                             int32_t remaining_items) {
    const wstring url = metadata.first(schema::kUrl);
    kTrace("{1}: url=<{2}>") % __func__ % url;

    Idle::AddOnce(std::bind(qs->callback, qs->source, qs->operation_id,
                       metadata.make_media(qs->keys), remaining_items,
                       qs->user_data, nullptr));
}

static void query_run(GrlSourceQuerySpec *qs, MediaIndex *media_index) {
    BooleanFilter filter;

    // TODO(M3): L10N
    if (not make_filter(qs->options, &filter)) {
        query_report_error_message(qs, "Cannot create filter from options");
        return;
    }

    if (qs->query) {
        filter.add_clause(QueryStringFilter(ToUnicode(qs->query)),
                          BooleanFilter::MUST);
    }

    if (media_index->Query(std::bind(&query_visit_item, qs, std::placeholders::_1, std::placeholders::_2), filter,
                           grl_operation_options_get_count(qs->options),
                           grl_operation_options_get_skip(qs->options))) {
        Idle::AddOnce(std::bind(qs->callback, qs->source, qs->operation_id,
                           nullptr, 0, qs->user_data, nullptr));
    } else {
        query_report_error_message(qs, media_index->error_message().c_str());
    }
}

static void query(GrlSource *source, GrlSourceQuerySpec *qs) {
    kTrace("{1}: opid={2}") % __func__ % qs->operation_id;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run browse operation
    push_task(media_scanner_source, qs->operation_id,
              std::bind(&query_run, qs, std::placeholders::_1), std::bind(&query_report_error, qs, std::placeholders::_1));
}

// Query Operation /////////////////////////////////////////////////////////////

static void search_report_error(const GrlSourceSearchSpec *ss,
                                Wrapper<GError> error) {
    Idle::AddOnce(std::bind(ss->callback, ss->source, ss->operation_id,
                       nullptr, 0, ss->user_data, error));
}

static void search_report_error_message(const GrlSourceSearchSpec *ss,
                                        const string &message) {
    search_report_error(ss, make_error(GRL_CORE_ERROR_SEARCH_FAILED, message));
}

static void search_visit_item(GrlSourceSearchSpec *ss,
                              const MediaInfo &metadata,
                             int32_t remaining_items) {
    Idle::AddOnce(std::bind(ss->callback, ss->source, ss->operation_id,
                       metadata.make_media(ss->keys), remaining_items,
                       ss->user_data, nullptr));
}

static void search_run(GrlSourceSearchSpec *ss,
                       GrlMediaScannerSearchMethod search_method,
                       MediaIndex *media_index) {
    BooleanFilter filter;

    // TODO(M3): L10N
    if (not make_filter(ss->options, &filter)) {
        search_report_error_message(ss, "Cannot create filter from options");
        return;
    }

    if (ss->text) {
        switch (search_method) {
        case GRL_MEDIA_SCANNER_SEARCH_SUBSTRING:
            filter.add_clause(SubStringFilter(ToUnicode(ss->text)),
                              BooleanFilter::MUST);
            break;

        case GRL_MEDIA_SCANNER_SEARCH_FULL_TEXT:
            filter.add_clause(FullTextFilter(ToUnicode(ss->text)),
                              BooleanFilter::MUST);
            break;
        }
    }

    if (media_index->Query(std::bind(&search_visit_item, ss, std::placeholders::_1, std::placeholders::_2), filter,
                           grl_operation_options_get_count(ss->options),
                           grl_operation_options_get_skip(ss->options))) {
        Idle::AddOnce(std::bind(ss->callback, ss->source, ss->operation_id,
                           nullptr, 0, ss->user_data, nullptr));
    } else {
        search_report_error_message(ss, media_index->error_message().c_str());
    }
}

static void search(GrlSource *source, GrlSourceSearchSpec *ss) {
    kTrace("{1}: opid={2}") % __func__ % ss->operation_id;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run search operation
    push_task(media_scanner_source, ss->operation_id,
              std::bind(&search_run, ss,
                   media_scanner_source->priv->search_method_, std::placeholders::_1),
              std::bind(&search_report_error, ss, std::placeholders::_1));
}

////////////////////////////////////////////////////////////////////////////////

static void store_real_report_result(const GrlSourceStoreSpec *ss,
                                     Wrapper<GList> failed_keys,
                                     Wrapper<GError> error) {
    ss->callback(ss->source, ss->media, failed_keys.get(),
                 ss->user_data, error);
}

static void store_report_result(const GrlSourceStoreSpec *ss,
                                GList *failed_keys,
                                Wrapper<GError> error) {
    Idle::AddOnce(std::bind(store_real_report_result, ss, take(failed_keys), error));
}

static void store_report_error_message(const GrlSourceStoreSpec *ss,
                                       GList *failed_keys,
                                       const string &message) {
    store_report_result(ss, failed_keys,
                        make_error(GRL_CORE_ERROR_STORE_FAILED, message));
}

static void store_run(GrlSourceStoreSpec *ss, MediaIndex *,
                      dbus::MediaScannerProxy *service) {
    const Wrapper<GList> keys = take(grl_data_get_keys(GRL_DATA(ss->media)));
    Wrapper<GList> failed_keys;
    string error_message;

    if (StoreMedia(service, wrap(ss->media), keys,
                   failed_keys.out_param(), &error_message)) {
        store_report_result(ss, nullptr, Wrapper<GError>());
    } else {
        store_report_error_message(ss, failed_keys.get(), error_message);
    }
}

static void store(GrlSource *source, GrlSourceStoreSpec *ss) {
    const string url = safe_string(grl_media_get_url(ss->media));
    kTrace("{1}: url=<{2}>") % __func__ % url;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run store operation
    dbus::MediaScannerProxy *const service =
            &media_scanner_source->priv->service_proxy_;

    push_task(media_scanner_source, TaskManager::kInstantly,
              std::bind(&store_run, ss, std::placeholders::_1, service),
              std::bind(&store_report_result, ss, nullptr, std::placeholders::_1));
}

////////////////////////////////////////////////////////////////////////////////

static void store_metadata_real_report_result
                        (const GrlSourceStoreMetadataSpec *ss,
                         Wrapper<GList> failed_keys, Wrapper<GError> error) {
    ss->callback(ss->source, ss->media, failed_keys.get(),
                 ss->user_data, error);
}

static void store_metadata_report_result
                                (const GrlSourceStoreMetadataSpec *ss,
                                 GList *failed_keys, Wrapper<GError> error) {
    Idle::AddOnce(std::bind(store_metadata_real_report_result,
                       ss, take(failed_keys), error));
}

static void store_metadata_report_error_message
                                (const GrlSourceStoreMetadataSpec *ss,
                                 GList *failed_keys, const string &message) {
    store_metadata_report_result
            (ss, failed_keys, make_error(GRL_CORE_ERROR_STORE_FAILED, message));
}

static void store_metadata_run(GrlSourceStoreMetadataSpec *ss, MediaIndex *,
                               dbus::MediaScannerProxy *service) {
    Wrapper<GList> failed_keys;
    string error_message;

    if (StoreMedia(service, wrap(ss->media), ss->keys,
                   failed_keys.out_param(), &error_message)) {
        store_metadata_report_result(ss, nullptr, Wrapper<GError>());
    } else {
        store_metadata_report_error_message(ss, failed_keys.get(),
                                            error_message);
    }
}

static void store_metadata(GrlSource *source, GrlSourceStoreMetadataSpec *ss) {
    const string url = safe_string(grl_media_get_url(ss->media));
    kTrace("{1}: url=<{2}>") % __func__ % url;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run store operation
    dbus::MediaScannerProxy *const service =
            &media_scanner_source->priv->service_proxy_;

    push_task(media_scanner_source, TaskManager::kInstantly,
              std::bind(&store_metadata_run, ss, std::placeholders::_1, service),
              std::bind(&store_metadata_report_result, ss, nullptr, std::placeholders::_1));
}

////////////////////////////////////////////////////////////////////////////////

static void remove_report_result(const GrlSourceRemoveSpec *rs,
                                 Wrapper<GError> error) {
    Idle::AddOnce(std::bind(rs->callback, rs->source, rs->media,
                       rs->user_data, error));
}

static void remove_report_error_message(const GrlSourceRemoveSpec *rs,
                                        const string &message) {
    remove_report_result(rs, make_error(GRL_CORE_ERROR_REMOVE_FAILED, message));
}

static void remove_run(const GrlSourceRemoveSpec *rs, MediaIndex *,
                       dbus::MediaScannerProxy *service) {
    Wrapper<GError> error;

    if (not service->handle()
            && not service->ConnectAndWait(Wrapper<GCancellable>(),
                                           error.out_param())) {
        const string error_message = to_string(error);
        kWarning("Mediascanner service not available: {1}") % error_message;
        remove_report_error_message(rs, error->message);
        return;
    }

    service->RemoveMediaInfo(rs->media_id, error.out_param());

    if (error) {
        const string error_message = to_string(error);
        kWarning("Remove operation failed: {1}") % error_message;
        remove_report_error_message(rs, error->message);
    } else {
        remove_report_result(rs, error);
    }
}

static void remove(GrlSource *source, GrlSourceRemoveSpec *rs) {
    kTrace("{1}: media={2}") % __func__ % rs->media_id;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != NULL);

    // Run remove operation
    dbus::MediaScannerProxy *const service =
            &media_scanner_source->priv->service_proxy_;

    push_task(media_scanner_source, TaskManager::kInstantly,
              std::bind(&remove_run, rs, std::placeholders::_1, service),
              std::bind(&remove_report_result, rs, std::placeholders::_1));
}

////////////////////////////////////////////////////////////////////////////////

static void notify_report_error_message(Wrapper<GError> error) {
    const string error_message = to_string(error);
    kWarning("While preparing change notification: {1}") % error_message;
}

static string to_string(GrlSourceChangeType change_type) {
    switch (change_type) {
    case GRL_CONTENT_ADDED:
        return "content-added";
    case GRL_CONTENT_CHANGED:
        return "content-changed";
    case GRL_CONTENT_REMOVED:
        return "content-removed";
    }

    std::ostringstream oss;
    oss << "unknown-content-change-" << change_type;
    return oss.str();
}

static void real_notify_changed_media(Wrapper<GrlSource> source,
                                      GrlSourceChangeType change_type,
                                      Wrapper<GPtrArray> changed_medias) {
    const string change_type_name = to_string(change_type);
    kTrace("{1}: {2} notification for {3} media(s)")
            % __func__ % change_type_name % changed_medias->len;

    grl_source_notify_change_list(source.get(),
                                  changed_medias.release(),
                                  change_type, false);
}

static void notify_changed_media(MediaIndex *media_index,
                                 Wrapper<GrlSource> source,
                                 GrlSourceChangeType change_type,
                                 const std::vector<string> &media_urls) {
    const string change_type_name = to_string(change_type);
    kTrace("{1}: {2} notification for {3}")
            % __func__ % change_type_name % media_urls;

    const Wrapper<GPtrArray> changed_medias =
            take(g_ptr_array_sized_new(media_urls.size()));

    for (const std::string &url: media_urls) {
        const MediaInfo metadata = media_index->Lookup(ToUnicode(url));

        // Properties might contain no URL. So we have to attach manually.
        Wrapper<GrlMedia> media = take(metadata.make_media(nullptr, url));
        grl_media_set_id(media.get(), url.c_str());
        g_ptr_array_add(changed_medias.get(), media.release());
    }

    // Wrap data to ensure it survives moving to main thread.
    Idle::AddOnce(std::bind(&real_notify_changed_media,
                              source, change_type, changed_medias),
                  G_PRIORITY_HIGH_IDLE);
}

static bool translate_change_type(dbus::MediaChangeType media_change_type,
                                  GrlSourceChangeType *source_change_type) {
    switch (media_change_type) {
    case dbus::MEDIA_INFO_CREATED:
        *source_change_type = GRL_CONTENT_ADDED;
        return true;

    case dbus::MEDIA_INFO_UPDATED:
        *source_change_type = GRL_CONTENT_CHANGED;
        return true;

    case dbus::MEDIA_INFO_REMOVED:
        *source_change_type = GRL_CONTENT_REMOVED;
        return true;
    }

    return false;
}

static void notify_changed_media_cb(GDBusConnection */*connection*/,
                                    const char */*sender_name*/,
                                    const char */*object_path*/,
                                    const char */*interface_name*/,
                                    const char */*signal_name*/,
                                    GVariant *parameters, void *data) {
    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(data);
    g_return_if_fail(media_scanner_source != NULL);

    typedef dbus::MediaScannerProxy::MediaInfoChangedSignal signal_type;

    const signal_type::args_type::value_type args =
            signal_type::args_type::make_value(parameters);

    GrlSourceChangeType change_type;
    g_return_if_fail(translate_change_type(args.get<0>(), &change_type));

    push_task(media_scanner_source, TaskManager::kInstantly,
              std::bind(&notify_changed_media, std::placeholders::_1,
                          wrap(GRL_SOURCE(media_scanner_source)),
                          change_type, args.get<1>()),
              notify_report_error_message);
}

gboolean notify_change_start(GrlSource *source, GError **error) {
    kTrace(__func__);

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_val_if_fail(media_scanner_source != NULL, false);

    if (media_scanner_source->priv->change_id_) {
        g_set_error_literal(error, GRL_CORE_ERROR,
                            GRL_CORE_ERROR_NOTIFY_CHANGED_FAILED,
                            "Already subsribed to change notifications");
        return false;
    }

    // Unsubscribe to D-Bus signal
    // FIXME(M5): Wrap nicely in <dbusutils.h>
    dbus::MediaScannerProxy *const service =
            &media_scanner_source->priv->service_proxy_;



    if (not service->handle()) {
        Wrapper<GError> dbus_error;

        if (not service->ConnectAndWait(Wrapper<GCancellable>(),
                                        dbus_error.out_param())) {
            g_set_error(error, GRL_CORE_ERROR,
                        GRL_CORE_ERROR_NOTIFY_CHANGED_FAILED,
                        "Mediascanner service not available: %s",
                        dbus_error->message);

            return false;
        }
    }

    media_scanner_source->priv->change_id_ = g_dbus_connection_signal_subscribe
            (service->connection().get(),
             service->service_name().c_str(),
             service->interface_name().c_str(),
             service->media_info_changed.name().c_str(),
             service->object_path().c_str(),
             nullptr, G_DBUS_SIGNAL_FLAGS_NONE,
             notify_changed_media_cb,
             g_object_ref(source),
             g_object_unref);

    return true;
}

gboolean notify_change_stop(GrlSource *source, GError **error) {
    kTrace(__func__);

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_val_if_fail(media_scanner_source != NULL, false);

    if (not media_scanner_source->priv->change_id_) {
        g_set_error_literal(error, GRL_CORE_ERROR,
                            GRL_CORE_ERROR_NOTIFY_CHANGED_FAILED,
                            "Not subsribed to change notifications");
        return false;
    }

    // Unsubscribe from D-Bus signal
    // FIXME(M5): Wrap nicely in <dbusutils.h>
    dbus::MediaScannerProxy *const service =
            &media_scanner_source->priv->service_proxy_;

    const unsigned id = media_scanner_source->priv->change_id_;
    media_scanner_source->priv->change_id_ = 0;

    g_dbus_connection_signal_unsubscribe(service->connection().get(), id);

    return true;
}

////////////////////////////////////////////////////////////////////////////////

static GrlSupportedOps supported_operations(GrlSource *source) {
    kTrace(__func__);

    // Check preconditions.
    g_return_val_if_fail(GRL_IS_SOURCE(source), GRL_OP_NONE);

    // Return supported operations.
    return GRL_OP_RESOLVE
            | GRL_OP_BROWSE
            | GRL_OP_SEARCH
            | GRL_OP_QUERY
            | GRL_OP_STORE
            | GRL_OP_STORE_METADATA
            | GRL_OP_REMOVE
            | GRL_OP_MEDIA_FROM_URI
            | GRL_OP_NOTIFY_CHANGE;
}

static bool collect_keys(const Property &p, GList **const keys) {
    *keys = g_list_prepend(*keys, GRLKEYID_TO_POINTER(p.metadata_key().id()));
    return false;
}

static const GList *supported_keys(GrlSource *source) {
    kTrace(__func__);

    // Check preconditions.
    g_return_val_if_fail(GRL_IS_SOURCE(source), NULL);

    // Collect ids of supported keys.
    GList *keys = NULL;
    Property::VisitAll(std::bind(&collect_keys, std::placeholders::_1, &keys));
    return keys;
}

static const GList *writable_keys(GrlSource *source) {
    kTrace(__func__);

    // Check preconditions.
    g_return_val_if_fail(GRL_IS_SOURCE(source), NULL);

    // FIXME(M3): Report only writable keys.
    return supported_keys(source);
}

static void cancel(GrlSource *source, unsigned opid) {
    kTrace("{1}: opid={2}") % __func__ % opid;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_if_fail(media_scanner_source != nullptr);

    // Cancel the task identified by opid.
    media_scanner_source->priv->task_manager_.CancelByGroupId(opid);
}

static bool collect_property_filters(const Property &property,
                                     GList **key_filter,
                                     GList **key_range_filter) {
    const gpointer key = GRLKEYID_TO_POINTER(property.metadata_key().id());

    *key_range_filter = g_list_prepend(*key_range_filter, key);
    *key_filter = g_list_prepend(*key_filter, key);

    return false;
}

static Wrapper<GrlCaps> make_filter_caps(void) {
    Wrapper<GrlCaps> caps = take(grl_caps_new());

    Wrapper<GList> key_filter;
    Wrapper<GList> key_range_filter;

    Property::VisitAll(std::bind(collect_property_filters,
                            std::placeholders::_1, key_filter.out_param(),
                            key_range_filter.out_param()));

    grl_caps_set_key_filter(caps.get(), key_filter.get());
    grl_caps_set_key_range_filter(caps.get(), key_range_filter.get());
    grl_caps_set_type_filter(caps.get(), kSupportedTypeFilters);

    return caps;
}

static GrlCaps *get_caps(GrlSource *source, GrlSupportedOps ops) {
    kTrace("{1}: ops={2}") % __func__ % ops;

    // Check preconditions.
    GrlMediaScannerSource *const media_scanner_source =
        GRL_MEDIA_SCANNER_SOURCE(source);
    g_return_val_if_fail(media_scanner_source != nullptr, nullptr);

    if (ops & ~(GRL_OP_BROWSE | GRL_OP_SEARCH | GRL_OP_QUERY)) {
        if (media_scanner_source->priv->simple_caps_ == nullptr)
            media_scanner_source->priv->simple_caps_ = take(grl_caps_new());

        return media_scanner_source->priv->simple_caps_.get();
    }

    if (media_scanner_source->priv->filter_caps_ == nullptr)
        media_scanner_source->priv->filter_caps_ = make_filter_caps();

    return media_scanner_source->priv->filter_caps_.get();
}

static void set_property(GObject *object, unsigned property_id,
                         const GValue *value, GParamSpec *pspec) {
    GrlMediaScannerSource *const source = GRL_MEDIA_SCANNER_SOURCE(object);

    switch (property_id) {
    case PROP_INDEX_PATH:
        grl_media_scanner_source_set_index_path(
            source, g_value_get_string(value));
        break;

    case PROP_SEARCH_METHOD:
        grl_media_scanner_source_set_search_method(source, static_cast
                <GrlMediaScannerSearchMethod>(g_value_get_enum(value)));
        break;

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
        break;
    }
}

static void get_property(GObject *object, unsigned property_id,
                         GValue *value, GParamSpec *pspec) {
    GrlMediaScannerSource *const source = GRL_MEDIA_SCANNER_SOURCE(object);

    switch (property_id) {
    case PROP_INDEX_PATH:
        g_value_set_string(
            value, grl_media_scanner_source_get_index_path(source));
        break;

    case PROP_SEARCH_METHOD:
        g_value_set_enum(
            value, grl_media_scanner_source_get_search_method(source));
        break;

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
        break;
    }
}

static void finalize(GObject *object) {
    GrlMediaScannerSource *const source = GRL_MEDIA_SCANNER_SOURCE(object);
    g_return_if_fail(source != nullptr);

    delete source->priv;
    source->priv = NULL;

    G_OBJECT_CLASS(grl_media_scanner_source_parent_class)->finalize(object);
}

static void grl_media_scanner_source_class_init
                                     (GrlMediaScannerSourceClass *klass) {
    GrlSourceClass *const source_class = GRL_SOURCE_CLASS(klass);
    source_class->supported_operations = supported_operations;
    source_class->supported_keys = supported_keys;
    source_class->writable_keys = writable_keys;
    source_class->get_caps = get_caps;

    source_class->resolve = resolve;
    source_class->test_media_from_uri = test_media_from_uri;
    source_class->media_from_uri = media_from_uri;
    source_class->browse = browse;
    source_class->query = query;
    source_class->search = search;
    source_class->store = store;
    source_class->store_metadata = store_metadata;
    source_class->remove = remove;
    source_class->cancel = cancel;
    source_class->notify_change_start = notify_change_start;
    source_class->notify_change_stop = notify_change_stop;

    GObjectClass *const object_class = G_OBJECT_CLASS(klass);
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->finalize = finalize;

    properties[PROP_INDEX_PATH] =
            g_param_spec_string("index-path", "Index Path",
                                "Local file system path of the media index",
                                NULL,
                                G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS);

    properties[PROP_SEARCH_METHOD] =
            g_param_spec_enum("search-method", "Search Method",
                              "Current search method of the source",
                              GRL_TYPE_MEDIA_SCANNER_SEARCH_METHOD,
                              GRL_MEDIA_SCANNER_SEARCH_SUBSTRING,
                              G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS);

    g_object_class_install_properties(object_class, N_PROPERTIES, properties);

    if (not mediascanner::CheckLocaleFacets())
        mediascanner::SetupLocale();
}

static void grl_media_scanner_source_init(GrlMediaScannerSource *source) {
    source->priv = new GrlMediaScannerSourcePrivate;
    source->priv->search_method_ = GRL_MEDIA_SCANNER_SEARCH_FULL_TEXT;
}

} // namespace mediascanner

GType grl_media_scanner_source_get_type() {
    return mediascanner::grl_media_scanner_source_get_type();
}

GrlMediaScannerSource *grl_media_scanner_source_new() {
    return GRL_MEDIA_SCANNER_SOURCE(g_object_new
                (GRL_TYPE_MEDIA_SCANNER_SOURCE,
                 "source-id", GRL_MEDIA_SCANNER_PLUGIN_ID,
                 "source-name", _("Media Scanner"),
                 "source-desc", _("A media source using Media Scanner indices"),
                 NULL));
}

void grl_media_scanner_source_set_index_path(GrlMediaScannerSource *source,
                                             const char *path) {
    g_return_if_fail(GRL_IS_MEDIA_SCANNER_SOURCE(source));
    const std::string safe_path = mediascanner::safe_string(path);

    if (safe_path != source->priv->task_facade_.media_index_path()) {
        source->priv->task_facade_.set_media_index_path(safe_path);
        g_object_notify_by_pspec(G_OBJECT(source), properties[PROP_INDEX_PATH]);
    }
}

const char *grl_media_scanner_source_get_index_path
                                      (GrlMediaScannerSource *source) {
    g_return_val_if_fail(GRL_IS_MEDIA_SCANNER_SOURCE(source), 0);
    return source->priv->task_facade_.media_index_path().string().c_str();
}

void grl_media_scanner_source_set_search_method
                                       (GrlMediaScannerSource *source,
                                        GrlMediaScannerSearchMethod method) {
    g_return_if_fail(GRL_IS_MEDIA_SCANNER_SOURCE(source));

    if (source->priv->search_method_ != method) {
        source->priv->search_method_ = method;
        g_object_notify_by_pspec(G_OBJECT(source),
                                 properties[PROP_SEARCH_METHOD]);
    }
}

GrlMediaScannerSearchMethod grl_media_scanner_source_get_search_method
                                            (GrlMediaScannerSource *source) {
    g_return_val_if_fail(GRL_IS_MEDIA_SCANNER_SOURCE(source),
                         GrlMediaScannerSearchMethod(~0));
    return source->priv->search_method_;
}
