/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_DBUSUTILS_H
#define MEDIASCANNER_DBUSUTILS_H

// Boost C++
#include <boost/bind.hpp>
#include <boost/tuple/tuple.hpp>

// C++ Standard Library
#include <map>
#include <string>
#include <memory>

// Media Scanner Library
#include "mediascanner/dbustypes.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/utilities.h"

namespace mediascanner {
namespace dbus {

////////////////////////////////////////////////////////////////////////////////

using boost::tuples::element;

template<typename InfoType> class Member;

typedef Member<GDBusMethodInfo> MethodInfo;
typedef std::shared_ptr<MethodInfo> MethodInfoPtr;

typedef Member<GDBusPropertyInfo> PropertyInfo;
typedef std::shared_ptr<PropertyInfo> PropertyInfoPtr;

typedef Member<GDBusSignalInfo> SignalInfo;
typedef std::shared_ptr<SignalInfo> SignalInfoPtr;

typedef std::shared_ptr<class InterfaceProxy> InterfaceProxyPtr;
typedef std::shared_ptr<class InterfaceSkeleton> InterfaceSkeletonPtr;
typedef std::shared_ptr<class MethodInvocation> MethodInvocationPtr;

////////////////////////////////////////////////////////////////////////////////

template<typename InfoType>
struct SkeletonTrait;

template<typename InfoType>
class Member {
public:
    /**
     * @brief Type of the GDBus introspection structure for this members.
     */
    typedef InfoType dbus_info_type;

    typedef typename SkeletonTrait<dbus_info_type>::type skeleton_type;

protected:
    Member(const std::string &name, const skeleton_type *skeleton)
        : name_(name)
        , skeleton_(skeleton) {
    }

public:
    virtual ~Member() {
    }

    const std::string& name() const {
        return name_;
    }

    const skeleton_type* skeleton() const {
        return skeleton_;
    }

    virtual Wrapper<dbus_info_type> info() const = 0;

private:
    std::string name_;
    const skeleton_type *const skeleton_;
};

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief Description of an D-Bus interface's method or signal argument.
 */
template<typename T>
class Argument {
public:
    typedef T value_type;
    typedef Type<T> dbus_type;

    explicit Argument(const std::string &name)
        : name_(name) {
    }

    const std::string& name() const {
        return name_;
    }

    const Signature& signature() const {
        return dbus_type::signature();
    }

    static GVariant* make_variant(const value_type &value) {
        return Type<value_type>::make_variant(value);
    }

    Wrapper<GDBusArgInfo> info() const {
        if (info_ == null_ptr) {
            const GDBusArgInfo info = {
                1, g_strdup(name().c_str()),
                g_variant_type_dup_string(signature()),
                null_ptr
            };

            info_ = wrap_static(&info);
        }

        return info_;
    }

private:
    std::string name_;
    mutable Wrapper<GDBusArgInfo> info_;
};

////////////////////////////////////////////////////////////////////////////////

namespace internal {

template<typename T, size_t N>
class ArgumentTail {
    typedef ArgumentTail<T, N - 1> predecessor;

public:
    static void ref(const T *args, GDBusArgInfo **info) {
        info[N - 1] = boost::tuples::get<N - 1>(*args).info().dup();
        predecessor::ref(args, info);
    }

    static void ref(GDBusArgInfo *const *other, GDBusArgInfo **info) {
        info[N - 1] = g_dbus_arg_info_ref(other[N - 1]);
        predecessor::ref(other, info);
    }

    static void unref(GDBusArgInfo **info) {
        g_dbus_arg_info_unref(info[N - 1]);
        predecessor::unref(info);
    }
};

template<typename T>
struct ArgumentTail<T, 0> {
    static void ref(const T *, GDBusArgInfo **) {
    }

    static void ref(GDBusArgInfo *const *, GDBusArgInfo **) {
    }

    static void unref(GDBusArgInfo **/*info*/) {
    }
};

template<typename T>
struct ArgumentTrait {
    typedef Argument<T> type;
};

template<>
struct ArgumentTrait<boost::tuples::null_type> {
    typedef boost::tuples::null_type type;
};

} // namespace internal

////////////////////////////////////////////////////////////////////////////////

template<typename T0 = boost::tuples::null_type,
         typename T1 = boost::tuples::null_type,
         typename T2 = boost::tuples::null_type,
         typename T3 = boost::tuples::null_type,
         typename T4 = boost::tuples::null_type,
         typename T5 = boost::tuples::null_type,
         typename T6 = boost::tuples::null_type,
         typename T7 = boost::tuples::null_type>
class ArgumentList : public boost::tuples::tuple
        <typename internal::ArgumentTrait<T0>::type,
         typename internal::ArgumentTrait<T1>::type,
         typename internal::ArgumentTrait<T2>::type,
         typename internal::ArgumentTrait<T3>::type,
         typename internal::ArgumentTrait<T4>::type,
         typename internal::ArgumentTrait<T5>::type,
         typename internal::ArgumentTrait<T6>::type,
         typename internal::ArgumentTrait<T7>::type> {
public:
    typedef boost::tuples::tuple
            <typename internal::ArgumentTrait<T0>::type,
             typename internal::ArgumentTrait<T1>::type,
             typename internal::ArgumentTrait<T2>::type,
             typename internal::ArgumentTrait<T3>::type,
             typename internal::ArgumentTrait<T4>::type,
             typename internal::ArgumentTrait<T5>::type,
             typename internal::ArgumentTrait<T6>::type,
             typename internal::ArgumentTrait<T7>::type> inherited;

    typedef boost::tuples::tuple<T0, T1, T2, T3, T4, T5, T6, T7> value_type;
    typedef boost::tuples::length<inherited> length;

    typedef T0 value0_type;
    typedef T1 value1_type;
    typedef T2 value2_type;
    typedef T3 value3_type;
    typedef T4 value4_type;
    typedef T5 value5_type;
    typedef T6 value6_type;
    typedef T7 value7_type;

private:
    typedef internal::ArgumentTail<ArgumentList, length::value> tail;
    typedef boost::tuples::null_type null_type;

public:
    ArgumentList()
        : inherited()
        , info_(null_ptr) {
    }

    explicit ArgumentList
            (const typename internal::ArgumentTrait<T0>::type &a0,
             const typename internal::ArgumentTrait<T1>::type &a1 = null_type(),
             const typename internal::ArgumentTrait<T2>::type &a2 = null_type(),
             const typename internal::ArgumentTrait<T3>::type &a3 = null_type(),
             const typename internal::ArgumentTrait<T4>::type &a4 = null_type(),
             const typename internal::ArgumentTrait<T5>::type &a5 = null_type(),
             const typename internal::ArgumentTrait<T6>::type &a6 = null_type(),
             const typename internal::ArgumentTrait<T7>::type &a7 = null_type())
        : inherited(a0, a1, a2, a3, a4, a5, a6, a7)
        , info_(null_ptr) {
    }

    ~ArgumentList() {
        if (info_) {
            tail::unref(info_);
            g_free(info_);
        }
    }

    GDBusArgInfo** info() const {
        if (info_ == null_ptr) {
            GDBusArgInfo **info = g_new(GDBusArgInfo *, length::value + 1);
            info[length::value] = null_ptr;
            tail::ref(this, info);
            info_ = info;
        }

        return info_;
    }

    static GDBusArgInfo** dup(GDBusArgInfo *const *const other) {
        GDBusArgInfo **info = g_new(GDBusArgInfo *, length::value + 1);
        info[length::value] = null_ptr;
        tail::ref(other, info);
        return info;
    }

    static value_type make_value(GVariant *variant) {
        return Type<value_type>::make_value(variant);
    }

    static GVariant* make_variant(const value_type &value) {
        return Type<value_type>::make_variant(value);
    }

    static const Signature& signature() {
        return Type<value_type>::signature();
    }

private:
    mutable GDBusArgInfo **info_;
};

////////////////////////////////////////////////////////////////////////////////

class MethodSkeleton {
public:
    virtual ~MethodSkeleton() {
    }

    virtual void Invoke(MethodInvocation call) const = 0;
};

template<> struct SkeletonTrait<GDBusMethodInfo> {
    typedef MethodSkeleton type;
};

template< typename InputArguments = ArgumentList<>,
          typename OutputArguments = ArgumentList<> >
class Method : public Member<GDBusMethodInfo> {
public:
    typedef InputArguments input_args_type;
    typedef OutputArguments output_args_type;

protected:
    Method(const std::string &name, MethodSkeleton *skeleton,
           const input_args_type &input = input_args_type(),
           const output_args_type &output = output_args_type())
        : Member(name, skeleton)
        , input_(input)
        , output_(output) {
    }

public:
    Wrapper<dbus_info_type> info() const {
        if (not info_) {
            const GDBusMethodInfo static_info = {
                1, g_strdup(name().c_str()),
                input_args_type::dup(input_.info()),
                output_args_type::dup(output_.info()),
                null_ptr
            };

            info_ = wrap_static(&static_info);
        }

        return info_;
    }

private:
    const input_args_type input_;
    const output_args_type output_;
    mutable Wrapper<dbus_info_type> info_;
};

class MethodInvocation {
public:
    explicit MethodInvocation(GDBusMethodInvocation *invocation)
        : invocation_(invocation)
        , origin_(g_thread_self()) {
    }

    bool is_active() const {
        return invocation_;
    }

    uint32_t serial() const {
        return g_dbus_message_get_serial(message());
    }

    std::string sender() const {
        BOOST_ASSERT(is_active());
        return g_dbus_method_invocation_get_sender(invocation_);
    }

    std::string target() const {
        BOOST_ASSERT(is_active());
        return g_dbus_method_invocation_get_object_path(invocation_);
    }

    std::string interface_name() const {
        BOOST_ASSERT(is_active());
        return g_dbus_method_invocation_get_interface_name(invocation_);
    }

    std::string method_name() const {
        BOOST_ASSERT(is_active());
        return g_dbus_method_invocation_get_method_name(invocation_);
    }

    GVariant* parameters() const {
        BOOST_ASSERT(is_active());
        return g_dbus_method_invocation_get_parameters(invocation_);
    }

    void return_error(GQuark domain, int code, const std::string &message);
    void return_dbus_error(const std::string &error_name,
                           const std::string &error_message);
    void return_error(Wrapper<GError> error);

    void emit_signal(const SignalInfo *signal, GVariant *args,
                     GError **error) const;

    GDBusMethodInvocation* dup() const;
    GDBusMethodInvocation* swap(GDBusMethodInvocation *other = null_ptr); // NOLINT:build/include_what_you_use

protected:
    GDBusConnection* connection() const {
        BOOST_ASSERT(is_active());
        return g_dbus_method_invocation_get_connection(invocation_);
    }

    GDBusMessage* message() const {
        BOOST_ASSERT(is_active());
        return g_dbus_method_invocation_get_message(invocation_);
    }

    void assert_called_from_original_thread() const {
        BOOST_ASSERT(g_thread_self() == origin_);
    }

private:
    GDBusMethodInvocation *invocation_;
    GThread *const origin_;
};

template<typename A0, typename A1, typename A2, typename A3,
         typename A4, typename A5, typename A6, typename A7,
         typename R0, typename R1, typename R2, typename R3,
         typename R4, typename R5, typename R6, typename R7,
         typename MethodProxy>
class MethodProxyCall {
};

using boost::tuples::null_type;

template<typename A0, typename MethodProxy>
class MethodProxyCall
        <A0, null_type, null_type, null_type,
         null_type, null_type, null_type, null_type,
         null_type, null_type, null_type, null_type,
         null_type, null_type, null_type, null_type, MethodProxy> {
public:
    void operator()(const A0 &a0, GError **error) const {
        const typename MethodProxy::input_value_type args(a0);
        static_cast<const MethodProxy *>(this)->CallAndWait(args, error);
    }
};

template<typename A0, typename A1, typename MethodProxy>
class MethodProxyCall
        <A0, A1, null_type, null_type,
         null_type, null_type, null_type, null_type,
         null_type, null_type, null_type, null_type,
         null_type, null_type, null_type, null_type, MethodProxy> {
public:
    void operator()(const A0 &a0, const A1 &a1, GError **error) const {
        const typename MethodProxy::input_value_type args(a0, a1);
        static_cast<const MethodProxy *>(this)->CallAndWait(args, error);
    }
};

template<typename A0, typename R0, typename MethodProxy>
class MethodProxyCall
        <A0, null_type, null_type, null_type,
         null_type, null_type, null_type, null_type,
         R0, null_type, null_type, null_type,
         null_type, null_type, null_type, null_type, MethodProxy> {
public:
    R0 operator()(const A0 &a0, GError **error) const {
        const typename MethodProxy::input_value_type args(a0);

        return boost::tuples::get<0>
                (static_cast<const MethodProxy *>(this)->
                 CallAndWait(args, error));
    }
};

template<typename MethodProxy,
         typename InputArguments,
         typename OutputArguments>
struct MethodProxyCallTrait {
    typedef Method<InputArguments, OutputArguments> method_type;
    typedef typename method_type::input_args_type input_args_type;
    typedef typename method_type::output_args_type output_args_type;

    typedef MethodProxyCall
            <typename input_args_type::value0_type,
             typename input_args_type::value1_type,
             typename input_args_type::value2_type,
             typename input_args_type::value3_type,
             typename input_args_type::value4_type,
             typename input_args_type::value5_type,
             typename input_args_type::value6_type,
             typename input_args_type::value7_type,
             typename output_args_type::value0_type,
             typename output_args_type::value1_type,
             typename output_args_type::value2_type,
             typename output_args_type::value3_type,
             typename output_args_type::value4_type,
             typename output_args_type::value5_type,
             typename output_args_type::value6_type,
             typename output_args_type::value7_type,
             MethodProxy> type;
};

template< typename InputArguments = ArgumentList<>,
          typename OutputArguments = ArgumentList<> >
class MethodProxy
        : public Method<InputArguments, OutputArguments>
        , public MethodProxyCallTrait<MethodProxy<InputArguments,
                                                  OutputArguments>,
                                      InputArguments, OutputArguments>::type {
public:
    typedef Method<InputArguments, OutputArguments> method_type;
    typedef typename method_type::input_args_type input_args_type;
    typedef typename input_args_type::value_type input_value_type;
    typedef typename method_type::output_args_type output_args_type;
    typedef typename output_args_type::value_type output_value_type;
    typedef boost::posix_time::time_duration timeout_type;

    MethodProxy(InterfaceProxy *proxy, const std::string &name,
                const input_args_type &input = input_args_type(),
                const output_args_type &output = output_args_type())
        : method_type(name, null_ptr, input, output)
        , proxy_(proxy)
        , flags_(G_DBUS_CALL_FLAGS_NONE)
        , timeout_(boost::date_time::not_a_date_time) {
    }

    output_value_type CallAndWait(const input_value_type &args,
                                  GError **error) const;

    template<typename T>
    MethodProxy& operator[](const T &value) {
        set_attribute(value);
        return *this;
    }

    template<typename T>
    MethodProxy operator[](const T &value) const {
        MethodProxy dup(*this);
        dup.set_attribute(value);
        return dup;
    }

private:
    void set_attribute(Wrapper<GCancellable> cancellable) {
        cancellable_ = cancellable;
    }

    void set_attribute(GDBusCallFlags flags) {
        flags_ = flags;
    }

    void set_attribute(timeout_type timeout) {
        timeout_ = timeout;
    }

    int timeout_ms() const {
        if (timeout_.is_not_a_date_time())
            return -1;

        return timeout_.total_milliseconds();
    }

    InterfaceProxy *const proxy_;
    Wrapper<GCancellable> cancellable_;
    GDBusCallFlags flags_;
    timeout_type timeout_;
};

template< typename InputArguments = ArgumentList<>,
          typename OutputArguments = ArgumentList<> >
class MethodImplementation
        : public Method<InputArguments, OutputArguments>
        , protected MethodSkeleton {
public:
    typedef Method<InputArguments, OutputArguments> method_type;
    typedef typename method_type::input_args_type input_args_type;
    typedef typename input_args_type::value_type input_value_type;
    typedef typename method_type::output_args_type output_args_type;
    typedef typename output_args_type::value_type output_value_type;

    class Invocation : public MethodInvocation {
    public:
        explicit Invocation(GDBusMethodInvocation *other)
            : MethodInvocation(other) {
        }

        explicit Invocation(MethodInvocation *other)
            : MethodInvocation(other->swap()) {
        }

        template<size_t N>
        typename element<N, input_value_type>::type arg() const {
            const Wrapper<GVariant> child_variant =
                    take(g_variant_get_child_value(parameters(), N));

            typedef typename element<N, input_value_type>::type return_type;
            return Type<return_type>::make_value(child_variant.get());
        }

        template<size_t N, typename ErrorDetailType>
        typename element<N, input_value_type>::type arg
                                            (ErrorDetailType *error) const {
            const Wrapper<GVariant> child_variant =
                    take(g_variant_get_child_value(parameters(), N));

            typedef typename element<N, input_value_type>::type return_type;
            return Type<return_type>::make_value(child_variant.get(), error);
        }

        void return_value(const output_value_type &value) {
            GVariant *const variant = output_args_type::make_variant(value);
            g_dbus_method_invocation_return_value(swap(), variant);
        }
    };

    typedef std::shared_ptr<Invocation> InvocationPtr;

protected:
    explicit MethodImplementation
        (InterfaceSkeleton *, const std::string &name,
         const input_args_type &input = input_args_type(),
         const output_args_type &output = output_args_type())
        : method_type(name, this, input, output) {
    }

    void Invoke(MethodInvocation invocation) const {
        Invoke(InvocationPtr(new Invocation(&invocation)));
    }

    virtual void Invoke(InvocationPtr invocation) const = 0;
};

////////////////////////////////////////////////////////////////////////////////

class PropertySkeleton {
public:
    virtual ~PropertySkeleton() {
    }

    virtual bool readable() const = 0;
    virtual bool writable() const = 0;

    /**
     * @brief Retrieves the value of this D-Bus property.
     * @param[in] sender The sender of this D-Bus request.
     * @param[in] target The target of this D-Bus request.
     * @param[out] error Location for storing a possible error.
     * @return The current value of this property.
     */
    virtual GVariant* GetValue(const std::string &sender,
                               const std::string &target,
                               GError **error) const = 0;

    /**
     * @brief Updates the value of this D-Bus property.
     * @param[in] sender The sender of this D-Bus request.
     * @param[in] target The target of this D-Bus request.
     * @param[in] value The new value for this property.
     * @param[out] error Location for storing a possible error.
     * @return @c true on success
     */
    virtual bool SetValue(const std::string &sender, const std::string &target,
                          GVariant *value, GError **error) const = 0;
};

template<> struct SkeletonTrait<GDBusPropertyInfo> {
    typedef PropertySkeleton type;
};

template<typename T, GDBusPropertyInfoFlags Flags>
class Property : public Member<GDBusPropertyInfo> {
public:
    typedef T value_type;
    typedef Type<T> dbus_type;

protected:
    Property(const std::string &name, PropertySkeleton *skeleton)
        : Member(name, skeleton) {
    }

public:
    /**
     * @brief Predicate if this property can be read.
     * @see GetValue(), @flags()
     */
    static bool readable() {
        return Flags & G_DBUS_PROPERTY_INFO_FLAGS_READABLE;
    }

    /**
     * @brief Predicate if this property can be written.
     * @see SetValue(), @flags()
     */
    static bool writable() {
        return Flags & G_DBUS_PROPERTY_INFO_FLAGS_WRITABLE;
    }

    Wrapper<dbus_info_type> info() const {
        if (not info_) {
            const GDBusPropertyInfo static_info = {
                1, g_strdup(name().c_str()),
                g_variant_type_dup_string(dbus_type::signature()),
                Flags, null_ptr
            };

            info_ = wrap_static(&static_info);
        }

        return info_;
    }

private:
    mutable Wrapper<dbus_info_type> info_;
};

template<typename T, GDBusPropertyInfoFlags Flags>
class PropertyProxy
        : public Property<T, Flags> {
public:
    typedef Property<T, Flags> property_type;

    explicit PropertyProxy(const std::string &name)
        : property_type(name, null_ptr) {
    }
};

template<typename T>
class ReadOnlyPropertyProxy
        : public PropertyProxy<T, G_DBUS_PROPERTY_INFO_FLAGS_READABLE> {
public:
    typedef
        PropertyProxy<T, G_DBUS_PROPERTY_INFO_FLAGS_READABLE>
        inherited;

    explicit ReadOnlyPropertyProxy(const std::string &name)
        : inherited(name) {
    }

    T ReadCachedValue(const InterfaceProxy *proxy) const;
    bool ReadValue(const InterfaceProxy *proxy, T *value) const;
};

template<typename T, GDBusPropertyInfoFlags Flags>
class PropertyImplementation
        : public Property<T, Flags>
        , protected PropertySkeleton {
public:
    typedef Property<T, Flags> property_type;

    explicit PropertyImplementation(const std::string &name)
        : property_type(name, this) {
    }

    bool readable() const {
        return property_type::readable();
    }

    bool writable() const {
        return property_type::writable();
    }
};

template<typename T>
class ReadOnlyPropertyImplementation
        : public PropertyImplementation
                <T, G_DBUS_PROPERTY_INFO_FLAGS_READABLE> {
public:
    typedef
        PropertyImplementation<T, G_DBUS_PROPERTY_INFO_FLAGS_READABLE>
        inherited;

    explicit ReadOnlyPropertyImplementation(const std::string &name)
        : inherited(name) {
    }

    bool SetValue(const std::string &, const std::string &,
                  GVariant *, GError **error) const {
        g_set_error(error, G_DBUS_ERROR, G_DBUS_ERROR_INVALID_ARGS,
                    "Property \"%s\" is for reading only",
                    inherited::name().c_str());
        return false;
    }
};

////////////////////////////////////////////////////////////////////////////////

class SignalSkeleton {
};

template<> struct SkeletonTrait<GDBusSignalInfo> {
    typedef SignalSkeleton type;
};

template< typename Arguments = ArgumentList<> >
class Signal : public Member<GDBusSignalInfo> {
public:
    typedef Arguments args_type;

protected:
    Signal(const std::string &name, SignalSkeleton *skeleton,
           const args_type &args = args_type())
        : Member(name, skeleton)
        , args_(args) {
    }

protected:
    Wrapper<dbus_info_type> info() const {
        if (not info_) {
            const GDBusSignalInfo static_info = {
                1, g_strdup(name().c_str()),
                args_type::dup(args_.info()),
                null_ptr
            };

            info_ = wrap_static(&static_info);
        }

        return info_;
    }

private:
    const args_type args_;
    mutable Wrapper<dbus_info_type> info_;
};

template< typename Arguments = ArgumentList<> >
class SignalProxy
        : public Signal<Arguments> {
public:
    typedef Signal<Arguments> signal_type;
    typedef typename signal_type::args_type args_type;
    typedef typename args_type::value_type value_type;

    explicit SignalProxy(const std::string &name,
                         const args_type &args = args_type())
        : Signal<Arguments>(name, null_ptr, args) {
    }
};

template< typename Arguments = ArgumentList<> >
class SignalImplementation
        : public Signal<Arguments>
        , protected SignalSkeleton {
public:
    typedef Signal<Arguments> signal_type;
    typedef typename signal_type::args_type args_type;
    typedef typename args_type::value_type value_type;

    explicit SignalImplementation(const std::string &name,
                                  const args_type &args = args_type())
        : Signal<Arguments>(name, this, args) {
    }

    void emit_result(const value_type &args,
                     MethodInvocationPtr invocation) const {
        // FIXME(M5): Handle errors on signal emission
        invocation->emit_signal(this, args_type::make_variant(args), null_ptr);
    }

    void emit_result_on_idle(const value_type &args,
                             MethodInvocationPtr invocation) const {
        Idle::AddOnce(boost::bind(&SignalImplementation::emit_result,
                                  this, args, invocation));
    }

    void emit_signal(const value_type &args,
                     const std::string &object_path,
                     const std::string &interface_name,
                     Wrapper<GDBusConnection> connection) const {
        // FIXME(M5): Handle errors on signal emission
        // FIXME(M5): Find better way to pass object_path and interface_name.
        Wrapper<GError> error;

        if (not g_dbus_connection_emit_signal(connection.get(), null_ptr,
                                              object_path.c_str(),
                                              interface_name.c_str(),
                                              Signal<Arguments>::name().c_str(),
                                              args_type::make_variant(args),
                                              error.out_param()))
            std::cerr << to_string(error)<< std::endl;
    }

    void emit_signal_on_idle(const value_type &args,
                             const std::string &object_path,
                             const std::string &interface_name,
                             Wrapper<GDBusConnection> connection) const {
        Idle::AddOnce(boost::bind(&SignalImplementation::emit_signal,
                                  this, args, object_path,
                                  interface_name, connection));
    }
};

////////////////////////////////////////////////////////////////////////////////

class InterfaceInfo {
protected:
    explicit InterfaceInfo(const std::string &name)
        : name_(name) {
    }

public:
    const std::string& name() const {
        return name_;
    }

    Wrapper<GDBusInterfaceInfo> info() const;

protected:
    template<typename T, typename A0>
    std::shared_ptr<T> register_method(const A0 &a0) {
        const std::shared_ptr<T> method(new T(a0));
        add_method(method);
        return method;
    }

    template<typename T, typename A0>
    std::shared_ptr<T> register_property(const A0 &a0) {
        const std::shared_ptr<T> property(new T(a0));
        add_property(property);
        return property;
    }

    template<typename T>
    std::shared_ptr<T> register_signal() {
        const std::shared_ptr<T> signal(new T);
        add_signal(signal);
        return signal;
    }

    void add_method(MethodInfoPtr method);
    void add_property(PropertyInfoPtr property);
    void add_signal(SignalInfoPtr signal);

    MethodInfoPtr find_method(const std::string &name) const;
    PropertyInfoPtr find_property(const std::string &name) const;
    SignalInfoPtr find_signal(const std::string &name) const;

private:
    typedef std::map<std::string, MethodInfoPtr> MethodInfoMap;
    typedef std::map<std::string, PropertyInfoPtr> PropertyInfoMap;
    typedef std::map<std::string, SignalInfoPtr> SignalInfoMap;

    std::string name_;
    MethodInfoMap methods_;
    PropertyInfoMap properties_;
    SignalInfoMap signals_;

    mutable Wrapper<GDBusInterfaceInfo> info_;
};

////////////////////////////////////////////////////////////////////////////////

class InterfaceProxy : public InterfaceInfo {
public:
    explicit InterfaceProxy(const std::string &name)
        : InterfaceInfo(name) {
    }
    virtual ~InterfaceProxy() {}

    bool ConnectAndWait(Wrapper<GDBusConnection> connection,
                        const std::string &service_name,
                        const std::string &object_path,
                        Wrapper<GCancellable> cancellable,
                        GError **error);

    bool ConnectAndWait(Wrapper<GDBusConnection> connection,
                        const std::string &service_name,
                        const std::string &object_path,
                        GError **error) {
        return ConnectAndWait(connection, service_name, object_path,
                              Wrapper<GCancellable>(), error);
    }

    void connect(GBusType bus_type,
                 const std::string &service_name,
                 const std::string &object_path,
                 Wrapper<GCancellable> cancellable = Wrapper<GCancellable>());

    void connect(Wrapper<GDBusConnection> connection,
                 const std::string &service_name,
                 const std::string &object_path,
                 Wrapper<GCancellable> cancellable = Wrapper<GCancellable>());

    Wrapper<GVariant> ReadPropertyValue(const std::string &property_name) const;

    Wrapper<GDBusProxy> handle() const {
        return handle_;
    }

    Wrapper<GDBusConnection> connection() const;
    std::string interface_name() const;
    std::string service_name() const;
    std::string object_path() const;

protected:
    virtual void connected();
    virtual void connect_failed(Wrapper<GError> error) const;

private:
    static void on_connect_for_bus(GObject *object,
                                   GAsyncResult *result, void *data);
    static void on_connect(GObject *object, GAsyncResult *result, void *data);

    Wrapper<GDBusProxy> handle_;
};

////////////////////////////////////////////////////////////////////////////////

class InterfaceSkeleton : public InterfaceInfo {
public:
    explicit InterfaceSkeleton(const std::string &name)
        : InterfaceInfo(name) {
    }

    const MethodSkeleton *find_method_skeleton(const std::string &name) const;
    const PropertySkeleton* find_property_skeleton
                                            (const std::string &name) const;

    void InvokeMethod(MethodInvocation invocation) const;

    GVariant* GetProperty(const std::string &sender,
                          const std::string &target,
                          const std::string &name,
                          GError **error) const;

    bool SetProperty(const std::string &sender,
                     const std::string &target,
                     const std::string &name,
                     GVariant *value,
                     GError **error) const;
};

////////////////////////////////////////////////////////////////////////////////

template< typename InterfaceType,
          typename InputArguments = ArgumentList<>,
          typename OutputArguments = ArgumentList<> >
struct MethodTrait;

template<typename InputArguments,
         typename OutputArguments>
struct MethodTrait<InterfaceProxy, InputArguments, OutputArguments> {
    typedef MethodProxy<InputArguments, OutputArguments> type;
};

template<typename InputArguments,
         typename OutputArguments>
struct MethodTrait<InterfaceSkeleton, InputArguments, OutputArguments> {
    typedef MethodImplementation<InputArguments, OutputArguments> type;
};

// TODO(M5): Maybe also use for PropertyInfo class?
enum PropertyAccess {
    ReadOnly = 1, WriteOnly = 2, ReadWrite = 3
};

template<typename InterfaceType,
         PropertyAccess AccessMode,
         typename ValueType>
struct PropertyTrait;

template<typename ValueType>
struct PropertyTrait<InterfaceProxy, ReadOnly, ValueType> {
    typedef ReadOnlyPropertyProxy<ValueType> type;
};

template<typename ValueType>
struct PropertyTrait<InterfaceSkeleton, ReadOnly, ValueType> {
    typedef ReadOnlyPropertyImplementation<ValueType> type;
};

template< typename InterfaceType,
          typename Arguments = ArgumentList<> >
struct SignalTrait;

template<typename Arguments>
struct SignalTrait<InterfaceProxy, Arguments> {
    typedef SignalProxy<Arguments> type;
};

template<typename Arguments>
struct SignalTrait<InterfaceSkeleton, Arguments> {
    typedef SignalImplementation<Arguments> type;
};

////////////////////////////////////////////////////////////////////////////////

class Service {
public:
    Service();
    virtual ~Service();

    Wrapper<GDBusConnection> connection() const;

    void Connect(const std::string &service_name);
    void Disconnect();

    void RegisterObject(const std::string &path,
                        InterfaceSkeletonPtr skeleton);

    InterfaceSkeletonPtr find_object(const std::string &name) const;

protected:
    virtual void Connected();
    virtual void NameAcquired();
    virtual void NameLost();

    virtual void InvokeMethod(GDBusMethodInvocation *call);

    virtual GVariant* GetProperty(const std::string &sender,
                                  const std::string &target,
                                  const std::string &interface,
                                  const std::string &property,
                                  GError **error);

    virtual bool SetProperty(const std::string &sender,
                             const std::string &target,
                             const std::string &interface,
                             const std::string &property,
                             GVariant *value, GError **error);

private:
    static void OnDBusConnected(GDBusConnection *connection,
                                const char *name, void *data);
    static void OnDBusNameAcquired(GDBusConnection *connection,
                                   const char *name, void *data);
    static void OnDBusNameLost(GDBusConnection *connection,
                               const char *name, void *data);

    static void OnMethodCall(GDBusConnection *, const char *sender,
                             const char *target, const char *interface,
                             const char *method, GVariant *parameters,
                             GDBusMethodInvocation *call, void *data);
    static GVariant* OnGetProperty(GDBusConnection *, const char *sender,
                                   const char *target, const char *interface,
                                   const char *property, GError **error,
                                   void *data);
    static gboolean OnSetProperty(GDBusConnection *, const char *sender,
                                  const char *target, const char *interface,
                                  const char *property, GVariant *value,
                                  GError **error, void *data);

private:
    unsigned name_id_;
    Wrapper<GDBusConnection> connection_;

    struct ObjectInfo {
        ObjectInfo(unsigned object_id, InterfaceSkeletonPtr skeleton)
            : object_id(object_id)
            , skeleton(skeleton) {
        }

        unsigned object_id;
        InterfaceSkeletonPtr skeleton;
    };

    typedef std::map<std::string, ObjectInfo> ObjectMap;
    ObjectMap objects_;
};

////////////////////////////////////////////////////////////////////////////////

template< typename In, typename Out>
inline typename MethodProxy<In, Out>::output_value_type
MethodProxy<In, Out>::CallAndWait(const input_value_type &args,
                                  GError **error) const {
    const Wrapper<GVariant> result = take
            (g_dbus_proxy_call_sync(proxy_->handle().get(),
                                    method_type::name().c_str(),
                                    input_args_type::make_variant(args),
                                    flags_, timeout_ms(), cancellable_.get(),
                                    error));

    return output_args_type::make_value(result.get());
}

template<typename T>
inline T ReadOnlyPropertyProxy<T>::ReadCachedValue
                                        (const InterfaceProxy *proxy) const {
    return Type<T>::make_value(g_dbus_proxy_get_cached_property
            (proxy->handle().get(), PropertyInfo::name().c_str()));
}

template<typename T>
inline bool ReadOnlyPropertyProxy<T>::ReadValue(const InterfaceProxy *proxy,
                                                T *value) const {
    const Wrapper<GVariant> boxed_value =
            proxy->ReadPropertyValue(PropertyInfo::name());

    if (not boxed_value)
        return false;

    const Wrapper<GVariant> unboxed_value =
            take(g_variant_get_variant(boxed_value.get()));

    *value = Type<T>::make_value(unboxed_value.get());

    return true;
}

std::string ConnectionName(Wrapper<GDBusConnection> connection);

} // namespace dbus
} // namespace mediascanner

#endif // MEDIASCANNER_DBUSUTILS_H
