/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_PROPERTY_H
#define MEDIASCANNER_PROPERTY_H

// Boost C++
#include <boost/date_time/posix_time/ptime.hpp>
#include <boost/function.hpp>
#include <boost/rational.hpp>
#include <boost/variant.hpp>

// C++ Standard Library
#include <map>
#include <set>
#include <string>
#include <utility>
#include <vector>
#include <memory>

// Media Scanner Library
#include "mediascanner/declarations.h"

namespace mediascanner {

enum FullTextSearchMode {
    DisableFullTextSearch,
    EnableFullTextSearch
};

/// The relational data type we support.
///
/// Fractions are used for instance to preciscely describe
/// a photo's exposure time.
typedef boost::rational<uint32_t> Fraction;

/// The date-time type we support.
typedef boost::posix_time::ptime DateTime;

/// The string type we support.
///
/// We use std::wstring instead of std::string to match Lucene++.
typedef std::wstring String;

/// Definition of a property value.
///
/// This class provides all the information to descibe data mappings between
/// all the involved subsystems such as [Lucene++], [Grilo] and [GStreamer].
/// It also provides facilities to convert property values among those systems.
///
/// Properties cannot be instantiated directly. Instead refer the constants
/// declared in the mediascanner::schema namespace.
///
/// [Lucene++]:  https://github.com/luceneplusplus/LucenePlusPlus
/// [Grilo]:     https://live.gnome.org/Grilo/
/// [GStreamer]: http://gstreamer.freedesktop.org/
class Property {
protected:
    /// Internal fields of a Property.
    class Private;

    /// Shared pointer to internal property fields.
    typedef std::shared_ptr<Private> PrivatePtr;

public:
    typedef boost::variant<boost::blank, bool, int32_t, uint64_t,
                           double, Fraction, DateTime, String> ValueBase;

    /// Container class for all supported property values.
    class Value : public ValueBase {
    public:
        Value()
            : ValueBase() {
        }

        template<typename T> Value(const T &v)
            : ValueBase(v) {
        }

        bool operator<(const Value &other) const {
            return ValueBase::operator<(static_cast<const ValueBase &>(other));
        }

        bool operator==(const Value &other) const {
            return ValueBase::operator==(static_cast<const ValueBase &>(other));
        }
    };

    class Category {
    protected:
        explicit Category(unsigned id)
            : id_(id) {
        }

    public:
        bool operator==(const Category &other) const {
            return id_ == other.id_;
        }

        bool operator<(const Category &other) const {
            return id_ < other.id_;
        }

        bool is_a(const Category &base) const {
            return (id_ & base.id_) == base.id_;
        }

        static const Category Generic;
        static const Category File;
        static const Category Media;
        static const Category Music;
        static const Category Image;
        static const Category Photo;
        static const Category Movie;

    private:
        unsigned id_;
    };

    /// A property value bound to an actual property.
    typedef std::pair<Property, Property::Value> BoundValue;

    /// Maps properties to their actual values.
    typedef std::map<Property, Value> ValueMap;

    /// A set of distinct properties.
    typedef std::set<Property> Set;

    /// Collection of Lucene++ fields.
    typedef Lucene::Collection<Lucene::FieldablePtr> LuceneFields;

    /// Boundaries of a range query.
    enum Boundary { Inclusive, Exclusive };

    /// How to resolve merge conflicts for this property.
    enum MergeStrategy { MergeAppend, MergeReplace, MergePreserve };

public:
    class MetadataKey {
    public:
        /// Functions of this kind are used to register custom metadata keys.
        typedef boost::function<GParamSpec *()> SpecFunction;

        /// Grilo metadata key for properties with pending custom metadata keys.
        ///
        /// Custom metadata keys need lazy initialization since they rely on the
        /// GObject type system which must be initialized with g_type_init(), or
        /// any other function such as grl_init() that implicitly calls this
        /// function.
        static const GrlKeyID kPending;

        /// Invalid Grilo metadata key.
        static const GrlKeyID kInvalid;

        MetadataKey(GrlKeyID id) // NOLINT: runtime/explicit
            : id_(id) {
        }

        explicit MetadataKey(const SpecFunction &make_spec)
            : id_(kPending)
            , make_spec_(make_spec) {
        }

        GrlKeyID id() const {
            return id_;
        }

        bool is_custom() const {
            return not make_spec_.empty();
        }

        std::string name() const;

        /// A human-readable description of this property.
        /// This is used only to generate documentation.
        std::string description() const;

        /// The type of the data stored in this property,
        /// such as G_TYPE_STRING, G_TYPE_UINT, etc.
        /// This is used only to generate documentation.
        GType gtype() const;

        bool RegisterCustomKey();

        const GList *relation() const;

    private:
        /// Numeric ID of the metadata key.
        ///
        /// Initialized with kPendingMetadataKey for custom metadata keys.
        GrlKeyID id_;

        /// Function for registering custom metadata keys.
        SpecFunction make_spec_;
    };

public:
    /// Functions of this kind are passed to the VisitAll() method.
    typedef boost::function<bool(const Property &property)> PropertyVisitor;

    /// Functions of this kind are merge the discoverer's stream information.
    typedef boost::function<bool(GstDiscovererInfo *info,
                                 GstDiscovererStreamInfo *stream,
                                 ValueMap *item)> StreamInfoFunction;

public:
    /// Constructs an unidentified property.
    ///
    /// Such properties are needed to support uninitialized variables and
    /// to support various container operations.
    Property() {
    }

    /// Constructs a copy of the other property.
    ///
    /// @param[in] other - the copied property
    Property(const Property &other)
        : d(other.d) {
    }

    ~Property();

protected:
    /// Constructs a new property instance.
    ///
    /// This constructor is used create specializations of the Property class.
    ///
    /// @param[in] impl - the private fields of the new property
    explicit Property(PrivatePtr impl);

public:
    /// Name of the Lucene++ field.
    String field_name() const;

    /// Id of the Grilo metadata key.
    const MetadataKey &metadata_key() const;

    /// Category of the property.
    Category category() const;

    /// Origins of this property.
    std::set<std::string> origins() const;

    /// Checks if the property is undefined (null).
    bool is_null() const {
        return d == 0;
    }

    /// Checks if the property permits full text searches.
    bool supports_full_text_search() const;

    /// How to resolve merge conflicts for this property.
    MergeStrategy merge_strategy() const;

public:
    /// Extracts property values from discoverer stream information.
    ///
    /// @param[in] info - the discoverer info to investigate
    /// @param[in] stream - the stream info to investigate
    /// @param[in,out] properties - the item to update
    /// @returns true if the property value got updated
    bool MergeStreamInfo(GstDiscovererInfo *media,
                         GstDiscovererStreamInfo *stream,
                         ValueMap *properties) const;

    /// Turns a property value into Lucene++ fields.
    ///
    /// @param[in] value - the property values to convert
    LuceneFields MakeFields(const Value &value) const;

    /// Extracts property values of the Lucene++ fields.
    ///
    /// @param[in] fields - the Lucene++ fields to convert
    Value TransformFields(const LuceneFields &fields) const;
    Value TransformSingleField(Lucene::FieldablePtr field) const;

    /// Constructs a Lucene term query for this property and value.
    ///
    /// @param[in] value - the value to search for
    Lucene::QueryPtr MakeTermQuery(const Value &value) const;

    /// Constructs a Lucene term query for this property and value.
    ///
    /// @param[in] lower_value - the lower value to search for
    /// @param[in] lower_boundary - boundery to use for lower_value
    /// @param[in] upper_value - the upper value to search for
    /// @param[in] upper_boundary - boundery to use for upper_value
    Lucene::QueryPtr MakeRangeQuery(const Value &lower_value,
                                    Boundary lower_boundary,
                                    const Value &upper_value,
                                    Boundary upper_boundary) const;

    /// Constructs a Lucene term query for this property and value.
    ///
    /// The constructed query searchs for lower_value <= x < upper_value.
    ///
    /// @param[in] lower_value - the lower value to search for
    /// @param[in] upper_value - the upper value to search for
    Lucene::QueryPtr MakeRangeQuery(const Value &lower_value,
                                    const Value &upper_value) {
        return MakeRangeQuery(lower_value, Inclusive, upper_value, Exclusive);
    }

    /// Converts a tag value into a canonical property value.
    ///
    /// @param[in] input - the tag value to transform
    /// @param[in/out] output - location for the transformed value
    /// @returns true on succcess
    template<typename T>
    bool TransformTagValue(const GValue *input, Value *output) const;

    /// Converts a tag value into a canonical property value.
    ///
    /// @param[in] input - the tag value to transform
    /// @param[in/out] output - location for the transformed value
    /// @returns true on succcess
    bool TransformGriloValue(const GValue *input, Value *output) const;

    bool TransformDBusVariant(GVariant *input, Value *output) const;

    /// Turns a property value into a Grilo value.
    ///
    /// @param[in] value - the property values to convert
    GValue* MakeGriloValue(const Value &value) const;

    /// Finds a property by its Lucene++ field name.
    ///
    /// @param[in] name - the field name by which to search
    static Property FromFieldName(const String &name);

    /// Finds a property by its Grilo metadata key.
    ///
    /// @param[in] key - the metadata key ID to search for.
    static Property FromMetadataKey(GrlKeyID key);

    /// Visits all property declarations known to the media scanner.
    ///
    /// @param[in] visit - a function that is called for each property
    static void VisitAll(const PropertyVisitor &visit);

public:
    /// Checks if two properties are the same.
    bool operator==(const Property &other) const {
        return d == other.d;
    }

    /// Checks if two properties are the different.
    bool operator!=(const Property &other) const {
        return d != other.d;
    }

    /// Safe boolean cast. The more natural operator bool()
    /// is problematic because it leads to unwanted implicit casts.
    operator const void*() const {
        return d ? reinterpret_cast<const void *>(1) : 0;
    }

    /// Define sort order for usage in std::map.
    bool operator<(const Property &other) const {
        return d < other.d;
    }

protected:
    static MetadataKey define_boolean(const char *name,
                                      const char *nick, const char *blurb,
                                      bool default_value);
    static MetadataKey define_datetime(const char *name,
                                       const char *nick, const char *blurb);
    static MetadataKey define_string(const char *name,
                                     const char *nick, const char *blurb,
                                     const char *default_value = 0);

    static bool merge_nothing(const GstDiscovererInfo *,
                              const GstDiscovererStreamInfo *, ValueMap *) {
        return false;
    }

    static bool MergeAny(GstDiscovererInfo *media,
                         GstDiscovererStreamInfo *stream,
                         const StreamInfoFunction &merge_first,
                         const StreamInfoFunction &merge_second,
                         ValueMap *item);

    StreamInfoFunction bind_any(const StreamInfoFunction &first,
                                const StreamInfoFunction &second) const;

    template<typename ValueType>
    bool MergeAttribute(GstDiscovererInfo *media,
                        GstDiscovererStreamInfo *stream,
                        ValueType (*get_attribute)(const GstDiscovererInfo *),
                        ValueMap *item) const;

    template<typename ValueType, typename InfoType>
    bool MergeAttribute(GstDiscovererInfo *media,
                        GstDiscovererStreamInfo *stream,
                        ValueType (*get_attribute)(const InfoType *),
                        ValueMap *item) const;

    template<typename ValueType>
    bool MergeTag(GstDiscovererInfo *media,
                  GstDiscovererStreamInfo *stream,
                  const char *tag_name, ValueMap *item) const;

    template<typename ValueType>
    StreamInfoFunction bind_attr(ValueType (*)(const GstDiscovererInfo *));

    template<typename ValueType, typename InfoType>
    StreamInfoFunction bind_attr(ValueType (*)(const InfoType *));

    template<typename ValueType>
    StreamInfoFunction bind_tag(const char *tag_name) const;

private:
    typedef std::map<String, PrivatePtr> FieldNameCache;
    typedef std::map<GrlKeyID, PrivatePtr> MetadataKeyCache;

    /// Private fields of a property.
    const PrivatePtr d;
    /// Properties by their field name.
    static FieldNameCache field_name_cache_;
    /// Properties by their metadata key.
    static MetadataKeyCache metadata_key_cache_;
};

/// This class adds some utilities for property implementations.
template<typename PropertyType, typename ValueType>
class GenericProperty : public Property {
public:
    typedef ValueType value_type;

    BoundValue bind_value(const ValueType &value) const {
        return BoundValue(*this, Value(value));
    }

protected:
    class Private;

    typedef value_type (*MediaInfoGetter)(const GstDiscovererInfo *);
    typedef value_type (*StreamInfoGetter)(const GstDiscovererStreamInfo *);
    typedef value_type (*AudioInfoGetter)(const GstDiscovererAudioInfo *);
    typedef value_type (*VideoInfoGetter)(const GstDiscovererVideoInfo *);

    explicit GenericProperty(Private *impl)
        : Property(PrivatePtr(impl)) {
    }

    StreamInfoFunction bind_attr(MediaInfoGetter get_attribute) {
        return Property::bind_attr(get_attribute);
    }

    StreamInfoFunction bind_attr(StreamInfoGetter get_attribute) {
        return Property::bind_attr(get_attribute);
    }

    StreamInfoFunction bind_attr(AudioInfoGetter get_attribute) {
        return Property::bind_attr(get_attribute);
    }

    StreamInfoFunction bind_attr(VideoInfoGetter get_attribute) {
        return Property::bind_attr(get_attribute);
    }

    StreamInfoFunction bind_tag(const char *tag_name) const {
        return Property::bind_tag<value_type>(tag_name);
    }
};

class DateTimeProperty
        : public GenericProperty<DateTimeProperty, DateTime> {
protected:
    typedef GenericProperty<DateTimeProperty, DateTime> inherited;

    friend class Private;
    class Private;

    // TODO(M3): merge with numeric property?
    DateTimeProperty(const String &field_name,
                     const MetadataKey &metadata_key,
                     Property::Category category,
                     MergeStrategy merge_strategy,
                     const StreamInfoFunction &stream_info);

    explicit DateTimeProperty(Private *impl);

    static MetadataKey define(const char *name,
                              const char *nick, const char *blurb) {
        return define_datetime(name, nick, blurb);
    }
};

template<typename T>
class NumericProperty
        : public GenericProperty<NumericProperty<T>, T> {
protected:
    typedef GenericProperty<NumericProperty<T>, T> inherited;

    // TODO(M3): Add explicit typedefs for all the numeric properties?
    // TODO(M5): Introduce Private class to avoid C&P in impl.
    NumericProperty(const String &field_name,
                    const Property::MetadataKey &metadata_key,
                    Property::Category category,
                    Property::MergeStrategy merge_strategy,
                    const Property::StreamInfoFunction &stream_info);

    explicit NumericProperty(typename inherited::Private *impl);

    static Property::MetadataKey define(const char *name,
                                        const char *nick, const char *blurb,
                                        T default_value = T());

    static Property::MetadataKey define(const char *name,
                                        const char *nick, const char *blurb,
                                        T minimum, T maximum,
                                        T default_value = T());
};

template<FullTextSearchMode fts>
class GenericStringProperty
        : public GenericProperty<GenericStringProperty<fts>, String> {
protected:
    typedef GenericProperty<GenericStringProperty<fts>, String> inherited;

    GenericStringProperty(const String &field_name,
                          const Property::MetadataKey &metadata_key,
                          Property::Category category,
                          Property::MergeStrategy merge_strategy,
                          const Property::StreamInfoFunction &stream_info);

    explicit GenericStringProperty(typename inherited::Private *impl);

    static Property::MetadataKey define(const char *name,
                                        const char *nick, const char *blurb,
                                        const char *default_value = 0) {
        return Property::define_string(name, nick, blurb, default_value);
    }
};

typedef GenericStringProperty<DisableFullTextSearch> StringProperty;
typedef GenericStringProperty<EnableFullTextSearch> TextProperty;

} // namespace mediascanner

#endif // MEDIASCANNER_PROPERTY_H
